"""
Callbacks para dashboard de análisis de ventas de prescripción.

Refactored from single file to modular structure for maintainability.

Modules:
- data_loading: API calls and data fetching
- filters: Date range, slider, and filter synchronization
- overview: KPIs, context panel, ATC distribution
- contributors: Top contributors table and waterfall click-to-filter
- evolution: Evolution chart with drill-down
- products: Products accordion and category/ingredient rendering
- seasonality: Tab 3 heatmap, monthly index, KPIs (Issue #489)

Issue #436 PR #438: Original callbacks
Refactoring: Split into modules (~450 lines each vs 2700+ original)
"""

import logging

from .data_loading import register_data_loading_callbacks
from .filters import register_filter_callbacks
from .overview import register_overview_callbacks
from .contributors import register_contributors_callbacks
from .evolution import register_evolution_callbacks
from .products import register_products_callbacks
from .seasonality import register_seasonality_callbacks

logger = logging.getLogger(__name__)


def register_prescription_callbacks(app):
    """
    Registra todos los callbacks del dashboard de prescripción.

    Mantiene compatibilidad con el registro anterior.

    Args:
        app: Instancia Dash
    """
    register_data_loading_callbacks(app)
    register_overview_callbacks(app)
    register_contributors_callbacks(app)

    logger.info("[prescription_callbacks] Core callbacks registered (data, overview, contributors)")


def register_prescription_new_callbacks(app):
    """
    Registra callbacks adicionales del nuevo layout de prescripción.

    Mantiene compatibilidad con el registro anterior.

    Args:
        app: Instancia Dash
    """
    register_filter_callbacks(app)
    register_evolution_callbacks(app)
    register_products_callbacks(app)
    register_seasonality_callbacks(app)  # Issue #489: Tab 3 Seasonality

    logger.info("[prescription_callbacks] Extended callbacks registered (filters, evolution, products, seasonality)")


# Re-export for backwards compatibility
__all__ = [
    "register_prescription_callbacks",
    "register_prescription_new_callbacks",
    "register_seasonality_callbacks",  # Issue #489
]
