# frontend/callbacks/inventory/table.py
"""
Callbacks para tabla de productos por rotación (Issue #471).

Consume medida: abc_classification (top_a_products).
"""

import logging

import dash_bootstrap_components as dbc
from dash import Input, Output, State, callback, html
from dash.exceptions import PreventUpdate

from components.inventory.rotation_table import (
    create_empty_rotation_table,
    create_product_row,
    create_rotation_table_header,
)
from utils.auth_helpers import get_auth_headers_from_tokens
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_inventory_table_callbacks(app):
    """
    Registrar callbacks para tabla de rotación.
    """
    _register_table_callbacks_for_prefix(app, "prescription")
    _register_table_callbacks_for_prefix(app, "ventalibre")


def _register_table_callbacks_for_prefix(app, id_prefix: str):
    """
    Registrar callbacks de tabla para un prefijo específico.
    """

    @app.callback(
        Output(f"{id_prefix}-inv-rotation-table", "children"),
        [
            Input(f"{id_prefix}-inv-table-top", "n_clicks"),
            Input(f"{id_prefix}-inv-table-bottom", "n_clicks"),
            Input(f"{id_prefix}-inv-table-sort", "value"),
            Input(f"{id_prefix}-inv-table-abc-filter", "value"),
            Input(f"{id_prefix}-tabs" if id_prefix == "ventalibre" else "prescription-tabs", "active_tab"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Token restoration
            State(f"{id_prefix}-inv-product-type", "data"),
        ],
        prevent_initial_call=True,
    )
    def update_rotation_table(
        top_clicks, bottom_clicks, sort_by, abc_filter, active_tab, auth_state, auth_tokens, product_type
    ):
        """
        Actualizar tabla de productos por rotación.
        """
        from dash import ctx
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if active_tab != "tab-inventario":
            raise PreventUpdate

        # Determinar si Top o Bottom
        triggered = ctx.triggered_id
        show_top = True
        if triggered == f"{id_prefix}-inv-table-bottom":
            show_top = False

        logger.info(
            f"[INVENTORY-TABLE-{id_prefix}] Loading table: "
            f"top={show_top}, sort={sort_by}, abc={abc_filter}"
        )

        # Obtener pharmacy_id (está en auth_state["user"]["pharmacy_id"])
        user = auth_state.get("user") if auth_state else None
        pharmacy_id = user.get("pharmacy_id") if user else None
        if not pharmacy_id:
            return create_empty_rotation_table("Sin datos de farmacia")

        # REGLA #7.6: Restaurar tokens para multi-worker (Render)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        # Llamar a la API de ABC (que incluye top_a_products)
        # Usar GET /calculate/{measure_name} con query params
        try:
            params = {
                "pharmacy_id": str(pharmacy_id),
                "product_type": product_type,  # Issue #500: Filtrar por tipo producto
            }

            # Issue #533: Pasar filtro ABC al backend para filtrar ANTES del límite
            if abc_filter and abc_filter != "all":
                params["abc_class"] = abc_filter

            data = request_coordinator.make_request(
                "/api/v1/measures/calculate/abc_classification",
                method="GET",
                params=params,
                timeout=30,
                auth_headers=auth_headers,  # Pass explicit auth headers for multi-worker
            )

            if data is None:
                logger.error(f"[INVENTORY-TABLE-{id_prefix}] API returned None")
                return create_empty_rotation_table("Error cargando datos")

            result = data.get("value", {})

            # Obtener todos los productos con métricas
            products = result.get("top_a_products", [])

            if not products:
                return create_empty_rotation_table()

            # Filtrar por clase ABC si se seleccionó una específica
            if abc_filter and abc_filter != "all":
                products = [p for p in products if p.get("abc_class") == abc_filter]

            if not products:
                return create_empty_rotation_table(f"No hay productos en clase {abc_filter}")

            # Mapeo de sort_by a campo del producto
            sort_field_map = {
                "rotation": "rotation",
                "days": "days_inventory",
                "gmroi": "gmroi",
                "sales": "sales",
                "dead_days": "days_inventory",  # Mismo campo, diferente orden
            }
            sort_field = sort_field_map.get(sort_by, "rotation")

            # Ordenar productos según métrica seleccionada
            # Para "Top": mayor es mejor (excepto days_inventory donde menor es mejor)
            # Para "Bottom": invertir el orden
            def get_sort_key(product):
                value = product.get(sort_field)
                # Si el valor es None, ponerlo al final
                if value is None:
                    return float('-inf') if show_top else float('inf')
                return value

            # Determinar si ordenar ascendente o descendente
            # - rotation, gmroi, sales: mayor es mejor → Top = descendente
            # - days_inventory: menor es mejor → Top = ascendente
            if sort_by in ["days", "dead_days"]:
                # Menos días = mejor rotación
                reverse = not show_top  # Top = ascendente (menos días primero)
            else:
                # Mayor valor = mejor
                reverse = show_top  # Top = descendente (más valor primero)

            sorted_products = sorted(products, key=get_sort_key, reverse=reverse)

            # Crear filas de tabla
            rows = []
            for product in sorted_products[:10]:  # Limitar a 10
                rows.append(create_product_row(
                    product_name=product.get("product_name", "Producto"),
                    codigo_nacional=product.get("codigo_nacional", "--"),
                    rotation=product.get("rotation"),
                    days_inventory=product.get("days_inventory"),
                    gmroi=product.get("gmroi"),
                    sales=product.get("sales", 0),
                    abc_class=product.get("abc_class", "A"),
                ))

            # Crear tabla
            table = html.Table([
                create_rotation_table_header(),
                html.Tbody(rows),
            ], className="table table-hover table-sm")

            return table

        except Exception as e:
            logger.error(f"[INVENTORY-TABLE-{id_prefix}] Error: {e}")
            return create_empty_rotation_table("Error al cargar productos")
