# frontend/callbacks/inventory/kpis.py
"""
Callbacks para KPIs de inventario (Issue #471, #487).

Consumen medidas: stock_value, rotation_index, gmroi, stock_coverage_days, dead_stock_value.
"""

import logging

from dash import Input, Output, State, callback, ctx, no_update
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens
from utils.helpers import format_currency
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Issue #487: Mapeo de alert_level a clases Bootstrap para borde de card
ALERT_LEVEL_TO_BORDER_CLASS = {
    "danger": "h-100 shadow-sm border-danger border-2",
    "warning": "h-100 shadow-sm border-warning border-2",
    "success": "h-100 shadow-sm border-success border-2",
    "info": "h-100 shadow-sm border-info border-2",
    "secondary": "h-100 shadow-sm border-secondary",
}
DEFAULT_CARD_CLASS = "h-100 shadow-sm"


def register_inventory_kpis_callbacks(app):
    """
    Registrar callbacks para KPIs de inventario.

    Callbacks registrados para ambos prefijos (prescription, ventalibre).
    """
    # Registrar para prescription
    _register_kpi_callbacks_for_prefix(app, "prescription")

    # Registrar para ventalibre
    _register_kpi_callbacks_for_prefix(app, "ventalibre")


def _register_kpi_callbacks_for_prefix(app, id_prefix: str):
    """
    Registrar callbacks de KPIs para un prefijo específico.

    Args:
        app: Aplicación Dash
        id_prefix: Prefijo de IDs ("prescription" o "ventalibre")
    """

    @app.callback(
        [
            Output(f"{id_prefix}-inv-kpi-stock-value", "children"),
            Output(f"{id_prefix}-inv-kpi-stock-value-interpretation", "children"),
            Output(f"{id_prefix}-inv-kpi-rotation", "children"),
            Output(f"{id_prefix}-inv-kpi-rotation-interpretation", "children"),
            Output(f"{id_prefix}-inv-kpi-gmroi", "children"),
            Output(f"{id_prefix}-inv-kpi-gmroi-interpretation", "children"),
            Output(f"{id_prefix}-inv-kpi-coverage", "children"),  # Issue #487
            Output(f"{id_prefix}-inv-kpi-coverage-interpretation", "children"),  # Issue #487
            Output(f"{id_prefix}-inv-kpi-coverage-card", "className"),  # Issue #487: Borde dinámico
            Output(f"{id_prefix}-inv-kpi-dead-stock", "children"),
            Output(f"{id_prefix}-inv-kpi-dead-stock-interpretation", "children"),
            Output(f"{id_prefix}-inv-no-data-alert", "is_open"),
        ],
        [
            Input("url", "pathname"),
            Input(f"{id_prefix}-tabs", "active_tab") if id_prefix == "ventalibre" else Input("prescription-tabs", "active_tab"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Token restoration
            State(f"{id_prefix}-inv-product-type", "data"),
        ],
        prevent_initial_call=True,
    )
    def update_inventory_kpis(pathname, active_tab, auth_state, auth_tokens, product_type):
        """
        Actualizar KPIs de inventario cuando se activa la pestaña.

        Solo carga datos cuando:
        1. El usuario está autenticado
        2. La pestaña activa es "tab-inventario"
        """
        # Verificar autenticación (REGLA #302)
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug(f"[INVENTORY-KPIS-{id_prefix}] User not authenticated - skipping")
            raise PreventUpdate

        # Solo cargar si estamos en la pestaña de inventario
        if active_tab != "tab-inventario":
            raise PreventUpdate

        # Verificar pathname correcto
        expected_path = "/ventalibre" if id_prefix == "ventalibre" else "/prescription"
        if pathname != expected_path:
            raise PreventUpdate

        logger.info(f"[INVENTORY-KPIS-{id_prefix}] Loading inventory KPIs for {product_type}")

        # REGLA #7.6: Restaurar tokens para multi-worker (Render)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        # Obtener pharmacy_id (está en auth_state["user"]["pharmacy_id"])
        user = auth_state.get("user") if auth_state else None
        pharmacy_id = user.get("pharmacy_id") if user else None
        if not pharmacy_id:
            logger.warning(f"[INVENTORY-KPIS-{id_prefix}] No pharmacy_id - showing no data alert")
            return ("--", "", "--", "", "--", "", "--", "", DEFAULT_CARD_CLASS, "--", "", True)

        # Llamar a la API de medidas
        try:
            data = request_coordinator.make_request(
                "/api/v1/measures/calculate/multiple",
                method="POST",
                data={
                    "pharmacy_id": str(pharmacy_id),  # En raíz, no en filters
                    "measure_names": [
                        "stock_value",
                        "rotation_index",
                        "gmroi",
                        "stock_coverage_days",  # Issue #487: Días de cobertura
                        "dead_stock_value",
                    ],
                    "filters": {
                        "product_type": product_type,  # Issue #500: Filtrar por tipo producto
                    },
                },
                timeout=30,
                auth_headers=auth_headers,  # Pass explicit auth headers for multi-worker
            )

            if data is None:
                logger.error(f"[INVENTORY-KPIS-{id_prefix}] API returned None")
                return ("Error", "", "Error", "", "Error", "", "Error", "", DEFAULT_CARD_CLASS, "Error", "", True)

            results = data.get("results", {})

            # Extraer valores de cada medida
            stock_value = results.get("stock_value", 0)
            rotation = results.get("rotation_index", {})
            gmroi = results.get("gmroi", {})
            coverage = results.get("stock_coverage_days", {})  # Issue #487
            dead_stock = results.get("dead_stock_value", {})

            # Formatear valores
            stock_value_str = format_currency(stock_value) if isinstance(stock_value, (int, float)) else format_currency(0)

            rotation_value = rotation.get("rotation_index", 0) if isinstance(rotation, dict) else rotation
            rotation_str = f"{rotation_value:.1f}x" if rotation_value else "--"
            rotation_interp = rotation.get("interpretation", "") if isinstance(rotation, dict) else ""

            gmroi_value = gmroi.get("gmroi", 0) if isinstance(gmroi, dict) else gmroi
            gmroi_str = f"{gmroi_value:.2f}" if gmroi_value else "--"
            gmroi_interp = gmroi.get("interpretation", "") if isinstance(gmroi, dict) else ""

            # Issue #487: Días de cobertura
            coverage_days = coverage.get("coverage_days", 0) if isinstance(coverage, dict) else coverage
            if coverage_days is None:
                coverage_str = "∞"  # Sin ventas = cobertura indefinida
            elif coverage_days:
                coverage_str = f"{coverage_days:.0f}d"
            else:
                coverage_str = "--"
            coverage_interp = coverage.get("interpretation", "") if isinstance(coverage, dict) else ""
            # Issue #487: Clase de borde dinámica según alert_level
            alert_level = coverage.get("alert_level", "") if isinstance(coverage, dict) else ""
            coverage_card_class = ALERT_LEVEL_TO_BORDER_CLASS.get(alert_level, DEFAULT_CARD_CLASS)

            dead_value = dead_stock.get("value", 0) if isinstance(dead_stock, dict) else dead_stock
            dead_str = format_currency(dead_value) if dead_value else "€0"
            dead_products = dead_stock.get("products_count", 0) if isinstance(dead_stock, dict) else 0
            dead_interp = f"{dead_products} productos" if dead_products else ""

            logger.info(f"[INVENTORY-KPIS-{id_prefix}] Loaded: stock={stock_value_str}, rotation={rotation_str}, coverage={coverage_str}")

            return (
                stock_value_str,
                "",  # No interpretation for stock value
                rotation_str,
                rotation_interp,
                gmroi_str,
                gmroi_interp,
                coverage_str,  # Issue #487
                coverage_interp,  # Issue #487
                coverage_card_class,  # Issue #487: Borde dinámico
                dead_str,
                dead_interp,
                False,  # Hide no-data alert
            )

        except Exception as e:
            logger.error(f"[INVENTORY-KPIS-{id_prefix}] Error loading KPIs: {e}")
            return ("--", "", "--", "", "--", "", "--", "", DEFAULT_CARD_CLASS, "--", "", True)
