# frontend/callbacks/inventory/coverage.py
"""
Callbacks para el gráfico de distribución de cobertura de stock (Issue #496).

Muestra un donut chart con la distribución de productos por bandas de cobertura:
- Crítico (<7 días)
- Bajo (7-14 días)
- Saludable (14-30 días)
- Holgado (30-60 días)
- Sobrestock (>60 días)
"""

import logging

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import Input, Output, State, dcc, html
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Bandas de cobertura y sus configuraciones
COVERAGE_BANDS = [
    {"id": "critical", "label": "Crítico (<7d)", "min": 0, "max": 7, "color": "#dc3545"},
    {"id": "low", "label": "Bajo (7-14d)", "min": 7, "max": 14, "color": "#fd7e14"},
    {"id": "healthy", "label": "Saludable (14-30d)", "min": 14, "max": 30, "color": "#198754"},
    {"id": "comfortable", "label": "Holgado (30-60d)", "min": 30, "max": 60, "color": "#0dcaf0"},
    {"id": "overstock", "label": "Sobrestock (>60d)", "min": 60, "max": float("inf"), "color": "#6c757d"},
    {"id": "unknown", "label": "Sin ventas", "min": None, "max": None, "color": "#adb5bd"},
]


def register_inventory_coverage_callbacks(app):
    """
    Registrar callbacks para gráfico de cobertura de stock.
    """
    _register_coverage_callbacks_for_prefix(app, "prescription")
    _register_coverage_callbacks_for_prefix(app, "ventalibre")


def _register_coverage_callbacks_for_prefix(app, id_prefix: str):
    """
    Registrar callbacks de cobertura para un prefijo específico.
    """

    @app.callback(
        Output(f"{id_prefix}-inv-coverage-chart", "children"),
        [
            Input("url", "pathname"),
            Input(f"{id_prefix}-tabs" if id_prefix == "ventalibre" else "prescription-tabs", "active_tab"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def update_coverage_chart(pathname, active_tab, auth_state, auth_tokens):
        """
        Actualizar gráfico de distribución de cobertura.
        """
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if active_tab != "tab-inventario":
            raise PreventUpdate

        expected_path = "/ventalibre" if id_prefix == "ventalibre" else "/prescription"
        if pathname != expected_path:
            raise PreventUpdate

        logger.info(f"[INVENTORY-COVERAGE-{id_prefix}] Loading coverage distribution")

        # REGLA #7.6: Restaurar tokens para multi-worker
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        user = auth_state.get("user") if auth_state else None
        pharmacy_id = user.get("pharmacy_id") if user else None
        if not pharmacy_id:
            return _create_empty_chart("Sin datos de farmacia")

        # Determinar product_type según la página
        product_type = "prescription" if id_prefix == "prescription" else "venta_libre"

        try:
            # Usar endpoint de stock_alerts que devuelve coverage_days por producto
            data = request_coordinator.make_request(
                "/api/v1/measures/calculate/stock_alerts",
                method="GET",
                params={
                    "pharmacy_id": str(pharmacy_id),
                    "product_type": product_type,
                },
                timeout=30,
                auth_headers=auth_headers,  # Pass explicit auth headers for multi-worker
            )

            if data is None:
                logger.error(f"[INVENTORY-COVERAGE-{id_prefix}] API returned None")
                return _create_empty_chart("Error cargando datos")

            result = data.get("value", {})

            # Issue #535: Usar coverage_distribution para TODOS los productos
            # (no solo alertas)
            coverage_dist = result.get("coverage_distribution", {})

            if coverage_dist:
                # Usar distribución completa desde backend
                bands_data = coverage_dist.get("bands", {})
                total = coverage_dist.get("total_products", 0)

                # Convertir a formato band_counts esperado
                band_counts = {
                    band_id: band_info.get("count", 0)
                    for band_id, band_info in bands_data.items()
                }
                # Mapear "healthy" y "comfortable" a nuestros IDs
                # (backend usa mismos nombres)
            else:
                # Fallback: usar alertas como antes (compatibilidad)
                alerts = result.get("alerts", [])
                band_counts = {band["id"]: 0 for band in COVERAGE_BANDS}

                for alert in alerts:
                    coverage_days = alert.get("coverage_days")
                    band_id = _get_coverage_band(coverage_days)
                    band_counts[band_id] += 1

                total = sum(band_counts.values())

            # Si no hay datos, mostrar mensaje
            if total == 0:
                logger.info(f"[INVENTORY-COVERAGE-{id_prefix}] No products with coverage data")
                return _create_empty_chart("Sin datos de cobertura")

            # Crear gráfico donut
            fig = _create_coverage_donut(band_counts, total)

            logger.info(
                f"[INVENTORY-COVERAGE-{id_prefix}] Loaded distribution: {band_counts} (total: {total})"
            )

            # Issue #535: Mostrar "productos" en lugar de "productos con alertas"
            return html.Div([
                dcc.Graph(
                    figure=fig,
                    config={"displayModeBar": False, "responsive": True},
                    style={"height": "280px"},
                ),
                # Leyenda con contexto y tooltip explicativo
                html.Div([
                    html.Small([
                        html.I(className="fas fa-info-circle me-1 text-muted"),
                        f"Distribución de {total} productos por cobertura",
                    ], className="text-muted", title="Días de cobertura = stock / ventas diarias promedio (90 días)"),
                ], className="text-center mt-2"),
            ])

        except Exception as e:
            logger.error(f"[INVENTORY-COVERAGE-{id_prefix}] Error: {e}")
            return _create_empty_chart("Error al cargar datos")


def _get_coverage_band(coverage_days) -> str:
    """
    Determinar la banda de cobertura para un valor de días.

    Args:
        coverage_days: Días de cobertura (None si sin ventas)

    Returns:
        ID de la banda correspondiente
    """
    if coverage_days is None:
        return "unknown"

    for band in COVERAGE_BANDS:
        if band["min"] is None:
            continue
        if band["min"] <= coverage_days < band["max"]:
            return band["id"]

    # Fallback (no debería llegar aquí)
    return "overstock"


def _create_coverage_donut(band_counts: dict, total: int) -> go.Figure:
    """
    Crear gráfico donut de distribución de cobertura.

    Args:
        band_counts: Diccionario {band_id: count}
        total: Total de productos

    Returns:
        Figura de Plotly
    """
    # Filtrar bandas con valores > 0
    labels = []
    values = []
    colors = []

    for band in COVERAGE_BANDS:
        count = band_counts.get(band["id"], 0)
        if count > 0:
            labels.append(band["label"])
            values.append(count)
            colors.append(band["color"])

    fig = go.Figure(data=[
        go.Pie(
            labels=labels,
            values=values,
            hole=0.5,
            marker_colors=colors,
            textinfo="percent",
            textposition="outside",
            textfont_size=11,
            hovertemplate="<b>%{label}</b><br>%{value} productos<br>%{percent}<extra></extra>",
            sort=False,  # Mantener orden de bandas
        )
    ])

    fig.update_layout(
        showlegend=True,
        legend=dict(
            orientation="h",
            yanchor="bottom",
            y=-0.2,
            xanchor="center",
            x=0.5,
            font=dict(size=10),
        ),
        margin=dict(t=20, b=60, l=20, r=20),
        annotations=[
            dict(
                text=f"<b>{total}</b><br>productos",
                x=0.5, y=0.5,
                font_size=14,
                showarrow=False,
            )
        ],
    )

    return fig


def _create_empty_chart(message: str = "Sin datos") -> html.Div:
    """
    Crear estado vacío para el gráfico de cobertura.
    """
    return html.Div([
        html.I(className="fas fa-chart-pie fa-3x text-muted mb-3"),
        html.P(message, className="text-muted small mb-0"),
        html.Small(
            "Sube un fichero de inventario para ver la distribución",
            className="text-muted",
        ),
    ], className="text-center py-4")
