# frontend/callbacks/inventory/alerts.py
"""
Callbacks para alertas de stock (Issue #470).

Consume medida: stock_alerts.
"""

import logging

import dash_bootstrap_components as dbc
from dash import Input, Output, State, html
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens
from utils.helpers import format_currency
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_inventory_alerts_callbacks(app):
    """
    Registrar callbacks para alertas de stock.
    """
    _register_alerts_callbacks_for_prefix(app, "prescription")
    _register_alerts_callbacks_for_prefix(app, "ventalibre")


def _register_alerts_callbacks_for_prefix(app, id_prefix: str):
    """
    Registrar callbacks de alertas para un prefijo específico.
    """

    @app.callback(
        [
            Output(f"{id_prefix}-inv-alerts-count", "children"),
            Output(f"{id_prefix}-inv-alerts-list", "children"),
        ],
        [
            Input("url", "pathname"),
            Input(f"{id_prefix}-tabs" if id_prefix == "ventalibre" else "prescription-tabs", "active_tab"),
            Input(f"{id_prefix}-inv-alerts-filter", "value"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def update_stock_alerts(pathname, active_tab, alert_filter, auth_state, auth_tokens):
        """
        Actualizar lista de alertas de stock.
        """
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if active_tab != "tab-inventario":
            raise PreventUpdate

        expected_path = "/ventalibre" if id_prefix == "ventalibre" else "/prescription"
        if pathname != expected_path:
            raise PreventUpdate

        logger.info(f"[INVENTORY-ALERTS-{id_prefix}] Loading stock alerts")

        # REGLA #7.6: Restaurar tokens para multi-worker
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        user = auth_state.get("user") if auth_state else None
        pharmacy_id = user.get("pharmacy_id") if user else None
        if not pharmacy_id:
            return "0", _create_empty_alerts("Sin datos de farmacia")

        # Determinar product_type según la página
        product_type = "prescription" if id_prefix == "prescription" else "venta_libre"

        try:
            data = request_coordinator.make_request(
                "/api/v1/measures/calculate/stock_alerts",
                method="GET",
                params={
                    "pharmacy_id": str(pharmacy_id),
                    "product_type": product_type,
                },
                timeout=30,
                auth_headers=auth_headers,  # Issue #542: Pass explicit auth headers for multi-worker
            )

            if data is None:
                logger.error(f"[INVENTORY-ALERTS-{id_prefix}] API returned None")
                return "0", _create_empty_alerts("Error cargando alertas")

            result = data.get("value", {})
            alerts = result.get("alerts", [])
            summary = result.get("summary", {})

            # Filtrar por tipo de alerta si se especifica
            if alert_filter and alert_filter != "all":
                alerts = [
                    a for a in alerts
                    if a.get("primary_alert") == alert_filter
                ]

            total_alerts = len(alerts)

            if not alerts:
                filter_labels = {
                    "dead_stock": "stock muerto",
                    "low_stock": "stock bajo",
                    "overstock": "sobrestock",
                }
                filter_label = filter_labels.get(alert_filter, "")
                message = f"Sin alertas de {filter_label}" if filter_label else "Sin alertas de stock"
                return "0", _create_empty_alerts(message)

            # Crear lista de alertas
            alert_items = []
            for alert in alerts[:20]:  # Limitar a 20 en UI
                alert_items.append(_create_alert_item(alert))

            # Mostrar resumen si hay más de 20
            if total_alerts > 20:
                alert_items.append(
                    html.Div(
                        f"... y {total_alerts - 20} alertas más",
                        className="text-muted text-center small py-2"
                    )
                )

            logger.info(
                f"[INVENTORY-ALERTS-{id_prefix}] Loaded {total_alerts} alerts "
                f"(filter={alert_filter}): "
                f"dead={summary.get('dead_stock', 0)}, "
                f"low={summary.get('low_stock', 0)}, "
                f"over={summary.get('overstock', 0)}"
            )

            return str(total_alerts), html.Div(alert_items)

        except Exception as e:
            logger.error(f"[INVENTORY-ALERTS-{id_prefix}] Error: {e}")
            return "0", _create_empty_alerts("Error al cargar alertas")


def _create_alert_item(alert: dict) -> dbc.Card:
    """
    Crear item de alerta individual.
    """
    product_name = alert.get("product_name", "Producto")
    product_code = alert.get("product_code", "--")
    stock_qty = alert.get("stock_quantity", 0)
    stock_value = alert.get("stock_value", 0)
    primary_severity = alert.get("primary_severity", "secondary")
    alerts_list = alert.get("alerts", [])

    # Color según severidad
    border_color = {
        "danger": "border-danger",
        "warning": "border-warning",
        "info": "border-info",
    }.get(primary_severity, "border-secondary")

    # Icono según tipo de alerta
    alert_icons = {
        "dead_stock": "fas fa-skull",
        "low_stock": "fas fa-exclamation-triangle",
        "overstock": "fas fa-boxes",
    }

    # Crear badges de alertas
    alert_badges = []
    for a in alerts_list:
        alert_type = a.get("type", "unknown")
        icon = alert_icons.get(alert_type, "fas fa-info-circle")
        color = {
            "danger": "danger",
            "warning": "warning",
            "info": "info",
        }.get(a.get("severity", "secondary"), "secondary")

        alert_badges.append(
            dbc.Badge(
                [html.I(className=f"{icon} me-1"), a.get("message", "")],
                color=color,
                className="me-1 mb-1",
            )
        )

    return dbc.Card(
        dbc.CardBody([
            # Nombre y código
            html.Div([
                html.Span(
                    product_name[:35] + "..." if len(product_name) > 35 else product_name,
                    className="fw-medium",
                ),
            ]),
            html.Small(product_code, className="text-muted"),
            # Stock info
            html.Div([
                html.Small(
                    f"{stock_qty} uds · {format_currency(stock_value)}",
                    className="text-muted",
                ),
            ], className="mt-1"),
            # Alert badges
            html.Div(alert_badges, className="mt-2"),
        ], className="p-2"),
        className=f"mb-2 {border_color} border-start border-3",
    )


def _create_empty_alerts(message: str = "Sin alertas") -> html.Div:
    """
    Crear estado vacío para alertas.
    """
    return html.Div([
        html.I(className="fas fa-check-circle fa-2x text-success mb-2"),
        html.P(message, className="text-muted small mb-0"),
    ], className="text-center py-4")
