# frontend/callbacks/inventory/abc.py
"""
Callbacks para gráfico ABC Classification (Issue #471).

Consume medida: abc_classification del backend.
"""

import logging

import plotly.express as px
import plotly.graph_objects as go
from dash import Input, Output, State, callback, html, dcc
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens
from utils.helpers import format_currency
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_inventory_abc_callbacks(app):
    """
    Registrar callbacks para clasificación ABC.
    """
    _register_abc_callbacks_for_prefix(app, "prescription")
    _register_abc_callbacks_for_prefix(app, "ventalibre")


def _register_abc_callbacks_for_prefix(app, id_prefix: str):
    """
    Registrar callbacks de ABC para un prefijo específico.
    """

    @app.callback(
        [
            Output(f"{id_prefix}-inv-abc-chart", "children"),
            Output(f"{id_prefix}-inv-abc-summary-a", "children"),
            Output(f"{id_prefix}-inv-abc-summary-b", "children"),
            Output(f"{id_prefix}-inv-abc-summary-c", "children"),
        ],
        [
            Input(f"{id_prefix}-inv-abc-view-bar", "n_clicks"),
            Input(f"{id_prefix}-inv-abc-view-treemap", "n_clicks"),
            Input(f"{id_prefix}-tabs" if id_prefix == "ventalibre" else "prescription-tabs", "active_tab"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Token restoration
            State(f"{id_prefix}-inv-product-type", "data"),
        ],
        prevent_initial_call=True,
    )
    def update_abc_chart(bar_clicks, treemap_clicks, active_tab, auth_state, auth_tokens, product_type):
        """
        Actualizar gráfico ABC cuando se cambia la vista o se activa la pestaña.
        """
        from dash import ctx
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if active_tab != "tab-inventario":
            raise PreventUpdate

        # Determinar tipo de vista
        triggered = ctx.triggered_id
        view_type = "bar"  # Default
        if triggered == f"{id_prefix}-inv-abc-view-treemap":
            view_type = "treemap"

        logger.info(f"[INVENTORY-ABC-{id_prefix}] Loading ABC chart ({view_type})")

        # Obtener pharmacy_id (está en auth_state["user"]["pharmacy_id"])
        user = auth_state.get("user") if auth_state else None
        pharmacy_id = user.get("pharmacy_id") if user else None
        if not pharmacy_id:
            return (
                _create_empty_chart("Sin datos de farmacia"),
                "--", "--", "--"
            )

        # REGLA #7.6: Restaurar tokens para multi-worker (Render)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        # Llamar a la API - Usar GET /calculate/{measure_name} con query params
        try:
            params = {
                "pharmacy_id": str(pharmacy_id),
                "product_type": product_type,  # Issue #500: Filtrar por tipo producto
            }

            data = request_coordinator.make_request(
                "/api/v1/measures/calculate/abc_classification",
                method="GET",
                params=params,
                timeout=30,
                auth_headers=auth_headers,  # Issue #542: Pass explicit auth headers for multi-worker
            )

            if data is None:
                logger.error(f"[INVENTORY-ABC-{id_prefix}] API returned None")
                return (
                    _create_empty_chart("Error cargando datos"),
                    "--", "--", "--"
                )

            result = data.get("value", {})

            # Extraer datos de clases
            class_a = result.get("class_a", {})
            class_b = result.get("class_b", {})
            class_c = result.get("class_c", {})

            # Crear gráfico
            if view_type == "treemap":
                chart = _create_abc_treemap(class_a, class_b, class_c)
            else:
                chart = _create_abc_bar_chart(class_a, class_b, class_c)

            # Crear resúmenes
            summary_a = _format_class_summary(class_a)
            summary_b = _format_class_summary(class_b)
            summary_c = _format_class_summary(class_c)

            return (chart, summary_a, summary_b, summary_c)

        except Exception as e:
            logger.error(f"[INVENTORY-ABC-{id_prefix}] Error: {e}")
            return (
                _create_empty_chart("Error al cargar clasificación ABC"),
                "--", "--", "--"
            )


def _create_abc_bar_chart(class_a: dict, class_b: dict, class_c: dict):
    """Crear gráfico de barras horizontales para ABC."""
    classes = ["Clase A", "Clase B", "Clase C"]
    products = [
        class_a.get("products", 0),
        class_b.get("products", 0),
        class_c.get("products", 0),
    ]
    sales_pct = [
        class_a.get("sales_pct", 0),
        class_b.get("sales_pct", 0),
        class_c.get("sales_pct", 0),
    ]
    colors = ["#28a745", "#ffc107", "#dc3545"]  # success, warning, danger

    fig = go.Figure()

    # Barras de productos
    fig.add_trace(go.Bar(
        name="Productos",
        y=classes,
        x=products,
        orientation="h",
        marker_color=colors,
        text=[f"{p} prods" for p in products],
        textposition="inside",
    ))

    fig.update_layout(
        title="Distribución ABC",
        xaxis_title="Número de Productos",
        yaxis_title="",
        height=300,
        margin=dict(t=40, b=40, l=80, r=20),
        showlegend=False,
    )

    return dcc.Graph(figure=fig, config={"displayModeBar": False})


def _create_abc_treemap(class_a: dict, class_b: dict, class_c: dict):
    """Crear treemap para ABC."""
    labels = ["Clase A", "Clase B", "Clase C"]
    parents = ["", "", ""]
    values = [
        class_a.get("sales", 0) or 1,  # Avoid 0 for treemap
        class_b.get("sales", 0) or 1,
        class_c.get("sales", 0) or 1,
    ]
    colors = ["#28a745", "#ffc107", "#dc3545"]

    fig = go.Figure(go.Treemap(
        labels=labels,
        parents=parents,
        values=values,
        marker_colors=colors,
        textinfo="label+value",
        texttemplate="%{label}<br>€%{value:,.0f}",
    ))

    fig.update_layout(
        title="Ventas por Clasificación ABC",
        height=300,
        margin=dict(t=40, b=20, l=20, r=20),
    )

    return dcc.Graph(figure=fig, config={"displayModeBar": False})


def _create_empty_chart(message: str):
    """Crear estado vacío para el gráfico."""
    return html.Div([
        html.I(className="fas fa-chart-pie fa-3x text-muted mb-3"),
        html.P(message, className="text-muted"),
    ], className="text-center py-5")


def _format_class_summary(class_data: dict) -> str:
    """Formatear resumen de una clase ABC."""
    products = class_data.get("products", 0)
    products_pct = class_data.get("products_pct", 0)
    sales = class_data.get("sales", 0)
    sales_pct = class_data.get("sales_pct", 0)

    return html.Div([
        html.Div(f"{products} prods ({products_pct:.0f}%)", className="small"),
        html.Div(f"€{sales:,.0f} ({sales_pct:.0f}%)", className="small text-muted"),
    ])
