# frontend/callbacks/insights/feedback.py
"""
Insight Engine Feedback Callbacks - Issue #506

Callbacks para gestionar feedback de insights (snooze/dismiss/resolve).
"""

import structlog
from dash import Input, Output, State, callback, ctx, MATCH, ALL, no_update
from dash.exceptions import PreventUpdate

from components.toast_manager import success_toast, error_toast
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.config import BACKEND_URL
from utils.request_coordinator import request_coordinator

logger = structlog.get_logger(__name__)


def register_insights_feedback_callbacks(app):
    """Registra callbacks de feedback de insights."""

    @app.callback(
        Output("toast-container", "children", allow_duplicate=True),
        Output("ventalibre-insights-refresh-btn", "n_clicks"),
        Input({"type": "insight-snooze-btn", "index": ALL, "hash": ALL, "rule": ALL}, "n_clicks"),
        State("auth-state", "data"),
        State("auth-tokens-store", "data"),
        State("ventalibre-insights-refresh-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def handle_snooze(snooze_clicks, auth_state, auth_tokens, current_refresh_clicks):
        """Maneja el botón de posponer (snooze)."""
        if not ctx.triggered or not any(snooze_clicks):
            raise PreventUpdate

        # Find which button was clicked
        trigger = ctx.triggered_id
        if not trigger or not isinstance(trigger, dict):
            raise PreventUpdate

        insight_hash = trigger.get("hash", "")
        rule_code = trigger.get("rule", "")

        if not insight_hash:
            raise PreventUpdate

        # Submit feedback
        success, message = _submit_feedback(
            auth_state=auth_state,
            auth_tokens=auth_tokens,
            insight_hash=insight_hash,
            action="snooze",
            rule_code=rule_code,
        )

        # Trigger refresh and show toast
        new_refresh = (current_refresh_clicks or 0) + 1

        if success:
            return (
                success_toast("Insight pospuesto por 7 días"),
                new_refresh,
            )
        else:
            return (
                error_toast(f"Error: {message}"),
                no_update,
            )

    @app.callback(
        Output("toast-container", "children", allow_duplicate=True),
        Output("ventalibre-insights-refresh-btn", "n_clicks", allow_duplicate=True),
        Input({"type": "insight-dismiss-btn", "index": ALL, "hash": ALL, "rule": ALL}, "n_clicks"),
        State("auth-state", "data"),
        State("auth-tokens-store", "data"),
        State("ventalibre-insights-refresh-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def handle_dismiss(dismiss_clicks, auth_state, auth_tokens, current_refresh_clicks):
        """Maneja el botón de descartar (dismiss)."""
        if not ctx.triggered or not any(dismiss_clicks):
            raise PreventUpdate

        # Find which button was clicked
        trigger = ctx.triggered_id
        if not trigger or not isinstance(trigger, dict):
            raise PreventUpdate

        insight_hash = trigger.get("hash", "")
        rule_code = trigger.get("rule", "")

        if not insight_hash:
            raise PreventUpdate

        # Submit feedback
        success, message = _submit_feedback(
            auth_state=auth_state,
            auth_tokens=auth_tokens,
            insight_hash=insight_hash,
            action="dismiss",
            rule_code=rule_code,
        )

        # Trigger refresh and show toast
        new_refresh = (current_refresh_clicks or 0) + 1

        if success:
            return (
                success_toast("Insight descartado"),
                new_refresh,
            )
        else:
            return (
                error_toast(f"Error: {message}"),
                no_update,
            )

    @app.callback(
        Output("url", "pathname", allow_duplicate=True),
        Input({"type": "insight-action-btn", "index": ALL}, "n_clicks"),
        State({"type": "insight-deeplink", "index": ALL}, "children"),
        prevent_initial_call=True,
    )
    def handle_action_click(action_clicks, deeplinks):
        """Maneja el click en el botón de acción principal (navegación)."""
        if not ctx.triggered or not any(action_clicks):
            raise PreventUpdate

        # Find which button was clicked
        trigger = ctx.triggered_id
        if not trigger or not isinstance(trigger, dict):
            raise PreventUpdate

        index = trigger.get("index", 0)

        # Get corresponding deeplink
        if deeplinks and len(deeplinks) > index:
            deeplink = deeplinks[index]
            if deeplink and isinstance(deeplink, str) and deeplink.startswith("/"):
                logger.info("insights.navigate", deeplink=deeplink)
                return deeplink

        raise PreventUpdate


def _submit_feedback(
    auth_state: dict,
    auth_tokens: dict,
    insight_hash: str,
    action: str,
    rule_code: str = None,
) -> tuple[bool, str]:
    """
    Envía feedback al backend.

    Returns:
        Tuple (success: bool, message: str)
    """
    # Auth check
    if not is_user_authenticated(auth_state):
        return False, "No autenticado"

    # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
    auth_headers = get_auth_headers_from_tokens(auth_tokens)
    if not auth_headers:
        logger.warning("insights.feedback_no_auth_headers")
        return False, "Error de autenticación"

    try:
        url = f"{BACKEND_URL}/api/v1/insights/feedback"
        payload = {
            "insight_hash": insight_hash,
            "action": action,
            "rule_code": rule_code,
        }

        response = request_coordinator.make_request(
            "POST",
            url,
            json=payload,
            timeout=15,
            auth_headers=auth_headers,
        )

        if response and response.get("success"):
            logger.info(
                "insights.feedback_submitted",
                insight_hash=insight_hash,
                action=action,
            )
            return True, "OK"
        else:
            message = response.get("message", "Error desconocido") if response else "Sin respuesta"
            logger.warning(
                "insights.feedback_failed",
                insight_hash=insight_hash,
                action=action,
                message=message,
            )
            return False, message

    except Exception as e:
        logger.error("insights.feedback_error", error=str(e), exc_info=True)
        return False, str(e)
