# frontend/callbacks/insights/data_loading.py
"""
Insight Engine Data Loading Callbacks - Issue #506

Callbacks para cargar insights desde el backend.

Patrón:
1. Trigger inicial o refresh -> API call
2. Guardar en store
3. Renderizar cards
"""

import structlog
from dash import Input, Output, State, callback, ctx, no_update
from dash.exceptions import PreventUpdate

from components.insights import create_empty_state, create_error_state, render_insights_list
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.config import BACKEND_URL
from utils.helpers import format_currency
from utils.request_coordinator import request_coordinator

logger = structlog.get_logger(__name__)


def register_insights_data_callbacks(app):
    """Registra callbacks de carga de datos de insights."""

    @app.callback(
        Output("ventalibre-insights-data-store", "data"),
        Output("ventalibre-insights-container", "children"),
        Output("ventalibre-insights-total-value", "children"),
        Output("ventalibre-insights-count-high", "children"),
        Output("ventalibre-insights-count-medium", "children"),
        Output("ventalibre-insights-count-low", "children"),
        Output("ventalibre-insights-suppressed-count", "children"),
        Output("ventalibre-insights-refresh-interval", "disabled"),
        Input("ventalibre-insights-refresh-btn", "n_clicks"),
        Input("ventalibre-insights-refresh-interval", "n_intervals"),
        Input("url", "pathname"),
        State("auth-state", "data"),
        State("auth-tokens-store", "data"),
        State("ventalibre-insights-filter-store", "data"),
        prevent_initial_call=False,
    )
    def load_insights(
        n_clicks,
        n_intervals,
        pathname,
        auth_state,
        auth_tokens,
        filter_store,
    ):
        """Carga insights desde el backend."""
        # Guard: Solo en página venta libre
        if pathname and "/ventalibre" not in pathname:
            raise PreventUpdate

        # Guard: Auth check (REGLA #7.6)
        if not is_user_authenticated(auth_state):
            logger.debug("insights.load_skipped", reason="not_authenticated")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("insights.no_auth_headers")
            raise PreventUpdate

        # Get pharmacy_id from auth_state
        pharmacy_id = auth_state.get("pharmacy_id") if auth_state else None
        if not pharmacy_id:
            logger.warning("insights.no_pharmacy_id")
            return (
                None,
                create_error_state("No se encontró farmacia asociada"),
                "--",
                "0", "0", "0",
                "",
                True,  # Disable interval
            )

        # Apply filters if any
        category = filter_store.get("category") if filter_store else None
        severity = filter_store.get("severity") if filter_store else None

        try:
            # Build URL with query params
            url = f"{BACKEND_URL}/api/v1/insights/{pharmacy_id}"
            params = {}
            if category:
                params["category"] = category
            if severity:
                params["severity"] = severity

            # Make API request
            response = request_coordinator.make_request(
                "GET",
                url,
                params=params if params else None,
                timeout=30,
                auth_headers=auth_headers,
            )

            if response and response.get("insights") is not None:
                insights = response.get("insights", [])
                total_opportunity = response.get("total_opportunity", 0)
                by_severity = response.get("by_severity", {})
                suppressed = response.get("suppressed_count", 0)

                logger.info(
                    "insights.loaded",
                    pharmacy_id=pharmacy_id,
                    insights_count=len(insights),
                    total_opportunity=total_opportunity,
                )

                # Render insights
                content = render_insights_list(insights, id_prefix="ventalibre")

                # Format total
                total_text = format_currency(total_opportunity) if total_opportunity else "0€"

                # Format suppressed
                suppressed_text = f"({suppressed} ocultos)" if suppressed > 0 else ""

                return (
                    response,  # Store data
                    content,
                    total_text,
                    str(by_severity.get("high", 0)),
                    str(by_severity.get("medium", 0)),
                    str(by_severity.get("low", 0)),
                    suppressed_text,
                    False,  # Enable interval
                )

            else:
                logger.warning("insights.empty_response", pharmacy_id=pharmacy_id)
                return (
                    None,
                    create_empty_state(),
                    "0€",
                    "0", "0", "0",
                    "",
                    False,
                )

        except Exception as e:
            logger.error("insights.load_error", error=str(e), exc_info=True)
            return (
                None,
                create_error_state(str(e)),
                "--",
                "0", "0", "0",
                "",
                True,  # Disable interval on error
            )

    @app.callback(
        Output("ventalibre-insights-filter-store", "data"),
        Input({"type": "ventalibre-insight-filter", "value": "all"}, "n_clicks"),
        Input({"type": "ventalibre-insight-filter", "value": "stock"}, "n_clicks"),
        Input({"type": "ventalibre-insight-filter", "value": "margin"}, "n_clicks"),
        Input({"type": "ventalibre-insight-filter", "value": "hhi"}, "n_clicks"),
        Input({"type": "ventalibre-insight-filter", "value": "trend"}, "n_clicks"),
        Input({"type": "ventalibre-insight-filter", "value": "surtido"}, "n_clicks"),
        prevent_initial_call=True,
    )
    def update_filter(*args):
        """Actualiza el filtro de categoría."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered_id
        if not trigger_id or not isinstance(trigger_id, dict):
            raise PreventUpdate

        category_value = trigger_id.get("value", "all")

        if category_value == "all":
            return {"category": None, "severity": None}
        else:
            return {"category": category_value, "severity": None}
