"""
Partners callbacks for Generics Panel.

Handles:
- Partners dropdown loading (load_partners_dropdown)
- Unified partners controller (selection, persistence, cache)
- Partners count initialization (BD-FIRST pattern)
"""

import logging

import requests
import requests.exceptions
from dash import Input, Output, State, ctx, no_update
from dash.exceptions import PreventUpdate

from utils.api_client import api_client
from utils.debug_flags import CACHE_DEBUG_ENABLED
from utils.helpers import format_currency, normalize_utf8_string, validate_session_storage_size, get_storage_recommendations
from utils.generics_helpers import (
    get_codes_with_cache,
    get_selected_partners_from_db,
    persist_partners_selection,
    simplify_lab_name,
)
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)


def register_partners_callbacks(app):
    """
    Register partners callbacks for the Generics Panel.

    Args:
        app: Dash application instance
    """

    # ============================================================================
    # 3a. CALLBACK CARGAR LISTA DE PARTNERS → Store Intermedio (REGLA #11)
    # ============================================================================

    @app.callback(
        [
            Output("partners-load-result-store", "data"),
            Output("laboratory-cache-store", "data"),  # HOTFIX: Actualizar cache persistente
        ],
        [
            Input("refresh-partners-list-btn", "n_clicks"),
            Input("url", "pathname"),
            Input("auth-ready", "data"),  # FIX RACE CONDITION: Volver a Input PERO con guards estrictos
        ],
        [
            State("partners-selection-store", "data"),
            State("auth-state", "data"),
            State("laboratory-cache-store", "data"),  # HOTFIX: Cache actual
        ],
        prevent_initial_call=False,
    )
    def load_partners_dropdown(refresh_clicks, pathname, auth_ready, current_store, auth_state, laboratory_cache_store):
        """
        Cargar dropdown de partners con persistencia de selección.
        Inicializa automáticamente si no existen partners.
        Retorna resultado a store intermedio.

        FIX DEPENDENCY CYCLE: Solo se ejecuta si estamos en /generics

        FIX RACE CONDITION (Partners Borrados Reaparecen):
        - Input("context-store", "data") REMOVIDO para prevenir race condition
        - Problema: context-store update disparaba reload ANTES de que persist complete commit
        - Solución: Partners solo recargan en: navegación a /generics, auth ready, o refresh manual
        - Partners persisten en BD inmediatamente, sin reload innecesario

        FIX REFRESH BUTTON (Restablecer 8 partners por defecto):
        - Refresh button ahora llama POST /initialize?force_refresh=true ANTES de GET
        - Recalcula top 8 partners por ventas y los marca como seleccionados
        - Funciona incluso si BD tiene partners vacíos o stale

        Issue #309 - Auth Protection: Verificar autenticación ANTES de cargar datos.
        Issue #344 - Auth Ready: Esperar señal de auth-ready ANTES de cargar datos (previene F5 race condition).
        Issue #345 - BD como Fuente de Verdad: Base de datos es SIEMPRE la fuente de verdad para selección de partners.
                     Store session solo se usa para UI temporal, NO para determinar estado inicial.
                     Elimina PRIORIDAD 3 (store local) de la lógica de carga para asegurar persistencia entre sesiones.

                     IMPORTANTE: Lista vacía [] es válida (usuario puede deseleccionar todos los partners).
                     Se distingue explícitamente entre:
                     - db_selected_partners = None → Primera carga (sin datos en BD)
                     - db_selected_partners = [] → Usuario deseleccionó todos (persiste lista vacía)
        """
        # Determinar qué disparó el callback
        trigger = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else "initial"

        # DEBUG: Log entrada del callback
        logger.info(f"[load_partners_dropdown DEBUG] Callback triggered:")
        logger.info(f"   - trigger: {trigger}")
        logger.info(f"   - pathname: {pathname}")
        logger.info(f"   - auth_ready: {auth_ready}")
        logger.info(f"   - auth_state: {auth_state}")

        # GUARD 1: Validar pathname PRIMERO (orden importa)
        if pathname != "/generics":
            logger.info(f"[load_partners_dropdown] Skipping - pathname is {pathname}, not /generics")
            raise PreventUpdate

        # GUARD 2: Verificar auth-ready DESPUÉS de pathname (previene F5 race condition)
        if not auth_ready:
            logger.info(f"[load_partners_dropdown] Skipping - auth_ready={auth_ready}, waiting for auth sync")
            raise PreventUpdate

        # GUARD 3: PREVENIR RELOAD EN TOKEN REFRESH (Issue #187)
        if trigger == "auth-ready":
            triggers_list = [t["prop_id"].split(".")[0] for t in ctx.triggered] if ctx.triggered else []

            if len(triggers_list) == 1 and triggers_list[0] == "auth-ready":
                if current_store and current_store.get("selected"):
                    logger.info(
                        f"[load_partners_dropdown] Skipping - auth-ready cambió (token refresh) "
                        f"pero partners ya están cargados (count={len(current_store.get('selected', []))})"
                    )
                    raise PreventUpdate
                logger.info("[load_partners_dropdown] auth-ready trigger detectado - Primera carga después de auth")
            else:
                logger.info(f"[load_partners_dropdown] auth-ready + otros triggers ({triggers_list}) - Carga legítima")

        # ISSUE #309 - Verificar autenticación ANTES de llamadas API
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.info(f"[load_partners_dropdown] Skipping - User not authenticated (auth_state={auth_state})")
            raise PreventUpdate

        logger.debug(f"[load_partners_dropdown] Callback triggered by: {trigger}")
        logger.debug(
            f"[load_partners_dropdown] auth_ready={auth_ready}, auth_state={auth_state.get('authenticated')}, proceeding with data load"
        )

        logger.info(f"[load_partners_dropdown] Cargando partners (trigger: {trigger})...")

        # DEBUG: Log del estado del cache actual (solo si CACHE_DEBUG_ENABLED=true)
        if CACHE_DEBUG_ENABLED:
            cache_size = len(laboratory_cache_store.get("data", {})) if laboratory_cache_store else 0
            cache_age = "N/A"
            if laboratory_cache_store and "timestamp" in laboratory_cache_store:
                from datetime import datetime
                try:
                    cache_time = datetime.fromisoformat(laboratory_cache_store["timestamp"])
                    cache_age_mins = (datetime.now() - cache_time).total_seconds() / 60
                    cache_age = f"{cache_age_mins:.1f} min"
                except (ValueError, KeyError, TypeError, AttributeError) as e:
                    logger.debug(f"No se pudo calcular cache_age para laboratory_cache_store: {e}")

            logger.info(
                f"[CACHE_DEBUG] laboratory-cache-store estado:\n"
                f"  - storage_type: local (fix producción)\n"
                f"  - cache_size: {cache_size} laboratorios\n"
                f"  - cache_age: {cache_age}\n"
                f"  - cache_exists: {laboratory_cache_store is not None}"
            )

        try:
            # Obtener pharmacy_id dinámicamente
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError as e:
                logger.error(f"[load_partners_dropdown] Error obteniendo pharmacy_id: {str(e)}")
                return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})

            # FIX REFRESH BUTTON: Si trigger es refresh-btn → force re-initialize
            if trigger == "refresh-partners-list-btn" and refresh_clicks and refresh_clicks > 0:
                logger.info(f"[load_partners_dropdown] Refresh button clicked (click #{refresh_clicks}) - forcing re-initialization")

                logger.info("[REFRESH_MANUAL] Iniciando recálculo de partners (usuario presionó botón)")
                init_response = api_client.post(
                    f"/api/v1/pharmacy-partners/initialize/{pharmacy_id}?force_refresh=true&user_initiated=true"
                )

                if not init_response:
                    logger.error("Error en inicialización forzada de partners")
                    return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})

                logger.info(f"Partners re-inicializados: {init_response.get('auto_suggested_count', 0)} sugeridos")

            # FLUJO NORMAL: GET partners (ya inicializados)
            response = api_client.get(f"/api/v1/pharmacy-partners/{pharmacy_id}")

            if not response:
                logger.error("Error cargando partners: respuesta vacía")
                return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})

            partners_data = response.get("partners", [])

            if not isinstance(partners_data, list):
                logger.error(f"Formato inválido de partners: esperado list, recibido {type(partners_data)}")
                return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})

            if not partners_data:
                logger.warning("No hay partners disponibles para esta farmacia")
                return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})

            # Validar estructura mínima de cada partner
            validated_partners = []
            for i, partner in enumerate(partners_data):
                if not isinstance(partner, dict) or "laboratory_name" not in partner:
                    logger.warning(f"Partner en índice {i} tiene formato inválido: {partner}")
                    continue
                validated_partners.append(partner)

            if not validated_partners:
                logger.error("Ningún partner tiene formato válido")
                return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})

            # Los partners sugeridos son los que tienen is_auto_suggested = True
            suggested_partners = [p["laboratory_name"] for p in validated_partners if p.get("is_auto_suggested", False)]

            # PERSISTENCIA DESDE BASE DE DATOS
            db_selected_partners = [p["laboratory_name"] for p in validated_partners if p.get("is_selected", False)]

            user_configured_count = sum(1 for p in validated_partners if p.get("user_configured", False))
            logger.info(
                f"[CARGA_BD] Partners cargados desde base de datos:\n"
                f"   - Total disponibles: {len(validated_partners)}\n"
                f"   - Seleccionados en BD: {len(db_selected_partners)}\n"
                f"   - user_configured: {user_configured_count} partners\n"
                f"   - Partners seleccionados: {db_selected_partners[:3]}{'...' if len(db_selected_partners) > 3 else ''}\n"
                f"   - ESPERADO: Si user_configured > 0, estos partners deberían persistir entre sesiones"
            )

            # Función para extraer el monto de ventas para sorting
            def get_sales_amount(p):
                if p.get("suggestion_sales_amount"):
                    try:
                        sales_str = p["suggestion_sales_amount"].replace("€", "").replace(",", "")
                        return float(sales_str)
                    except (ValueError, AttributeError):
                        return 0
                return p.get("total_sales", 0)

            # PRIORIDAD ÚNICA: Base de datos (FUENTE DE VERDAD ÚNICA - Issue #345 + Fix Refresh)
            if db_selected_partners is not None:
                selected_partners = db_selected_partners
                logger.info(f"[Issue #345] BD como fuente de verdad: {len(db_selected_partners)} partners seleccionados")
                logger.debug(f"   Partners: {db_selected_partners[:3]}{'...' if len(db_selected_partners) > 3 else ''}")
                logger.debug("   Store session ignorado (BD es fuente de verdad)")
            else:
                selected_partners = suggested_partners
                logger.info(f"Primera carga sin BD: Pre-seleccionando {len(suggested_partners)} partners auto-sugeridos")
                logger.debug("   BD retornó None, usando auto-sugeridos como fallback")

            # FIX ORDEN: Ordenar selected_partners por ventas (no alfabético)
            partners_sales_map = {p["laboratory_name"]: get_sales_amount(p) for p in validated_partners}
            selected_partners_sorted = sorted(
                selected_partners,
                key=lambda name: partners_sales_map.get(name, 0),
                reverse=True
            )
            logger.info(f"[SORTING] Selected partners ordenados por ventas: {selected_partners_sorted[:3] if len(selected_partners_sorted) >= 3 else selected_partners_sorted}")
            selected_partners = selected_partners_sorted

            # Calcular total de ventas de todos los partners para porcentajes dinámicos
            total_all_partners_sales = 0
            for p in partners_data:
                if p.get("suggestion_sales_amount"):
                    try:
                        sales_str = p["suggestion_sales_amount"].replace("€", "").replace(",", "")
                        total_all_partners_sales += float(sales_str)
                    except (ValueError, AttributeError):
                        pass
                else:
                    total_all_partners_sales += p.get("total_sales", 0)

            # Crear opciones para dropdown con información adicional y porcentajes dinámicos
            dropdown_options = []

            # DEBUG: Log sorting para verificar orden
            sorted_partners = sorted(partners_data, key=get_sales_amount, reverse=True)
            logger.info(f"[SORTING DEBUG] Top 10 partners por ventas:")
            for i, p in enumerate(sorted_partners[:10]):
                logger.info(f"  {i+1}. {p['laboratory_name'][:30]:<30} - {format_currency(get_sales_amount(p))}")

            for partner in sorted_partners:
                lab_name = normalize_utf8_string(partner["laboratory_name"])
                lab_name_simplified = simplify_lab_name(lab_name)
                sales_amount = get_sales_amount(partner)
                is_suggested = lab_name in suggested_partners

                # Calcular porcentaje sobre el total de partners
                sales_percentage = (
                    (sales_amount / total_all_partners_sales * 100) if total_all_partners_sales > 0 else 0
                )

                # Formato con icono para partners sugeridos y nombre simplificado
                if is_suggested:
                    label = f"⭐ {lab_name_simplified} ({sales_percentage:.1f}% - {format_currency(sales_amount)})"
                else:
                    label = f"{lab_name_simplified} ({sales_percentage:.1f}% - {format_currency(sales_amount)})"

                dropdown_options.append(
                    {"label": label, "value": lab_name}
                )

            # FIX ENCODING: Normalizar UTF-8 también en el store para matching consistente
            store_data = {
                "selected": [normalize_utf8_string(p) for p in selected_partners],
                "available": [normalize_utf8_string(p["laboratory_name"]) for p in partners_data],
                "suggested": [normalize_utf8_string(p) for p in suggested_partners],
                "sales_map": {normalize_utf8_string(p["laboratory_name"]): get_sales_amount(p) for p in validated_partners}
            }

            # DEBUG: Log del cache que se está retornando (solo si CACHE_DEBUG_ENABLED=true)
            if CACHE_DEBUG_ENABLED:
                cache_return_info = {
                    "cache_exists": laboratory_cache_store is not None,
                    "cache_size": len(laboratory_cache_store.get("data", {})) if laboratory_cache_store else 0,
                    "cache_has_timestamp": "timestamp" in laboratory_cache_store if laboratory_cache_store else False
                }
                logger.info(
                    f"[CACHE_DEBUG][load_partners_dropdown] Retornando cache:\n"
                    f"  - cache_info: {cache_return_info}\n"
                    f"  - partners_loaded: {len(dropdown_options)}\n"
                    f"  - partners_selected: {len(selected_partners)}"
                )

            # VALIDACIÓN: Verificar tamaño de sessionStorage antes de retornar
            is_valid, size_mb = validate_session_storage_size(
                store_data,
                "partners-selection-store"
            )

            if not is_valid:
                recommendations = get_storage_recommendations(size_mb, "partners-selection-store")
                logger.warning(
                    f"[STORAGE_OPTIMIZATION] partners-selection-store excede límite: {size_mb:.2f}MB\n"
                    f"Recomendaciones: {recommendations}"
                )

            # DEBUG: Log antes de retornar para confirmar que llegamos aquí
            logger.info(f"[load_partners_dropdown] Retornando datos correctamente:")
            logger.info(f"   - dropdown_options count: {len(dropdown_options)}")
            logger.info(f"   - dropdown_value (selected): {len(selected_partners)} - {selected_partners[:3] if len(selected_partners) > 3 else selected_partners}")
            logger.info(f"   - store_data keys: {store_data.keys() if store_data else 'None'}")

            return (
                {
                    "success": True,
                    "dropdown_options": dropdown_options,
                    "dropdown_value": selected_partners,
                    "store_data": store_data,
                },
                laboratory_cache_store or {}
            )

        except requests.exceptions.RequestException as e:
            logger.error(f"Error de conexión cargando partners: {str(e)}")
            return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})
        except requests.exceptions.Timeout as e:
            logger.error(f"Timeout cargando partners: {str(e)}")
            return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})
        except ValueError as e:
            logger.error(f"Error de formato en partners: {str(e)}")
            return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})
        except Exception as e:
            logger.error(f"Error inesperado cargando partners: {str(e)}")
            return ({"success": False, "dropdown_options": [], "dropdown_value": [], "store_data": {"selected": []}}, laboratory_cache_store or {})

    # ============================================================================
    # 3b. CALLBACK UNIFICADO: Controla outputs compartidos (REGLA #11)
    # FIX CICLO ISSUE #303: Escucha dropdown.value DIRECTAMENTE (sin store intermedio)
    # ============================================================================

    @app.callback(
        [
            Output("partners-dropdown", "options"),
            Output("partners-dropdown", "value"),
            Output("partners-selection-store", "data"),
            Output("selected-partners-count", "children"),
            Output("laboratory-cache-store", "data", allow_duplicate=True),
        ],
        [Input("partners-load-result-store", "data"), Input("partners-dropdown", "value")],
        [State("partners-selection-store", "data"), State("context-store", "data"), State("laboratory-cache-store", "data")],
        prevent_initial_call=True,
    )
    def unified_partners_controller(load_result, dropdown_value, current_store, context_data, laboratory_cache_store):
        """
        Callback unificado que escucha stores intermedios Y dropdown.value DIRECTAMENTE.
        Previene duplicación de outputs (REGLA #11).

        FIX CICLO ISSUE #303: Elimina store intermedio partners-selection-result-store
        y escucha dropdown.value directamente. Incluye toda la lógica de persistencia BD
        y recálculo de porcentajes dinámicos.

        FIX REGLA #11: Combinado con cache_laboratory_codes para evitar duplicación de Input.
        Ahora este callback maneja TANTO la selección de partners COMO el cacheo de códigos.

        Issue #303 - Mejora #4: Logging detallado para debugging.
        """
        trigger = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else ""
        logger.debug(f"[unified_partners_controller] Trigger: {trigger}")

        # Guard: Si no hay datos de ventas, no procesar partners
        if context_data and context_data.get("no_data"):
            logger.debug("[unified_partners_controller] No sales data available - skipping")
            raise PreventUpdate

        # DEBUG: Estado del cache al inicio del callback (solo si CACHE_DEBUG_ENABLED=true)
        if CACHE_DEBUG_ENABLED:
            cache_info = {
                "exists": laboratory_cache_store is not None,
                "size": len(laboratory_cache_store.get("data", {})) if laboratory_cache_store else 0,
                "has_timestamp": "timestamp" in laboratory_cache_store if laboratory_cache_store else False
            }
            logger.info(f"[CACHE_DEBUG][unified_partners_controller] Cache entrada: {cache_info}")

        # DEBUG: Log estado completo al inicio
        logger.info(f"[unified_partners_controller DEBUG] Estado completo:")
        logger.info(f"   - Trigger: {trigger}")
        logger.info(f"   - load_result type: {type(load_result)}, success: {load_result.get('success') if isinstance(load_result, dict) else 'N/A'}")
        logger.info(f"   - dropdown_value: {len(dropdown_value) if isinstance(dropdown_value, list) else 'N/A'}")
        logger.info(f"   - current_store type: {type(current_store)}")

        # FIX: CASO 2 DEBE TENER PRIORIDAD sobre CASO 1
        current_selected_in_store = current_store.get("selected", []) if current_store else []
        is_auto_trigger = (not dropdown_value and current_selected_in_store)

        # DEBUG PERSISTENCIA: Log detallado para detectar por qué CASO 2 no ejecuta
        logger.info(f"[PERSISTENCIA DEBUG] Evaluando CASO 2:")
        logger.info(f"  - trigger: {trigger}")
        logger.info(f"  - current_store exists: {current_store is not None}")
        logger.info(f"  - has 'available': {current_store.get('available') if current_store else 'N/A'}")
        logger.info(f"  - is_auto_trigger: {is_auto_trigger}")
        logger.info(f"  - dropdown_value: {dropdown_value[:2] if dropdown_value else None}...")

        if trigger == "partners-dropdown" and current_store and current_store.get("available") and not is_auto_trigger:
            logger.info("[unified_partners_controller] Usuario cambió selección en dropdown")
            logger.info(
                f"[DEBUG] CASO 2 - Estado inicial:\n"
                f"  - current_store type: {type(current_store)}\n"
                f"  - dropdown_value type: {type(dropdown_value)}, length: {len(dropdown_value) if isinstance(dropdown_value, list) else 'N/A'}\n"
                f"  - context_data type: {type(context_data)}, "
                f"has_labs: {'laboratories_in_universe' in context_data if isinstance(context_data, dict) else 'N/A'}"
            )
            try:
                # Validar que key "selected" existe (aunque puede estar vacía)
                if "selected" not in current_store:
                    logger.debug(
                        "[unified_partners_controller] Ignorando trigger de dropdown: "
                        "store no tiene key 'selected' (inicialización incompleta)"
                    )
                    raise PreventUpdate

                logger.debug(
                    f"[unified_partners_controller] Store validado: "
                    f"{len(current_store.get('available', []))} disponibles, "
                    f"{len(current_store.get('selected', []))} seleccionados"
                )

                # === VALIDACIÓN DE ENTRADA ===
                if dropdown_value is None:
                    logger.debug("[unified_partners_controller] dropdown_value es None, inicializando con lista vacía")
                    dropdown_value = []
                elif not isinstance(dropdown_value, list):
                    logger.error(
                        f"[unified_partners_controller] dropdown_value debe ser lista, recibido: {type(dropdown_value)}"
                    )
                    raise PreventUpdate

                # Usar selección del dropdown directamente
                selected_partners = dropdown_value or []
                logger.info(f"[unified_partners_controller] Usuario seleccionó {len(selected_partners)} partners")

                # FIX ENCODING: Normalizar UTF-8 para matching consistente
                selected_partners = [normalize_utf8_string(p) for p in selected_partners]

                # FIX ORDEN: Ordenar selected_partners por ventas (CASO 2)
                sales_map = current_store.get("sales_map", {})
                if sales_map and selected_partners:
                    selected_partners_sorted = sorted(
                        selected_partners,
                        key=lambda name: sales_map.get(name, 0),
                        reverse=True
                    )
                    logger.info(f"[SORTING CASO 2] Partners ordenados por ventas: {selected_partners_sorted[:3] if len(selected_partners_sorted) >= 3 else selected_partners_sorted}")
                    selected_partners = selected_partners_sorted

                # Normalizar available_partners del store
                available_partners_list = current_store.get("available", [])
                available_partners_normalized = [normalize_utf8_string(p) for p in available_partners_list]

                # Validar que todos los valores están en available_partners (después de normalización)
                invalid_selections = [p for p in selected_partners if p not in available_partners_normalized]

                if invalid_selections:
                    logger.warning(
                        f"[unified_partners_controller] Selecciones inválidas ignoradas (después de normalización UTF-8): {invalid_selections}"
                    )
                    selected_partners = [p for p in selected_partners if p in available_partners_normalized]
                    logger.info(
                        f"[unified_partners_controller] Después de filtrar: {len(selected_partners)} partners válidos"
                    )

                # Actualizar store
                updated_store = current_store.copy()
                updated_store["selected"] = selected_partners
                logger.debug(
                    f"[unified_partners_controller] Store actualizado con {len(selected_partners)} seleccionados"
                )

                # === PERSISTENCIA EN BASE DE DATOS ===
                try:
                    pharmacy_id = get_current_pharmacy_id()

                    available_partners = current_store.get("available", [])

                    if not available_partners:
                        if context_data and "laboratories_in_universe" in context_data:
                            available_partners = [
                                lab["laboratory_name"]
                                for lab in context_data["laboratories_in_universe"]
                                if isinstance(lab, dict) and "laboratory_name" in lab
                            ]
                            logger.warning(
                                f"[unified_partners_controller] Store vacío, usando context_data como fallback: "
                                f"{len(available_partners)} partners"
                            )
                        else:
                            logger.warning(
                                "[unified_partners_controller] No hay available_partners - skipping persistence"
                            )
                            raise ValueError("No available_partners to persist")
                    else:
                        logger.debug(
                            f"[unified_partners_controller] Usando {len(available_partners)} partners "
                            f"desde current_store (fuente de verdad para dropdown)"
                        )

                    logger.debug(f"[unified_partners_controller] Persistiendo selección para farmacia {pharmacy_id}")
                    persist_partners_selection(pharmacy_id, selected_partners, available_partners)
                except ValueError as ve:
                    logger.warning(
                        f"[unified_partners_controller] No se puede persistir selección: {str(ve)}"
                    )
                except Exception as pe:
                    logger.warning(
                        f"[unified_partners_controller] Error al persistir selección: {str(pe)}"
                    )

                # Preparar outputs
                available_labs = current_store.get("available", [])
                count_text = f"{len(selected_partners)} de {len(available_labs)} seleccionados"

                # FIX Issue #XXX: Usar current_store (fuente de verdad) en lugar de context_data (stale)
                # El context_data["laboratories_in_universe"] se carga una vez y NO se actualiza
                # cuando el usuario añade/quita partners. El current_store SÍ tiene todos los labs.
                sales_map = current_store.get("sales_map", {})
                suggested_partners = current_store.get("suggested", [])

                if not available_labs:
                    logger.warning(
                        f"[unified_partners_controller] No hay labs disponibles en store"
                    )
                    return (no_update, no_update, updated_store, count_text, no_update)

                # FIX: Calcular total sobre TODOS los labs (universo completo)
                # Antes: solo partners seleccionados → labs no seleccionados mostraban 0%
                # Ahora: todos los labs → usuarios ven importancia real de cada lab
                total_universe_sales = sum(
                    sales_map.get(lab_name, 0)
                    for lab_name in available_labs
                )

                # Recrear opciones con porcentajes actualizados
                dropdown_options = []

                # Ordenar por ventas (descendente)
                for lab_name in sorted(available_labs, key=lambda name: sales_map.get(name, 0), reverse=True):
                    if not lab_name or not isinstance(lab_name, str):
                        logger.warning(
                            f"[unified_partners_controller] Ignorando lab con nombre inválido: {lab_name}"
                        )
                        continue

                    lab_name_simplified = simplify_lab_name(lab_name)
                    sales_amount = sales_map.get(lab_name, 0)
                    is_suggested = lab_name in suggested_partners

                    # FIX: Calcular porcentaje sobre TOTAL del universo (no solo seleccionados)
                    # Permite ver importancia de cada lab ANTES de seleccionarlo
                    if total_universe_sales > 0:
                        sales_percentage = sales_amount / total_universe_sales * 100
                    else:
                        sales_percentage = 0

                    if is_suggested:
                        label = (
                            f"⭐ {lab_name_simplified} ({sales_percentage:.1f}% - {format_currency(sales_amount)})"
                        )
                    else:
                        label = f"{lab_name_simplified} ({sales_percentage:.1f}% - {format_currency(sales_amount)})"

                    dropdown_options.append({"label": label, "value": lab_name})

                # DEBUG: Log orden de opciones ANTES de retornar
                logger.info(f"[CASO 2 ORDEN DEBUG] Primeras 10 opciones a retornar:")
                for i, opt in enumerate(dropdown_options[:10]):
                    logger.info(f"  {i+1}. {opt['label'][:60]}")

                return (
                    dropdown_options,
                    no_update,  # NO actualizar dropdown.value (ya fue actualizado por el usuario)
                    updated_store,
                    count_text,
                    no_update,  # NO actualizar caché de códigos (ya está cargado en CASO 1)
                )

            except PreventUpdate:
                raise
            except Exception as e:
                import traceback
                tb_str = traceback.format_exc()
                logger.error(
                    f"[ERROR] CASO 2 - Exception capturada:\n"
                    f"  - Error type: {type(e).__name__}\n"
                    f"  - Error message: {str(e)}\n"
                    f"  - Error args: {e.args}\n"
                    f"  - Traceback:\n{tb_str}"
                )
                logger.error(
                    f"[ERROR] CASO 2 - Estado en momento de error:\n"
                    f"  - current_store: {current_store}\n"
                    f"  - dropdown_value: {dropdown_value}\n"
                    f"  - context_data keys: {list(context_data.keys()) if isinstance(context_data, dict) else 'N/A'}"
                )
                return no_update, no_update, current_store, "Error", no_update

        # CASO 1: Cargar partners desde API
        elif load_result and load_result.get("success"):
            logger.info("[unified_partners_controller] CASO 1: Cargando partners desde API...")
            logger.info(
                f"[unified_partners_controller] Partners cargados: {len(load_result['dropdown_options'])} disponibles, {len(load_result['dropdown_value'])} seleccionados"
            )
            logger.info(f"[unified_partners_controller CASO 1 DEBUG] dropdown_value a retornar: {load_result['dropdown_value'][:3] if len(load_result.get('dropdown_value', [])) > 3 else load_result.get('dropdown_value', [])}")

            # === CACHEO DE CÓDIGOS DE LABORATORIO (Fix Rate Limiting - P1) ===
            if laboratory_cache_store is None:
                laboratory_cache_store = {}
            try:
                dropdown_options = load_result.get('dropdown_options', [])

                if dropdown_options:
                    lab_names = [opt['value'] for opt in dropdown_options if isinstance(opt, dict) and 'value' in opt]

                    if lab_names:
                        logger.info(f"[unified_partners_controller] Cacheando códigos para {len(lab_names)} laboratorios")

                        _, laboratory_cache_store = get_codes_with_cache(lab_names, laboratory_cache_store)

                        logger.info(f"Cache de códigos creado: {len(laboratory_cache_store)} laboratorios mapeados")
                    else:
                        logger.warning("[unified_partners_controller] No se pudieron extraer nombres de laboratorios")
                else:
                    logger.warning("[unified_partners_controller] No hay opciones de laboratorios para cachear")
            except Exception as e:
                logger.error(f"Error cacheando códigos de laboratorios: {str(e)}")

            # DEBUG: Confirmar que el cache se está retornando (solo si CACHE_DEBUG_ENABLED=true)
            if CACHE_DEBUG_ENABLED:
                cache_return = {
                    "exists": laboratory_cache_store is not None,
                    "size": len(laboratory_cache_store.get("data", {})) if laboratory_cache_store else 0,
                }
                logger.info(f"[CACHE_DEBUG][unified_partners CASO 1] Cache retornado: {cache_return}")

            # VALIDACIÓN: Verificar tamaño de sessionStorage antes de retornar
            is_valid, size_mb = validate_session_storage_size(
                load_result["store_data"],
                "partners-selection-store"
            )

            if not is_valid:
                recommendations = get_storage_recommendations(size_mb, "partners-selection-store")
                logger.warning(
                    f"[STORAGE_OPTIMIZATION] partners-selection-store excede límite: {size_mb:.2f}MB\n"
                    f"Recomendaciones: {recommendations}"
                )

            # DEBUG: Log orden de opciones ANTES de retornar
            logger.info(f"[CASO 1 ORDEN DEBUG] Primeras 10 opciones a retornar:")
            for i, opt in enumerate(load_result["dropdown_options"][:10]):
                logger.info(f"  {i+1}. {opt['label'][:60]}")

            return (
                load_result["dropdown_options"],
                load_result["dropdown_value"],
                load_result["store_data"],
                f"{len(load_result['dropdown_value'])} de {len(load_result['dropdown_options'])} seleccionados",
                laboratory_cache_store,
            )

        # CASO 3: Error o estado inicial
        else:
            return no_update, no_update, no_update, no_update, no_update

    # ============================================================================
    # 3c. CALLBACK INICIALIZAR CONTADOR PARTNERS (BD-FIRST PATTERN)
    # ============================================================================

    @app.callback(
        Output("selected-partners-count", "children", allow_duplicate=True),
        [Input("url", "pathname"), Input("auth-ready", "data")],
        [State("auth-state", "data")],
        prevent_initial_call=True,
    )
    def initialize_partners_count(pathname, auth_ready, auth_state):
        """
        Inicializar contador de partners seleccionados al entrar a /generics.

        BD-FIRST PATTERN: Lee partners SIEMPRE desde BD (independiente de stores).

        Notes:
            - Se ejecuta al cambiar pathname a /generics
            - Usa get_selected_partners_from_db() con cache (2s TTL)
            - Independiente de unified_partners_controller (no race conditions)
            - allow_duplicate porque unified_partners_controller también actualiza este Output

        Returns:
            str: Contador en formato "X de Y seleccionados" o "-- de --" si error
        """
        trigger = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else "unknown"

        # GUARD 1: Solo ejecutar si estamos en /generics
        if pathname != "/generics":
            raise PreventUpdate

        # GUARD 2: Verificar auth-ready
        if not auth_ready:
            raise PreventUpdate

        # GUARD 3: PREVENIR RELOAD EN TOKEN REFRESH (Issue #187)
        if trigger == "auth-ready":
            triggers_list = [t["prop_id"].split(".")[0] for t in ctx.triggered] if ctx.triggered else []
            if len(triggers_list) == 1 and triggers_list[0] == "auth-ready":
                logger.debug("[initialize_partners_count] Skipping - auth-ready solo (token refresh detected)")
                raise PreventUpdate

        # GUARD 4: Verificar autenticación
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("[initialize_partners_count] User not authenticated")
            raise PreventUpdate

        try:
            # BD-FIRST PATTERN: Leer partners desde BD
            logger.info("[initialize_partners_count] Reading partners count from DB (BD-first pattern)")
            result = get_selected_partners_from_db()

            if not result["success"]:
                logger.error(
                    f"[initialize_partners_count] ERROR fetching partners from DB\n"
                    f"   - Error: {result.get('error', 'Unknown')}"
                )
                return "-- de --"

            selected_partners = result["partners"]

            # Obtener total de partners disponibles
            try:
                pharmacy_id = get_current_pharmacy_id()
                response = api_client.get(f"/api/v1/pharmacy-partners/{pharmacy_id}")

                if response and response.get("partners"):
                    total_available = len(response["partners"])
                else:
                    total_available = len(selected_partners)
                    logger.warning(
                        f"[initialize_partners_count] Could not fetch available partners, "
                        f"using selected count as fallback"
                    )
            except Exception as e:
                logger.error(f"[initialize_partners_count] Error fetching available partners: {str(e)}")
                total_available = len(selected_partners)

            count_text = f"{len(selected_partners)} de {total_available} seleccionados"

            logger.info(
                f"[initialize_partners_count] Counter initialized from DB\n"
                f"   - Selected: {len(selected_partners)}\n"
                f"   - Available: {total_available}\n"
                f"   - Source: {result['source'].upper()} (BD-first pattern with caching)\n"
                f"   - Display: {count_text}"
            )

            return count_text

        except Exception as e:
            logger.error(f"[initialize_partners_count] Unexpected error: {str(e)}")
            return "-- de --"
