"""
Optimal partners callbacks for Generics Panel.

Handles:
- Optimal partners section (compact list)
- Partner penetration gauge
"""

import logging

from dash import Input, Output, State, callback
from dash.exceptions import PreventUpdate

from utils.api_client import api_client

logger = logging.getLogger(__name__)


def register_optimal_partners_callbacks(app):
    """
    Register optimal partners callbacks for the Generics Panel.

    Args:
        app: Dash application instance

    Note:
        Uses standalone @callback decorators for module-level callbacks.
    """
    pass  # Callbacks are registered via @callback decorators below


# ============================================================================
# 14. CALLBACK PARTNERS ÓPTIMOS
# ============================================================================

@callback(
    Output("optimal-partners-compact-list", "children"),
    [
        Input("analysis-store", "data"),
        Input("optimal-partners-count-selector", "value"),
    ],
    [
        State("generics-date-range", "start_date"),
        State("generics-date-range", "end_date"),
        State("generics-employee-filter", "value"),
        State("auth-state", "data"),
    ],
    prevent_initial_call=True,
)
def update_optimal_partners_section(
    analysis_data, top_n, start_date, end_date, employee_names, auth_state
):
    """
    Actualizar sección de Partners Óptimos con lista compacta.

    Issue #428: Versión simplificada - solo lista con cobertura por partner.

    Args:
        analysis_data: Datos del análisis actual (trigger)
        top_n: Cantidad de partners a analizar (1-10)
        start_date: Fecha inicio del período
        end_date: Fecha fin del período
        employee_names: Filtro de empleados (opcional)
        auth_state: Estado de autenticación

    Returns:
        html.Div: Lista compacta de partners
    """
    from utils.auth_helpers import is_user_authenticated
    from utils.pharmacy_context import get_current_pharmacy_id
    from components.optimal_partners_section import (
        create_compact_partners_list,
        _create_empty_list,
    )

    # Guard: Verificar análisis válido
    if not analysis_data or "error" in analysis_data:
        return _create_empty_list()

    # Guard: Verificación proactiva de autenticación
    if not is_user_authenticated(auth_state):
        logger.debug("[OPTIMAL_PARTNERS] User not authenticated - skipping")
        raise PreventUpdate

    # Guard: Validar parámetros
    if not start_date or not end_date:
        raise PreventUpdate

    try:
        pharmacy_id = get_current_pharmacy_id()
        if not pharmacy_id:
            logger.error("[OPTIMAL_PARTNERS] No pharmacy_id found")
            raise PreventUpdate

        # Construir parámetros API
        params = {
            "start_date": start_date,
            "end_date": end_date,
            "max_partners": top_n or 5,
        }
        if employee_names:
            params["employee_names"] = employee_names

        logger.info(
            "[OPTIMAL_PARTNERS] Fetching",
            extra={"pharmacy_id": pharmacy_id, "max_partners": params["max_partners"]}
        )

        data = api_client.get(
            f"/api/v1/optimal-partners/{pharmacy_id}",
            params=params
        )

        # Verificar datos válidos
        if not data or "error" in data:
            logger.warning("[OPTIMAL_PARTNERS] API returned empty or error response")
            return _create_empty_list()

        # Crear lista compacta con datos reales
        partners_list = create_compact_partners_list(data)

        partners_count = len(data.get("optimal_partners", []))
        logger.info(
            "[OPTIMAL_PARTNERS] Success",
            extra={"partners_count": partners_count}
        )

        return partners_list

    except Exception as e:
        logger.error("[OPTIMAL_PARTNERS] Error", extra={"error": str(e)})
        return _create_empty_list()


# ============================================================================
# 15. CALLBACK GAUGE DE PENETRACIÓN
# ============================================================================

@callback(
    [
        Output("partner-penetration-gauge-chart", "figure"),
        Output("penetration-gauge-legend", "children"),
    ],
    [Input("analysis-store", "data")],
    [State("url", "pathname")],
    prevent_initial_call=True,
)
def update_penetration_gauge(analysis_data, pathname):
    """
    Actualizar gauge de penetración basado en datos del análisis.

    Issue #428: Gauge muestra % de ventas sustituibles cubiertas por partners.

    Métrica: (€ partners / € total analizable) * 100

    Args:
        analysis_data: Datos del análisis con partners vs no-partners (incluye selected partners)

    Returns:
        Tuple: (gauge_figure, legend_div)
    """
    from components.partner_penetration_gauge import (
        create_empty_gauge,
        create_penetration_gauge,
        create_partners_legend,
        _create_empty_legend,
    )

    # Guard 0: Pathname guard - solo ejecutar en /generics
    if pathname and pathname.lower() not in ["/generics", "/genericos"]:
        raise PreventUpdate

    # Helper para retornar estado vacío
    def _empty_response():
        return create_empty_gauge(), _create_empty_legend()

    # Guard: Verificar datos válidos
    if not analysis_data or "error" in analysis_data:
        return _empty_response()

    try:
        # Extraer métricas del análisis
        partner_performance = analysis_data.get("partner_performance", {})
        analyzable_universe = analysis_data.get("analyzable_universe", {})
        selected_partner_codes = analysis_data.get("selected_partner_codes", [])

        # Guard: Verificar partners seleccionados
        if not selected_partner_codes:
            return _empty_response()

        partners_amount = partner_performance.get("partner_revenue", 0.0)
        total_analyzable = analyzable_universe.get("total_revenue", 0.0)
        penetration_percentage = partner_performance.get("penetration_percentage", 0.0)

        # Convertir códigos a nombres de laboratorio para mostrar en la leyenda
        from utils.laboratory_mapping import LaboratoryMapper
        mapper = LaboratoryMapper()
        code_to_name_mapping = mapper.codes_to_names(selected_partner_codes)

        # Crear lista de nombres en el mismo orden que los códigos
        selected_partner_names = [
            code_to_name_mapping.get(code, code)
            for code in selected_partner_codes
        ]

        # Crear gauge y leyenda (ahora con nombres, no códigos)
        gauge_fig = create_penetration_gauge(
            penetration_percentage,
            partners_amount,
            total_analyzable,
            selected_partner_names
        )
        legend_div = create_partners_legend(selected_partner_names, partners_amount)

        logger.info(
            "[PENETRATION_GAUGE] Updated",
            extra={
                "penetration_pct": round(penetration_percentage, 1),
                "partners_count": len(selected_partner_codes),
                "partners_amount": round(partners_amount, 2)
            }
        )

        return gauge_fig, legend_div

    except Exception as e:
        logger.error("[PENETRATION_GAUGE] Error", extra={"error": str(e)})
        return _empty_response()
