"""
Homogeneous detail callbacks for Generics Panel.

Handles:
- Homogeneous set selection
- Detail chart visualization
- Drill-down navigation for individual sets
"""

import json
import logging
from typing import Dict, List, Optional, Tuple

from dash import ALL, Input, Output, State, callback, ctx, dcc, html
from dash.exceptions import PreventUpdate

from components.common import create_alert
from components.generics import create_stacked_bar_chart
from utils.generics_helpers import (
    get_homogeneous_detail_from_store,
    get_selected_partners_from_db,
    get_temporal_breakdown_data,
)

logger = logging.getLogger(__name__)


def register_homogeneous_detail_callbacks(app):
    """
    Register homogeneous detail callbacks for the Generics Panel.

    Args:
        app: Dash application instance

    Note:
        Uses standalone @callback decorators for module-level callbacks.
    """
    pass  # Callbacks are registered via @callback decorators below


# ============================================================================
# ISSUE #346: SELECCIÓN DE CONJUNTO HOMOGÉNEO Y GRÁFICO DETALLADO
# ============================================================================


@callback(
    [
        Output("selected-homogeneous-store", "data"),
        Output({"type": "select-homogeneous-btn", "code": ALL}, "color"),
        Output({"type": "select-homogeneous-btn", "code": ALL}, "outline"),
    ],
    Input({"type": "select-homogeneous-btn", "code": ALL}, "n_clicks"),
    State("selected-homogeneous-store", "data"),
    prevent_initial_call=True,
)
def handle_homogeneous_selection(
    n_clicks_list: List[Optional[int]], current_selection: Optional[str]
) -> Tuple[Optional[str], List[str], List[bool]]:
    """
    Manejar selección de conjunto homogéneo para visualización detallada.

    Issue #346: Implementa selección única (solo un conjunto a la vez).
    Toggle: Click en seleccionado lo deselecciona.

    Returns:
        - selected_code: Código del conjunto seleccionado (None si deselección)
        - colors: Lista de colores para todos los botones
        - outlines: Lista de estados outline para todos los botones
    """
    if not ctx.triggered:
        raise PreventUpdate

    logger.debug(
        f"[HOMOGENEOUS_SELECTION] Callback triggered. "
        f"n_clicks_list length: {len(n_clicks_list) if n_clicks_list else 0}, "
        f"current_selection: {current_selection}"
    )

    # Obtener el botón que se clickeó
    triggered_prop_id = ctx.triggered[0]["prop_id"]
    # Issue #346: Usar rsplit para evitar cortar en puntos dentro del código
    triggered_id = triggered_prop_id.rsplit(".", 1)[0]

    if not triggered_id or triggered_id == "{}":
        raise PreventUpdate

    # Parsear el ID del botón clickeado
    try:
        button_id = json.loads(triggered_id)
        clicked_code = str(button_id["code"])
    except (json.JSONDecodeError, KeyError) as e:
        logger.error(
            f"[HOMOGENEOUS_SELECTION] Invalid button ID. "
            f"Raw prop_id: {triggered_prop_id[:100]}, "
            f"Extracted ID: {triggered_id[:100]}, "
            f"Error: {str(e)}"
        )
        raise PreventUpdate

    # Toggle: Si ya está seleccionado, deseleccionar
    if current_selection == clicked_code:
        logger.info(f"[HOMOGENEOUS_SELECTION] Deselecting: {clicked_code}")
        new_selection = None
    else:
        logger.info(f"[HOMOGENEOUS_SELECTION] Selecting: {clicked_code}")
        new_selection = clicked_code

    # Actualizar colores y outlines de TODOS los botones
    num_buttons = len(n_clicks_list)
    colors = []
    outlines = []

    # Obtener los códigos de todos los botones desde ctx.outputs_list
    all_codes = [output["id"]["code"] for output in ctx.outputs_list[1]]

    for code in all_codes:
        if new_selection and code == new_selection:
            # Botón seleccionado: verde, sin outline
            colors.append("success")
            outlines.append(False)
        else:
            # Botones no seleccionados: light, con outline
            colors.append("light")
            outlines.append(True)

    return new_selection, colors, outlines


@callback(
    Output("homogeneous-detail-chart-container", "children"),
    [Input("selected-homogeneous-store", "data"), Input("homogeneous-drill-store", "data")],
    [State("analysis-store", "data"), State("partners-selection-store", "data"), State("auth-state", "data"), State("codes-cache-store", "data")],
    prevent_initial_call=True,
)
def update_homogeneous_detail_chart(
    selected_code: Optional[str],
    homogeneous_drill_data: Optional[Dict],
    analysis_data: Optional[Dict],
    partners_store: Optional[Dict],
    auth_state: Optional[Dict],
    codes_cache: Optional[Dict],
) -> html.Div:
    """
    Actualizar gráfico temporal del conjunto homogéneo seleccionado.

    BD-FIRST PATTERN: Lee partners SIEMPRE desde BD (ignora partners_store).

    Issue #346:
    - Drill-down INDEPENDIENTE con homogeneous-drill-store (NO sincronizado con temporal-drill-store)
    - Título mejorado: 11px, multi-línea, margen superior 70px

    Args:
        selected_code: Código del conjunto homogéneo seleccionado
        homogeneous_drill_data: Datos de drill temporal INDEPENDIENTE del conjunto
        analysis_data: Datos completos del análisis (contiene homogeneous_groups_detail)
        partners_store: IGNORED - kept for backwards compatibility but value ignored
        auth_state: Estado de autenticación (Issue #302)
        codes_cache: Cache de códigos de laboratorio

    Returns:
        html.Div con gráfico temporal o mensaje de estado
    """
    if not ctx.triggered:
        raise PreventUpdate

    # Issue #302: Verificar autenticación antes de llamadas API
    from utils.auth_helpers import is_user_authenticated

    if not is_user_authenticated(auth_state):
        logger.debug("[GENERICS_CHART] User not authenticated - skipping chart update")
        raise PreventUpdate

    # Sin selección: Mostrar placeholder
    if not selected_code:
        return html.Div(
            [
                html.I(className="fas fa-mouse-pointer me-2 text-info"),
                html.P(
                    "Selecciona un conjunto de la matriz para ver su evolución temporal",
                    className="text-muted text-center mb-0",
                ),
            ],
            className="text-center py-3",
        )

    # Verificar que existe en analysis_data
    if not analysis_data or "homogeneous_groups_detail" not in analysis_data:
        return create_alert("No hay datos de análisis disponibles", "warning")

    # Obtener detalle del conjunto homogéneo desde analysis-store
    group_detail = get_homogeneous_detail_from_store(analysis_data, selected_code)

    if not group_detail:
        return create_alert(f"No se encontraron datos para el conjunto: {selected_code}", "warning")

    homogeneous_name = group_detail.get("homogeneous_name", "Sin nombre")

    # NUEVO: Llamar al backend CON homogeneous_code (Issue #346)
    try:
        # BD-FIRST PATTERN: Leer partners desde BD (ignorar store)
        logger.info("[HOMOGENEOUS_CHART] Reading partners from DB (BD-first pattern)")
        result = get_selected_partners_from_db()

        if not result["success"]:
            logger.error(
                f"[HOMOGENEOUS_CHART] ERROR fetching partners from DB\n"
                f"   - Error: {result.get('error', 'Unknown')}"
            )
            return create_alert("Error cargando partners. Por favor, intenta de nuevo.", "danger")

        selected_partners = result["partners"]
        logger.info(
            f"[HOMOGENEOUS_CHART] Partners loaded from DB\n"
            f"   - Count: {len(selected_partners)}\n"
            f"   - Source: {result['source'].upper()}"
        )

        if not selected_partners:
            return create_alert("Debes seleccionar al menos un partner para ver detalle", "info")

        # Issue #346: Obtener nivel temporal y path del drill-store INDEPENDIENTE
        drill_level = homogeneous_drill_data.get("level", "quarter") if homogeneous_drill_data else "quarter"
        drill_path = homogeneous_drill_data.get("path", []) if homogeneous_drill_data else []

        # Inicializar cache si no existe
        if codes_cache is None:
            codes_cache = {}

        # Obtener datos temporales FILTRADOS por conjunto homogéneo
        filtered_temporal_data, updated_cache = get_temporal_breakdown_data(
            level=drill_level,
            path=drill_path,
            partners_store=partners_store,
            codes_cache=codes_cache,
            homogeneous_code=selected_code
        )

        if not filtered_temporal_data:
            return create_alert(
                f"No hay datos temporales disponibles para {homogeneous_name}",
                "warning"
            )

        logger.info(
            f"[HOMOGENEOUS_CHART] Received {len(filtered_temporal_data)} periods for group {selected_code}"
        )

        # Crear gráfico temporal apilado con datos filtrados
        units_fig = create_stacked_bar_chart(
            filtered_temporal_data,
            "units",
            "Unidades",
            drill_level
        )

        # Issue #346: Título mejorado (11px, multi-línea, margen superior 70px)
        units_fig.update_layout(
            title={
                'text': f"{homogeneous_name}<br>Evolución Temporal",
                'x': 0.5,
                'xanchor': 'center',
                'font': {'size': 11, 'color': '#333'},
                'xref': 'paper',
                'yref': 'paper'
            },
            height=350,
            margin=dict(l=40, r=40, t=70, b=60),
        )

        # Issue #346: Agregar ID para capturar clickData en drill-down independiente
        return dcc.Graph(
            id="homogeneous-detail-chart",
            figure=units_fig,
            config={"displayModeBar": False, "responsive": True}
        )

    except Exception as e:
        logger.error(f"[CHART_UPDATE] Error generando gráfico temporal: {str(e)}")
        return create_alert(f"Error al cargar evolución temporal: {str(e)}", "danger")


@callback(
    [
        Output("homogeneous-drill-store", "data"),
        Output("homogeneous-drill-up-btn", "disabled")
    ],
    Input("homogeneous-detail-chart", "clickData"),
    State("homogeneous-drill-store", "data"),
    prevent_initial_call=True
)
def handle_homogeneous_drill_down(click_data, current_drill_state):
    """
    Manejar drill-down independiente para el gráfico de conjunto homogéneo.

    Issue #346: Drill-down NO sincronizado con gráficos principales (temporal-drill-store).
    Cada gráfico tiene su propio path de navegación temporal.
    Habilita el botón drill-up cuando se hace drill-down.

    Args:
        click_data: Datos del click en el gráfico (contiene período clickeado)
        current_drill_state: Estado actual del drill (nivel y path)

    Returns:
        Tuple: (nuevo_estado_drill, is_button_disabled)
    """
    from utils.drill_down_helpers import (
        parse_drill_down_click,
        calculate_next_drill_level,
        initialize_drill_state
    )

    # Parsear período clickeado
    clicked_period = parse_drill_down_click(click_data)
    if not clicked_period:
        raise PreventUpdate

    # Inicializar drill state si no existe
    if not current_drill_state:
        current_drill_state = initialize_drill_state()

    # Calcular siguiente nivel usando helper
    new_level, new_path = calculate_next_drill_level(
        current_drill_state["level"],
        current_drill_state["path"],
        clicked_period
    )

    if new_level is None:
        # Ya estamos en fortnight, no más drill-down
        raise PreventUpdate

    logger.info(f"[HOMOGENEOUS_DRILL] Drilling from {current_drill_state['level']} to {new_level}: {new_path}")

    # Habilitar botón drill-up (disabled=False) porque ahora podemos volver
    return {"level": new_level, "path": new_path}, False


@callback(
    [
        Output("homogeneous-drill-store", "data", allow_duplicate=True),
        Output("homogeneous-drill-up-btn", "disabled", allow_duplicate=True)
    ],
    Input("homogeneous-drill-up-btn", "n_clicks"),
    State("homogeneous-drill-store", "data"),
    prevent_initial_call=True
)
def handle_homogeneous_drill_up(n_clicks, current_drill_state):
    """
    Volver al nivel anterior en drill-down del gráfico de conjunto.

    Issue #346: Drill-up independiente (fortnight → month → quarter).
    Deshabilita el botón cuando llegamos a quarter (nivel raíz).

    Args:
        n_clicks: Número de clicks en el botón (trigger)
        current_drill_state: Estado actual del drill (nivel y path)

    Returns:
        Tuple: (nuevo_estado_drill, is_button_disabled)
    """
    from utils.drill_down_helpers import calculate_previous_drill_level

    if not current_drill_state or len(current_drill_state.get("path", [])) == 0:
        raise PreventUpdate

    # Calcular nivel anterior usando helper
    new_level, new_path, is_disabled = calculate_previous_drill_level(
        current_drill_state["path"]
    )

    new_state = {"level": new_level, "path": new_path}

    logger.info(f"[HOMOGENEOUS_DRILL_UP] From {current_drill_state['level']} to {new_level}: {new_path}")

    return new_state, is_disabled
