"""
Homogeneous callbacks for Generics Panel.

Handles:
- Search options autocomplete
- PowerBI-style matrix
- Partner references table
- Expansion/contraction unified handler
- Discount simulation
- CSV export
"""

import json
import logging
from datetime import datetime
from typing import Any, Dict, List, Optional, Tuple

import pandas as pd
import requests
import requests.exceptions
from dash import ALL, Input, Output, State, ctx, dcc, html, no_update
from dash.exceptions import PreventUpdate

from components.common import create_alert
from components.generics import (
    create_partner_refs_table,
    create_powerbi_matrix,
    create_products_detail_table,
)
from utils.api_client import api_client
from utils.debug_flags import CACHE_DEBUG_ENABLED
from utils.generics_helpers import (
    DEFAULT_DISCOUNT_RATE,
    ICON_COLLAPSED,
    ICON_EXPANDED,
    create_expand_button_content,
    get_codes_with_cache,
    get_homogeneous_detail_from_store,
    get_selected_partners_from_db,
    has_placeholder,
)
from utils.helpers import format_compact_number, format_currency, format_number
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)


def register_homogeneous_callbacks(app):
    """
    Register homogeneous callbacks for the Generics Panel.

    Args:
        app: Dash application instance
    """

    # ============================================================================
    # 7A. CALLBACK OPCIONES AUTOCOMPLETADO BÚSQUEDA
    # ============================================================================

    @app.callback(
        Output("homogeneous-search-input", "options"),
        Input("analysis-store", "data"),
        State("url", "pathname"),
        prevent_initial_call=False
    )
    def update_search_options(analysis_data, pathname):
        """
        Actualizar opciones de autocompletado basadas en conjuntos disponibles.

        Issue: Normalización de nombres para búsqueda robusta.
        Issue #484: Añadido pathname guard para evitar ejecución en otras páginas.
        """
        # Guard: Solo ejecutar en /generics
        if pathname and pathname.lower() not in ["/generics", "/genericos"]:
            return []

        try:
            if not analysis_data or "error" in analysis_data:
                return []

            homogeneous_data = analysis_data.get("homogeneous_groups_detail", [])
            if not homogeneous_data:
                return []

            # Crear opciones únicas de nombres de conjuntos con normalización
            options = []
            for group in homogeneous_data:
                name = group.get("homogeneous_name", "")
                if name:
                    normalized_name = " ".join(name.split())
                    options.append({
                        "label": normalized_name,
                        "value": normalized_name
                    })

            # Ordenar alfabéticamente
            options.sort(key=lambda x: x["label"])

            logger.info(f"[SEARCH_OPTIONS] Total opciones: {len(options)}")
            if options:
                logger.info(f"[SEARCH_OPTIONS] Primeras 10: {[opt['label'][:60] for opt in options[:10]]}")

            return options

        except (ValueError, KeyError, TypeError, AttributeError) as e:
            logger.error(f"Error actualizando opciones búsqueda: {str(e)}")
            return []

    # ============================================================================
    # 7B. CALLBACK MATRIZ CONJUNTOS HOMOGÉNEOS (PowerBI-style)
    # ============================================================================

    @app.callback(
        [
            Output("homogeneous-matrix-container", "children"),
            Output("homogeneous-matrix-summary", "children"),
            Output("matrix-sort-store", "data"),
        ],
        [
            Input("analysis-store", "data"),
            Input("homogeneous-search-input", "value"),
            Input("expand-all-homogeneous-btn", "n_clicks"),
            Input({"type": "matrix-header", "column": ALL}, "n_clicks"),
        ],
        [State("matrix-sort-store", "data"), State("partner-discount-slider", "value"), State("url", "pathname")],
        prevent_initial_call=False,
    )
    def update_homogeneous_matrix(
        analysis_data, search_value, expand_all_clicks, header_clicks, sort_state, discount_percentage, pathname
    ):
        """
        Actualizar matriz PowerBI-style de conjuntos homogéneos con jerarquía expandible.
        Ordenación mediante clicks en headers de columnas.

        FIX REGLA #11: Discount slider moved to State (not Input) to avoid duplicate Input.
        Issue #484: Añadido pathname guard para evitar ejecución en otras páginas.
        """
        # Guard: Solo ejecutar en /generics
        if pathname and pathname.lower() not in ["/generics", "/genericos"]:
            return no_update, no_update, no_update

        trigger = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else "initial_load"
        logger.info(f"[update_homogeneous_matrix] CALLED - Trigger: {trigger}")
        logger.info(f"[update_homogeneous_matrix] analysis_data type: {type(analysis_data)}, has_data: {bool(analysis_data)}")
        if analysis_data:
            logger.info(f"[update_homogeneous_matrix] analysis_data keys: {list(analysis_data.keys()) if isinstance(analysis_data, dict) else 'NOT_DICT'}")

        try:
            if not analysis_data or "error" in analysis_data:
                logger.warning("[update_homogeneous_matrix] NO DATA or ERROR - Returning loading alert")
                return create_alert("Cargando análisis de conjuntos...", "info"), "", no_update

            # FIX STALE CACHE: Si no hay datos de ventas, mostrar mensaje vacío
            if analysis_data.get("no_data"):
                logger.info("[update_homogeneous_matrix] no_data=True - mostrando vacío")
                return (
                    create_alert("No hay datos de ventas. Sube tu archivo ERP para ver conjuntos homogéneos.", "info"),
                    "",
                    no_update
                )

            # Obtener datos de conjuntos homogéneos
            homogeneous_data = analysis_data.get("homogeneous_groups_detail", [])
            if not homogeneous_data:
                return create_alert("No hay datos de conjuntos homogéneos disponibles", "warning"), "", no_update

            # Determinar ordenación por clicks en headers
            triggered_id = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else None

            # Estado de ordenación por defecto
            if not sort_state:
                sort_state = {"column": "savings_base", "direction": "desc"}

            # Si se clickeó un header, cambiar ordenación
            if triggered_id and "matrix-header" in str(triggered_id):
                try:
                    header_data = json.loads(triggered_id)
                    clicked_column = header_data.get("column")

                    if sort_state.get("column") == clicked_column:
                        sort_state["direction"] = "asc" if sort_state["direction"] == "desc" else "desc"
                    else:
                        sort_state["column"] = clicked_column
                        sort_state["direction"] = "desc"
                except:
                    pass

            # Aplicar búsqueda
            filtered_data = homogeneous_data
            if search_value:
                search_lower = search_value.lower()
                filtered_data = [
                    group for group in homogeneous_data if search_lower in group.get("homogeneous_name", "").lower()
                ]

            # Aplicar ordenación
            column = sort_state.get("column", "savings_base")
            direction = sort_state.get("direction", "desc")
            reverse = direction == "desc"

            # Mapeo de columnas a campos
            column_map = {
                "name": "homogeneous_name",
                "total_revenue": "total_revenue",
                "total_units": "total_units",
                "partner_penetration": "partner_penetration",
                "savings_base": "savings_base",
            }

            sort_field = column_map.get(column, "savings_base")
            filtered_data = sorted(filtered_data, key=lambda x: x.get(sort_field, 0), reverse=reverse)

            # Vista matriz PowerBI con descuento aplicado
            matrix_content = create_powerbi_matrix(
                filtered_data, expand_all_clicks, discount_percentage or 0, sort_state
            )

            # Crear resumen con ahorro CON descuento aplicado (coherente con panel)
            total_groups = len(filtered_data)
            if analysis_data and "opportunity_metrics" in analysis_data:
                total_savings = analysis_data["opportunity_metrics"].get("estimated_savings_with_discount", 0)
            else:
                total_savings = sum(group.get("savings_base", 0) for group in filtered_data)

            total_units = sum(group.get("total_units", 0) for group in filtered_data)

            summary = f"📊 {total_groups} conjuntos • 💰 {format_currency(total_savings)} ahorro con descuento • 📦 {format_compact_number(total_units)} unidades"

            return matrix_content, summary, sort_state

        except Exception as e:
            logger.error(f"Error actualizando matriz homogéneos: {str(e)}")
            return create_alert(f"Error: {str(e)}", "danger"), "", no_update

    # ============================================================================
    # 8. CALLBACK TABLA REFERENCIAS PARTNERS
    # ============================================================================

    @app.callback(
        [
            Output("partner-references-container", "children"),
            Output("partner-refs-count", "children"),
            Output("partner-refs-sales", "children"),
        ],
        [
            Input("partners-selection-store", "data"),
            Input("partner-refs-search", "value"),
            Input("context-store", "data"),
        ],
        [State("laboratory-cache-store", "data"), State("auth-state", "data"), State("url", "pathname")],
        prevent_initial_call=False,
    )
    def update_partner_references_table(partners_store, search_value, context_data, codes_cache, auth_state, pathname):
        """
        Actualizar tabla de referencias de productos partners.

        BD-FIRST PATTERN: Lee partners SIEMPRE desde BD (ignora partners_store).

        Issue #403: Verificación proactiva de autenticación antes de llamadas API.
        Issue #484: Añadido pathname guard para evitar ejecución en otras páginas.
        """
        # Guard: Solo ejecutar en /generics
        if pathname and pathname.lower() not in ["/generics", "/genericos"]:
            raise PreventUpdate

        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("[update_partner_references_table] User not authenticated - skipping API calls")
            raise PreventUpdate

        # Issue #438: Validate pharmacy_id matches current user
        try:
            current_pharmacy_id = str(get_current_pharmacy_id())
            cached_pharmacy_id = context_data.get("_pharmacy_id") if context_data else None
            if cached_pharmacy_id and cached_pharmacy_id != current_pharmacy_id:
                logger.warning("[update_partner_references_table] Stale cache detected - waiting")
                raise PreventUpdate
        except ValueError:
            raise PreventUpdate

        # FIX STALE CACHE: Si contexto indica "sin datos", mostrar mensaje vacío
        if context_data and context_data.get("no_data"):
            logger.info("[update_partner_references_table] Context indica no_data=True - mostrando vacío")
            return (
                create_alert("No hay datos de ventas. Sube tu archivo ERP para ver referencias.", "info"),
                "0",
                format_currency(0)
            )

        # Guard: Si contexto no está disponible, esperar
        if not context_data or context_data.get("error"):
            raise PreventUpdate

        codes_cache = codes_cache or {}

        try:
            trigger = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else "initial_load"
            logger.info(f"[PARTNER_REFS] Callback ejecutado - trigger: {trigger}, search: {search_value}")

            # BD-FIRST PATTERN: Leer partners SIEMPRE desde BD (ignorar store)
            logger.info(
                "[PARTNER_REFS] IGNORING partners_store (may be stale/None)\n"
                "   - Reading from DB as single source of truth..."
            )

            result = get_selected_partners_from_db()

            if not result["success"]:
                logger.error(
                    f"[PARTNER_REFS] Failed reading partners from DB\n"
                    f"   - Error: {result['error_message']}"
                )
                return create_alert("Error cargando selección de partners", "danger"), "0", format_currency(0)

            selected_partners = result["partners"]

            logger.info(
                f"[PARTNER_REFS] Partners loaded from DB:\n"
                f"   - Total selected: {len(selected_partners)}\n"
                f"   - Partners: {selected_partners[:3]}{'...' if len(selected_partners) > 3 else ''}\n"
                f"   - Source: {result['source'].upper()} (BD-first pattern with caching)\n"
                f"   - Store present but ignored: {partners_store is not None} (prevents race conditions)"
            )

            # Edge case: Si no hay partners seleccionados, mostrar mensaje informativo
            if len(selected_partners) == 0:
                logger.info("[PARTNER_REFS] Lista de partners vacía (0 seleccionados)")
                return create_alert("Selecciona al menos un partner para ver referencias", "info"), "0", format_currency(0)

            # Llamar al endpoint de referencias partners
            try:
                try:
                    pharmacy_id = get_current_pharmacy_id()
                    logger.info(f"[PARTNER_REFS] Pharmacy ID obtenido: {pharmacy_id}")
                except ValueError as e:
                    logger.error(f"[PARTNER_REFS] Error obteniendo pharmacy_id: {str(e)}")
                    return create_alert("Error obteniendo farmacia del usuario", "danger"), "0", format_currency(0)

                # Convertir nombres a códigos (P1 Fix - Cache-first)
                logger.info(f"[PARTNER_REFS] Intentando convertir {len(selected_partners)} partners a códigos")
                logger.info(f"[PARTNER_REFS] Partners input: {selected_partners[:3]}{'...' if len(selected_partners) > 3 else ''}")
                logger.info(f"[PARTNER_REFS] Cache actual tiene {len(codes_cache) if codes_cache else 0} entradas")

                selected_partner_codes, codes_cache = get_codes_with_cache(selected_partners, codes_cache)

                logger.info(
                    f"[PARTNER_REFS] Códigos convertidos: {selected_partner_codes} (total: {len(selected_partner_codes) if selected_partner_codes else 0})"
                )

                if not selected_partner_codes:
                    logger.error("[PARTNER_REFS] FALLO conversión partners a códigos")
                    logger.error(f"[PARTNER_REFS]    Partners input: {selected_partners}")
                    logger.error(f"[PARTNER_REFS]    Cache size: {len(codes_cache) if codes_cache else 0}")
                    if selected_partners and isinstance(selected_partners, list) and len(selected_partners) > 0:
                        logger.error(f"[PARTNER_REFS]    Tipo partners[0]: {type(selected_partners[0])}")
                        logger.error(f"[PARTNER_REFS]    Valor partners[0]: '{selected_partners[0]}'")
                    return create_alert(
                        "No se pudo procesar la selección de laboratorios. "
                        "Por favor, intente refrescar la página o contacte soporte.",
                        "danger"
                    ), "0", format_currency(0)

                logger.info(
                    f"[PARTNER_REFS] Llamando API /partner-products/{pharmacy_id} con codes: {selected_partner_codes}"
                )

                response = api_client.get(
                    f"/api/v1/analysis/partner-products/{pharmacy_id}",
                    params={"partner_codes": selected_partner_codes, "period_months": 12},
                )

                logger.info(f"[PARTNER_REFS] API response type: {type(response)}")
                if response:
                    logger.info(f"[PARTNER_REFS] API response keys: {list(response.keys())}")
                else:
                    logger.warning("[PARTNER_REFS] API response es None o vacío")

                if not response:
                    logger.error("[PARTNER_REFS] Response vacío desde API")
                    return create_alert("Error cargando referencias", "danger"), "0", format_currency(0)

                partner_refs = response.get("products", [])
                logger.info(f"[PARTNER_REFS] Productos obtenidos: {len(partner_refs)}")

                if partner_refs:
                    partner_labs = {ref.get("laboratory", "N/A") for ref in partner_refs}
                    logger.info(f"[PARTNER_REFS] Laboratorios únicos en respuesta: {partner_labs}")
                    logger.info(f"[PARTNER_REFS] Laboratorios esperados (seleccionados): {set(selected_partners)}")

                    unexpected_labs = partner_labs - set(selected_partners)
                    if unexpected_labs:
                        logger.warning(f"[PARTNER_REFS] Laboratorios inesperados en respuesta: {unexpected_labs}")
                    else:
                        logger.info("[PARTNER_REFS] Filtrado correcto - solo partners seleccionados")

                    logger.info(f"[PARTNER_REFS] Primer producto: {partner_refs[0]}")
                else:
                    logger.warning("[PARTNER_REFS] Lista de productos vacía (API retornó 0 productos)")

            except requests.exceptions.ConnectionError as e:
                logger.error(f"Error de conexión con API referencias: {str(e)}")
                return (
                    create_alert("Error de conexión con el servidor. Verifique que el backend esté activo.", "danger"),
                    "0",
                    format_currency(0),
                )
            except requests.exceptions.Timeout as e:
                logger.error(f"Timeout en endpoint referencias: {str(e)}")
                return (
                    create_alert("El servidor tardó demasiado en responder. Intente nuevamente.", "warning"),
                    "0",
                    format_currency(0),
                )
            except requests.exceptions.HTTPError as e:
                logger.error(f"Error HTTP en endpoint referencias: {str(e)}")
                return create_alert(f"Error del servidor: {e.response.status_code}", "danger"), "0", format_currency(0)
            except Exception as e:
                logger.error(f"Error inesperado llamando endpoint referencias: {str(e)}")
                return create_alert(f"Error inesperado: {str(e)}", "danger"), "0", format_currency(0)

            if not partner_refs:
                return create_alert("No hay referencias de partners disponibles", "warning"), "0", format_currency(0)

            # Aplicar filtro de búsqueda
            if search_value:
                search_lower = search_value.lower()
                partner_refs = [
                    ref
                    for ref in partner_refs
                    if (
                        search_lower in ref.get("national_code", "").lower()
                        or search_lower in ref.get("product_name", "").lower()
                        or search_lower in ref.get("laboratory", "").lower()
                    )
                ]

            # Crear tabla de referencias
            refs_table = create_partner_refs_table(partner_refs)

            # Calcular totales
            total_refs = len(partner_refs)
            total_sales = sum(ref.get("total_sales", 0) for ref in partner_refs)

            return refs_table, format_number(total_refs), format_currency(total_sales)

        except Exception as e:
            logger.error(f"Error actualizando referencias partners: {str(e)}")
            return create_alert(f"Error: {str(e)}", "danger"), "Error", "Error"

    # ============================================================================
    # 9. CALLBACK UNIFICADO EXPANDIR/CONTRAER CONJUNTOS HOMOGÉNEOS
    # FIX ISSUE #249: Combinar 3 callbacks en 1 unificado (REGLA #11)
    # ============================================================================

    @app.callback(
        [
            Output({"type": "detail-row", "index": ALL}, "style"),
            Output({"type": "expand-icon", "index": ALL}, "className"),
            Output({"type": "detail-content", "code": ALL}, "children"),
            Output("expand-all-homogeneous-btn", "children"),
            Output("homogeneous-expansion-state", "data"),
        ],
        [Input({"type": "expand-btn", "index": ALL}, "n_clicks"), Input("expand-all-homogeneous-btn", "n_clicks")],
        [
            State({"type": "detail-row", "index": ALL}, "style"),
            State({"type": "expand-icon", "index": ALL}, "className"),
            State({"type": "detail-content", "code": ALL}, "id"),
            State({"type": "detail-content", "code": ALL}, "children"),
            State("partners-selection-store", "data"),
            State("partner-discount-slider", "value"),
            State("homogeneous-expansion-state", "data"),
            State("analysis-store", "data"),
            State("laboratory-cache-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_homogeneous_expansion(
        expand_clicks: List[Optional[int]],
        expand_all_clicks: Optional[int],
        current_styles: List[Dict[str, str]],
        current_icons: List[str],
        detail_ids: List[Dict[str, str]],
        current_children: List[Any],
        partners_store: Optional[Dict[str, Any]],
        discount_percentage: Optional[float],
        expansion_state: Optional[Dict[str, Any]],
        analysis_data: Optional[Dict[str, Any]],
        codes_cache: Optional[Dict[str, str]],
    ) -> Tuple[List[Dict[str, str]], List[str], List[Any], List[Any], Dict[str, Any]]:
        """
        CALLBACK UNIFICADO (Issue #249) - REGLA #11: ONE INPUT ONE CALLBACK

        Maneja toda la lógica de expansión/contracción de conjuntos homogéneos:
        1. Toggle de estilos (display: block/none)
        2. Cambio de iconos (+/- square)
        3. Carga de datos cuando se expande
        4. Actualización del botón "Expandir Todo" / "Contraer Todo"
        5. Persistencia del estado en store
        """
        # FIX: Inicializar codes_cache si es None
        if codes_cache is None:
            codes_cache = {}

        try:
            logger.info("[ACCORDION_DEBUG] ====== CALLBACK TRIGGERED ======")
            logger.info(f"[ACCORDION_DEBUG] ctx.triggered: {ctx.triggered}")
            logger.info(f"[ACCORDION_DEBUG] expand_clicks: {expand_clicks}")
            logger.info(f"[ACCORDION_DEBUG] expand_all_clicks: {expand_all_clicks}")

            if not ctx.triggered:
                logger.info("[ACCORDION_DEBUG] NO ctx.triggered - PreventUpdate")
                raise PreventUpdate

            triggered_id = ctx.triggered[0]["prop_id"].split(".")[0]
            logger.info(f"[ACCORDION_DEBUG] triggered_id (raw): {triggered_id}")

            # Inicializar estado si no existe
            if not expansion_state:
                expansion_state = {"all_expanded": False, "individual_states": {}}

            # Crear placeholder para resetear contenido
            def create_placeholder():
                return [
                    dcc.Loading(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-info-circle me-2 text-primary"),
                                    "Haz click en el botón (+) para cargar el detalle de productos...",
                                ],
                                className="text-muted text-center py-3",
                            )
                        ],
                        type="default",
                    )
                ]

            # ===================================================================
            # CASO 1: Click en "Expandir Todo"
            # ===================================================================
            if triggered_id == "expand-all-homogeneous-btn":
                any_expanded = any(style.get("display") == "block" for style in current_styles)

                if any_expanded:
                    # ====== CONTRAER TODAS ======
                    new_styles = [{"display": "none", "marginLeft": "0", "marginRight": "0"} for _ in current_styles]
                    new_icons = [ICON_COLLAPSED for _ in current_icons]
                    new_content = [create_placeholder() for _ in detail_ids]

                    button_content = create_expand_button_content(all_expanded=False)
                    expansion_state = {"all_expanded": False, "individual_states": {}}

                    return new_styles, new_icons, new_content, button_content, expansion_state
                else:
                    # ====== EXPANDIR TODAS ======
                    new_styles = [{"display": "block", "marginLeft": "0", "marginRight": "0"} for _ in current_styles]
                    new_icons = [ICON_EXPANDED for _ in current_icons]

                    new_content = []

                    # BD-FIRST PATTERN: Leer partners desde BD (ignorar store)
                    logger.info("[ACCORDION_EXPAND_ALL] Reading partners from DB (BD-first pattern)")
                    result = get_selected_partners_from_db()

                    if not result["success"]:
                        logger.error(
                            f"[ACCORDION_EXPAND_ALL] ERROR fetching partners from DB\n"
                            f"   - Error: {result.get('error', 'Unknown')}"
                        )
                        new_content = [no_update for _ in detail_ids]
                    else:
                        selected_partners = result["partners"]
                        logger.info(
                            f"[ACCORDION_EXPAND_ALL] Partners loaded from DB\n"
                            f"   - Count: {len(selected_partners)}\n"
                            f"   - Source: {result['source'].upper()}"
                        )

                        try:
                            pharmacy_id = get_current_pharmacy_id()
                        except ValueError as e:
                            logger.error(f"Error obteniendo pharmacy_id: {str(e)}")
                            new_content = [create_alert("Error obteniendo farmacia", "danger") for _ in detail_ids]

                        for idx, (detail_id, current_child) in enumerate(zip(detail_ids, current_children)):
                            if has_placeholder(current_child):
                                homogeneous_code = detail_id["code"]

                                # PRIORIDAD 1: Intentar desde analysis-store (rápido, datos actuales)
                                detail_data = get_homogeneous_detail_from_store(analysis_data, homogeneous_code)

                                if detail_data and detail_data.get("products_detail"):
                                    products_detail = detail_data.get("products_detail")
                                    group_summary = detail_data.get("group_summary", {})
                                    logger.info(f"{homogeneous_code}: Datos desde analysis-store (sin API)")

                                    if products_detail:
                                        detail_content = create_products_detail_table(
                                            products_detail, group_summary, discount_percentage or 0
                                        )
                                        new_content.append(detail_content)
                                    else:
                                        new_content.append(
                                            create_alert("No hay productos en este conjunto", "info")
                                        )
                                else:
                                    # FALLBACK: Datos NO en store, llamar API
                                    logger.warning(f"{homogeneous_code}: NO en store, llamando API...")
                                    try:
                                        selected_partner_codes, codes_cache = get_codes_with_cache(selected_partners, codes_cache)
                                        if not selected_partner_codes:
                                            logger.error(f"Error convirtiendo partners: {selected_partners}")
                                            new_content.append(
                                                create_alert("Error al procesar selección de partners", "danger")
                                            )
                                            continue

                                        response = api_client.get(
                                            f"/api/v1/analysis/homogeneous-detail/{pharmacy_id}/{homogeneous_code}",
                                            params={"partner_codes": selected_partner_codes, "period_months": 12},
                                        )

                                        if response:
                                            products_detail = response.get("products_detail", [])
                                            group_summary = response.get("group_summary", {})

                                            if products_detail:
                                                detail_content = create_products_detail_table(
                                                    products_detail, group_summary, discount_percentage or 0
                                                )
                                                new_content.append(detail_content)
                                            else:
                                                new_content.append(
                                                    create_alert("No hay productos en este conjunto", "info")
                                                )
                                        else:
                                            new_content.append(create_alert("Error cargando detalle", "danger"))
                                    except Exception as e:
                                        logger.error(f"Error cargando detalle {homogeneous_code}: {str(e)}")
                                        new_content.append(create_alert(f"Error: {str(e)}", "danger"))
                            else:
                                new_content.append(no_update)

                    button_content = create_expand_button_content(all_expanded=True)

                    expansion_state = {
                        "all_expanded": True,
                        "individual_states": {i: True for i in range(len(current_styles))},
                    }

                    return new_styles, new_icons, new_content, button_content, expansion_state

            # ===================================================================
            # CASO 2: Click individual en botón +/-
            # ===================================================================
            valid_clicks = [c for c in expand_clicks if c is not None and c > 0]
            logger.info(f"[ACCORDION_DEBUG] valid_clicks: {valid_clicks} (total: {len(valid_clicks)})")

            if not valid_clicks:
                logger.info("[ACCORDION_DEBUG] No valid clicks - PreventUpdate")
                raise PreventUpdate

            # Encontrar qué botón se clickeó
            clicked_index = None
            try:
                logger.info(f"[ACCORDION_DEBUG] Parsing triggered_id: {triggered_id}")
                triggered_json = json.loads(triggered_id)
                logger.info(f"[ACCORDION_DEBUG] triggered_json parsed: {triggered_json}")
                clicked_index = triggered_json.get("index")
                logger.info(f"[ACCORDION_DEBUG] clicked_index extracted: {clicked_index}")
            except Exception as e:
                logger.error(f"[ACCORDION_DEBUG] Error parsing triggered_id: {str(e)}")
                raise PreventUpdate

            if clicked_index is None:
                raise PreventUpdate

            # Preparar arrays de salida
            new_styles = list(current_styles)
            new_icons = list(current_icons)
            new_content = [no_update] * len(detail_ids)

            # Toggle de la fila clickeada
            if current_styles[clicked_index].get("display") == "none":
                # ====== EXPANDIENDO ======
                new_styles[clicked_index] = {"display": "block", "marginLeft": "0", "marginRight": "0"}
                new_icons[clicked_index] = ICON_EXPANDED

                # BD-FIRST PATTERN: Leer partners desde BD (ignorar store)
                logger.info("[ACCORDION_INDIVIDUAL] Reading partners from DB (BD-first pattern)")
                result = get_selected_partners_from_db()

                if not result["success"]:
                    logger.error(
                        f"[ACCORDION_INDIVIDUAL] ERROR fetching partners from DB\n"
                        f"   - Error: {result.get('error', 'Unknown')}"
                    )
                    new_content[clicked_index] = create_alert("Error cargando partners", "danger")
                else:
                    selected_partners = result["partners"]
                    logger.info(
                        f"[ACCORDION_INDIVIDUAL] Partners loaded from DB\n"
                        f"   - Count: {len(selected_partners)}\n"
                        f"   - Source: {result['source'].upper()}"
                    )
                    homogeneous_code = detail_ids[clicked_index]["code"]

                    # PRIORIDAD 1: Intentar desde analysis-store
                    detail_data = get_homogeneous_detail_from_store(analysis_data, homogeneous_code)

                    if detail_data and detail_data.get("products_detail"):
                        products_detail = detail_data.get("products_detail")
                        group_summary = detail_data.get("group_summary", {})
                        logger.info(f"{homogeneous_code}: {len(products_detail)} productos desde analysis-store (sin API)")

                        new_content[clicked_index] = create_products_detail_table(
                            products_detail, group_summary, discount_percentage or 0
                        )
                    else:
                        # FALLBACK: Datos NO en store, llamar API
                        logger.warning(f"{homogeneous_code}: NO en store, llamando API...")
                        try:
                            try:
                                pharmacy_id = get_current_pharmacy_id()
                            except ValueError as e:
                                logger.error(f"Error obteniendo pharmacy_id: {str(e)}")
                                new_content[clicked_index] = create_alert("Error obteniendo farmacia", "danger")

                            selected_partner_codes, codes_cache = get_codes_with_cache(selected_partners, codes_cache)
                            if not selected_partner_codes:
                                logger.error(f"Error convirtiendo partners: {selected_partners}")
                                new_content[clicked_index] = create_alert(
                                    "Error al procesar selección de partners", "danger"
                                )
                            else:
                                response = api_client.get(
                                    f"/api/v1/analysis/homogeneous-detail/{pharmacy_id}/{homogeneous_code}",
                                    params={"partner_codes": selected_partner_codes, "period_months": 12},
                                )

                                if response:
                                    products_detail = response.get("products_detail", [])
                                    group_summary = response.get("group_summary", {})

                                    if products_detail:
                                        new_content[clicked_index] = create_products_detail_table(
                                            products_detail, group_summary, discount_percentage or 0
                                        )
                                    else:
                                        new_content[clicked_index] = create_alert(
                                            "No hay productos en este conjunto", "warning"
                                        )
                                else:
                                    new_content[clicked_index] = create_alert("Error cargando detalle", "danger")
                        except Exception as e:
                            logger.error(f"Error cargando detalle: {str(e)}")
                            new_content[clicked_index] = create_alert(f"Error: {str(e)}", "danger")

                # Actualizar estado individual
                expansion_state["individual_states"][clicked_index] = True
            else:
                # ====== CONTRAYENDO ======
                new_styles[clicked_index] = {"display": "none", "marginLeft": "0", "marginRight": "0"}
                new_icons[clicked_index] = ICON_COLLAPSED
                new_content[clicked_index] = create_placeholder()

                expansion_state["individual_states"][clicked_index] = False
                expansion_state["all_expanded"] = False

            # Determinar estado del botón global
            any_expanded_now = any(style.get("display") == "block" for style in new_styles)
            button_content = create_expand_button_content(all_expanded=any_expanded_now)

            return new_styles, new_icons, new_content, button_content, expansion_state

        except PreventUpdate:
            raise
        except Exception as e:
            logger.error(f"Error en callback unificado de expansión: {str(e)}")
            raise PreventUpdate

    # ============================================================================
    # 10. CALLBACK SIMULACIÓN DESCUENTOS
    # ============================================================================

    @app.callback(
        Output("discount-additional-savings", "children"),
        Input("analysis-store", "data"),
        State("url", "pathname"),
        prevent_initial_call=False
    )
    def update_discount_simulation(analysis_data, pathname):
        """
        Actualizar simulación de ahorros adicionales con descuentos.

        FIX REGLA #11: Only listens to analysis-store (which already contains discount-aware data).
        Issue #484: Añadido pathname guard para evitar ejecución en otras páginas.
        """
        # Guard: Solo ejecutar en /generics
        if pathname and pathname.lower() not in ["/generics", "/genericos"]:
            return no_update

        try:
            if not analysis_data or "error" in analysis_data:
                logger.debug("[DISCOUNT] No analysis_data o contiene error")
                return "--"

            opportunity_metrics = analysis_data.get("opportunity_metrics", {})
            estimated_savings = opportunity_metrics.get("estimated_savings_with_discount", 0)

            logger.debug(f"[DISCOUNT] opportunity_metrics keys: {list(opportunity_metrics.keys())}")
            logger.debug(f"[DISCOUNT] estimated_savings_with_discount: {estimated_savings}")
            logger.debug(f"[DISCOUNT] opportunity_revenue: {opportunity_metrics.get('opportunity_revenue', 0)}")
            logger.debug(f"[DISCOUNT] potential_savings_base: {opportunity_metrics.get('potential_savings_base', 0)}")

            # TODO(#339): Remove this fallback once backend correctly calculates
            if estimated_savings == 0:
                potential_base = opportunity_metrics.get("potential_savings_base", 0)
                if potential_base > 0:
                    logger.warning(
                        f"[DISCOUNT] Backend didn't calculate discount, using fallback. "
                        f"potential_base={potential_base}. This should be fixed in backend. See issue #339"
                    )
                    estimated_savings = potential_base * DEFAULT_DISCOUNT_RATE
                    logger.info(f"[DISCOUNT] Using fallback rate {DEFAULT_DISCOUNT_RATE*100}%: {estimated_savings}")

            return format_currency(estimated_savings)

        except Exception as e:
            logger.error(f"Error en simulación descuentos: {str(e)}")
            return "Error"

    # ============================================================================
    # 11. CALLBACK EXPORTAR MATRIZ A CSV
    # ============================================================================

    @app.callback(
        Output("download-homogeneous-csv", "data"),
        [Input("export-homogeneous-btn", "n_clicks")],
        [State("analysis-store", "data")],
        prevent_initial_call=True,
    )
    def export_homogeneous_matrix_to_csv(n_clicks, analysis_data):
        """
        Exportar matriz de conjuntos homogéneos a CSV.
        """
        try:
            if not n_clicks or not analysis_data or "error" in analysis_data:
                raise PreventUpdate

            homogeneous_data = analysis_data.get("homogeneous_groups_detail", [])
            if not homogeneous_data:
                logger.warning("No hay datos para exportar")
                raise PreventUpdate

            # Convertir a DataFrame con tipos explícitos
            df = pd.DataFrame(
                [
                    {
                        "Conjunto Homogéneo": str(group.get("homogeneous_name", "")),
                        "Código": str(group.get("homogeneous_code", "")),
                        "Unidades Totales": int(group.get("total_units", 0)),
                        "Ventas Totales (€)": float(group.get("total_revenue", 0)),
                        "Unidades Partners": int(group.get("partner_units", 0)),
                        "Unidades Oportunidad": int(group.get("opportunity_units", 0)),
                        "Ahorro Base (€)": float(group.get("savings_base", 0)),
                        "Penetración Partners (%)": float(group.get("partner_penetration", 0)),
                    }
                    for group in homogeneous_data
                ]
            )

            # Generar CSV con formato español correcto
            csv_string = df.to_csv(
                index=False,
                encoding="utf-8-sig",
                sep=";",
                decimal=",",
                float_format="%.2f",
            )

            # Crear nombre de archivo con timestamp
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            filename = f"conjuntos_homogeneos_partners_{timestamp}.csv"

            logger.info(f"Exportando {len(homogeneous_data)} conjuntos a CSV: {filename}")

            return dict(content=csv_string, filename=filename)

        except PreventUpdate:
            raise
        except Exception as e:
            logger.error(f"Error exportando matriz: {str(e)}")
            raise PreventUpdate
