"""
Data loading callbacks for Generics Panel.

Handles:
- Interval control (enable/disable based on page)
- Employee options loading
- Substitutable universe context loading
"""

import logging

import requests
import requests.exceptions
from dash import Input, Output, State
from dash.exceptions import PreventUpdate

from styles.design_tokens import COLORS as DS_COLORS
from utils.api_client import api_client
from utils.constants import PAGE_PATHS
from utils.helpers import format_number
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)

# Style for disabled controls (FREE tier)
DISABLED_CONTROL_STYLE = {
    "opacity": "0.6",
    "cursor": "not-allowed",
    "backgroundColor": DS_COLORS["bg_secondary"],
}


def register_data_loading_callbacks(app):
    """
    Register data loading callbacks for the Generics Panel.

    Args:
        app: Dash application instance
    """

    # ============================================================================
    # 0. CALLBACK CONTROL INTERVAL - Habilitar/deshabilitar según página
    # ============================================================================
    @app.callback(
        Output("context-refresh-interval", "disabled"),
        Input("url", "pathname"),
        prevent_initial_call=False,
    )
    def control_context_interval(pathname):
        """
        Controlar estado del interval según la página actual.
        Solo habilitado en /generics para prevenir ejecución en otras páginas.

        Issue #303 - Mejora #6: Usar constante PAGE_PATHS para pathname.
        Issue #303 - Mejora #7: Documentación de edge cases.

        Edge Cases Manejados:
        ---------------------
        1. Navegación Rápida entre Páginas:
           El callback puede recibir múltiples actualizaciones consecutivas cuando
           el usuario navega rápidamente. El diseño actual maneja esto correctamente
           ya que el interval se deshabilita inmediatamente al salir de /generics,
           previniendo ejecuciones innecesarias.

        2. Carga Inicial:
           El interval se crea con disabled=True por defecto en el layout y solo
           se habilita cuando pathname="/generics". Esto previene ejecución prematura
           antes de que el usuario navegue a la página correcta.

        3. Pathname None o Vacío:
           Si pathname es None o '', el callback lo trata como "no /generics" y
           deshabilita el interval (retorna True). Esto es seguro y previene
           ejecución en estados indeterminados.

        4. Cambio de Pathname Durante Ejecución del Interval:
           Si el usuario cambia de página mientras el interval está ejecutándose,
           el callback load_substitutable_universe_context verifica pathname y
           hace PreventUpdate si no está en /generics, evitando procesamiento innecesario.

        Returns:
            bool: False (habilitar interval) si pathname="/generics",
                  True (deshabilitar interval) en cualquier otro caso
        """
        if pathname == PAGE_PATHS["generics"]:
            logger.debug("Habilitando context-refresh-interval en /generics")
            return False  # Habilitar
        else:
            logger.debug(f"Deshabilitando context-refresh-interval fuera de /generics (pathname: {pathname})")
            return True  # Deshabilitar

    # ============================================================================
    # 0.5 CALLBACK CARGAR EMPLEADOS - Issue #402: Employee filtering integration
    # ============================================================================
    @app.callback(
        [
            Output("generics-employee-filter", "options"),
            Output("generics-employee-filter", "disabled"),
            Output("generics-employee-filter", "style"),
            Output("generics-employee-filter", "placeholder"),
            Output("generics-employee-filter-badge", "style"),
        ],
        [Input("url", "pathname"), Input("auth-ready", "data")],
        [State("auth-state", "data")],
        prevent_initial_call=False,
    )
    def load_employee_options(pathname, auth_ready, auth_state):
        """
        Cargar opciones del dropdown de empleados para el filtro.

        Issue #402: Employee filtering integration - Las opciones se cargan
        dinámicamente desde /api/v1/employees/{pharmacy_id}.
        Issue #435: FREE tier UX - Deshabilitar dropdown con placeholder PRO.

        Incluye valor especial "__sin_empleado__" para ventas sin empleado asignado.

        PRO Gating: El endpoint requiere plan PRO+. Si el usuario es FREE,
        el dropdown queda deshabilitado con placeholder indicativo.
        """
        # Guard: Solo ejecutar en /generics
        if pathname != PAGE_PATHS["generics"]:
            raise PreventUpdate

        # Guard: Esperar auth-ready
        if not auth_ready:
            logger.debug("[load_employee_options] Waiting for auth-ready")
            raise PreventUpdate

        # Guard: Verificar autenticación
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("[load_employee_options] User not authenticated")
            raise PreventUpdate

        # Guard: Verificar tier ANTES de llamar API (evita 403 innecesarios)
        user_data = auth_state.get("user", {})
        subscription_plan = user_data.get("subscription_plan", "").lower()

        # Issue #435: FREE tier - Deshabilitar dropdown con placeholder PRO
        BADGE_VISIBLE_STYLE = {"fontSize": "0.7rem", "opacity": "0.8", "display": "inline-block"}
        BADGE_HIDDEN_STYLE = {"display": "none"}

        if subscription_plan == "free":
            logger.debug("[load_employee_options] FREE tier - disabling employee filter")
            return [], True, DISABLED_CONTROL_STYLE, "👑 Disponible en plan PRO", BADGE_VISIBLE_STYLE

        # Usuario PRO/MAX: Cargar empleados desde API
        try:
            pharmacy_id = get_current_pharmacy_id()
            logger.info(f"[load_employee_options] Loading employees for pharmacy {pharmacy_id}")

            response = api_client.get(f"/api/v1/employees/{pharmacy_id}")

            if not response:
                logger.warning("[load_employee_options] Empty response from employees API")
                return [], False, {}, "Todos los empleados", BADGE_HIDDEN_STYLE

            employees = response.get("employees", [])

            # Construir opciones para el dropdown
            options = []

            # Opción especial para ventas sin empleado
            options.append({"label": "🚫 Sin empleado asignado", "value": "__sin_empleado__"})

            # Agregar cada empleado con su contador de ventas
            for emp in employees:
                name = emp.get("name", "Desconocido")
                sales_count = emp.get("sales_count", 0)
                options.append({"label": f"{name} ({format_number(sales_count)} ventas)", "value": name})

            logger.info(f"[load_employee_options] Loaded {len(options)} employee options")
            return options, False, {}, "Todos los empleados", BADGE_HIDDEN_STYLE

        except requests.exceptions.HTTPError as e:
            # 403 = Usuario FREE sin acceso a esta función PRO (fallback si guard falla)
            if e.response is not None and e.response.status_code == 403:
                logger.info("[load_employee_options] User is FREE tier (403) - employee filter disabled")
                return [], True, DISABLED_CONTROL_STYLE, "👑 Disponible en plan PRO", BADGE_VISIBLE_STYLE
            logger.error(f"[load_employee_options] HTTP error: {e}")
            return [], False, {}, "Todos los empleados", BADGE_HIDDEN_STYLE
        except Exception as e:
            logger.error(f"[load_employee_options] Error loading employees: {e}")
            return [], False, {}, "Todos los empleados", BADGE_HIDDEN_STYLE

    # ============================================================================
    # 1. CALLBACK CONTEXTO FIJO - Se carga UNA vez al inicio
    # ============================================================================
    @app.callback(
        Output("context-store", "data"),
        [
            Input("url", "pathname"),
            Input("context-refresh-interval", "n_intervals"),
            Input("auth-ready", "data"),
            Input("generics-apply-btn", "n_clicks"),
            Input("generics-date-range-store", "data"),
        ],
        [
            State("auth-state", "data"),
            State("generics-date-range", "start_date"),
            State("generics-date-range", "end_date"),
        ],
        prevent_initial_call=False,
    )
    def load_substitutable_universe_context(
        pathname, n_intervals, auth_ready, apply_clicks, date_range_store, auth_state, start_date, end_date
    ):
        """
        Cargar contexto del universo sustituible con filtros de fecha.
        Se ejecuta cuando navegamos a /generics, cada 15 minutos, cuando auth sincroniza,
        cuando el usuario aplica filtros de fecha, o cuando las fechas se inicializan.

        FIX RACE CONDITION: Usar pathname como Input (no State) para disparar
        cuando el usuario navega a /generics.
        Issue #303 - Mejora #6: Usar constante PAGE_PATHS para pathname.
        Issue #309 - Auth Protection: Verificar autenticación ANTES de cargar datos.
        Issue #344 - Auth Ready: Esperar señal de auth-ready ANTES de cargar datos.
        FIX RENDER EMPTY CONTEXT: auth-ready como Input (no State) dispara callback cuando auth sincroniza.
        FIX DISCREPANCIA FECHAS: Esperar a date_range_store.ready antes de llamar API.
        """
        # ✅ ISSUE #344: Verificar auth-ready ANTES de continuar
        if not auth_ready:
            logger.debug(
                f"[load_substitutable_universe_context] auth_ready={auth_ready}, skipping data load (waiting for auth sync)"
            )
            raise PreventUpdate

        # ✅ FIX RACE CONDITION: Solo ejecutar si estamos en la página de generics
        if pathname != PAGE_PATHS["generics"]:
            raise PreventUpdate

        # ✅ ISSUE #309: Verificación proactiva de autenticación ANTES de llamadas API
        from utils.auth_helpers import is_user_authenticated

        logger.debug(
            f"[load_substitutable_universe_context] "
            f"auth_ready={auth_ready}, "
            f"auth_state={auth_state}, "
            f"auth_state.authenticated={auth_state.get('authenticated') if auth_state else None}, "
            f"auth_state.user={auth_state.get('user') if auth_state else None}, "
            f"is_authenticated={is_user_authenticated(auth_state)}"
        )

        if not is_user_authenticated(auth_state):
            logger.debug("[load_substitutable_universe_context] User not authenticated - PreventUpdate")
            raise PreventUpdate

        # ✅ FIX DISCREPANCIA FECHAS: Esperar a que date_range_store tenga fechas listas
        if not date_range_store or not date_range_store.get("ready"):
            logger.debug(
                f"[load_substitutable_universe_context] date_range_store not ready, waiting for dates to be initialized"
            )
            raise PreventUpdate

        logger.debug(
            f"[load_substitutable_universe_context] ✅ All checks passed - proceeding with data load (interval: {n_intervals})"
        )

        try:
            logger.info(f"Cargando contexto de universo sustituible (interval: {n_intervals})")

            # Obtener pharmacy_id dinámicamente del usuario autenticado
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError as e:
                logger.error(f"Error obteniendo pharmacy_id: {str(e)}")
                return {"error": "No se pudo identificar la farmacia del usuario", "error_type": "auth_error"}

            # ✅ ERROR HANDLING ROBUSTO: Usar requests directamente para capturar status codes específicos
            from utils.auth import auth_manager
            from utils.config import BACKEND_URL

            token = auth_manager.get_access_token()
            headers = {"Authorization": f"Bearer {token}"} if token else {}

            # Issue #478: Pasar fechas directas al backend (endpoint solo acepta start_date/end_date)
            params = {}
            if start_date and end_date:
                # Normalizar formato de fecha a YYYY-MM-DD
                params["start_date"] = start_date[:10] if isinstance(start_date, str) else start_date.strftime("%Y-%m-%d")
                params["end_date"] = end_date[:10] if isinstance(end_date, str) else end_date.strftime("%Y-%m-%d")
                logger.info(
                    f"[load_substitutable_universe_context] Using direct dates: {params['start_date']} to {params['end_date']}"
                )
            else:
                # FIX Issue #478 Review: Calcular fechas cuando no se proporcionan (endpoint no soporta period_months)
                from datetime import datetime, timedelta

                end_dt = datetime.now()
                start_dt = end_dt - timedelta(days=365)  # Últimos 12 meses
                params["start_date"] = start_dt.strftime("%Y-%m-%d")
                params["end_date"] = end_dt.strftime("%Y-%m-%d")
                logger.info(
                    f"[load_substitutable_universe_context] No dates provided, calculated 12-month default: "
                    f"{params['start_date']} to {params['end_date']}"
                )

            # Issue #478 Fase 2: Usar nuevo endpoint basado en medida SubstitutableUniverseSummary
            http_response = requests.get(
                f"{BACKEND_URL}/api/v1/generic-analysis/universe-summary/{pharmacy_id}",
                params=params,
                headers=headers,
                timeout=30,
            )

            # ✅ Capturar error 503 específico (nomenclator no sincronizado)
            if http_response.status_code == 503:
                try:
                    error_detail = http_response.json().get("detail", "Servicio no disponible")
                except Exception:
                    error_detail = "Servicio no disponible temporalmente"
                logger.warning(f"[CONTEXT] Service unavailable (503): {error_detail}")
                return {
                    "error": error_detail,
                    "error_type": "service_unavailable",
                }

            http_response.raise_for_status()
            response = http_response.json()

            # Validación estricta de estructura de respuesta esperada
            if not response:
                logger.error("Error al cargar contexto: respuesta vacía")
                return {
                    "error": "Error al cargar datos. Si el problema persiste, contacta al administrador.",
                    "error_type": "empty_response",
                }

            # Validar estructura mínima esperada
            required_fields = ["universe_summary", "homogeneous_groups", "laboratories_in_universe"]
            missing_fields = [field for field in required_fields if field not in response]

            if missing_fields:
                logger.error(f"Respuesta API incompleta. Campos faltantes: {missing_fields}")
                return {
                    "error": f"Respuesta incompleta del servidor. Campos faltantes: {', '.join(missing_fields)}",
                    "error_type": "incomplete_response",
                    "missing_fields": missing_fields,
                }

            # Issue: Detectar "sin datos" desde la fuente (API)
            universe_summary = response.get("universe_summary", {})
            total_groups = universe_summary.get("total_substitutable_groups", 0)
            total_revenue = universe_summary.get("total_substitutable_revenue", 0)

            if total_groups == 0 and total_revenue == 0:
                logger.info("[CONTEXT] Sin datos de ventas - devolviendo flag no_data para limpiar cache")
                return {"no_data": True, "message": "No hay datos de ventas cargados"}

            logger.info("Contexto de universo sustituible cargado exitosamente")
            # Issue #438: Include pharmacy_id in stored data to prevent stale cache
            response["_pharmacy_id"] = str(pharmacy_id)
            return response

        except requests.exceptions.RequestException as e:
            logger.error(f"Error de conexión al cargar contexto: {str(e)}")
            return {"error": "Error de conexión con el servidor", "error_type": "connection_error", "details": str(e)}
        except requests.exceptions.Timeout as e:
            logger.error(f"Timeout al cargar contexto: {str(e)}")
            return {"error": "Tiempo de espera agotado al cargar datos", "error_type": "timeout_error"}
        except ValueError as e:
            logger.error(f"Error de formato en respuesta: {str(e)}")
            return {"error": "Formato de respuesta inválido", "error_type": "format_error", "details": str(e)}
        except Exception as e:
            logger.error(f"Excepción inesperada al cargar contexto: {str(e)}")
            return {"error": "Error inesperado al cargar datos", "error_type": "unexpected_error", "details": str(e)}
