"""
Context display callbacks for Generics Panel.

Handles:
- Context panel update (substitutable universe metrics)
- Context treemap visualization
"""

import logging
from typing import Any, Dict

import plotly.graph_objects as go
import requests
from dash import Input, Output, State, no_update
from dash.exceptions import PreventUpdate

from components.toast_manager import info_toast
from utils.generics_helpers import get_selected_partners_from_db
from utils.helpers import format_compact_number, format_currency
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)


def register_context_display_callbacks(app):
    """
    Register context display callbacks for the Generics Panel.

    Args:
        app: Dash application instance
    """

    # ============================================================================
    # 2. CALLBACK ACTUALIZACIÓN PANEL CONTEXTO
    # ============================================================================

    @app.callback(
        [
            Output("context-total-substitutable", "children"),
            Output("context-total-percentage", "children"),
            Output("context-analyzable-amount", "children"),
            Output("context-analyzable-percentage", "children"),
            Output("context-current-partners", "children"),
            Output("context-current-percentage", "children"),
            Output("context-opportunity", "children"),
            Output("context-opportunity-percentage", "children"),
            Output("toast-trigger-store", "data", allow_duplicate=True),
        ],
        [Input("context-store", "data"), Input("analysis-store", "data"), Input("partners-selection-store", "data")],
        [State("auth-state", "data"), State("url", "pathname")],
        prevent_initial_call="initial_duplicate",
    )
    def update_context_panel(context_data, analysis_data, partners_store, auth_state, pathname):
        """
        Actualizar el panel de contexto con datos del universo sustituible y análisis dinámico.

        BD-FIRST PATTERN: Lee partners SIEMPRE desde BD (ignora partners_store).

        Notes:
            - partners_store se mantiene como Input para reactividad inmediata
            - PERO su valor se IGNORA, siempre se lee desde BD (fuente de verdad)
            - Garantiza actualización inmediata cuando cambian partners + datos frescos en F5

        Issue #403: Verificación proactiva de autenticación antes de acceso a BD.
        Issue: Toast informativo cuando no hay datos de ventas.
        """
        # Guard 0: Pathname guard - solo ejecutar en /generics
        if pathname and pathname.lower() not in ["/generics", "/genericos"]:
            raise PreventUpdate

        # Guard 1: Verificar datos válidos
        if not context_data:
            return ["--"] * 8 + [no_update]

        # Guard 1.1: Manejar error específico de servicio no disponible (503)
        if context_data.get("error_type") == "service_unavailable":
            error_msg = context_data.get("error", "Servicio no disponible")
            logger.warning(f"[CONTEXT_PANEL] Service unavailable: {error_msg}")
            from components.toast_manager import warning_toast

            toast_data = warning_toast(error_msg, "Catálogo no disponible")
            return [
                "Error",
                "Catálogo pendiente",
                "--",
                "--",
                "--",
                "--",
                "--",
                "--",
                toast_data,
            ]

        # Guard 1.2: Otros errores genéricos
        if "error" in context_data:
            return ["--"] * 8 + [no_update]

        # Guard 1.5: Issue #438 - Validate pharmacy_id matches current user
        try:
            current_pharmacy_id = str(get_current_pharmacy_id())
            cached_pharmacy_id = context_data.get("_pharmacy_id")
            if cached_pharmacy_id and cached_pharmacy_id != current_pharmacy_id:
                logger.warning(
                    f"[CONTEXT_PANEL] Stale cache detected - pharmacy mismatch: "
                    f"cached={cached_pharmacy_id[:8]}... vs current={current_pharmacy_id[:8]}..."
                )
                return ["--"] * 8 + [no_update]
        except ValueError:
            return ["--"] * 8 + [no_update]

        # Guard 2: Manejar "sin datos" - usuario FREE sin ventas cargadas
        if context_data.get("no_data"):
            logger.info("[CONTEXT_PANEL] Sin datos de ventas - mostrando estado vacío con toast")
            toast_data = info_toast(
                "No tienes datos de ventas cargados. Ve a 'Gestión de Datos' para subir tu archivo ERP.",
                "Sin datos de ventas",
            )
            return [
                "Sin datos",
                "Sube tu archivo ERP",
                "--",
                "--",
                "--",
                "--",
                "--",
                "--",
                toast_data,
            ]

        # Issue #403: Guard 2: Verificación proactiva de autenticación ANTES de acceso a BD
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("[update_context_panel] User not authenticated - skipping DB access")
            raise PreventUpdate

        try:
            # Datos base del universo sustituible
            universe_summary = context_data.get("universe_summary", {})
            total_substitutable = universe_summary.get("total_substitutable_revenue", 0)
            total_groups = universe_summary.get("total_substitutable_groups", 0)
            total_units = universe_summary.get("total_substitutable_units", 0)

            # Issue #330: Usar total REAL de farmacia (no estimación)
            total_pharmacy_revenue = universe_summary.get("total_pharmacy_revenue", 0)

            logger.info(
                f"[CONTEXT_PANEL] total_substitutable={total_substitutable}, "
                f"total_pharmacy_revenue={total_pharmacy_revenue}, "
                f"keys_in_summary={list(universe_summary.keys())}"
            )

            # Fallback a estimación solo si no hay total real (backward compatibility)
            if total_pharmacy_revenue == 0:
                logger.warning("[CONTEXT_PANEL] total_pharmacy_revenue is 0 - usando estimación")
                total_pharmacy_revenue = total_substitutable * 1.5

            substitutable_percentage = (
                (total_substitutable / total_pharmacy_revenue * 100) if total_pharmacy_revenue > 0 else 0
            )

            # Si tenemos datos de análisis dinámico, mostrar métricas de partners
            if analysis_data and "error" not in analysis_data:
                analyzable = analysis_data.get("analyzable_universe", {})
                partner_perf = analysis_data.get("partner_performance", {})
                opportunity = analysis_data.get("opportunity_metrics", {})

                analyzable_revenue = analyzable.get("total_revenue", 0)
                analyzable_groups = analyzable.get("groups_count", 0)

                current_partner_revenue = partner_perf.get("partner_revenue", 0)
                partner_penetration = partner_perf.get("penetration_percentage", 0)

                opportunity_revenue = opportunity.get("opportunity_revenue", 0)

                # BD-FIRST PATTERN: Leer partners SIEMPRE desde BD (ignorar store)
                logger.info(
                    f"[CONTEXT_PANEL] IGNORING partners_store (may be stale/None)\n"
                    f"   - Reading from DB as single source of truth..."
                )

                result = get_selected_partners_from_db()

                if not result["success"]:
                    logger.error(
                        f"[CONTEXT_PANEL] ERROR fetching partners from DB\n"
                        f"   - Error: {result.get('error', 'Unknown')}\n"
                        f"   - Falling back to 0 partners for display"
                    )
                    selected_partners_list = []
                else:
                    selected_partners_list = result["partners"]
                    logger.info(
                        f"[CONTEXT_PANEL] Partners loaded from DB (BD-first pattern)\n"
                        f"   - Count: {len(selected_partners_list)}\n"
                        f"   - Partners: {selected_partners_list[:3]}{'...' if len(selected_partners_list) > 3 else ''}\n"
                        f"   - Source: {result['source'].upper()}"
                    )

                selected_partners = len(selected_partners_list)

                # Calcular porcentajes relativos
                analyzable_pct_substitutable = (
                    (analyzable_revenue / total_substitutable * 100) if total_substitutable > 0 else 0
                )
                opportunity_pct_analyzable = (
                    (opportunity_revenue / analyzable_revenue * 100) if analyzable_revenue > 0 else 0
                )

                return [
                    format_currency(total_substitutable),
                    f"{substitutable_percentage:.1f}% del total farmacia",
                    format_currency(analyzable_revenue),
                    f"{analyzable_pct_substitutable:.1f}% del sustituible ({analyzable_groups} grupos)",
                    f"Partners: {selected_partners} laboratorios",
                    f"Penetración: {partner_penetration:.1f}% ({format_currency(current_partner_revenue)})",
                    format_currency(opportunity_revenue),
                    f"{opportunity_pct_analyzable:.1f}% del analizable",
                    no_update,
                ]
            else:
                # Sin análisis dinámico - mostrar solo contexto
                return [
                    format_currency(total_substitutable),
                    f"{substitutable_percentage:.1f}% del total farmacia",
                    format_compact_number(total_units),
                    f"{total_groups} conjuntos homogéneos",
                    "No seleccionados",
                    "Seleccionar partners",
                    "--",
                    "Esperando análisis...",
                    no_update,
                ]

        except Exception as e:
            logger.error(f"Error actualizando panel contexto: {str(e)}")
            return ["Error"] * 8 + [no_update]

    # ============================================================================
    # 2b. CALLBACK CONTEXT TREEMAP - Issue #415
    # ============================================================================

    @app.callback(
        Output("generics-context-treemap", "figure"),
        [
            Input("analysis-store", "data"),
            Input("context-store", "data"),
        ],
        [
            State("generics-date-range", "start_date"),
            State("generics-date-range", "end_date"),
            State("generics-employee-filter", "value"),
            State("auth-state", "data"),
            State("url", "pathname"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Token restoration
        ],
        prevent_initial_call="initial_duplicate",  # FIX: Allow initial population
    )
    def update_context_treemap(analysis_data, context_data, start_date, end_date, employee_filter, auth_state, pathname, auth_tokens):
        """
        Actualizar el treemap de contexto de ventas.

        Issue #415: Visualización jerárquica del universo de ventas.
        Issue #426: Terminología actualizada:
        - "Sustituible" → "Con Conjunto Homogéneo"
        - "Analizable" → "En Vademécum"

        Issue #415: Visualización jerárquica del universo de ventas.
        Issue #426: Terminología actualizada.
        Issue #472: Migrado a GenericContextTreemap measure (TotalVentas + filtros composables).
                    Usa grupos homogéneos/genéricos (datos correctos), NO ATC codes.
        """
        from utils.auth_helpers import is_user_authenticated
        from utils.config import BACKEND_URL

        from utils.auth import auth_manager

        logger.info(
            f"[update_context_treemap] CALLBACK TRIGGERED - "
            f"pathname={pathname}, has_analysis={analysis_data is not None}, "
            f"has_context={context_data is not None}"
        )

        # Guard 0: Pathname guard - solo ejecutar en /generics
        if pathname and pathname.lower() not in ["/generics", "/genericos"]:
            logger.debug(f"[update_context_treemap] Not on generics page ({pathname}) - skipping")
            raise PreventUpdate

        # REGLA #7.6: Restore tokens FIRST in multi-worker environment (Render)
        # Must be before any auth_manager or get_current_pharmacy_id() calls
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        # Guard 1: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[update_context_treemap] User not authenticated - showing empty treemap")
            return _create_empty_treemap("Inicia sesión para ver datos")

        # Issue #438: Validate pharmacy_id matches current user
        try:
            current_pharmacy_id = str(get_current_pharmacy_id())
            cached_pharmacy_id = context_data.get("_pharmacy_id") if context_data else None
            if cached_pharmacy_id and cached_pharmacy_id != current_pharmacy_id:
                logger.warning("[update_context_treemap] Stale cache detected - waiting for fresh data")
                return _create_empty_treemap("Cargando datos...")
        except ValueError:
            return _create_empty_treemap("Cargando datos...")

        # Guard 2: Verificar si hay datos
        logger.info(
            f"[update_context_treemap] Guard 2 check - "
            f"context_data keys: {list(context_data.keys()) if context_data else None}, "
            f"no_data: {context_data.get('no_data') if context_data else None}"
        )
        if context_data and context_data.get("no_data"):
            logger.info("[update_context_treemap] No sales data - showing empty treemap")
            return _create_empty_treemap("Sin datos de ventas")

        # Guard 3: Obtener pharmacy_id
        logger.info("[update_context_treemap] Guard 3: Getting pharmacy_id...")
        pharmacy_id = get_current_pharmacy_id()
        logger.info(f"[update_context_treemap] pharmacy_id = {pharmacy_id}")
        if not pharmacy_id:
            logger.warning("[update_context_treemap] No pharmacy_id available")
            return _create_empty_treemap("Farmacia no configurada")

        try:
            logger.info("[update_context_treemap] Getting access token...")
            token = auth_manager.get_access_token()
            logger.info(f"[update_context_treemap] Token obtained: {bool(token)}")
            if not token:
                return _create_empty_treemap("Sesión expirada")

            # Construir parámetros
            params = {}
            if start_date:
                params["start_date"] = start_date[:10] if isinstance(start_date, str) else start_date.strftime("%Y-%m-%d")
            if end_date:
                params["end_date"] = end_date[:10] if isinstance(end_date, str) else end_date.strftime("%Y-%m-%d")
            if employee_filter and len(employee_filter) > 0:
                params["employee_names"] = employee_filter

            # Issue #472: Usar nuevo endpoint con GenericContextTreemap measure
            # La medida usa TotalVentas + filtros composables (grupos homogéneos, NO ATC codes)
            response = requests.get(
                f"{BACKEND_URL}/api/v1/generic-analysis/context-treemap/{pharmacy_id}",
                params=params,
                headers={"Authorization": f"Bearer {token}"},
                timeout=30,
            )

            if response.status_code == 200:
                data = response.json()
                return _create_treemap_figure(data)
            elif response.status_code == 401:
                logger.warning("[update_context_treemap] Token expired or invalid")
                return _create_empty_treemap("Sesión expirada")
            else:
                logger.error(f"[update_context_treemap] API error: {response.status_code}")
                return _create_empty_treemap("Error cargando datos")

        except requests.exceptions.Timeout:
            logger.error("[update_context_treemap] Timeout calling treemap endpoint")
            return _create_empty_treemap("Tiempo de espera agotado")
        except Exception as e:
            logger.error(f"[update_context_treemap] Error: {str(e)}")
            return _create_empty_treemap("Error interno")


def _create_empty_treemap(message: str) -> go.Figure:
    """Crear treemap vacío con mensaje."""
    fig = go.Figure()
    fig.add_annotation(
        text=message,
        showarrow=False,
        font={"size": 14, "color": "#6c757d"},
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
    )
    fig.update_layout(
        margin={"t": 10, "b": 10, "l": 10, "r": 10},
        height=180,
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )
    return fig


def _create_treemap_figure(data: Dict[str, Any]) -> go.Figure:
    """
    Crear figura treemap desde datos del backend.

    Issue #415: Visualización jerárquica con colores semánticos.
    Issue #426: Terminología actualizada en visualización.
    """
    treemap_data = data.get("treemap_data", {})

    if not treemap_data or not treemap_data.get("labels"):
        return _create_empty_treemap("Sin datos de ventas")

    fig = go.Figure(
        go.Treemap(
            labels=treemap_data.get("labels", []),
            parents=treemap_data.get("parents", []),
            values=treemap_data.get("values", []),
            text=treemap_data.get("text", []),
            marker={"colors": treemap_data.get("colors", [])},
            textinfo="label+text",
            textfont={"size": 11},
            hovertemplate="<b>%{label}</b><br>%{text}<extra></extra>",
            branchvalues="total",
            maxdepth=-1,
        )
    )

    fig.update_layout(
        margin={"t": 5, "b": 5, "l": 5, "r": 5},
        height=180,
        paper_bgcolor="rgba(0,0,0,0)",
        font={"family": "Arial, sans-serif"},
    )

    return fig
