"""
Analysis callbacks for Generics Panel.

Handles:
- Dynamic partner analysis with selected partners
- Drill-down temporal charts (quarter → month → fortnight)
"""

import logging
import time

from dash import Input, Output, State, ctx, no_update
from dash.exceptions import PreventUpdate

from components.generics import create_empty_chart, create_stacked_bar_chart
from utils.api_client import api_client
from utils.generics_helpers import (
    get_codes_with_cache,
    get_selected_partners_from_db,
    get_temporal_breakdown_data,
)
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)


def register_analysis_callbacks(app):
    """
    Register analysis callbacks for the Generics Panel.

    Args:
        app: Dash application instance
    """

    # ============================================================================
    # 5. CALLBACK ANÁLISIS DINÁMICO CON PARTNERS SELECCIONADOS
    # ============================================================================

    @app.callback(
        Output("analysis-store", "data"),
        [
            Input("partners-selection-store", "data"),
            Input("url", "pathname"),
            Input("partner-discount-slider", "value"),
            Input("context-store", "data"),
        ],
        [
            State("laboratory-cache-store", "data"),
            State("auth-state", "data"),
            State("generics-date-range", "start_date"),
            State("generics-date-range", "end_date"),
            State("generics-employee-filter", "value"),
        ],
        prevent_initial_call=False,
    )
    def perform_dynamic_analysis(partners_store, pathname, discount_value, context_data, codes_cache, auth_state, start_date, end_date, employee_filter):
        """
        Realizar análisis dinámico automáticamente al cambiar partners o descuento.

        BD-FIRST PATTERN: Lee partners SIEMPRE desde BD (ignora partners_store).

        Args:
            partners_store: Partner selection store (reactivity trigger, valor ignorado)
            pathname: URL pathname (trigger al entrar a /generics)
            discount_value: Descuento aplicado (0-100)
            context_data: Datos del contexto (universo sustituible) - verifica no_data
            codes_cache: Cache de códigos de laboratorio
            auth_state: Estado de autenticación
            start_date: Fecha inicio del rango de análisis
            end_date: Fecha fin del rango de análisis
            employee_filter: Lista de nombres de empleados seleccionados (None = todos)

        Notes:
            - partners_store se mantiene como Input para reactividad (dispara callback)
            - PERO su valor se IGNORA, siempre se lee desde BD (fuente de verdad)
        """
        callback_start = time.time()

        # COMBINED GUARDS: Validar pathname y auth en un solo check (performance)
        from utils.auth_helpers import is_user_authenticated

        if pathname != "/generics" or not is_user_authenticated(auth_state):
            logger.debug(
                f"[perform_dynamic_analysis] Skipping - pathname: {pathname}, "
                f"auth: {is_user_authenticated(auth_state)}"
            )
            raise PreventUpdate

        # FIX STALE CACHE: Si contexto indica "sin datos", limpiar analysis-store
        if context_data and context_data.get("no_data"):
            logger.info(
                "[perform_dynamic_analysis] Context indica no_data=True - limpiando analysis-store"
            )
            return {"no_data": True, "message": "No hay datos de ventas cargados"}

        # Issue #438: Validate pharmacy_id matches current user (prevents stale cache)
        try:
            current_pharmacy_id = str(get_current_pharmacy_id())
            cached_pharmacy_id = context_data.get("_pharmacy_id") if context_data else None
            if cached_pharmacy_id and cached_pharmacy_id != current_pharmacy_id:
                logger.warning(
                    f"[perform_dynamic_analysis] Stale context cache detected - waiting for fresh data"
                )
                raise PreventUpdate
        except ValueError:
            logger.debug("[perform_dynamic_analysis] No pharmacy_id yet - waiting for auth")
            raise PreventUpdate

        # Guard: Si contexto tiene error o no está disponible, no proceder
        if not context_data or context_data.get("error"):
            logger.debug(
                f"[perform_dynamic_analysis] Context not ready or has error - waiting"
            )
            raise PreventUpdate

        codes_cache = codes_cache or {}
        trigger = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else "initial_load"
        logger.info(f"[perform_dynamic_analysis] Trigger: {trigger}, Starting execution...")

        try:
            # BD-FIRST PATTERN: Leer partners SIEMPRE desde BD (ignorar store)
            logger.info(
                "[perform_dynamic_analysis] IGNORING partners_store (may be stale/None)\n"
                "   - Reading from DB as single source of truth..."
            )

            result = get_selected_partners_from_db()

            if not result["success"]:
                logger.error(
                    f"[perform_dynamic_analysis] Failed reading partners from DB\n"
                    f"   - Error: {result['error_message']}"
                )
                return {"error": "No se pudo cargar la selección de partners"}

            selected_partners = result["partners"]

            db_read_time = time.time() - callback_start
            logger.info(
                f"[perform_dynamic_analysis] Partners loaded from DB:\n"
                f"   - Total selected: {len(selected_partners)}\n"
                f"   - Partners: {selected_partners[:3]}{'...' if len(selected_partners) > 3 else ''}\n"
                f"   - Source: {result['source'].upper()} (BD-first pattern with caching)\n"
                f"   - Store present but ignored: {partners_store is not None} (prevents race conditions)\n"
                f"   - [TIMING] DB read: {db_read_time:.2f}s"
            )

            # Convertir nombres a códigos para backend (P1 Fix - Cache-first)
            conversion_start = time.time()
            selected_partner_codes, codes_cache = get_codes_with_cache(selected_partners, codes_cache)
            conversion_time = time.time() - conversion_start

            logger.info(
                f"Partners converted: {len(selected_partners)} names → {len(selected_partner_codes)} codes\n"
                f"   - [TIMING] Conversion: {conversion_time:.2f}s (total: {time.time() - callback_start:.2f}s)"
            )
            logger.debug(f"Partner codes: {selected_partner_codes}")

            # Obtener pharmacy_id dinámicamente
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError as e:
                logger.error(f"Error obteniendo pharmacy_id: {str(e)}")
                return {"error": "No se pudo identificar la farmacia"}

            # Construir payload con fechas directas
            payload = {
                "selected_partner_codes": selected_partner_codes,
                "discount_percentage": discount_value,
                "selected_employee_names": employee_filter if employee_filter else None,
            }

            if start_date and end_date:
                payload["start_date"] = start_date[:10] if isinstance(start_date, str) else start_date.strftime("%Y-%m-%d")
                payload["end_date"] = end_date[:10] if isinstance(end_date, str) else end_date.strftime("%Y-%m-%d")
                logger.info(f"[perform_dynamic_analysis] Using direct dates: {payload['start_date']} to {payload['end_date']}")
            else:
                payload["period_months"] = 12
                logger.info("[perform_dynamic_analysis] No dates provided, using period_months=12 fallback")

            logger.debug(f"Payload details: {payload}")
            logger.debug(f"discount_value type: {type(discount_value)}, value: {discount_value}")

            api_start = time.time()
            analysis_data = api_client.post(f"/api/v1/analysis/partner-dynamic/{pharmacy_id}", data=payload)
            api_time = time.time() - api_start

            total_time = time.time() - callback_start

            if analysis_data:
                logger.info(
                    f"Análisis dinámico completado exitosamente\n"
                    f"   - [TIMING] API call: {api_time:.2f}s\n"
                    f"   - [TIMING] Total callback: {total_time:.2f}s"
                )

                if total_time > 120:
                    logger.warning(
                        f"[TIMEOUT_WARNING] Callback execution time {total_time:.2f}s approaching "
                        f"limit (180s timeout). Consider optimization or Background Callback."
                    )

                return analysis_data
            else:
                logger.error(
                    f"Error en análisis dinámico: respuesta vacía\n"
                    f"   - [TIMING] API call: {api_time:.2f}s\n"
                    f"   - [TIMING] Total callback: {total_time:.2f}s"
                )
                return {"error": "Error del servidor"}

        except Exception as e:
            total_time = time.time() - callback_start
            logger.error(
                f"Excepción en análisis dinámico: {str(e)}\n"
                f"   - [TIMING] Failed at: {total_time:.2f}s"
            )
            return {"error": str(e)}

    # ============================================================================
    # 6. CALLBACK GRÁFICOS DRILL-DOWN TEMPORALES
    # ============================================================================

    @app.callback(
        [
            Output("sales-drill-down-chart", "figure"),
            Output("units-drill-down-chart", "figure"),
            Output("drill-level-indicator", "children"),
            Output("drill-up-btn", "disabled"),
            Output("temporal-drill-store", "data"),
        ],
        [
            Input("analysis-store", "data"),
            Input("sales-drill-down-chart", "clickData"),
            Input("units-drill-down-chart", "clickData"),
            Input("drill-up-btn", "n_clicks"),
        ],
        [
            State("temporal-drill-store", "data"),
            State("partners-selection-store", "data"),
            State("codes-cache-store", "data"),
            State("auth-state", "data"),
            State("generics-employee-filter", "value"),
        ],
        prevent_initial_call=False,
    )
    def update_drill_down_charts(
        analysis_data, sales_click, units_click, drill_up_clicks, current_drill_state, partners_store, codes_cache, auth_state, employee_names
    ):
        """
        Actualizar gráficos de drill-down temporal (trimestre → mes → quincena).

        Issue #403: Verificación proactiva de autenticación antes de llamadas API.
        """
        try:
            # Guard 1: Verificar datos válidos
            if not analysis_data or "error" in analysis_data:
                empty_fig = create_empty_chart("Cargando análisis temporal...")
                return empty_fig, empty_fig, "Nivel: --", True, {"level": "quarter", "path": []}

            # Guard 1.5: Verificar si no hay datos de ventas (usuario borró datos)
            if analysis_data.get("no_data"):
                empty_fig = create_empty_chart("No hay datos de ventas. Sube tu archivo ERP.")
                return empty_fig, empty_fig, "Sin datos", True, {"level": "quarter", "path": []}

            # Issue #403: Guard 2: Verificación proactiva de autenticación ANTES de llamadas API
            from utils.auth_helpers import is_user_authenticated

            if not is_user_authenticated(auth_state):
                logger.debug("[update_drill_down_charts] User not authenticated - skipping API calls")
                raise PreventUpdate

            # Inicializar drill state si no existe
            if not current_drill_state:
                from utils.drill_down_helpers import initialize_drill_state
                current_drill_state = initialize_drill_state()

            triggered_id = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else ""

            # Manejar navegación drill-down
            if triggered_id in ["sales-drill-down-chart", "units-drill-down-chart"]:
                from utils.drill_down_helpers import parse_drill_down_click, calculate_next_drill_level

                click_data = sales_click or units_click
                clicked_period = parse_drill_down_click(click_data)

                if clicked_period:
                    new_level, new_path = calculate_next_drill_level(
                        current_drill_state["level"],
                        current_drill_state["path"],
                        clicked_period
                    )

                    if new_level is not None:
                        current_drill_state = {"level": new_level, "path": new_path}

            elif triggered_id == "drill-up-btn":
                from utils.drill_down_helpers import calculate_previous_drill_level

                if len(current_drill_state["path"]) > 0:
                    new_level, new_path, _ = calculate_previous_drill_level(
                        current_drill_state["path"]
                    )
                    current_drill_state = {"level": new_level, "path": new_path}

            # Inicializar codes_cache si no existe
            if codes_cache is None:
                codes_cache = {}

            # Obtener datos temporales según el nivel actual (P1 Fix - Cache-first)
            temporal_data, codes_cache = get_temporal_breakdown_data(
                current_drill_state["level"],
                current_drill_state["path"],
                partners_store,
                codes_cache,
                homogeneous_code=None,
                employee_names=employee_names
            )

            if not temporal_data:
                empty_fig = create_empty_chart("No hay datos para este período")
                return (
                    empty_fig,
                    empty_fig,
                    f"Nivel: {current_drill_state['level']}",
                    len(current_drill_state["path"]) == 0,
                    current_drill_state,
                )

            # Crear gráficos
            sales_fig = create_stacked_bar_chart(
                temporal_data, "sales_amount", "Ventas (€)", current_drill_state["level"]
            )

            units_fig = create_stacked_bar_chart(temporal_data, "units", "Unidades", current_drill_state["level"])

            # Actualizar indicadores
            level_names = {"quarter": "Trimestre", "month": "Mes", "fortnight": "Quincena"}
            level_indicator = f"Nivel: {level_names.get(current_drill_state['level'], 'Desconocido')}"
            drill_up_disabled = len(current_drill_state["path"]) == 0

            return sales_fig, units_fig, level_indicator, drill_up_disabled, current_drill_state

        except Exception as e:
            logger.error(f"Error en drill-down charts: {str(e)}")
            empty_fig = create_empty_chart(f"Error: {str(e)}")
            return empty_fig, empty_fig, "Nivel: Error", True, {"level": "quarter", "path": []}
