"""
Callbacks para el Panel de Análisis de Partners - REDESIGN 2025.

Refactored from single file to modular structure for maintainability.

Modules:
- data_loading: Context loading, employee options, interval control
- context_display: Context panel, treemap visualization
- partners: Partners dropdown, controller, count initialization
- analysis: Dynamic analysis, drill-down charts
- homogeneous: Matrix, expansion, discount simulation, export
- homogeneous_detail: Selection, detail chart, drill-down navigation
- optimal_partners: Optimal partners section, penetration gauge

Original: callbacks/generics.py (3,524 lines)
Refactoring: Split into modules (~200-800 lines each)
"""

import logging

from .data_loading import register_data_loading_callbacks
from .context_display import register_context_display_callbacks
from .partners import register_partners_callbacks
from .analysis import register_analysis_callbacks
from .homogeneous import register_homogeneous_callbacks
from .homogeneous_detail import register_homogeneous_detail_callbacks
from .optimal_partners import register_optimal_partners_callbacks

logger = logging.getLogger(__name__)


def register_generics_callbacks(app):
    """
    Registrar todos los callbacks para el Panel de Análisis de Partners.

    Mantiene compatibilidad con el registro anterior.

    Args:
        app: Instancia de la aplicación Dash
    """
    # Core data loading (context, employees)
    register_data_loading_callbacks(app)

    # Context display (panel, treemap)
    register_context_display_callbacks(app)

    # Partners management (dropdown, controller)
    register_partners_callbacks(app)

    # Dynamic analysis and drill-down
    register_analysis_callbacks(app)

    # Homogeneous groups (matrix, expansion, export)
    register_homogeneous_callbacks(app)

    # Homogeneous detail (selection, chart, drill)
    register_homogeneous_detail_callbacks(app)

    # Optimal partners section
    register_optimal_partners_callbacks(app)

    logger.info("[generics_callbacks] All callbacks registered (7 modules)")


# Re-export for backwards compatibility
__all__ = [
    "register_generics_callbacks",
]
