# frontend/callbacks/encargos/data_loading.py
"""
Cazador de Encargos - Data Loading Callbacks

Load order summary and critical orders list from backend API.
Local mode: No JWT required (PIN auth handled by middleware).

See: docs/CONTROL_MODULES_ARCHITECTURE.md
"""

import logging

from dash import Input, Output, State, ctx, no_update
from dash.exceptions import PreventUpdate

from components.encargos import create_encargos_summary_kpis, create_orders_list
from utils.config import BACKEND_URL
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_encargos_data_callbacks(app):
    """
    Register callbacks for loading encargos data.

    Callbacks:
    - Load summary KPIs on page load and after sync
    - Load critical orders list
    """

    @app.callback(
        [
            Output("encargos-widget-summary", "children"),
            Output("encargos-widget-list", "children"),
        ],
        [
            Input("url", "pathname"),
            Input("encargos-widget-sync-btn", "n_clicks"),
        ],
        prevent_initial_call=False,
    )
    def load_encargos_data(pathname, sync_clicks):
        """
        Load encargos summary and critical orders list.

        Triggers:
        - On dashboard page load (pathname == "/" or "/dashboard")
        - After sync button click

        Local mode: No auth headers needed (PIN session validated by middleware).
        """
        # Only load on dashboard pages
        if pathname not in ["/", "/dashboard", "/encargos"]:
            raise PreventUpdate

        trigger = ctx.triggered_id if ctx.triggered else None
        logger.info(f"[ENCARGOS] Loading data, trigger: {trigger}, pathname: {pathname}")

        # Load summary
        try:
            summary_response = request_coordinator.make_request(
                "/api/v1/customer-orders/summary",
                method="GET",
                timeout=15,
                cache_ttl=30,  # Cache for 30s
            )

            if summary_response and not summary_response.get("error"):
                summary_component = create_encargos_summary_kpis(summary_response)
            else:
                logger.warning(f"[ENCARGOS] Summary API error: {summary_response}")
                summary_component = _create_error_message("Error cargando resumen")

        except Exception as e:
            logger.error(f"[ENCARGOS] Error loading summary: {e}")
            summary_component = _create_error_message("Error de conexion")

        # Load critical orders (severity=critical, limit=10)
        try:
            orders_response = request_coordinator.make_request(
                "/api/v1/customer-orders/pending",
                method="GET",
                params={"severity": "critical", "limit": 10},
                timeout=15,
                cache_ttl=30,
            )

            if orders_response and not orders_response.get("error"):
                orders = orders_response.get("orders", [])
                orders_component = create_orders_list(orders)
            else:
                logger.warning(f"[ENCARGOS] Orders API error: {orders_response}")
                orders_component = _create_error_message("Error cargando encargos")

        except Exception as e:
            logger.error(f"[ENCARGOS] Error loading orders: {e}")
            orders_component = _create_error_message("Error de conexion")

        return summary_component, orders_component


def _create_error_message(message: str):
    """Create an error message component."""
    from dash import html
    from styles.design_tokens import COLORS

    return html.Div(
        [
            html.I(
                className="fas fa-exclamation-triangle me-2",
                style={"color": COLORS["warning"]},
            ),
            html.Span(message, className="text-muted"),
        ],
        className="text-center py-3",
    )
