# frontend/callbacks/encargos/actions.py
"""
Cazador de Encargos - Action Callbacks

Handle user actions: sync, mark contacted, mark for return.
Local mode: No JWT required (PIN auth handled by middleware).

See: docs/CONTROL_MODULES_ARCHITECTURE.md
"""

import logging

from dash import Input, Output, State, ctx, no_update, ALL, MATCH
from dash.exceptions import PreventUpdate

from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_encargos_action_callbacks(app):
    """
    Register callbacks for encargos actions.

    Callbacks:
    - Sync from ERP
    - Mark order as contacted
    - Mark order for return
    """

    @app.callback(
        [
            Output("encargos-widget-toast", "children"),
            Output("encargos-widget-toast", "is_open"),
            Output("encargos-widget-toast", "icon"),
            Output("encargos-widget-sync-btn", "disabled"),
        ],
        Input("encargos-widget-sync-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def sync_from_erp(n_clicks):
        """
        Trigger sync from ERP when sync button is clicked.

        Shows toast notification with result.
        """
        if not n_clicks:
            raise PreventUpdate

        logger.info("[ENCARGOS] Starting ERP sync...")

        try:
            # Call sync endpoint
            response = request_coordinator.make_request(
                "/api/v1/customer-orders/sync",
                method="POST",
                timeout=60,  # Sync can take time
                bypass_cache=True,
            )

            if response and not response.get("error"):
                new_orders = response.get("new", 0)
                updated = response.get("updated", 0)
                resolved = response.get("resolved", 0)
                total = response.get("total_pending", 0)

                message = f"Sincronizado: {new_orders} nuevos, {updated} actualizados, {resolved} resueltos. Total: {total}"
                logger.info(f"[ENCARGOS] Sync success: {message}")
                return message, True, "success", False

            else:
                error_msg = response.get("message", "Error desconocido") if response else "Sin respuesta"
                logger.warning(f"[ENCARGOS] Sync failed: {error_msg}")
                return f"Error: {error_msg}", True, "danger", False

        except Exception as e:
            logger.error(f"[ENCARGOS] Sync error: {e}")
            return f"Error de conexion: {str(e)[:50]}", True, "danger", False

    @app.callback(
        [
            Output("encargos-widget-toast", "children", allow_duplicate=True),
            Output("encargos-widget-toast", "is_open", allow_duplicate=True),
            Output("encargos-widget-toast", "icon", allow_duplicate=True),
        ],
        Input({"type": "encargos-widget-contact-btn", "index": ALL}, "n_clicks"),
        prevent_initial_call=True,
    )
    def mark_order_contacted(n_clicks_list):
        """
        Mark an order as contacted when contact button is clicked.

        Uses pattern-matching callbacks for dynamic order buttons.
        """
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Find which button was clicked
        triggered = ctx.triggered_id
        if not triggered or not isinstance(triggered, dict):
            raise PreventUpdate

        erp_order_id = triggered.get("index")
        if not erp_order_id:
            raise PreventUpdate

        logger.info(f"[ENCARGOS] Marking order {erp_order_id} as contacted")

        try:
            response = request_coordinator.make_request(
                f"/api/v1/customer-orders/order/{erp_order_id}/contacted",
                method="POST",
                data={"notes": "Marcado desde widget dashboard"},
                timeout=10,
                bypass_cache=True,
            )

            if response and response.get("success"):
                return f"Encargo {erp_order_id[:20]}... marcado como contactado", True, "success"
            else:
                error_msg = response.get("message", "Error") if response else "Sin respuesta"
                return f"Error: {error_msg}", True, "warning"

        except Exception as e:
            logger.error(f"[ENCARGOS] Contact action error: {e}")
            return f"Error: {str(e)[:30]}", True, "danger"

    @app.callback(
        [
            Output("encargos-widget-toast", "children", allow_duplicate=True),
            Output("encargos-widget-toast", "is_open", allow_duplicate=True),
            Output("encargos-widget-toast", "icon", allow_duplicate=True),
        ],
        Input({"type": "encargos-widget-return-btn", "index": ALL}, "n_clicks"),
        prevent_initial_call=True,
    )
    def mark_order_for_return(n_clicks_list):
        """
        Mark an order for return to supplier when return button is clicked.
        """
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        triggered = ctx.triggered_id
        if not triggered or not isinstance(triggered, dict):
            raise PreventUpdate

        erp_order_id = triggered.get("index")
        if not erp_order_id:
            raise PreventUpdate

        logger.info(f"[ENCARGOS] Marking order {erp_order_id} for return")

        try:
            response = request_coordinator.make_request(
                f"/api/v1/customer-orders/order/{erp_order_id}/to-return",
                method="POST",
                data={"notes": "Marcado para devolucion desde widget"},
                timeout=10,
                bypass_cache=True,
            )

            if response and response.get("success"):
                return f"Encargo {erp_order_id[:20]}... marcado para devolver", True, "warning"
            else:
                error_msg = response.get("message", "Error") if response else "Sin respuesta"
                return f"Error: {error_msg}", True, "danger"

        except Exception as e:
            logger.error(f"[ENCARGOS] Return action error: {e}")
            return f"Error: {str(e)[:30]}", True, "danger"

    @app.callback(
        Output("url", "pathname", allow_duplicate=True),
        Input("encargos-widget-view-all-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def navigate_to_encargos_page(n_clicks):
        """
        Navigate to full encargos management page.
        """
        if not n_clicks:
            raise PreventUpdate

        return "/encargos"
