# frontend/callbacks/dashboard.py
"""
Callbacks para el dashboard farmacéutico principal según Issue #4.
Sistema de routing basado en URLs (callbacks antiguos de navegación fueron eliminados).
"""

import logging

from dash import html
from utils.helpers import format_number

logger = logging.getLogger(__name__)


def register_dashboard_callbacks(app):
    """
    Registrar callbacks para el dashboard especializado.

    Args:
        app: Instancia de la aplicación Dash

    Note:
        Sistema migrado a routing basado en URLs.
        Callbacks de navegación antiguos fueron eliminados (Issue #248).
        Callbacks de progreso de enriquecimiento registrados globalmente en __init__.py
    """
    pass  # No callbacks activos - solo routing con URLs


def create_section_header(icon: str, title: str, description: str) -> html.Div:
    """
    Crear header de sección especializada.

    Args:
        icon: Emoji o icono de la sección
        title: Título de la sección
        description: Descripción de la sección

    Returns:
        Header component para la sección
    """

    return html.Div(
        [
            html.H1(
                [html.Span(icon, className="me-3"), title], className="specialized-section-title text-primary mb-2"
            ),
            html.P(description, className="specialized-section-description text-muted mb-0"),
        ],
        className="specialized-section-header",
    )


def register_enrichment_callbacks(app):
    """
    Registrar callbacks específicos para enriquecimiento de datos.

    Args:
        app: Instancia de la aplicación Dash

    Note:
        Callbacks de enriquecimiento ahora manejados por componentes específicos.
        specialized-interval ya no existe en el layout.
    """
    pass  # No callbacks activos


def create_api_error_content(api_name: str, error_message: str) -> html.Div:
    """
    Crear contenido de error para APIs que fallan.

    Args:
        api_name: Nombre de la API que falló
        error_message: Mensaje de error

    Returns:
        Contenido de error amigable para el usuario
    """

    return html.Div(
        [
            html.Div(
                [
                    html.I(className="fas fa-exclamation-triangle fa-3x text-warning mb-3"),
                    html.H4("Error de Conectividad", className="text-warning mb-2"),
                    html.P(
                        [
                            f"No se pudo obtener datos de {api_name}. ",
                            "Esto puede deberse a que aún no hay datos procesados o hay un problema de conectividad.",
                        ],
                        className="text-muted mb-3",
                    ),
                    html.Details(
                        [
                            html.Summary("Detalles técnicos", className="text-muted small"),
                            html.P(error_message, className="text-muted small mt-2"),
                        ],
                        className="mb-3",
                    ),
                    html.Div(
                        [
                            html.Button(
                                html.Span([html.I(className="fas fa-sync-alt me-2"), "Reintentar"]),
                                className="btn btn-outline-warning",
                                id="retry-generics-btn",
                            ),
                            html.A(
                                [html.I(className="fas fa-upload me-2"), "Cargar Datos"],
                                className="btn btn-primary ms-2",
                                href="/cargar-datos",
                            ),
                        ]
                    ),
                ],
                className="text-center py-5",
            )
        ]
    )


def create_critical_error_content(error_message: str) -> html.Div:
    """
    Crear contenido para errores críticos.

    Args:
        error_message: Mensaje de error crítico

    Returns:
        Contenido de error crítico
    """

    return html.Div(
        [
            html.Div(
                [
                    html.I(className="fas fa-bug fa-3x text-danger mb-3"),
                    html.H4("Error del Sistema", className="text-danger mb-2"),
                    html.P(
                        [
                            "Ha ocurrido un error inesperado en el sistema. ",
                            "Por favor, contacta con soporte técnico si el problema persiste.",
                        ],
                        className="text-muted mb-3",
                    ),
                    html.Details(
                        [
                            html.Summary("Error técnico", className="text-muted small"),
                            html.Code(error_message, className="text-danger small"),
                        ]
                    ),
                ],
                className="text-center py-5",
            )
        ]
    )


def create_enrichment_in_progress_content(enrichment_rate: float, enriched: int, total_sales: int) -> html.Div:
    """
    Crear contenido informativo cuando el enriquecimiento está en progreso.

    Args:
        enrichment_rate: Porcentaje de enriquecimiento completado
        enriched: Número de registros enriquecidos
        total_sales: Total de registros de ventas

    Returns:
        Contenido informativo de progreso
    """

    pending = total_sales - enriched

    return html.Div(
        [
            html.Div(
                [
                    html.I(className="fas fa-cogs fa-3x text-info mb-3"),
                    html.H4("Procesando Datos de Genéricos", className="text-info mb-3"),
                    html.P(
                        [
                            "El sistema está enriqueciendo los datos de ventas con información oficial de CIMA y nomenclator. ",
                            "Este proceso es necesario para generar el estudio de oportunidades de genéricos.",
                        ],
                        className="mb-4",
                    ),
                    # Barra de progreso
                    html.Div(
                        [
                            html.Div(
                                [
                                    html.Div(
                                        [
                                            html.Span(
                                                f"{enrichment_rate:.1f}%".replace(".", ","), className="progress-text"
                                            )
                                        ],
                                        className="progress-bar progress-bar-striped progress-bar-animated bg-info",
                                        style={"width": f"{enrichment_rate}%"},
                                    )
                                ],
                                className="progress mb-3",
                                style={"height": "25px"},
                            ),
                            html.P(
                                [
                                    format_number(total_sales),
                                    html.Small(format_number(pending), className="text-muted"),
                                ],
                                className="mb-4",
                            ),
                        ]
                    ),
                    # Información adicional
                    html.Div(
                        [
                            html.H6("¿Qué está ocurriendo?", className="text-primary mb-2"),
                            html.Ul(
                                [
                                    html.Li("Búsqueda por código nacional en nomenclator oficial"),
                                    html.Li("Matching por código EAN para productos no encontrados"),
                                    html.Li("Búsqueda fuzzy por nombre comercial como último recurso"),
                                    html.Li("Identificación de laboratorios y alternativas genéricas"),
                                ],
                                className="text-left small mb-3",
                            ),
                            html.Div(
                                [
                                    html.I(className="fas fa-clock me-2 text-warning"),
                                    "El proceso se completará automáticamente. La página se actualizará cada 10 segundos.",
                                ],
                                className="text-warning small",
                            ),
                        ],
                        className="bg-light p-3 rounded",
                    ),
                ],
                className="text-center py-5",
            )
        ]
    )
