# frontend/callbacks/config.py
"""
Callbacks para la página de configuración.
"""

from components.common import create_alert
from dash import Input, Output, State, ctx, html, no_update
from dash.exceptions import PreventUpdate
from utils.api_client import backend_client


def register_config_callbacks(app):
    """
    Registrar callbacks de configuración.

    Args:
        app: Instancia de la aplicación Dash
    """

    # ✅ UNIFIED CALLBACK: Combines 3 callbacks listening to config-tabs (REGLA #11)
    # Previously separate callbacks:
    # - update_system_status_basic (line 24-119)
    # - check_system_alerts (line 198-242)
    # - load_pharmacy_config (line 372-421)

    @app.callback(
        [
            Output("system-status-basic", "children"),
            Output("system-alerts", "children", allow_duplicate=True),
            Output("pharmacy-name", "value"),
            Output("pharmacy-code", "value"),
            Output("pharmacy-address", "value"),
            Output("pharmacy-phone", "value"),
            Output("primary-erp-type", "value"),
            Output("erp-version", "value"),
        ],
        Input("config-tabs", "active_tab"),
        prevent_initial_call=True,
    )
    def handle_config_tab_changes(active_tab):
        """
        ✅ UNIFIED CALLBACK: Maneja cambios de tabs en config (REGLA #11)

        Combina lógica de:
        1. update_system_status_basic (system-tab)
        2. check_system_alerts (system-tab)
        3. load_pharmacy_config (pharmacy-tab)

        Args:
            active_tab: Tab activo

        Returns:
            tuple: (system_status, system_alerts, pharmacy_name, pharmacy_code,
                    pharmacy_address, pharmacy_phone, erp_type, erp_version)
        """

        # Default values for no_update
        default_pharmacy = ("", "", "", "", "farmatic", "")

        # === SYSTEM TAB ===
        if active_tab == "system-tab":
            try:
                # Verificar estado del sistema
                response = backend_client.check_health()

                if response.success:
                    data = response.data

                    # Extraer información del sistema
                    status = data.get("status", "unknown") if isinstance(data, dict) else "healthy"
                    database = data.get("database", "unknown") if isinstance(data, dict) else "connected"

                    status_items = []

                    # Estado general
                    if status == "healthy":
                        status_items.append(
                            html.Div(
                                [
                                    html.I(className="fas fa-check-circle text-success me-2"),
                                    html.Strong("Sistema: "),
                                    "Operativo",
                                ],
                                className="mb-2",
                            )
                        )
                    else:
                        status_items.append(
                            html.Div(
                                [
                                    html.I(className="fas fa-exclamation-triangle text-warning me-2"),
                                    html.Strong("Sistema: "),
                                    "Con problemas",
                                ],
                                className="mb-2",
                            )
                        )

                    # Estado de base de datos
                    if database == "healthy":
                        status_items.append(
                            html.Div(
                                [
                                    html.I(className="fas fa-database text-success me-2"),
                                    html.Strong("Base de datos: "),
                                    "Conectada",
                                ],
                                className="mb-2",
                            )
                        )
                    else:
                        status_items.append(
                            html.Div(
                                [
                                    html.I(className="fas fa-database text-danger me-2"),
                                    html.Strong("Base de datos: "),
                                    "Error de conexión",
                                ],
                                className="mb-2",
                            )
                        )

                    # Información adicional
                    status_items.extend(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-server text-info me-2"),
                                    html.Strong("API: "),
                                    "Respondiendo correctamente",
                                ],
                                className="mb-2",
                            ),
                            html.Div(
                                [
                                    html.I(className="fas fa-clock text-muted me-2"),
                                    html.Strong("Última verificación: "),
                                    "Ahora",
                                ],
                                className="mb-0",
                            ),
                        ]
                    )

                    system_status = html.Div(status_items)

                    # Check system alerts
                    alerts = []
                    if not response.success:
                        alerts.append(create_alert("⚠️ Sistema no responde - Verifica la conexión de red", "warning"))
                    system_alerts = html.Div(alerts) if alerts else html.Div()

                    return (system_status, system_alerts, *default_pharmacy)

                else:
                    error_status = create_alert(f"Error al conectar con el sistema: {response.message}", "danger")
                    alert_content = create_alert("⚠️ Sistema no responde - Verifica la conexión de red", "warning")
                    return (error_status, alert_content, *default_pharmacy)

            except Exception as e:
                error_status = create_alert("Error inesperado al verificar el sistema", "danger")
                error_alert = create_alert("Error al verificar alertas del sistema", "danger")
                return (error_status, error_alert, *default_pharmacy)

        # === PHARMACY TAB ===
        elif active_tab == "pharmacy-tab":
            try:
                # Usar farmacia activa dinámicamente
                from utils.pharmacy_context import get_current_pharmacy_id

                pharmacy_id = get_current_pharmacy_id()

                # Obtener datos de la farmacia
                response = backend_client.get_pharmacy_data(pharmacy_id)

                if response.success and response.data:
                    data = response.data.get("data", {})
                    pharmacy_data = (
                        data.get("name", ""),
                        data.get("code", ""),
                        data.get("address", ""),
                        data.get("phone", ""),
                        data.get("erp_type", "farmatic"),
                        data.get("erp_version", ""),
                    )
                    return (no_update, no_update, *pharmacy_data)
                else:
                    # Si no hay datos, usar valores por defecto
                    return (no_update, no_update, *default_pharmacy)

            except Exception as e:
                print(f"ERROR cargando configuración farmacia: {e}")
                # En caso de error, usar valores por defecto
                return (no_update, no_update, *default_pharmacy)

        # === OTHER TABS ===
        else:
            # No need to update anything for other tabs
            raise PreventUpdate

    @app.callback(
        Output("maintenance-status", "children"),
        [
            Input("clear-cache-btn", "n_clicks"),
            Input("update-system-btn", "n_clicks"),
            Input("check-db-btn", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def handle_maintenance_actions(clear_cache_clicks, update_clicks, check_db_clicks):
        """
        Manejar acciones de mantenimiento.

        Args:
            clear_cache_clicks: Clics en limpiar cache
            update_clicks: Clics en actualizar sistema
            check_db_clicks: Clics en verificar BD

        Returns:
            Estado de la acción de mantenimiento
        """

        # Determinar qué botón fue presionado
        if not ctx.triggered:
            raise PreventUpdate

        button_id = ctx.triggered[0]["prop_id"].split(".")[0]

        try:
            if button_id == "clear-cache-btn":
                # Simular limpieza de cache (por ahora)
                return create_alert("Cache limpiado exitosamente", "success", duration=5000)

            elif button_id == "update-system-btn":
                # Simular actualización de sistema
                return create_alert("Sistema actualizado - No hay actualizaciones disponibles", "info", duration=5000)

            elif button_id == "check-db-btn":
                # Verificar base de datos realmente
                response = backend_client.check_health()

                if response.success:
                    data = response.data
                    db_status = data.get("database", "unknown") if isinstance(data, dict) else "connected"

                    if db_status == "connected":
                        return create_alert("Base de datos funcionando correctamente", "success", duration=5000)
                    else:
                        return create_alert("Problemas detectados en la base de datos", "warning", duration=5000)
                else:
                    return create_alert("No se puede conectar para verificar la base de datos", "danger", duration=5000)

        except Exception as e:
            return create_alert(f"Error en la operación de mantenimiento: {str(e)}", "danger", duration=5000)

    # ❌ REMOVED: Duplicate callback for system-alerts (unified in handle_config_tab_changes above)
    # This callback violated REGLA #11: ONE INPUT ONE CALLBACK
    # Now handled by handle_config_tab_changes callback with multiple outputs

    @app.callback(
        [Output("save-general-config", "children"), Output("save-general-config", "color")],
        Input("save-general-config", "n_clicks"),
        [
            State("theme-selector", "value"),
            State("language-selector", "value"),
            State("refresh-interval", "value"),
            State("notifications-system", "value"),
            State("notifications-stock", "value"),
            State("notifications-daily", "value"),
            State("notification-email", "value"),
        ],
        prevent_initial_call=True,
    )
    def save_general_config(
        n_clicks, theme, language, refresh_interval, notify_system, notify_stock, notify_daily, email
    ):
        """
        Guardar configuración general.

        Args:
            n_clicks: Número de clics
            theme: Tema seleccionado
            language: Idioma seleccionado
            refresh_interval: Intervalo de actualización
            notify_system: Notificaciones de sistema
            notify_stock: Notificaciones de stock
            notify_daily: Resumen diario
            email: Email para notificaciones

        Returns:
            Texto y color del botón
        """

        if n_clicks is None:
            raise PreventUpdate

        try:
            # TODO: Implementar guardado real de configuración
            # Por ahora simular guardado exitoso

            config_data = {
                "theme": theme,
                "language": language,
                "refresh_interval": int(refresh_interval),
                "notifications": {
                    "system": notify_system,
                    "stock": notify_stock,
                    "daily": notify_daily,
                    "email": email,
                },
            }

            # Simular delay de guardado
            import time

            time.sleep(0.5)

            return "✓ Configuración Guardada", "success"

        except Exception as e:
            return "✗ Error al Guardar", "danger"

    # Callback del botón original eliminado - ya no se usa

    # Callbacks del sistema de confirmación original eliminados - código limpiado

    @app.callback(
        [Output("save-pharmacy-config", "children"), Output("save-pharmacy-config", "color")],
        Input("save-pharmacy-config", "n_clicks"),
        [
            State("pharmacy-name", "value"),
            State("pharmacy-code", "value"),
            State("pharmacy-address", "value"),
            State("pharmacy-phone", "value"),
            State("primary-erp-type", "value"),
            State("erp-version", "value"),
            State("date-format", "value"),
            State("decimal-separator", "value"),
        ],
        prevent_initial_call=True,
    )
    def save_pharmacy_config(n_clicks, name, code, address, phone, erp_type, erp_version, date_format, decimal_sep):
        """
        Guardar configuración de farmacia.

        Args:
            n_clicks: Número de clics
            name: Nombre de farmacia
            code: Código de farmacia
            address: Dirección
            phone: Teléfono
            erp_type: Tipo de ERP
            erp_version: Versión de ERP
            date_format: Formato de fecha
            decimal_sep: Separador decimal

        Returns:
            Texto y color del botón
        """

        if n_clicks is None:
            raise PreventUpdate

        try:
            # Usar farmacia activa dinámicamente
            from utils.pharmacy_context import get_current_pharmacy_id

            pharmacy_id = get_current_pharmacy_id()

            pharmacy_data = {
                "name": name,
                "code": code,
                "address": address,
                "phone": phone,
                "erp_type": erp_type,
                "erp_version": erp_version,
                # date_format y decimal_sep podrían ir en una tabla de configuración separada
            }

            # Llamar al backend para guardar
            response = backend_client.update_pharmacy_data(pharmacy_id, pharmacy_data)

            if response.success:
                return "✓ Configuración Guardada", "success"
            else:
                return f"✗ Error: {response.message}", "danger"

        except Exception as e:
            print(f"ERROR guardando configuración farmacia: {e}")
            return "✗ Error de Conexión", "danger"

    # ❌ REMOVED: Duplicate callback for pharmacy config loading (unified in handle_config_tab_changes above)
    # This callback violated REGLA #11: ONE INPUT ONE CALLBACK
    # Now handled by handle_config_tab_changes callback with multiple outputs (pharmacy-tab case)
