# frontend/callbacks/common.py
"""
Callbacks comunes compartidos entre páginas.
"""

import logging
from typing import Any, Dict, Optional

import dash
import dash_bootstrap_components as dbc
from components.common import create_alert, create_connection_status
from constants import PUBLIC_ROUTES  # Import at module level (Issue #141 fix)
from dash import Input, Output, State, html
from dash.exceptions import PreventUpdate
from utils.api_client import backend_client

logger = logging.getLogger(__name__)


def register_common_callbacks(app):
    """
    Registrar callbacks comunes.

    Args:
        app: Instancia de la aplicación Dash
    """

    @app.callback(
        Output("backend-status", "children"),
        Input("health-check-interval", "n_intervals"),
        [
            State("auth-state", "data"),  # Issue #301: Verificar autenticación
            State("auth-context-sync", "data"),  # Issue #333: JWT timing protection
        ],
        prevent_initial_call=True,  # Optimización Issue #147: solo ejecutar cuando intervalo dispara
    )
    def update_backend_status(n_intervals, auth_state, auth_context_sync):
        """
        Actualizar estado de conexión con el backend.

        Issue #301: Verifica autenticación ANTES de hacer llamadas API.
        Issue #333: Protección contra JWT timing race condition.

        Args:
            n_intervals: Número de intervalos transcurridos
            auth_state: Estado de autenticación del usuario
            auth_context_sync: Estado de sincronización del token JWT

        Returns:
            Componente con estado de conexión
        """
        from utils.auth_helpers import is_user_authenticated

        # Issue #301: Verificar autenticación ANTES de hacer llamadas API
        # Aunque /health es público, evitamos ejecutar callbacks innecesarios
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Issue #333: Verificar que el token JWT esté sincronizado
        # Previene errores HTTP 401 por race condition en auth_context_sync
        if not auth_context_sync or not auth_context_sync.get("synced"):
            logger.debug("[update_backend_status] Token not synced yet (auth-context-sync pending) - skipping health check")
            raise PreventUpdate

        try:
            # Verificar conexión con backend
            response = backend_client.check_health()

            if response.success:
                # Verificar si la respuesta contiene información del sistema
                if isinstance(response.data, dict):
                    status = response.data.get("status", "unknown")
                    database = response.data.get("database", "unknown")

                    if status == "healthy" and database == "connected":
                        return create_connection_status(is_connected=True, service_name="Sistema xfarma")
                    else:
                        return create_alert(message="Sistema parcialmente operativo", alert_type="warning")
                else:
                    return create_connection_status(is_connected=True, service_name="Backend")
            else:
                return create_alert(message=f"Error de conexión: {response.message}", alert_type="danger")

        except Exception as e:
            return create_alert(message="No se puede conectar al servidor", alert_type="danger")

    @app.callback(Output("current-page-store", "data"), Input("url", "pathname"))
    def store_current_page(pathname):
        """
        Almacenar página actual para uso en otros callbacks.

        Args:
            pathname: Ruta actual

        Returns:
            Diccionario con información de página
        """

        # Mapear rutas a nombres de página
        page_mapping = {
            "/": "dashboard",
            "/dashboard": "dashboard",
            "/upload": "upload",
            "/cargar": "upload",
            "/config": "config",
            "/configuracion": "config",
        }

        page_name = page_mapping.get(pathname, "unknown")

        return {"pathname": pathname, "page": page_name, "timestamp": str(__import__("datetime").datetime.now())}

    # Callback para actualizar el system-status-store - DESHABILITADO (banner eliminado)
    # @app.callback(
    #     Output("system-status-store", "data"),
    #     Input("system-status-interval", "n_intervals"),
    #     prevent_initial_call=True
    # )
    def update_system_status_DISABLED(n_intervals):
        """
        Obtiene el estado actual del sistema desde el backend usando RequestCoordinator.
        Se ejecuta con cache inteligente para evitar rate limiting.
        """
        from utils.request_coordinator import get_system_status

        try:
            # Usar RequestCoordinator con cache extendido
            data = get_system_status()

            if data:
                # Extraer solo la información de inicialización
                return data.get("initialization", {})
            else:
                return {"error": True, "message": "No se pudo obtener el estado del sistema"}

        except Exception as e:
            logger.error(f"Error consultando estado del sistema: {str(e)}")
            return {"error": True, "message": "Error al obtener el estado del sistema"}

    # Callback para actualizar la visualización del banner - DESHABILITADO (banner eliminado)
    # @app.callback(
    #     [
    #         Output("system-status-banner", "style"),
    #         Output("system-status-alert", "style"),
    #         Output("status-message", "children"),
    #         Output("system-progress-bar", "value")
    #     ],
    #     Input("system-status-store", "data")
    # )
    def update_banner_display_DISABLED(status_data: Optional[Dict[str, Any]]):
        """
        Actualiza la visualización del banner según el estado del sistema.
        """
        if not status_data:
            return (
                {"display": "none", "position": "absolute", "top": "4px", "left": "0", "right": "0", "zIndex": "1020"},
                {"backgroundColor": "#fff3cd"},
                "",
                0,
            )

        # Si hay error de conexión
        if status_data.get("error"):
            return (
                {"display": "block", "position": "absolute", "top": "4px", "left": "0", "right": "0", "zIndex": "1020"},
                {"backgroundColor": "#f8d7da"},
                [
                    html.I(className="bi bi-exclamation-triangle-fill me-2"),
                    status_data.get("message", "Error desconocido"),
                ],
                0,
            )

        overall_status = status_data.get("overall_status", "unknown")
        overall_progress = status_data.get("overall_progress", 0)
        overall_message = status_data.get("overall_message", "")

        # Sistema completamente listo - ocultar banner
        if overall_status == "ready":
            return (
                {"display": "none", "position": "absolute", "top": "4px", "left": "0", "right": "0", "zIndex": "1020"},
                {"backgroundColor": "#d4edda"},
                "",
                100,
            )

        # Sistema requiere inicialización
        if overall_status == "requires_initialization":
            return (
                {"display": "block", "position": "absolute", "top": "4px", "left": "0", "right": "0", "zIndex": "1020"},
                {"backgroundColor": "#fff3cd"},
                [
                    html.I(className="bi bi-info-circle-fill me-2"),
                    html.Span(overall_message),
                    html.Span(" "),
                    dbc.Button(
                        "Inicializar Ahora", id="initialize-system-btn", color="primary", size="sm", className="ms-3"
                    ),
                ],
                0,
            )

        # Sistema inicializando
        if overall_status == "initializing":
            # Mostrar detalles de componentes
            components_info = []
            components = status_data.get("components", {})

            for comp_name, comp_data in components.items():
                if comp_data.get("status") == "initializing":
                    comp_msg = comp_data.get("message", f"Procesando {comp_name}...")
                    components_info.append(comp_msg)

            # Construir contenido del mensaje
            spinner_and_text = [
                dbc.Spinner(size="sm"),
                html.Span(overall_message if overall_message else " ".join(components_info), className="ms-2"),
            ]

            # Si hay tiempo estimado, mostrarlo
            for comp_data in components.values():
                if comp_data.get("estimated_time_remaining"):
                    spinner_and_text.append(
                        html.Span(
                            f" - Tiempo estimado: {comp_data['estimated_time_remaining']}", className="text-muted ms-2"
                        )
                    )
                    break

            message_content = html.Div(spinner_and_text, className="d-flex align-items-center")

            return (
                {"display": "block", "position": "absolute", "top": "4px", "left": "0", "right": "0", "zIndex": "1020"},
                {"backgroundColor": "#d1ecf1"},
                message_content,
                overall_progress,
            )

        # Sistema parcialmente operativo - Mostrar detalles de componentes
        if overall_status == "partial":
            components = status_data.get("components", {})
            component_statuses = []

            for comp_name, comp_data in components.items():
                status = comp_data.get("status", "unknown")
                if status == "ready":
                    icon = "✅"
                elif status == "error":
                    icon = "❌"
                elif status == "initializing":
                    icon = "⏳"
                else:
                    icon = "⚠️"

                comp_display_name = {
                    "catalog": "Catálogo",
                    "nomenclator": "Nomenclátor",
                    "cima": "CIMA",
                    "homogeneous_groups": "Grupos Homogéneos",
                }.get(comp_name, comp_name)

                component_statuses.append(f"{icon} {comp_display_name}")

            return (
                {"display": "block", "position": "absolute", "top": "4px", "left": "0", "right": "0", "zIndex": "1020"},
                {"backgroundColor": "#fff3cd"},
                [
                    html.I(className="bi bi-exclamation-circle-fill me-2"),
                    html.Span("Estado del sistema: ", className="fw-bold"),
                    html.Span(" | ".join(component_statuses)),
                ],
                overall_progress,
            )

        # Estado desconocido
        return (
            {"display": "none", "position": "absolute", "top": "4px", "left": "0", "right": "0", "zIndex": "1020"},
            {"backgroundColor": "#e2e3e5"},
            "",
            0,
        )

    # Callback adicional para ocultar el banner en rutas públicas - DESHABILITADO (banner eliminado)
    # @app.callback(
    #     Output("system-status-banner", "style", allow_duplicate=True),
    #     Input("url", "pathname"),
    #     prevent_initial_call='initial_duplicate'
    # )
    def hide_banner_on_public_routes_DISABLED(pathname):
        """
        Oculta el banner de estado del sistema en páginas públicas.
        Solución para Issue #141 - evita 401 errors en Render.
        """
        # Normalizar pathname
        if pathname is None:
            pathname = "/"

        # Ocultar banner en rutas públicas
        if pathname in PUBLIC_ROUTES:
            return {
                "display": "none",
                "position": "absolute",
                "top": "4px",
                "left": "0",
                "right": "0",
                "zIndex": "1020",
            }

        # No cambiar el estado del banner en rutas protegidas
        # (dejar que el callback principal lo maneje)
        return dash.no_update

    # Callback para el botón de inicialización - TEMPORALMENTE DESHABILITADO
    # ISSUE: Causa sincronizaciones automáticas no deseadas
    # El botón se recrea frecuentemente por re-renders del banner
    # @app.callback(
    #     Output("initialize-system-btn", "n_clicks"),
    #     Input("initialize-system-btn", "n_clicks"),
    #     prevent_initial_call=True
    # )
    def handle_initialize_button_DISABLED(n_clicks: int):
        """
        Maneja el click del botón de inicialización usando RequestCoordinator.
        """
        if n_clicks:
            from utils.request_coordinator import request_coordinator

            try:
                # Usar RequestCoordinator para POST con bypass_cache
                response = request_coordinator.make_request(
                    "/api/v1/system/sync",
                    method="POST",
                    params={"target": "all"},
                    bypass_cache=True,  # No cachear operaciones POST
                )

                if response:
                    logger.info("Inicialización del sistema iniciada")
                else:
                    logger.error("Error iniciando el sistema")

            except Exception as e:
                logger.error(f"Error llamando API de inicialización: {str(e)}")

        return 0  # Reset clicks
