"""
UMAP Visualization Callbacks (Issue #458)

Callbacks for:
- Loading UMAP data from backend API
- Updating scatter plot on filter changes
- Handling point selection
- Managing ghost points
- Cleaning ghost points on version change
"""

import structlog
from dash import callback, ctx, no_update
from dash.dependencies import Input, Output, State
from dash.exceptions import PreventUpdate

from components.clustering import create_umap_scatter
from layouts.clustering import create_product_detail_card
from utils.auth import auth_manager
from utils.config import BACKEND_URL
from utils.request_coordinator import request_coordinator

logger = structlog.get_logger(__name__)


def register_umap_callbacks(app):
    """Register all UMAP-related callbacks."""

    # =========================================================================
    # CALLBACK: Load UMAP data on page mount or filter change
    # =========================================================================
    @app.callback(
        Output("umap-data-store", "data"),
        Output("ghost-points-store", "data"),
        Output("umap-necesidad-filter", "options"),
        Input("url", "pathname"),
        Input("umap-necesidad-filter", "value"),
        Input("umap-options", "value"),
        State("auth-state", "data"),
        State("auth-tokens-store", "data"),
        State("ghost-points-store", "data"),
        State("umap-data-store", "data"),
        prevent_initial_call=False,
    )
    def load_umap_data(
        pathname,
        necesidad_filter,
        options,
        auth_state,
        auth_tokens,
        current_ghosts,
        previous_umap_data,
    ):
        """
        Load UMAP data from backend API.

        Handles:
        - Initial page load
        - Filter changes (NECESIDAD, verified_only)
        - Ghost points cleanup on version change
        """
        # Only process for clustering page
        if pathname != "/clustering":
            raise PreventUpdate

        # Auth check (REGLA #7.6)
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("umap_callbacks.not_authenticated")
            raise PreventUpdate

        # Restore tokens for multi-worker (REGLA #7.6)
        if auth_tokens and "tokens" in auth_tokens:
            try:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])
            except Exception as e:
                logger.warning("umap_callbacks.token_restore_failed", error=str(e))

        # Build API params
        params = {"limit": 5000}

        if necesidad_filter:
            params["necesidad"] = necesidad_filter

        if options and "verified_only" in options:
            params["verified_only"] = True

        # Fetch data from backend
        try:
            response = request_coordinator.make_request(
                "GET",
                f"{BACKEND_URL}/api/v1/clustering/umap-data",
                params=params,
                timeout=30,
            )

            if response.status_code != 200:
                logger.error(
                    "umap_callbacks.api_error",
                    status=response.status_code,
                    body=response.text[:200],
                )
                return no_update, no_update, no_update

            data = response.json()

            # Check if version changed -> clear ghost points
            new_version = data.get("metadata", {}).get("version")
            old_version = (
                previous_umap_data.get("metadata", {}).get("version")
                if previous_umap_data
                else None
            )

            ghost_points = current_ghosts or []
            if new_version != old_version and old_version is not None:
                logger.info(
                    "umap_callbacks.version_changed",
                    old=old_version,
                    new=new_version,
                )
                ghost_points = []  # Clear stale ghosts

            # Build NECESIDAD filter options
            necesidades = set()
            for point in data.get("points", []):
                if point.get("necesidad"):
                    necesidades.add(point["necesidad"])

            filter_options = [
                {"label": n.replace("_", " ").title(), "value": n}
                for n in sorted(necesidades)
            ]

            logger.info(
                "umap_callbacks.data_loaded",
                points=len(data.get("points", [])),
                version=new_version,
            )

            return data, ghost_points, filter_options

        except Exception as e:
            logger.error("umap_callbacks.fetch_error", error=str(e))
            return no_update, no_update, no_update

    # =========================================================================
    # CALLBACK: Render UMAP scatter + Update KPIs (REGLA #11: Combined)
    # Both outputs depend on umap-data-store, so combined into single callback
    # =========================================================================
    @app.callback(
        Output("umap-scatter-container", "children"),
        Output("kpi-total-products", "children"),
        Output("kpi-with-coords", "children"),
        Output("kpi-coverage", "children"),
        Output("kpi-verified", "children"),
        Output("umap-version-badge", "children"),
        Input("umap-data-store", "data"),
        Input("ghost-points-store", "data"),
        Input("umap-color-by", "value"),
        State("umap-options", "value"),  # Changed to State (REGLA #11: umap-options Input only in load_umap_data)
        State("selected-product-store", "data"),
        prevent_initial_call=True,
    )
    def render_umap_and_kpis(umap_data, ghost_points, color_by, options, selected_id):
        """
        Render UMAP scatter plot and update KPIs.

        Combined callback per REGLA #11: umap-data-store Input in single callback.
        """
        from dash import html

        # Default KPI values
        kpi_defaults = ("--", "--", "--", "--", "--")

        if not umap_data or not umap_data.get("points"):
            empty_scatter = html.Div(
                "No hay datos UMAP disponibles. Ejecuta el script de entrenamiento.",
                className="text-center text-muted py-5",
            )
            return (empty_scatter,) + kpi_defaults

        # Render scatter
        a11y_mode = options and "a11y" in options
        scatter = create_umap_scatter(
            points=umap_data["points"],
            ghost_points=ghost_points,
            color_by=color_by or "necesidad",
            selected_id=selected_id,
            a11y_mode=a11y_mode,
            height=500,
        )

        # Calculate KPIs
        metadata = umap_data.get("metadata", {})
        points = umap_data.get("points", [])

        total = metadata.get("total_products", 0)
        with_coords = metadata.get("products_with_coords", 0)
        coverage = metadata.get("coverage_percent", 0)
        version = metadata.get("version", "N/A")

        # Count verified from points
        verified = sum(1 for p in points if p.get("human_verified"))

        return (
            scatter,
            f"{total:,}".replace(",", "."),
            f"{with_coords:,}".replace(",", "."),
            f"{coverage:.1f}%",
            f"{verified:,}".replace(",", "."),
            version,
        )

    # =========================================================================
    # CALLBACK: Handle point click -> show detail
    # =========================================================================
    @app.callback(
        Output("selected-product-store", "data"),
        Output("product-detail-panel", "children"),
        Input("umap-scatter", "clickData"),
        State("umap-data-store", "data"),
        prevent_initial_call=True,
    )
    def on_point_click(click_data, umap_data):
        """Handle click on UMAP point - show product detail."""
        if not click_data or not umap_data:
            raise PreventUpdate

        try:
            # Get clicked point index
            point_index = click_data["points"][0]["pointIndex"]
            points = umap_data.get("points", [])

            if point_index >= len(points):
                raise PreventUpdate

            product = points[point_index]

            logger.debug(
                "umap_callbacks.point_clicked",
                product_id=product.get("id"),
                name=product.get("product_name", "")[:50],
            )

            return product.get("id"), create_product_detail_card(product)

        except (KeyError, IndexError) as e:
            logger.warning("umap_callbacks.click_parse_error", error=str(e))
            raise PreventUpdate

    # =========================================================================
    # CALLBACK: Add ghost point when product is moved
    # (Placeholder for W15 - Move functionality)
    # =========================================================================
    @app.callback(
        Output("ghost-points-store", "data", allow_duplicate=True),
        Input("btn-move-product", "n_clicks"),
        State("selected-product-store", "data"),
        State("umap-data-store", "data"),
        State("ghost-points-store", "data"),
        State("target-cluster-dropdown", "value"),
        State("auth-tokens-store", "data"),  # REGLA #7.6: Ready for W15 API calls
        prevent_initial_call=True,
    )
    def add_ghost_point(n_clicks, product_id, umap_data, ghost_points, target_cluster, auth_tokens):
        """
        Add ghost point when product is moved.

        This callback is a placeholder for W15 functionality.
        The btn-move-product and target-cluster-dropdown don't exist yet.
        """
        if not n_clicks or not product_id or not target_cluster:
            raise PreventUpdate

        # Find product in current data
        points = umap_data.get("points", []) if umap_data else []
        product = next((p for p in points if p.get("id") == product_id), None)

        if not product:
            raise PreventUpdate

        # Add to ghost points
        from datetime import datetime, timezone

        ghost_points = ghost_points or []
        ghost_points.append(
            {
                "id": product_id,
                "product_name": product.get("product_name"),
                "original_umap_x": product.get("umap_x"),
                "original_umap_y": product.get("umap_y"),
                "moved_to_cluster": target_cluster,
                "timestamp": datetime.now(timezone.utc).isoformat(),
            }
        )

        logger.info(
            "umap_callbacks.ghost_point_added",
            product_id=product_id,
            target=target_cluster,
        )

        return ghost_points
