"""
Callbacks para mostrar el progreso REAL de sincronización del catálogo.
Elimina completamente el "teatro" de progreso simulado.
"""


import dash_bootstrap_components as dbc
from dash import Input, Output, State, callback, html
from dash.exceptions import PreventUpdate
from utils.auth_helpers import get_auth_headers_from_tokens  # REGLA #7.6: Multi-worker token restoration


@callback(
    [Output("catalog-real-progress-display", "children"), Output("catalog-real-progress-interval", "disabled")],
    [Input("catalog-real-progress-interval", "n_intervals")],
    [
        State("backend-url", "data"),
        State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
    ],
)
def update_real_progress(n_intervals, backend_url, auth_tokens):
    """
    Actualiza el display de progreso con datos REALES del backend.
    NO simulación, NO teatro, solo datos reales.
    """
    if not backend_url:
        return html.Div("Backend no configurado"), True

    # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
    auth_headers = get_auth_headers_from_tokens(auth_tokens)

    try:
        # Usar RequestCoordinator para evitar rate limiting
        from utils.request_coordinator import request_coordinator

        # Cache corto para progreso en tiempo real
        # REGLA #7.6: Pass explicit auth_headers for multi-worker support
        data = request_coordinator.make_request(
            "/api/v1/system/sync/real-progress", cache_ttl=10, auth_headers=auth_headers
        )

        if data:
            sync_data = data.get("sync", {})
            error_data = sync_data.get("errors", {})

            if not sync_data.get("is_active", False):
                # No hay sincronización activa - verificar si hay errores previos
                if error_data.get("last_error"):
                    # Mostrar último error antes de completar
                    return (
                        html.Div(
                            [
                                dbc.Alert(
                                    [
                                        html.I(className="fas fa-exclamation-triangle me-2"),
                                        html.Strong("Sincronización completada con errores"),
                                        html.Br(),
                                        html.Small(
                                            f"Último error: {error_data.get('user_message', 'Error no especificado')}"
                                        ),
                                    ],
                                    color="warning",
                                    className="text-center",
                                ),
                                html.Div(
                                    [
                                        html.Small(
                                            f"Total de errores: {error_data.get('error_count', 1)}",
                                            className="text-muted",
                                        )
                                    ],
                                    className="text-center mt-2",
                                ),
                            ]
                        ),
                        True,
                    )  # Desactivar interval
                else:
                    # Sincronización exitosa sin errores
                    return (
                        html.Div(
                            dbc.Alert(
                                "✅ Sistema sincronizado - No hay operaciones activas",
                                color="success",
                                className="text-center",
                            )
                        ),
                        True,
                    )  # Desactivar interval

            # Hay sincronización activa - mostrar datos REALES
            primary_sync = sync_data.get("primary_sync", data.get("primary_sync", {}))
            if not primary_sync:
                return html.Div("Error obteniendo datos de sincronización"), False

            # Extraer métricas REALES
            component = primary_sync["component"].upper()
            progress = primary_sync.get("progress", 0)

            # Mensaje adaptado según errores
            if error_data.get("last_error"):
                error_message = error_data.get("user_message", error_data.get("last_error", ""))
                error_count = error_data.get("error_count", 1)
                base_message = primary_sync.get("message", "Procesando...")
                if error_data.get("recoverable", True):
                    message = f"⚠️ {error_message} (continuando...)"
                else:
                    message = f"❌ {error_message} (pausado)"
            else:
                message = primary_sync.get("message", "Procesando...")

            # Métricas reales si están disponibles
            real_metrics = primary_sync.get("real_metrics", {})

            # Crear display de progreso REAL
            progress_display = []

            # Header con componente activo
            progress_display.append(
                html.Div(
                    [
                        html.I(className="fas fa-sync fa-spin me-2 text-primary"),
                        html.H5(f"Sincronización {component}", className="d-inline mb-0"),
                    ],
                    className="mb-3",
                )
            )

            # Barra de progreso REAL con color adaptado a errores
            progress_color = "primary"
            if error_data.get("last_error"):
                progress_color = "warning" if error_data.get("recoverable", True) else "danger"
            elif progress >= 100:
                progress_color = "success"

            progress_display.append(
                dbc.Progress(
                    value=progress,
                    label=f"{progress}%",
                    color=progress_color,
                    striped=True,
                    animated=True,
                    className="mb-2",
                    style={"height": "30px"},
                )
            )

            # Mensaje actual
            progress_display.append(html.P(message, className="text-center mb-3"))

            # Información de errores si existe
            if error_data.get("last_error"):
                error_alert_color = "warning" if error_data.get("recoverable", True) else "danger"
                progress_display.append(
                    dbc.Alert(
                        [
                            html.Div(
                                [
                                    html.I(
                                        className=f"fas fa-{'exclamation-triangle' if error_data.get('recoverable') else 'times-circle'} me-2"
                                    ),
                                    html.Strong("Información de errores:"),
                                ],
                                className="mb-2",
                            ),
                            html.Div(
                                [
                                    html.Small(
                                        [
                                            f"Errores detectados: {error_data.get('error_count', 1)}",
                                            html.Br(),
                                            f"Estado: {'Recuperable automáticamente' if error_data.get('recoverable', True) else 'Requiere intervención'}",
                                            html.Br(),
                                            f"Último error: {error_data.get('user_message', error_data.get('last_error', ''))[:100]}...",
                                        ],
                                        className="text-muted",
                                    )
                                ]
                            ),
                        ],
                        color=error_alert_color,
                        className="mb-3",
                        style={"fontSize": "0.9rem"},
                    )
                )

            # Métricas detalladas si están disponibles
            if real_metrics:
                metrics_cards = []

                # Items procesados
                if "processed" in real_metrics and "total" in real_metrics:
                    metrics_cards.append(
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.H4(f"{real_metrics['processed']:,}", className="mb-0 text-primary"),
                                        html.Small(f"de {real_metrics['total']:,}", className="text-muted"),
                                        html.Small("Items procesados", className="d-block text-muted"),
                                    ],
                                    className="text-center",
                                )
                            ],
                            width=6,
                            md=3,
                        )
                    )

                # Chunks
                if "current_chunk" in real_metrics and "total_chunks" in real_metrics:
                    metrics_cards.append(
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.H4(
                                            f"{real_metrics['current_chunk']}/{real_metrics['total_chunks']}",
                                            className="mb-0 text-info",
                                        ),
                                        html.Small("Chunks", className="text-muted"),
                                    ],
                                    className="text-center",
                                )
                            ],
                            width=6,
                            md=3,
                        )
                    )

                # Velocidad
                if "speed" in real_metrics and real_metrics["speed"] > 0:
                    metrics_cards.append(
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.H4(f"{real_metrics['speed']:.1f}", className="mb-0 text-warning"),
                                        html.Small("Items/seg", className="text-muted"),
                                    ],
                                    className="text-center",
                                )
                            ],
                            width=6,
                            md=3,
                        )
                    )

                # ETA
                if "eta_seconds" in real_metrics and real_metrics["eta_seconds"]:
                    eta_min = real_metrics["eta_seconds"] // 60
                    eta_sec = real_metrics["eta_seconds"] % 60
                    metrics_cards.append(
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.H4(f"{eta_min}m {eta_sec}s", className="mb-0 text-success"),
                                        html.Small("Tiempo restante", className="text-muted"),
                                    ],
                                    className="text-center",
                                )
                            ],
                            width=6,
                            md=3,
                        )
                    )

                if metrics_cards:
                    progress_display.append(
                        html.Div(
                            [
                                html.Hr(),
                                html.H6("📊 Métricas REALES", className="text-muted mb-3"),
                                dbc.Row(metrics_cards),
                            ]
                        )
                    )

                # Tiempo transcurrido
                if "elapsed_seconds" in real_metrics:
                    elapsed_min = real_metrics["elapsed_seconds"] // 60
                    elapsed_sec = real_metrics["elapsed_seconds"] % 60
                    progress_display.append(
                        html.Div(
                            html.Small(f"⏱️ Tiempo transcurrido: {elapsed_min}m {elapsed_sec}s", className="text-muted"),
                            className="text-center mt-3",
                        )
                    )

            return html.Div(progress_display), False  # Mantener interval activo

        else:
            return (
                html.Div(dbc.Alert("Error obteniendo progreso: No se pudo conectar al servidor", color="danger")),
                False,
            )

    except Exception as e:
        return html.Div(dbc.Alert(f"Error obteniendo progreso: {str(e)}", color="danger")), False


@callback(
    Output("catalog-sync-details", "children"),
    Input("view-sync-details-btn", "n_clicks"),
    [
        State("backend-url", "data"),
        State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
    ],
    prevent_initial_call=True,
)
def show_sync_details(n_clicks, backend_url, auth_tokens):
    """
    Muestra detalles completos de la sincronización con datos REALES.
    """
    if not n_clicks or not backend_url:
        raise PreventUpdate

    # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
    auth_headers = get_auth_headers_from_tokens(auth_tokens)

    try:
        # Usar RequestCoordinator para evitar rate limiting
        from utils.request_coordinator import request_coordinator

        # Cache corto para detalles en tiempo real
        # REGLA #7.6: Pass explicit auth_headers for multi-worker support
        data = request_coordinator.make_request(
            "/api/v1/system/sync/real-progress", cache_ttl=5, auth_headers=auth_headers
        )

        if data:
            components = data.get("components", {})

            details_cards = []

            for comp_name, comp_data in components.items():
                # Color basado en estado
                status = comp_data.get("status", "unknown")
                color_map = {
                    "ready": "success",
                    "updating": "primary",
                    "initializing": "info",
                    "error": "danger",
                    "not_initialized": "secondary",
                }
                color = color_map.get(status, "secondary")

                # Crear card para cada componente
                card_content = [
                    html.H5(comp_name.upper(), className="card-title"),
                    html.P(f"Estado: {status}", className=f"text-{color}"),
                ]

                if comp_data.get("message"):
                    card_content.append(html.P(comp_data["message"], className="text-muted small"))

                if comp_data.get("real_metrics"):
                    metrics = comp_data["real_metrics"]
                    metrics_text = []

                    if "processed" in metrics:
                        metrics_text.append(f"Procesados: {metrics['processed']:,}")
                    if "total" in metrics:
                        metrics_text.append(f"Total: {metrics['total']:,}")
                    if "speed" in metrics and metrics["speed"] > 0:
                        metrics_text.append(f"Velocidad: {metrics['speed']:.1f} items/seg")
                    if "errors" in metrics and metrics["errors"] > 0:
                        metrics_text.append(f"Errores: {metrics['errors']}")

                    if metrics_text:
                        card_content.append(html.Ul([html.Li(text) for text in metrics_text], className="small"))

                details_cards.append(
                    dbc.Col(
                        dbc.Card(dbc.CardBody(card_content), color=color, outline=True),
                        width=12,
                        md=4,
                        className="mb-3",
                    )
                )

            return dbc.Row(details_cards)

        else:
            return dbc.Alert("Error obteniendo detalles: No se pudo conectar al servidor", color="danger")

    except Exception as e:
        return dbc.Alert(f"Error: {str(e)}", color="danger")
