"""
Callbacks específicos para manejo transparente de errores de sincronización del catálogo.
Implementa la FASE 3 del plan de transparencia: Error Transparency en UI.
"""

from datetime import datetime
from typing import Dict

import dash_bootstrap_components as dbc
import requests
from dash import Input, Output, State, callback, ctx, html, no_update
from dash.exceptions import PreventUpdate
from utils.auth_helpers import get_auth_headers_from_tokens  # REGLA #7.6: Multi-worker token restoration

# Mensajes de error específicos para farmacéuticos españoles
PHARMACY_ERROR_MESSAGES = {
    "cima_connection": "Error de conexión con CIMA (base de datos de medicamentos oficial)",
    "cima_timeout": "CIMA responde lentamente. Reintentando automáticamente...",
    "nomenclator_format": "Archivo nomenclátor con formato incorrecto del Ministerio",
    "nomenclator_download": "Error descargando nomenclátor oficial del Ministerio de Sanidad",
    "product_matching": "Algunos medicamentos no se encuentran en catálogos oficiales",
    "memory_limit": "Procesando en lotes más pequeños por limitaciones del servidor",
    "database_lock": "Base de datos temporalmente ocupada. Reintentando...",
    "chunk_processing": "Error procesando lote de productos. Continuando con el siguiente...",
    "api_rate_limit": "Límite de peticiones alcanzado. Esperando antes de continuar...",
    "unknown": "Error inesperado durante la sincronización",
}

# Colores y estados basados en tipo de error
ERROR_UI_CONFIG = {
    "recoverable": {
        "color": "warning",
        "icon": "fas fa-exclamation-triangle",
        "actions": ["retry", "wait", "continue"],
    },
    "critical": {"color": "danger", "icon": "fas fa-times-circle", "actions": ["contact_support", "check_connection"]},
    "temporary": {"color": "info", "icon": "fas fa-clock", "actions": ["wait", "automatic_retry"]},
    "connection": {"color": "warning", "icon": "fas fa-wifi", "actions": ["check_connection", "retry"]},
}


def create_error_modal(error_data: Dict) -> dbc.Modal:
    """
    Crea modal de error transparente con información real del backend.

    Args:
        error_data: Datos de error del endpoint /api/v1/system/sync/real-progress
    """
    if not error_data:
        return dbc.Modal(id="catalog-error-modal", is_open=False)

    last_error = error_data.get("last_error", "Error desconocido")
    error_count = error_data.get("error_count", 0)
    recoverable = error_data.get("recoverable", True)
    user_message = error_data.get("user_message", last_error)
    technical_details = error_data.get("technical_details", "")
    suggested_actions = error_data.get("suggested_actions", [])

    # Determinar configuración UI basada en tipo de error
    error_type = "recoverable" if recoverable else "critical"
    if "timeout" in last_error.lower() or "connection" in last_error.lower():
        error_type = "connection"
    elif "temporary" in last_error.lower() or "retry" in last_error.lower():
        error_type = "temporary"

    ui_config = ERROR_UI_CONFIG.get(error_type, ERROR_UI_CONFIG["recoverable"])

    # Traducir mensaje técnico a mensaje de farmacia
    pharmacy_message = user_message
    for key, message in PHARMACY_ERROR_MESSAGES.items():
        if key in last_error.lower():
            pharmacy_message = message
            break

    return dbc.Modal(
        [
            dbc.ModalHeader(
                [
                    html.I(className=f"{ui_config['icon']} me-2 text-{ui_config['color']}"),
                    html.Span("Error en Sincronización del Catálogo"),
                ],
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    # Mensaje principal para el farmacéutico
                    html.Div(
                        [
                            html.H5("¿Qué pasó?", className="text-dark mb-2"),
                            html.P(pharmacy_message, className="mb-3"),
                            # Estadísticas de error si hay múltiples errores
                            (
                                dbc.Alert(
                                    [
                                        html.I(className="fas fa-info-circle me-2"),
                                        f"Este es el error número {error_count}. ",
                                        (
                                            "El sistema está reintentando automáticamente."
                                            if recoverable
                                            else "Requiere intervención manual."
                                        ),
                                    ],
                                    color=ui_config["color"],
                                    className="mb-3",
                                )
                                if error_count > 1
                                else None
                            ),
                        ]
                    ),
                    # Acciones recomendadas
                    html.Div(
                        [
                            html.H5("¿Qué puedes hacer?", className="text-dark mb-2"),
                            html.Ul(
                                [
                                    html.Li(action_text)
                                    for action_text in [
                                        "Reintentar la operación" if "retry" in suggested_actions else None,
                                        (
                                            "Esperar - el sistema reintenta automáticamente"
                                            if "wait" in suggested_actions
                                            else None
                                        ),
                                        (
                                            "Verificar conexión a internet"
                                            if "check_connection" in suggested_actions
                                            else None
                                        ),
                                        (
                                            "Contactar soporte si el problema persiste"
                                            if "contact_support" in suggested_actions
                                            else None
                                        ),
                                        (
                                            "El proceso continuará automáticamente"
                                            if "continue" in suggested_actions
                                            else None
                                        ),
                                    ]
                                    if action_text
                                ],
                                className="mb-3",
                            ),
                        ]
                    ),
                    # Detalles técnicos colapsibles
                    (
                        html.Div(
                            [
                                dbc.Button(
                                    html.Span([html.I(className="fas fa-code me-2"), "Ver detalles técnicos"]),
                                    id="error-technical-toggle",
                                    color="light",
                                    outline=True,
                                    size="sm",
                                    className="mb-2",
                                ),
                                dbc.Collapse(
                                    [
                                        html.H6("Detalles técnicos:", className="text-muted mt-2"),
                                        html.Code(
                                            technical_details or last_error,
                                            className="d-block bg-light p-2 rounded small text-dark",
                                        ),
                                        html.Small(
                                            f"Timestamp: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}",
                                            className="text-muted d-block mt-2",
                                        ),
                                    ],
                                    id="error-technical-collapse",
                                ),
                            ]
                        )
                        if technical_details
                        else None
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button("Cerrar", id="error-modal-close", color="secondary", outline=True, className="me-2"),
                    (
                        dbc.Button(
                            html.Span([html.I(className="fas fa-redo me-2"), "Reintentar Ahora"]),
                            id="error-modal-retry",
                            variant="primary",
                        )
                        if recoverable and "retry" in suggested_actions
                        else None
                    ),
                ]
            ),
        ],
        id="catalog-error-modal",
        size="lg",
        centered=True,
        is_open=True,
        backdrop="static",
        keyboard=False,
    )


@callback(
    [
        Output("catalog-error-display", "children"),
        Output("catalog-error-modal", "is_open"),
        Output("catalog-error-modal", "children"),
        Output("error-technical-collapse", "is_open"),  # ✅ UNIFIED: Added toggle output
    ],
    [
        Input("catalog-real-progress-interval", "n_intervals"),
        Input("error-modal-close", "n_clicks"),
        Input("error-technical-toggle", "n_clicks"),
    ],
    [
        State("backend-url", "data"),
        State("error-technical-collapse", "is_open"),
        State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
    ],
    prevent_initial_call=True,
)
def handle_sync_errors(n_intervals, close_clicks, toggle_clicks, backend_url, collapse_open, auth_tokens):
    """
    Maneja la detección y display de errores de sincronización usando datos REALES.
    NO simulación, NO teatro - solo transparencia total.

    ✅ UNIFIED CALLBACK: Combines error handling + toggle collapse (REGLA #11)
    """
    if not backend_url:
        raise PreventUpdate

    triggered_id = ctx.triggered_id

    # Cerrar modal si se hace clic en cerrar
    if triggered_id == "error-modal-close":
        return html.Div(), False, [], no_update  # ✅ Added no_update for toggle

    # ✅ UNIFIED: Toggle de detalles técnicos (moved from separate callback)
    if triggered_id == "error-technical-toggle":
        return no_update, no_update, no_update, not collapse_open  # Only toggle collapse

    # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
    auth_headers = get_auth_headers_from_tokens(auth_tokens)

    try:
        # Usar RequestCoordinator para evitar rate limiting
        from utils.request_coordinator import request_coordinator

        # Cache muy corto para errores en tiempo real
        # REGLA #7.6: Pass explicit auth_headers for multi-worker support
        data = request_coordinator.make_request(
            "/api/v1/system/sync/real-progress", cache_ttl=8, auth_headers=auth_headers
        )

        if not data:
            # Error de API - mostrar error de conexión
            error_display = dbc.Alert(
                html.Div([html.I(className="fas fa-exclamation-triangle me-2"), "Error de conexión con backend"]), color="danger"
            )

            return error_display, False, [], no_update  # ✅ Added no_update for toggle
        sync_data = data.get("sync", {})
        error_data = sync_data.get("errors", {})

        # ¿Hay errores REALES?
        if error_data.get("last_error"):

            # Display inline de error (no modal aún)
            error_alert = dbc.Alert(
                [
                    html.Div(
                        [
                            html.I(className="fas fa-exclamation-triangle me-2"),
                            html.Strong("Error detectado: "),
                            error_data.get("user_message", error_data.get("last_error", "Error desconocido")),
                        ]
                    ),
                    html.Div(
                        [
                            html.Small(f"Errores totales: {error_data.get('error_count', 1)}", className="me-3"),
                            html.Small(
                                (
                                    "Se reintenta automáticamente"
                                    if error_data.get("recoverable", True)
                                    else "Requiere intervención"
                                ),
                                className="fw-bold",
                            ),
                        ],
                        className="mt-1",
                    ),
                ],
                color="warning" if error_data.get("recoverable", True) else "danger",
                dismissable=False,
            )

            # Crear modal de error con todos los detalles
            error_modal = create_error_modal(error_data)

            return error_alert, True, error_modal.children, no_update  # ✅ Added no_update for toggle

        else:
            # No hay errores - sistema funcionando correctamente
            return html.Div(), False, [], no_update  # ✅ Added no_update for toggle

    except requests.RequestException as e:
        # Error de conexión
        connection_error = dbc.Alert(
            [
                html.I(className="fas fa-wifi me-2"),
                html.Strong("Error de conexión: "),
                f"No se puede conectar con el backend: {str(e)}",
            ],
            color="danger",
        )

        return connection_error, False, [], no_update  # ✅ Added no_update for toggle

    except Exception as e:
        # Error inesperado
        unexpected_error = dbc.Alert(
            [html.I(className="fas fa-bug me-2"), html.Strong("Error inesperado: "), str(e)], color="danger"
        )

        return unexpected_error, False, [], no_update  # ✅ Added no_update for toggle


# ❌ REMOVED: Duplicate callback for error-technical-toggle (unified above)
# This callback violated REGLA #11: ONE INPUT ONE CALLBACK
# Now handled by handle_sync_errors callback with multiple outputs


@callback(
    Output("catalog-sync-status-indicator", "children"),
    Input("catalog-real-progress-interval", "n_intervals"),
    [
        State("backend-url", "data"),
        State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
    ],
    prevent_initial_call=True,
)
def update_sync_status_indicator(n_intervals, backend_url, auth_tokens):
    """
    Indicador visual del estado de sincronización con errores.
    Para sidebar o área de estado general.
    """
    if not backend_url:
        return html.Div()

    # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
    auth_headers = get_auth_headers_from_tokens(auth_tokens)

    try:
        # Usar RequestCoordinator para evitar rate limiting
        from utils.request_coordinator import request_coordinator

        # Cache muy corto para errores en tiempo real
        # REGLA #7.6: Pass explicit auth_headers for multi-worker support
        data = request_coordinator.make_request(
            "/api/v1/system/sync/real-progress", cache_ttl=8, auth_headers=auth_headers
        )

        if data:
            sync_data = data.get("sync", {})
            error_data = sync_data.get("errors", {})
            is_syncing = sync_data.get("is_active", False)

            if error_data.get("last_error"):
                # Hay errores - mostrar indicador de warning/error
                error_count = error_data.get("error_count", 1)
                recoverable = error_data.get("recoverable", True)

                return html.Div(
                    [
                        html.I(
                            className=f"fas fa-exclamation-triangle text-{'warning' if recoverable else 'danger'} me-2"
                        ),
                        html.Small(
                            f"Error en sync ({error_count})", className=f"text-{'warning' if recoverable else 'danger'}"
                        ),
                    ],
                    className="d-flex align-items-center",
                )

            elif is_syncing:
                # Sincronizando sin errores
                return html.Div(
                    [
                        html.I(className="fas fa-sync fa-spin text-primary me-2"),
                        html.Small("Sincronizando...", className="text-primary"),
                    ],
                    className="d-flex align-items-center",
                )

            else:
                # Todo OK
                return html.Div(
                    [
                        html.I(className="fas fa-check-circle text-success me-2"),
                        html.Small("Sincronizado", className="text-success"),
                    ],
                    className="d-flex align-items-center",
                )

        else:
            # Error de API
            return html.Div(
                [
                    html.I(className="fas fa-exclamation-circle text-danger me-2"),
                    html.Small("Error API", className="text-danger"),
                ],
                className="d-flex align-items-center",
            )

    except:
        # Error de conexión
        return html.Div(
            [html.I(className="fas fa-wifi text-muted me-2"), html.Small("Sin conexión", className="text-muted")],
            className="d-flex align-items-center",
        )


@callback(
    Output("catalog-real-progress-interval", "disabled"),
    Input("catalog-error-modal", "is_open"),
    State("catalog-real-progress-interval", "disabled"),
)
def manage_polling_during_errors(modal_open, current_disabled):
    """
    Mantiene el polling activo incluso durante errores para mostrar recuperación.
    """
    # Siempre mantener polling activo durante errores para transparencia
    return False  # Nunca deshabilitar durante errores
