"""
Callbacks para la gestión del catálogo de productos.
Maneja actualizaciones en tiempo real, progreso y control del catálogo.

REFACTORIZACIÓN REGLA #11 (2025-10-20):
- Sistema de Intermediate Store Decoupling para evitar outputs duplicados
- Stores intermedios: catalog-modal-action, catalog-toast-action, catalog-progress-action
- Callbacks unificados que centralizan control de outputs finales
"""

import logging
from datetime import datetime
from typing import Dict, List, Optional, Tuple

import dash_bootstrap_components as dbc
from components.catalog_progress import create_catalog_progress_component, start_catalog_update, update_catalog_progress
from components.toast_manager import error_toast, info_toast, success_toast, warning_toast
from dash import Input, Output, State, callback, ctx, html, no_update
from dash.exceptions import PreventUpdate
from utils.request_coordinator import request_coordinator
from utils.auth_helpers import get_auth_headers_from_tokens  # REGLA #7.6: Multi-worker token restoration

# Logger para debugging
logger = logging.getLogger(__name__)


# ==============================================================================
# CALLBACKS UNIFICADOS - CONTROL CENTRALIZADO DE OUTPUTS
# Estos callbacks leen de stores intermedios y controlan outputs finales
# ==============================================================================


@callback(
    [Output("catalog-update-modal", "is_open"), Output("catalog-progress-modal", "is_open")],
    Input("catalog-modal-action-store", "data"),
    prevent_initial_call=True,
)
def unified_catalog_modal_controller(action_data: Optional[Dict]) -> Tuple[bool, bool]:
    """
    Controlador unificado para TODOS los modales de catálogo.

    REGLA #11: Este es el ÚNICO callback que controla:
    - catalog-update-modal.is_open
    - catalog-progress-modal.is_open

    Acciones soportadas:
    - open_confirmation: Abrir modal de confirmación
    - close_confirmation: Cerrar modal de confirmación
    - open_progress: Abrir modal de progreso
    - close_progress: Cerrar modal de progreso
    """
    if not action_data:
        raise PreventUpdate

    action = action_data.get("action")

    if action == "open_confirmation":
        return True, no_update
    elif action == "close_confirmation":
        return False, no_update
    elif action == "open_progress":
        return no_update, True
    elif action == "close_progress":
        return no_update, False
    elif action == "start_sync":
        # Cuando inicia sincronización: cerrar confirmación, abrir progreso
        return False, True

    raise PreventUpdate


@callback(Output("toast-trigger-store", "data"), Input("catalog-toast-action-store", "data"), prevent_initial_call=True)
def unified_catalog_toast_controller(action_data: Optional[Dict]) -> Dict:
    """
    Controlador unificado para TODOS los toasts de catálogo.

    REGLA #11: Este es el ÚNICO callback que escribe a toast-trigger-store
    desde callbacks de catálogo.

    Formato de action_data:
    {
        "action": "show_toast",
        "toast_type": "success" | "error" | "warning" | "info",
        "message": "Mensaje del toast",
        "title": "Título opcional"
    }
    """
    if not action_data:
        raise PreventUpdate

    action = action_data.get("action")

    if action == "show_toast":
        toast_type = action_data.get("toast_type", "info")
        message = action_data.get("message", "")
        title = action_data.get("title", "")

        if toast_type == "success":
            return success_toast(message, title)
        elif toast_type == "error":
            return error_toast(message, title)
        elif toast_type == "warning":
            return warning_toast(message, title)
        else:
            return info_toast(message, title)

    raise PreventUpdate


@callback(
    [
        Output("catalog-progress-store", "data"),
        Output("sidebar-progress-store", "data"),
        Output("catalog-info-store", "data"),
    ],
    Input("catalog-progress-action-store", "data"),
    [
        State("catalog-progress-store", "data"),
        State("sidebar-progress-store", "data"),
        State("catalog-info-store", "data"),
    ],
    prevent_initial_call=True,
)
def unified_catalog_progress_controller(
    action_data: Optional[Dict],
    current_progress: Optional[Dict],
    current_sidebar: Optional[Dict],
    current_catalog_info: Optional[Dict],
) -> Tuple[Dict, Dict, Dict]:
    """
    Controlador unificado para TODOS los stores de progreso de catálogo.

    REGLA #11: Este es el ÚNICO callback que escribe a:
    - catalog-progress-store.data
    - sidebar-progress-store.data
    - catalog-info-store.data

    Acciones soportadas:
    - start_sync: Iniciar sincronización
    - update_progress: Actualizar progreso
    - complete_sync: Marcar como completado
    - error_sync: Marcar error
    """
    if not action_data:
        raise PreventUpdate

    action = action_data.get("action")

    if action == "start_sync":
        # Inicializar progreso
        progress_data = start_catalog_update()

        # Configurar sidebar
        sidebar_progress = {
            "active": True,
            "delegate_to": "catalog-progress-store",
            "message": f"Actualizando {action_data.get('update_type', 'catálogo')}...",
        }

        # Actualizar estado del catálogo
        catalog_info = current_catalog_info.copy() if current_catalog_info else {}
        catalog_info.update({"is_syncing": True, "sync_message": "Iniciando sincronización..."})

        return progress_data, sidebar_progress, catalog_info

    elif action == "update_progress":
        # Actualizar progreso existente
        if not current_progress:
            raise PreventUpdate

        updated_progress = update_catalog_progress(
            current_progress,
            phase=action_data.get("phase"),
            progress=action_data.get("progress"),
            message=action_data.get("message"),
            stats_update=action_data.get("stats_update"),
        )

        return updated_progress, no_update, no_update

    elif action == "complete_sync":
        # Marcar sincronización completada
        progress_data = current_progress.copy() if current_progress else {}
        progress_data.update({"active": False, "current_phase": "completed", "progress": 100})

        sidebar_progress = {"active": False}

        catalog_info = current_catalog_info.copy() if current_catalog_info else {}
        catalog_info.update({"is_syncing": False, "sync_message": ""})

        return progress_data, sidebar_progress, catalog_info

    elif action == "error_sync":
        # Marcar error en sincronización
        progress_data = current_progress.copy() if current_progress else {}
        progress_data.update({"active": False, "current_phase": "error"})

        sidebar_progress = {"active": False}

        catalog_info = current_catalog_info.copy() if current_catalog_info else {}
        catalog_info.update({"is_syncing": False, "sync_message": ""})

        return progress_data, sidebar_progress, catalog_info

    raise PreventUpdate


# ==============================================================================
# CALLBACKS DE LÓGICA DE NEGOCIO - ESCRIBEN A STORES INTERMEDIOS
# Estos callbacks contienen la lógica pero NO controlan outputs finales
# ==============================================================================


@callback(
    [
        Output("catalog-modal-action-store", "data", allow_duplicate=True),
        Output("last-catalog-update-time", "children"),
        Output("catalog-update-status", "children"),
    ],
    [Input("catalog-update-button", "n_clicks"), Input("catalog-update-cancel", "n_clicks")],
    [State("catalog-info-store", "data"), State("catalog-update-modal", "is_open")],
    prevent_initial_call=True,
)
def manage_catalog_update_modal(
    update_clicks: Optional[int], cancel_clicks: Optional[int], catalog_info: Optional[Dict], is_open: bool
) -> Tuple[Dict, str, str]:
    """
    Gestiona el modal de confirmación para actualizar el catálogo.

    REGLA #11: Este callback escribe al store intermedio catalog-modal-action-store.
    El callback unified_catalog_modal_controller lee de ese store y controla los modales.
    """
    if not ctx.triggered:
        raise PreventUpdate

    trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

    # Verificar que realmente hay un click (no None)
    if trigger_id == "catalog-update-button" and not update_clicks:
        raise PreventUpdate
    if trigger_id == "catalog-update-cancel" and not cancel_clicks:
        raise PreventUpdate

    # Cerrar modal si se cancela
    if trigger_id == "catalog-update-cancel":
        modal_action = {"action": "close_confirmation", "timestamp": datetime.now().isoformat()}
        return modal_action, no_update, no_update

    # Abrir modal y actualizar información SOLO si no está ya abierto
    elif trigger_id == "catalog-update-button":
        # Si el modal ya está abierto, no hacer nada
        if is_open:
            raise PreventUpdate

        last_update = "Nunca"
        status_text = "El catálogo requiere actualización inicial"

        if catalog_info and catalog_info.get("last_update"):
            try:
                last_update_dt = datetime.fromisoformat(catalog_info["last_update"])
                last_update = last_update_dt.strftime("%d/%m/%Y a las %H:%M")

                days_since = (datetime.now() - last_update_dt).days
                if days_since == 0:
                    status_text = "✅ El catálogo está actualizado"
                elif days_since < 5:
                    status_text = f"✅ Actualizado hace {days_since} días"
                elif days_since < 10:
                    status_text = f"⚠️ Recomendado actualizar (hace {days_since} días)"
                else:
                    status_text = f"⚠️ Necesario actualizar (hace {days_since} días)"
            except:
                pass

        modal_action = {"action": "open_confirmation", "timestamp": datetime.now().isoformat()}
        return modal_action, last_update, status_text

    raise PreventUpdate


@callback(
    [
        Output("catalog-progress-action-store", "data", allow_duplicate=True),
        Output("catalog-modal-action-store", "data", allow_duplicate=True),
        Output("catalog-toast-action-store", "data", allow_duplicate=True),
    ],
    [
        Input("catalog-update-confirm", "n_clicks"),
        Input("update-cima-button", "n_clicks"),
        Input("update-nomenclator-button", "n_clicks"),
    ],
    [State("auth-tokens-store", "data")],  # REGLA #7.6: Multi-worker token restoration
    prevent_initial_call=True,
    suppress_callback_exceptions=True,
)
def start_catalog_update_process(
    confirm_clicks: Optional[int], cima_clicks: Optional[int], nomenclator_clicks: Optional[int],
    auth_tokens: Optional[Dict]  # REGLA #7.6
) -> Tuple[Dict, Dict, Dict]:
    """
    Inicia el proceso de actualización del catálogo.

    REGLA #11: Este callback escribe a stores intermedios:
    - catalog-progress-action-store: Para controlar progreso
    - catalog-modal-action-store: Para cerrar modal de confirmación
    - catalog-toast-action-store: Para mostrar notificaciones

    CRÍTICO: Solo se ejecuta con clicks REALES del usuario, no con actualizaciones
    automáticas de stores, intervalos, o renderizaciones.
    """
    # PRIMERA BARRERA: Verificar que hay un trigger
    if not ctx.triggered:
        logger.debug("[CATALOG SYNC] Prevented - No trigger")
        raise PreventUpdate

    # SEGUNDA BARRERA: Verificar que solo hay UN trigger (no múltiples simultáneos)
    if len(ctx.triggered) > 1:
        logger.warning(f"[CATALOG SYNC] Prevented - Multiple triggers: {len(ctx.triggered)}")
        raise PreventUpdate

    # Obtener el botón que fue clickeado
    trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
    trigger_prop = ctx.triggered[0]["prop_id"]
    trigger_value = ctx.triggered[0]["value"]

    # LOGGING DETALLADO: Registrar TODOS los intentos de ejecución
    logger.warning(
        f"[CATALOG SYNC DEBUG] Callback triggered - ID: {trigger_id}, Prop: {trigger_prop}, Value: {trigger_value}, Type: {type(trigger_value)}"
    )

    # CRÍTICO: Verificar que el trigger es un botón (n_clicks property)
    # y que tiene un valor válido > 0 (clicks reales)
    if not trigger_prop.endswith(".n_clicks"):
        # El trigger no es un click de botón, prevenir ejecución
        logger.warning(f"[CATALOG SYNC DEBUG] PREVENTED - Not a button click: {trigger_prop}")
        raise PreventUpdate

    if trigger_value is None or trigger_value <= 0:
        # Sin clicks reales o valor inválido, prevenir ejecución
        logger.warning(f"[CATALOG SYNC DEBUG] PREVENTED - Invalid click value: {trigger_value}")
        raise PreventUpdate

    # Debug: verificar qué botón se detectó
    logger.info(f"[CATALOG SYNC] ✅ ACCEPTED - Trigger: {trigger_id} with value: {trigger_value}")

    # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
    auth_headers = get_auth_headers_from_tokens(auth_tokens)

    # Solo procesar el botón específico que fue clickeado
    if trigger_id not in ["catalog-update-confirm", "update-cima-button", "update-nomenclator-button"]:
        raise PreventUpdate

    # Determinar qué endpoint llamar según el botón - Usar endpoint consolidado
    endpoint = "/api/v1/system/sync"

    # Mapeo explícito de botones a targets
    button_mapping = {
        "update-cima-button": ("cima", "CIMA"),
        "update-nomenclator-button": ("nomenclator", "Nomenclátor"),
        "catalog-update-confirm": ("all", "Catálogo completo"),
        "catalog-update-button": ("all", "Catálogo completo"),  # Por si acaso
    }

    # Obtener target y tipo basado en el botón
    if trigger_id in button_mapping:
        target, update_type = button_mapping[trigger_id]
        payload = {"target": target}
    else:
        # Fallback - no debería llegar aquí
        print(f"[WARNING] Unknown trigger_id: {trigger_id}")
        payload = {"target": "all"}
        update_type = "Catálogo completo"

    # Iniciar actualización via API real del backend
    try:
        print(f"[FRONTEND DEBUG] Sending to {endpoint}: {payload}")
        print(f"[FRONTEND DEBUG] Trigger ID was: {trigger_id}")
        # Usar request_coordinator para enviar con autenticación
        # REGLA #7.6: Pass explicit auth_headers for multi-worker support
        response_data = request_coordinator.make_request(
            endpoint,
            method="POST",
            data=payload,
            cache_ttl=0,  # No cachear operaciones de actualización
            bypass_cache=True,
            auth_headers=auth_headers,
        )
        success = response_data is not None
        if not success:
            response_data = {"message": "Error al conectar con el servidor"}
    except Exception as e:
        success = False
        response_data = {"message": str(e)}

    if success:
        # Preparar acciones para stores intermedios
        progress_action = {"action": "start_sync", "update_type": update_type, "timestamp": datetime.now().isoformat()}

        modal_action = {
            "action": "start_sync",  # Cierra confirmación, abre progreso
            "timestamp": datetime.now().isoformat(),
        }

        toast_action = {
            "action": "show_toast",
            "toast_type": "info",
            "message": f"Sincronizando {update_type}. Este proceso puede tardar varios minutos.",
            "title": f"🔄 Actualización de {update_type} iniciada",
            "timestamp": datetime.now().isoformat(),
        }

        return progress_action, modal_action, toast_action
    else:
        # Error al iniciar
        error_msg = response_data.get("message", "Error desconocido")

        # No cambiar progreso (no iniciado)
        progress_action = {}

        # Cerrar modal de confirmación
        modal_action = {"action": "close_confirmation", "timestamp": datetime.now().isoformat()}

        # Mostrar toast de error
        toast_action = {
            "action": "show_toast",
            "toast_type": "error",
            "message": error_msg,
            "title": "❌ No se pudo iniciar la actualización",
            "timestamp": datetime.now().isoformat(),
        }

        return progress_action, modal_action, toast_action


@callback(
    [
        Output("catalog-progress-action-store", "data", allow_duplicate=True),
        Output("catalog-progress-content", "children"),
        Output("catalog-progress-alerts", "children"),
        Output("catalog-update-interval", "disabled"),
    ],
    Input("catalog-update-interval", "n_intervals"),
    [State("catalog-progress-store", "data"), State("auth-state", "data")],  # Issue #302: Agregar auth_state
    prevent_initial_call=True,
)
def update_catalog_progress_display(
    n_intervals: Optional[int], progress_data: Optional[Dict], auth_state: Optional[Dict]  # Issue #302
) -> Tuple[Dict, html.Div, html.Div, bool]:
    """
    Actualiza el progreso de la sincronización del catálogo.

    REGLA #11: Escribe a catalog-progress-action-store (intermedio)
    para actualizar los stores de progreso.

    Issue #302: Verificación proactiva de autenticación antes de llamadas API.
    """
    # Issue #302: Verificación proactiva de autenticación
    from utils.auth_helpers import is_user_authenticated

    if not is_user_authenticated(auth_state):
        logger.debug("[CATALOG_PROGRESS] User not authenticated - skipping progress check")
        raise PreventUpdate

    # Validación adicional para evitar IndexError en producción
    if n_intervals is None:
        raise PreventUpdate

    if not progress_data or not progress_data.get("active"):
        raise PreventUpdate

    # Obtener estado actual usando RequestCoordinator optimizado
    try:
        from utils.request_coordinator import get_catalog_status

        # Usar helper específico con cache extendido
        full_data = get_catalog_status()
        success = full_data is not None
        # Extraer solo la información del catálogo
        response_data = full_data.get("catalog", {}) if full_data else {}
    except Exception as e:
        success = False
        response_data = {}

    if not success:
        # Error obteniendo estado
        alert = dbc.Alert("Error obteniendo el estado de actualización", color="danger")
        return {}, no_update, alert, False

    # Buscar componentes en sincronización desde el response completo
    components = full_data.get("components", {})

    # Determinar qué componente está activo y su fase
    active_component = None
    active_message = "Sincronizando..."
    active_progress = 0

    # Prioridad: CIMA > NOMENCLATOR > CATALOG
    for comp_name, priority in [("cima", 1), ("nomenclator", 2), ("catalog", 3)]:
        comp_data = components.get(comp_name, {})
        if comp_data.get("status") == "initializing":
            if active_component is None or priority < active_component[1]:
                active_component = (comp_name.upper(), priority)
                active_message = comp_data.get("message", f"Actualizando {comp_name}...")
                active_progress = comp_data.get("progress", 0)

    # Mapear a fase de UI
    phase_map = {"NOMENCLATOR": "processing_nomenclator", "CIMA": "downloading_cima", "CATALOG": "updating_database"}

    if active_component:
        current_phase = phase_map.get(active_component[0], "checking")
    else:
        current_phase = "checking"

    # Preparar acción para actualizar progreso
    progress_action = {
        "action": "update_progress",
        "phase": current_phase,
        "progress": active_progress,
        "message": active_message,
        "stats_update": {
            "products_checked": response_data.get("total_products", 0),
            "products_updated": 0,  # El backend actual no proporciona esto
        },
        "timestamp": datetime.now().isoformat(),
    }

    # Verificar si completó o hay error
    is_syncing = full_data.get("sync", {}).get("in_progress", False)

    if not is_syncing and active_component is None:
        # Verificar si completó exitosamente verificando el estado del sistema
        has_error = any(comp.get("status") == "error" for comp in components.values())

        if has_error:
            progress_action["action"] = "error_sync"
            alert = dbc.Alert(
                html.Div([html.I(className="fas fa-times-circle me-2"), html.Span("Error durante la sincronización")]),
                color="danger",
            )
        else:
            progress_action["action"] = "complete_sync"
            alert = dbc.Alert(
                html.Div(
                    [html.I(className="fas fa-check-circle me-2"), html.Span("Catálogo actualizado exitosamente")]
                ),
                color="success",
            )

        # Deshabilitar interval
        interval_disabled = True
    else:
        # En progreso
        alert = None
        interval_disabled = False

    # Crear componente de progreso actualizado con datos actuales
    updated_progress = update_catalog_progress(
        progress_data,
        phase=current_phase,
        progress=active_progress,
        message=active_message,
        stats_update={"products_checked": response_data.get("total_products", 0), "products_updated": 0},
    )
    progress_component = create_catalog_progress_component(updated_progress)

    return progress_action, progress_component, alert, interval_disabled


@callback(
    [
        Output("catalog-modal-action-store", "data", allow_duplicate=True),
        Output("catalog-toast-action-store", "data", allow_duplicate=True),
    ],
    [Input("catalog-progress-background-btn", "n_clicks"), Input("catalog-progress-cancel-btn", "n_clicks")],
    [
        State("catalog-progress-store", "data"),
        State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
    ],
    prevent_initial_call=True,
)
def handle_progress_modal_actions(
    background_clicks: Optional[int], cancel_clicks: Optional[int], progress_data: Optional[Dict],
    auth_tokens: Optional[Dict]  # REGLA #7.6
) -> Tuple[Dict, Dict]:
    """
    Maneja las acciones del modal de progreso.

    REGLA #11: Escribe a stores intermedios para controlar modal y toasts.
    """
    if not ctx.triggered:
        raise PreventUpdate

    trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

    if trigger_id == "catalog-progress-background-btn":
        # Ejecutar en segundo plano
        modal_action = {"action": "close_progress", "timestamp": datetime.now().isoformat()}

        toast_action = {
            "action": "show_toast",
            "toast_type": "info",
            "message": "Proceso en curso",
            "title": "La actualización continúa en segundo plano",
            "timestamp": datetime.now().isoformat(),
        }

        return modal_action, toast_action

    elif trigger_id == "catalog-progress-cancel-btn":
        # Cancelar actualización
        # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        try:
            result = request_coordinator.make_request(
                "/api/v1/admin/catalog/cancel", method="POST", cache_ttl=0, bypass_cache=True,
                auth_headers=auth_headers,  # REGLA #7.6
            )

            if result:
                if result.get("status") == "cancelled":
                    toast_type = "warning"
                    title = "Proceso detenido"
                    message = "Actualización del catálogo cancelada"
                elif result.get("status") == "nothing_to_cancel":
                    toast_type = "info"
                    title = "Información"
                    message = result.get("message", "No hay sincronización activa")
                else:
                    toast_type = "warning"
                    title = "Advertencia"
                    message = result.get("message", "Estado desconocido")
            else:
                toast_type = "error"
                title = "Error de conexión"
                message = "Error al cancelar la actualización"

        except Exception as e:
            logger.error(f"Error cancelando sincronización: {str(e)}")
            toast_type = "error"
            title = "Error de conexión"
            message = "No se pudo cancelar la actualización"

        modal_action = {"action": "close_progress", "timestamp": datetime.now().isoformat()}

        toast_action = {
            "action": "show_toast",
            "toast_type": toast_type,
            "message": message,
            "title": title,
            "timestamp": datetime.now().isoformat(),
        }

        return modal_action, toast_action

    raise PreventUpdate


# ==============================================================================
# CALLBACKS DE INTERFAZ - SIN OUTPUTS DUPLICADOS
# ==============================================================================


@callback(
    Output("sidebar", "children", allow_duplicate=True),
    [Input("catalog-info-store", "data"), Input("catalog-progress-store", "data")],
    State("sidebar", "children"),
    prevent_initial_call=True,
)
def update_sidebar_catalog_badge(
    catalog_info: Optional[Dict], progress_data: Optional[Dict], current_sidebar: List
) -> List:
    """
    Actualiza el badge de estado del catálogo en el sidebar.
    """
    # Esta es una versión simplificada
    # En producción, reconstruirías el sidebar con el nuevo estado

    # Por ahora retornamos sin cambios para evitar conflictos
    return no_update


@callback(
    [
        Output("catalog-info-store", "data", allow_duplicate=True),
        Output("catalog-panel-unified-store", "data", allow_duplicate=True),
    ],
    [
        Input("catalog-info-interval", "n_intervals"),
        Input("catalog-initial-load-interval", "n_intervals"),  # Añadir trigger inicial
        Input("catalog-status-store", "data"),
    ],
    [
        State("catalog-panel-unified-store", "data"),
        State("dashboard-data-store", "data"),
        State("auth-state", "data"),
    ],  # Issue #301: Verificar autenticación
    prevent_initial_call=True,
)
def refresh_catalog_info(
    n_intervals: int,
    initial_load: int,  # Nuevo parámetro para carga inicial
    catalog_status: Optional[Dict],
    current_store: Optional[Dict],
    dashboard_data: Optional[Dict],
    auth_state: Optional[Dict],  # Issue #301
) -> Tuple[Dict, Dict]:
    """
    Actualiza la información del catálogo desde múltiples fuentes:
    - Interval inicial (500ms después de cargar - solo una vez)
    - Interval periódico (cada 30 segundos)
    - Cambios en el estado del catálogo (catalog-status-store)
    - Datos del dashboard (dashboard-data-store) cuando estén disponibles (State)

    Este callback consolida la lógica que antes estaba dividida entre
    homepage.py y catalog_callbacks.py, simplificando el flujo de datos.

    El dashboard-data-store se usa como State para evitar errores cuando
    el store no existe (en páginas que no sean homepage).

    Issue #301: Verifica autenticación ANTES de hacer llamadas API.
    """
    from datetime import datetime

    from utils.auth_helpers import is_user_authenticated

    # Issue #301: Verificar autenticación ANTES de hacer llamadas API
    # Evita errores 401 cuando usuario no autenticado navega a rutas protegidas
    if not is_user_authenticated(auth_state):
        logger.debug("[CATALOG_CALLBACKS] User not authenticated - skipping catalog refresh")
        raise PreventUpdate

    # Si tenemos datos del dashboard del nuevo endpoint unificado
    if dashboard_data and dashboard_data.get("catalog"):
        catalog_info = dashboard_data["catalog"]
        # Agregar información del sistema si está disponible
        if dashboard_data.get("overall_status"):
            catalog_info["_system_status"] = dashboard_data["overall_status"]
    else:
        # Fallback: obtener información del endpoint optimizado para catálogo
        try:
            from utils.request_coordinator import get_catalog_status

            # Usar helper específico con cache extendido
            full_data = get_catalog_status()

            if not full_data:
                logger.warning("Error obteniendo info del catálogo")
                raise PreventUpdate

            catalog_info = full_data.get("catalog", {})

        except Exception as e:
            logger.error(f"Error obteniendo info del catálogo: {str(e)}")
            raise PreventUpdate

    # Determinar el estado real del catálogo basado en los datos
    total_products = catalog_info.get("total_products", 0)

    # Si el sistema dice "ready" pero no hay productos, el catálogo NO está sincronizado
    if catalog_info.get("_system_status") == "ready":
        if total_products > 0:
            catalog_info["status"] = "SINCRONIZADO"
        else:
            catalog_info["status"] = "NO_INICIALIZADO"

    # Actualizar store unificado
    if not current_store:
        current_store = {
            "source": None,
            "catalog_info": {},
            "dashboard_data": {},
            "catalog_status": {},
            "last_update": None,
        }

    new_store = current_store.copy()
    new_store["source"] = "dashboard" if dashboard_data else "interval"
    new_store["catalog_info"] = catalog_info
    new_store["dashboard_data"] = dashboard_data or {}
    new_store["catalog_status"] = catalog_status or {}
    new_store["last_update"] = datetime.now().isoformat()

    # Si hay información de estado del catálogo, fusionarla con catalog_info
    if catalog_status:
        catalog_info.update(
            {
                "is_syncing": catalog_status.get("is_syncing", False),
                "sync_progress": catalog_status.get("progress", 0),
                "sync_message": catalog_status.get("message", ""),
            }
        )
        new_store["catalog_info"] = catalog_info

    return catalog_info, new_store


@callback(
    Output("catalog-toast-action-store", "data", allow_duplicate=True),
    Input("catalog-sync-complete-store", "data"),
    prevent_initial_call=True,
    allow_duplicate=True,
)
def show_catalog_sync_notification(sync_result: Optional[Dict]) -> Dict:
    """
    Muestra notificaciones cuando se completa la sincronización del catálogo.

    REGLA #11: Escribe al store intermedio catalog-toast-action-store.
    """
    if not sync_result:
        raise PreventUpdate

    if sync_result.get("success"):
        stats = sync_result.get("stats", {})
        products_updated = stats.get("products_updated", 0)
        duration = stats.get("duration", "N/A")

        message = "✅ Actualización completada exitosamente\n"
        message += f"• {products_updated:,} productos actualizados\n"
        message += f"• Tiempo: {duration}"

        return {
            "action": "show_toast",
            "toast_type": "success",
            "message": message,
            "title": "🎉 Catálogo Sincronizado",
            "timestamp": datetime.now().isoformat(),
        }
    else:
        error_msg = sync_result.get("error", "Error desconocido")
        return {
            "action": "show_toast",
            "toast_type": "error",
            "message": f"❌ La sincronización falló\nError: {error_msg}\nPor favor, intenta nuevamente más tarde.",
            "title": "Error de Catálogo",
            "timestamp": datetime.now().isoformat(),
        }


@callback(
    [Output("auto-update-trigger", "data"), Output("catalog-toast-action-store", "data", allow_duplicate=True)],
    Input("auto-update-check-interval", "n_intervals"),
    State("catalog-info-store", "data"),
    prevent_initial_call=True,
    allow_duplicate=True,
)
def check_auto_update_needed(n_intervals: int, catalog_info: Optional[Dict]) -> Tuple[Dict, Dict]:
    """
    Verifica si es necesaria una actualización automática del catálogo.

    REGLA #11: Escribe al store intermedio catalog-toast-action-store.
    """
    if not catalog_info or catalog_info.get("is_syncing"):
        raise PreventUpdate

    last_update = catalog_info.get("last_update")
    if not last_update:
        # Sin actualización previa, sugerir actualización
        toast_action = {
            "action": "show_toast",
            "toast_type": "warning",
            "message": "⚠️ El catálogo requiere actualización inicial para funcionar correctamente.",
            "title": "Actualización requerida",
            "timestamp": datetime.now().isoformat(),
        }
        return {}, toast_action

    try:
        last_update_dt = datetime.fromisoformat(last_update)
        days_since = (datetime.now() - last_update_dt).days

        if days_since >= 10:
            # Iniciar actualización automática
            toast_action = {
                "action": "show_toast",
                "toast_type": "info",
                "message": f"Iniciando actualización automática (último update hace {days_since} días)",
                "title": "Auto-actualización",
                "timestamp": datetime.now().isoformat(),
            }

            trigger_data = {"auto_update": True, "timestamp": datetime.now().isoformat()}
            return trigger_data, toast_action
    except:
        pass

    raise PreventUpdate


# NUEVO: Callback único para renderizar el panel de control del catálogo
# Este es el ÚNICO punto donde se actualiza catalog-control-panel-container
@callback(
    Output("catalog-control-panel-container", "children"),
    Input("catalog-panel-unified-store", "data"),
    prevent_initial_call=False,
)
def render_catalog_control_panel(unified_data: Optional[Dict]) -> html.Div:
    """
    Callback único que renderiza el panel de control del catálogo.

    Este callback soluciona el problema de múltiples callbacks intentando
    actualizar el mismo output, siguiendo las mejores prácticas de Dash.

    El store unificado es actualizado por múltiples fuentes:
    - dashboard-data-store (desde homepage.py cuando esté disponible)
    - catalog-info-interval (actualización periódica)

    Args:
        unified_data: Datos unificados del store con información de todas las fuentes

    Returns:
        Panel de control del catálogo renderizado
    """
    from components.catalog_control import create_catalog_control_panel

    # Si no hay datos, mostrar panel con estado inicial "cargando"
    # NO mostrar ERROR inicialmente, solo un estado de carga
    if not unified_data:
        # En lugar de skeleton, mostrar el panel con estado inicial "cargando"
        return create_catalog_control_panel(
            {
                "status": "CARGANDO",
                "total_products": 0,
                "with_cima": 0,
                "with_nomenclator": 0,
                "last_update": None,
                "message": "Cargando información del catálogo...",
            }
        )

    # Usar datos del unified store
    catalog_info = unified_data.copy() if unified_data else {}

    # Prioridad de fuentes: interval > dashboard > default
    # El interval tiene prioridad porque es la fuente más actualizada
    if unified_data.get("catalog_info"):
        catalog_info.update(unified_data["catalog_info"])

    # Los datos del dashboard pueden complementar si hay información adicional
    if unified_data.get("dashboard_data"):
        # El nuevo endpoint envía los datos directamente en catalog
        catalog = unified_data["dashboard_data"].get("catalog", {})

        # Actualizar con los datos correctos del nuevo endpoint
        if catalog:
            catalog_info["total_products"] = catalog.get("total_products", 0)
            catalog_info["with_cima"] = catalog.get("with_cima", 0)
            catalog_info["with_nomenclator"] = catalog.get("with_nomenclator", 0)
            catalog_info["status"] = catalog.get("status", "DESCONOCIDO")
            catalog_info["last_update"] = catalog.get("last_update")
            catalog_info["coverage"] = catalog.get("coverage", {})

    # Aplicar información del catalog_status si está disponible
    if unified_data.get("catalog_status"):
        catalog_info.update(unified_data["catalog_status"])

    # IMPORTANTE: Determinar el estado real basándose en los datos funcionales
    total_products = catalog_info.get("total_products", 0)
    with_cima = catalog_info.get("with_cima", 0)
    with_nomenclator = catalog_info.get("with_nomenclator", 0)

    # Lógica inteligente: si hay datos, el sistema ES funcional independientemente de errores técnicos
    if total_products > 0:
        # HAY DATOS = SISTEMA OPERATIVO
        # Determinar nivel de operatividad según calidad de datos y estado técnico
        coverage_cima = (with_cima / total_products * 100) if total_products > 0 else 0
        coverage_nomenclator = (with_nomenclator / total_products * 100) if total_products > 0 else 0

        if catalog_info.get("_system_status") == "ready":
            # Todo perfecto
            catalog_info["status"] = "SINCRONIZADO"
        elif coverage_cima > 80 and coverage_nomenclator > 20:
            # Datos suficientes para operar, aunque haya advertencias técnicas
            catalog_info["status"] = "OPERATIVO"
            # Obtener información de la última sincronización exitosa vs fallida
            # Esto ayuda al usuario a entender qué pasó
            last_sync_info = unified_data.get("dashboard_data", {}).get("sync", {}).get("last_updates", {})
            cima_info = last_sync_info.get("cima", {})
            nomen_info = last_sync_info.get("nomenclator", {})

            # Construir mensaje informativo
            messages = []
            if cima_info.get("status") == "error":
                cima_date = (
                    cima_info.get("last_sync", "").split("T")[0] if cima_info.get("last_sync") else "fecha desconocida"
                )
                messages.append(f"CIMA: sincronización del {cima_date} interrumpida")
            if nomen_info.get("status") == "error":
                nomen_date = (
                    nomen_info.get("last_sync", "").split("T")[0]
                    if nomen_info.get("last_sync")
                    else "fecha desconocida"
                )
                messages.append(f"Nomenclátor: error en sincronización del {nomen_date}")

            if messages:
                catalog_info["status_message"] = (
                    f"Operativo con {total_products:,} productos. " + ". ".join(messages) + "."
                )
            else:
                catalog_info["status_message"] = (
                    f"Operativo con {total_products:,} productos. Advertencias técnicas menores."
                )
        else:
            # Pocos datos pero algo hay
            catalog_info["status"] = "DATOS_LIMITADOS"
            catalog_info["status_message"] = (
                f"Funcionando con datos limitados ({total_products:,} productos). Se recomienda sincronización completa."
            )
    else:
        # NO HAY DATOS = REALMENTE NO INICIALIZADO
        catalog_info["status"] = "NO_INICIALIZADO"
        catalog_info["status_message"] = "El catálogo está vacío. Requiere sincronización inicial."

    # Log para debugging (opcional)
    import logging

    logger = logging.getLogger(__name__)
    logger.debug(
        f"Renderizando panel de catálogo - Total productos: {total_products}, Estado: {catalog_info.get('status')}"
    )

    return create_catalog_control_panel(catalog_info)


# Callback para control inteligente de polling del catálogo
@callback(
    Output("catalog-info-interval", "disabled"),
    Input("catalog-panel-unified-store", "data"),
    prevent_initial_call=False,
)
def control_catalog_polling(unified_data: Optional[Dict]) -> bool:
    """
    Control inteligente del polling del catálogo:
    - Habilitado (5s) SOLO durante sincronización activa
    - Deshabilitado cuando no hay sincronización (ahorra recursos)

    Returns:
        bool: True para deshabilitar interval, False para habilitarlo
    """
    # Verificar si hay sincronización activa desde unified_data
    is_syncing = False

    # Fuente: unified_data - información del catálogo
    if unified_data and unified_data.get("catalog_info", {}).get("is_syncing"):
        is_syncing = True

    # DEBUG opcional: Log del estado para debugging
    # logger.debug(f"[CATALOG-POLLING] Sync active: {is_syncing} -> interval {'enabled' if is_syncing else 'disabled'}")

    # Retornar lo contrario: True=deshabilitar, False=habilitar
    return not is_syncing


# ================================
# CALLBACKS DEL PANEL ADMIN
# ================================
# Los callbacks de administración ahora están organizados en el módulo admin/
# para mejor modularidad. Ver: frontend/callbacks/admin/sync_operations.py


# NOTA: El callback update_admin_catalog_statistics fue removido para evitar duplicación
# con callbacks/admin/catalog_management.py. Los callbacks de admin ahora están
# organizados en el módulo admin/ para mejor modularidad.
