# frontend/callbacks/auth_guard.py
"""
Auth Guard - Sistema de protección contra errores 401 con refresh automático.

Este módulo implementa un guard centralizado que:
1. Detecta tokens YA expirados e intenta refresh de emergencia
2. Monitorea el estado del token y lo renueva ANTES de que expire (< 3 min)
3. Detecta errores 401 del request_coordinator
4. Intenta renovar el token con refresh_token antes de hacer logout
5. Realiza logout solo si el refresh falla (refresh_token expirado)
6. Redirige al usuario a REDIRECT_UNAUTHENTICATED (configurable, default "/")

Soluciona Issue #187: Auth redirect on 401 errors + Auto refresh tokens
Fix adicional: Redirección inmediata a landing cuando sesión expira
"""

import logging
from typing import Optional, Tuple

from constants import REDIRECT_UNAUTHENTICATED
from dash import Input, Output, no_update
from utils.auth import auth_manager
from utils.request_coordinator import request_coordinator
from utils.token_refresh_service import is_token_expired, refresh_access_token, should_refresh_token

logger = logging.getLogger(__name__)


def _attempt_token_refresh(reason: str) -> Optional[Tuple]:
    """
    Intenta refresh de emergencia del access token.

    Args:
        reason: Descripción del motivo del refresh (para logging)

    Returns:
        - None si refresh exitoso (caller debe retornar no_update)
        - Tuple con (auth_state, pathname, clear_data) si refresh falla (logout)
    """
    refresh_token = auth_manager.get_refresh_token()

    if refresh_token:
        refresh_result = refresh_access_token(refresh_token)

        if refresh_result:
            # Refresh exitoso: Actualizar tokens usando save_tokens() para mantener
            # consistencia entre access_token y _encrypted_tokens
            new_access_token = refresh_result["access_token"]
            auth_manager.save_tokens(new_access_token, refresh_token)
            logger.info(f"[AUTH_GUARD] Token refresh successful: {reason}")
            return None  # Éxito - caller debe retornar no_update

    # Refresh falló o no hay refresh_token: Logout y redirigir
    logger.warning(f"[AUTH_GUARD] Token refresh failed ({reason}) - logging out and redirecting")
    auth_manager.logout()
    return (
        {"authenticated": False, "user": None},
        REDIRECT_UNAUTHENTICATED,
        True,  # clear_data = True limpia auth-tokens-store
    )


def register_auth_guard_callbacks(app):
    """
    Registra el auth guard como callback con intervalo.

    Args:
        app: Instancia de la aplicación Dash
    """

    @app.callback(
        [
            Output("auth-state", "data", allow_duplicate=True),
            Output("url", "pathname", allow_duplicate=True),
            Output("auth-tokens-store", "clear_data", allow_duplicate=True),
        ],
        Input("auth-guard-interval", "n_intervals"),
        prevent_initial_call=True,
    )
    def auth_guard_check_and_refresh(n_intervals):
        """
        Callback de auth guard que:
        1. Intenta renovar el token si está próximo a expirar (< 3 min)
        2. Verifica errores 401 y hace logout si el refresh falla

        Se ejecuta cada 5-8 segundos via auth-guard-interval.

        Flujo de renovación automática:
        1. Si el access_token expira en < 3 min → Intenta refresh
        2. Si refresh exitoso → Actualiza access_token en auth_manager
        3. Si refresh falla → Procede con logout (refresh_token expirado)

        Flujo de detección de 401:
        1. Detecta error 401 del request_coordinator
        2. Intenta refresh como último recurso
        3. Si falla → Logout y redirección a login

        Args:
            n_intervals: Número de intervalos transcurridos

        Returns:
            Tuple con (auth_state, pathname, clear_data)
        """
        # PASO 0: DETECCIÓN DE TOKEN YA EXPIRADO
        # =======================================
        # Si el token ya expiró, intentar refresh de emergencia o logout.
        # Fix: Detectar tokens expirados que pasaron el threshold de refresh proactivo.
        # Nota: is_token_expired() retorna False para tokens malformados (safe fallback),
        # estos serán capturados por PASO 2 cuando generen 401.
        access_token = auth_manager.get_access_token()

        if access_token and is_token_expired(access_token):
            logger.warning("[AUTH_GUARD] Token already expired - attempting emergency refresh")
            result = _attempt_token_refresh("token already expired")
            if result is not None:
                return result  # Logout + redirect
            return (no_update, no_update, no_update)  # Refresh exitoso

        # PASO 1: REFRESH PROACTIVO (antes de que expire)
        # ================================================
        # Reutilizando access_token obtenido en PASO 0.
        # Tokens ya desencriptados por sync_auth_context callback.
        if access_token and should_refresh_token(access_token):
            logger.info("[AUTH_GUARD] Token approaching expiration (<3 min), attempting proactive refresh")
            result = _attempt_token_refresh("proactive refresh - token expiring soon")
            if result is not None:
                return result  # Logout + redirect
            return (no_update, no_update, no_update)  # Refresh exitoso

        # PASO 2: DETECCIÓN DE 401s (errores de API)
        # ============================================
        # Thread-safety: get_and_clear_401_error() es atómico (usa Lock internamente)
        error_401 = request_coordinator.get_and_clear_401_error()

        if error_401:
            logger.warning(f"[AUTH_GUARD] 401 detected on endpoint: {error_401.get('endpoint')}")
            result = _attempt_token_refresh(f"401 error on {error_401.get('endpoint')}")
            if result is not None:
                return result  # Logout + redirect
            return (no_update, no_update, no_update)  # Refresh exitoso

        # PASO 3: TODO OK - No hay tokens expirados, ni próximos a expirar, ni 401s
        # =========================================================================
        return no_update, no_update, no_update
