"""
Auth Context Callbacks - Sincronización de token entre navegador y servidor Python.

Este callback es CRÍTICO para resolver errores 401:
1. Escucha cambios en auth-tokens-store (localStorage del navegador)
2. Cuando el usuario hace login → token se guarda en localStorage
3. Este callback se dispara y sincroniza el token al auth_context (servidor Python)
4. api_client y request_coordinator usan auth_context para incluir Authorization header
5. Todas las requests HTTP incluyen el JWT automáticamente

Sin este callback:
- Token solo existe en navegador (localStorage)
- Servidor Python no puede acceder a localStorage
- Requests HTTP no incluyen Authorization header → 401 Unauthorized

IMPORTANTE (Issue #xxx - Auth Singleton Fix):
- auth_ready indica si el sync COMPLETÓ, no si hay sesión válida
- auth_ready = True + synced = False → sync completó, NO hay sesión (browser sin tokens)
- auth_ready = True + synced = True → sync completó, SÍ hay sesión válida
- auth_ready = False → sync aún no ejecutó (solo en el primer render)

CRÍTICO: Cuando no hay tokens en el browser, DEBEMOS limpiar el singleton auth_manager
para evitar que browsers diferentes compartan estado de autenticación (race condition).
"""

import logging

from dash import Input, Output
from utils.auth import auth_manager
from utils.auth_context import auth_context

logger = logging.getLogger(__name__)


def register_auth_context_callbacks(app):
    """Registrar callback de sincronización de auth context."""

    @app.callback(
        [Output("auth-context-sync", "data"), Output("auth-ready", "data")],
        Input("auth-tokens-store", "data"),
        prevent_initial_call=False,
    )
    def sync_auth_context(auth_tokens):
        """
        Sincronizar token del navegador al contexto Python global.

        Este callback se ejecuta:
        1. Al cargar la app (prevent_initial_call=False) → restaura sesión si hay token
        2. Al hacer login → guarda nuevo token
        3. Al hacer logout → limpia token

        Issue #344: Agrega señal auth-ready para prevenir race conditions en F5 refresh.

        Args:
            auth_tokens: Dict con tokens encriptados del navegador (localStorage)

        Returns:
            Tuple[Dict, bool]: (estado de sincronización, auth-ready flag)
        """
        logger.info(f"[AUTH_CONTEXT_SYNC] CALLBACK EXECUTED - auth_tokens type: {type(auth_tokens)}, has_tokens: {bool(auth_tokens and 'tokens' in auth_tokens) if isinstance(auth_tokens, dict) else False}")

        if auth_tokens and "tokens" in auth_tokens:
            # Intentar restaurar tokens encriptados
            if auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"]):
                token = auth_manager.get_access_token()
                if token:
                    # Token válido → sincronizar a AMBOS sistemas (Issue #348 RBAC migration)
                    auth_context.set_token(token)
                    logger.info(
                        f"[AUTH_CONTEXT_SYNC] ✅ Token sincronizado exitosamente (longitud: {len(token)}) - setting auth-ready=True"
                    )
                    return {"synced": True, "timestamp": auth_tokens.get("timestamp")}, True
                else:
                    logger.warning("[AUTH_CONTEXT_SYNC] ⚠️ auth_manager.get_access_token() retornó None")
            else:
                logger.warning("[AUTH_CONTEXT_SYNC] ⚠️ auth_manager.restore_from_encrypted_tokens() retornó False")

        # No hay token o es inválido → limpiar auth_context
        # NOTA: NO llamamos auth_manager.logout() porque el singleton es compartido
        # entre todos los browsers y limpiar afectaría a sesiones válidas de otros browsers.
        # El fix real está en routing.py: usar auth_state (per-browser) en vez de
        # auth_manager.is_authenticated() (singleton compartido).
        auth_context.clear_token()
        logger.info("[AUTH_CONTEXT_SYNC] ❌ Token limpiado del auth_context - setting auth-ready=True, synced=False")
        # auth_ready = True indica que sync COMPLETÓ (aunque no haya sesión)
        # routing.py debe verificar auth_state.authenticated, no auth_ready
        return {"synced": False}, True
