# frontend/callbacks/auth_callbacks.py
"""
Callbacks para manejo de autenticación en el frontend
"""

import re
from urllib.parse import parse_qs

import dash_bootstrap_components as dbc
import structlog
from constants import REDIRECT_AFTER_LOGIN, REDIRECT_AFTER_LOGOUT, REDIRECT_UNAUTHENTICATED, PUBLIC_ROUTES
from dash import Input, Output, State, html, no_update
from dash.exceptions import PreventUpdate
from utils.auth import auth_manager

# Logger inicializado a nivel de módulo para evitar overhead
logger = structlog.get_logger()


# Callback para login tradicional con almacenamiento persistente
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def handle_login(n_clicks, email, password, remember_me):
    """
    Maneja el proceso de login tradicional
    """
    if not n_clicks:
        raise PreventUpdate

    # Validación básica
    if not email or not password:
        alert = dbc.Alert("Por favor completa todos los campos", color="warning", dismissable=True)
        return alert, no_update, no_update, no_update, False, "fas fa-sign-in-alt me-2", "Iniciar Sesión"

    # Intentar login
    result = auth_manager.login(email, password)

    if result["success"]:
        # Login exitoso - guardar tokens y redirigir a home
        encrypted_tokens = auth_manager.get_encrypted_tokens()
        user_info = result["data"].get("user")  # Issue #398: Get user info from login response

        # Determinar el tipo de almacenamiento según remember_me
        storage_type = "session" if not remember_me else "local"

        tokens_data = {
            "tokens": encrypted_tokens,
            "storage_type": storage_type,
            "user": user_info  # Issue #398: Include user info
        }

        # Issue #398: Return auth_state immediately populated
        auth_state = {"authenticated": True, "user": user_info}

        logger.info(f"[ISSUE#398] handle_login user_info available: {bool(user_info)}")

        return no_update, REDIRECT_AFTER_LOGIN, tokens_data, auth_state, False, "fas fa-check me-2", "Ingresando..."
    else:
        # Mostrar error
        alert = dbc.Alert(result.get("error", "Error al iniciar sesión"), color="danger", dismissable=True)
        return alert, no_update, no_update, no_update, False, "fas fa-sign-in-alt me-2", "Iniciar Sesión"


# Callback unificado para registro (REGLA #11: ONE INPUT ONE CALLBACK)
# Combina validación de registro, términos, y validación de contraseñas
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def handle_register_and_password_validation(
    n_clicks,
    accept_terms,
    password,
    password_confirm,
    email,
    username,
    fullname,
    phone,
    pharmacy_name,
    pharmacy_code,
    pharmacy_email,
    pharmacy_address,
    pharmacy_city,
    pharmacy_postal_code,
    pharmacy_phone,
):
    """
    Maneja el proceso de registro con creación automática de farmacia.
    También actualiza validación de contraseña en tiempo real.

    REGLA #11: Un solo callback escucha los Inputs de register-password y register-password-confirm.
    """
    # PASO 1: Calcular password strength (para validación en tiempo real)
    if not password:
        strength, color, req_text = 0, "danger", "La contraseña es requerida"
    else:
        strength = 0
        requirements = []

        # Longitud
        if len(password) >= 8:
            strength += 25
        else:
            requirements.append("Mínimo 8 caracteres")

        # Mayúsculas
        if re.search(r"[A-Z]", password):
            strength += 25
        else:
            requirements.append("Al menos una mayúscula")

        # Números
        if re.search(r"\d", password):
            strength += 25
        else:
            requirements.append("Al menos un número")

        # Caracteres especiales
        if re.search(r'[!@#$%^&*(),.?":{}|<>]', password):
            strength += 25
        else:
            requirements.append("Al menos un carácter especial")

        # Determinar color
        if strength < 50:
            color = "danger"
        elif strength < 75:
            color = "warning"
        else:
            color = "success"

        # Mensaje de requisitos
        if requirements:
            req_text = "Falta: " + ", ".join(requirements)
        else:
            req_text = "✓ Contraseña fuerte"

    # PASO 2: Validar coincidencia de contraseñas
    if not password_confirm:
        password_valid, password_invalid = False, False
    elif password_confirm == password:
        password_valid, password_invalid = True, False
    else:
        password_valid, password_invalid = False, True

    # PASO 3: Manejar lógica de registro
    # Habilitar/deshabilitar botón según términos
    if not accept_terms:
        return no_update, True, no_update, strength, color, req_text, password_valid, password_invalid

    # Si no se ha clickeado, solo actualizar estado del botón y validaciones
    if not n_clicks:
        return no_update, False, no_update, strength, color, req_text, password_valid, password_invalid

    # Validaciones de registro
    errors = []

    if not email or not username or not password:
        errors.append("Los campos marcados con * son obligatorios")

    if not pharmacy_name:
        errors.append("El nombre de la farmacia es obligatorio")

    if password and password_confirm and password != password_confirm:
        errors.append("Las contraseñas no coinciden")

    if password and len(password) < 8:
        errors.append("La contraseña debe tener al menos 8 caracteres")

    if username and len(username) < 3:
        errors.append("El nombre de usuario debe tener al menos 3 caracteres")

    if pharmacy_name and len(pharmacy_name) < 3:
        errors.append("El nombre de la farmacia debe tener al menos 3 caracteres")

    # Validar email
    email_pattern = r"^[\w\.-]+@[\w\.-]+\.\w+$"
    if email and not re.match(email_pattern, email):
        errors.append("El email no es válido")

    if errors:
        alert = dbc.Alert([html.Ul([html.Li(error) for error in errors])], color="danger", dismissable=True)
        return alert, False, no_update, strength, color, req_text, password_valid, password_invalid

    # Intentar registro con datos de farmacia
    result = auth_manager.register(
        email=email,
        username=username,
        password=password,
        full_name=fullname,
        phone=phone,
        # Pharmacy fields
        pharmacy_name=pharmacy_name,
        pharmacy_code=pharmacy_code,
        pharmacy_email=pharmacy_email,
        pharmacy_address=pharmacy_address,
        pharmacy_city=pharmacy_city,
        pharmacy_postal_code=pharmacy_postal_code,
        pharmacy_phone=pharmacy_phone,
    )

    if result["success"]:
        # Registro exitoso
        alert = dbc.Alert(
            [
                html.H5("¡Registro exitoso!", className="alert-heading"),
                html.P(result["data"].get("message", "Por favor verifica tu email para activar tu cuenta.")),
            ],
            color="success",
            dismissable=True,
        )
        # Redirigir a landing page después de 3 segundos
        return alert, False, REDIRECT_UNAUTHENTICATED, strength, color, req_text, password_valid, password_invalid
    else:
        # Mostrar error
        alert = dbc.Alert(result.get("error", "Error al registrar usuario"), color="danger", dismissable=True)
        return alert, False, no_update, strength, color, req_text, password_valid, password_invalid


# Callback para recuperación de contraseña
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def handle_forgot_password(n_clicks, email):
    """
    Maneja la recuperación de contraseña
    """
    if not n_clicks:
        raise PreventUpdate

    if not email:
        alert = dbc.Alert("Por favor ingresa tu email", color="warning", dismissable=True)
        return alert

    # Enviar solicitud
    result = auth_manager.forgot_password(email)

    if result["success"]:
        alert = dbc.Alert(
            [
                html.I(className="fas fa-check-circle me-2"),
                result.get("message", "Si el email existe, recibirás instrucciones para resetear tu contraseña."),
            ],
            color="success",
            dismissable=True,
        )
    else:
        alert = dbc.Alert(result.get("error", "Error al procesar la solicitud"), color="danger", dismissable=True)

    return alert


# Callback para manejar OAuth callback con validación CSRF
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def handle_oauth_callback(search, pathname):
    """
    Maneja el callback de OAuth desde proveedores externos con validación CSRF
    """
    if pathname != "/auth/callback":
        raise PreventUpdate

    if not search:
        return "/auth/error", no_update

    # Parsear query parameters
    params = parse_qs(search.lstrip("?"))

    state_key = params.get("state_key", [None])[0]
    provider = params.get("provider", [None])[0]
    state = params.get("state", [None])[0]
    error = params.get("error", [None])[0]

    if error:
        # Hubo un error en OAuth
        return "/auth/error", no_update

    if state_key:
        # Procesar callback con validación de estado CSRF
        success = auth_manager.handle_oauth_callback(state_key, provider or "google", state)
        if success:
            # Obtener tokens encriptados para almacenamiento persistente
            encrypted_tokens = auth_manager.get_encrypted_tokens()
            user_info = auth_manager.user_info  # Issue #398: Get user info from auth_manager

            tokens_data = {
                "tokens": encrypted_tokens,
                "storage_type": "local",  # OAuth siempre usa almacenamiento local
                "user": user_info  # Issue #398: Include user info
            }

            logger.info(f"[ISSUE#398] OAuth callback user_info available: {bool(user_info)}")

            return REDIRECT_AFTER_LOGIN, tokens_data
        else:
            return "/auth/error", no_update

    return "/auth/error", no_update


# Callback para logout (Issue #154: Dos botones - navbar y sidebar)
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def handle_logout(navbar_clicks, sidebar_clicks):
    """
    Maneja el proceso de logout desde navbar o sidebar y limpia el almacenamiento persistente

    Issue #154: Escucha logout desde navbar dropdown y sidebar footer
    FIX Race Condition: Limpia auth-state inmediatamente para prevenir auto-restauración

    Args:
        navbar_clicks: Clicks del botón logout en navbar dropdown
        sidebar_clicks: Clicks del botón logout en sidebar footer (backward compatibility)

    Returns:
        tuple: (pathname, clear_tokens, auth_state_cleaned)
    """
    # Si ninguno de los botones fue clickeado, no hacer nada
    if not navbar_clicks and not sidebar_clicks:
        raise PreventUpdate

    # Cerrar sesión en memoria
    auth_manager.logout()

    # Limpiar tokens almacenados, auth-state, y redirigir a landing page
    # FIX: Retornar auth-state limpio para prevenir race condition con check_authentication
    return REDIRECT_AFTER_LOGOUT, True, {"authenticated": False, "user": None}


# Callback para restauración de tokens desde almacenamiento persistente
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def restore_tokens_from_storage(tokens_data):
    """
    Restaura los tokens desde el almacenamiento persistente al cargar la app
    """
    if not tokens_data or not tokens_data.get("tokens"):
        return no_update

    # Intentar restaurar tokens
    success = auth_manager.restore_from_encrypted_tokens(tokens_data["tokens"])

    if success and auth_manager.is_authenticated():
        # Issue #398: Leer user_info de tokens_data primero (no HTTP call)
        user_info = tokens_data.get("user")

        if user_info:
            logger.info(f"[ISSUE#398] restore_tokens using cached user: {bool(user_info)}")
            return {"authenticated": True, "user": user_info}

        # Fallback: Solo si no hay user en tokens_data (legacy storage)
        logger.info("[ISSUE#398] get_current_user called (legacy fallback)")
        user_info = auth_manager.get_current_user()  # HTTP call solo como fallback
        return {"authenticated": True, "user": user_info}

    return {"authenticated": False, "user": None}


# Callback para verificación de autenticación
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def check_authentication(pathname, auth_state, tokens_data):
    """
    Verifica si el usuario está autenticado para acceder a la página.

    Logging optimizado: Solo registra eventos críticos (warning+) para reducir overhead
    en callbacks de alta frecuencia.

    Security: Uses centralized PUBLIC_ROUTES constant from constants.py to ensure
    consistency across auth guards and prevent bypass vulnerabilities.
    """
    # Si es una ruta pública, permitir acceso sin autenticación
    if pathname in PUBLIC_ROUTES:
        return auth_state, {"redirect": False, "path": pathname}

    # Intentar restaurar tokens si no están en memoria pero sí en almacenamiento
    if not auth_manager.is_authenticated() and tokens_data and tokens_data.get("tokens"):
        success = auth_manager.restore_from_encrypted_tokens(tokens_data["tokens"])
        if success:
            # Obtener información del usuario
            user_info = auth_manager.get_current_user()
            # Log solo en caso de restauración exitosa (evento importante)
            logger.info("auth_token_restored", user_email=user_info.get("email") if user_info else None)
            return {"authenticated": True, "user": user_info}, {"redirect": False, "path": pathname}

    # Verificar si hay un token válido
    if auth_manager.is_authenticated():
        # Usuario autenticado
        if not auth_state.get("authenticated"):
            # Actualizar estado con información del usuario
            user_info = auth_manager.get_current_user()
            auth_state = {"authenticated": True, "user": user_info}
        return auth_state, {"redirect": False, "path": pathname}
    else:
        # Usuario no autenticado, redirigir a landing page
        # Log solo en redirección (evento crítico)
        logger.warning("unauthenticated_redirect", pathname=pathname)
        return {"authenticated": False, "user": None}, {"redirect": True, "path": REDIRECT_UNAUTHENTICATED}


# Callback para manejo de redirecciones
# NOTA: No usar @callback aquí - se registra en register_auth_callbacks()
def handle_redirect(redirect_data):
    """
    Maneja las redirecciones de autenticación.

    Logging optimizado: Solo registra redirecciones activas (warning).
    """
    if redirect_data.get("redirect"):
        # Log solo cuando hay redirección real (evento crítico)
        logger.warning("auth_redirect", path=redirect_data.get("path", REDIRECT_UNAUTHENTICATED))
        return redirect_data.get("path", REDIRECT_UNAUTHENTICATED)

    return no_update


# Callback para iniciar OAuth con generación de estado CSRF
# NOTA: Temporalmente comentado porque los botones OAuth no están en la landing page
# @callback(
#     Output('oauth-state-store', 'data'),
#     Input('oauth-login-button', 'n_clicks'),
#     State('oauth-provider', 'value'),
#     prevent_initial_call=True
# )
# def initiate_oauth_flow(n_clicks, provider):
#     """
#     Inicia el flujo OAuth generando un token de estado CSRF
#     """
#     if not n_clicks:
#         raise PreventUpdate
#
#     # Generar estado CSRF único
#     state_token = auth_manager.generate_oauth_state()
#
#     # Retornar el estado para ser usado en el enlace OAuth
#     return {
#         'state': state_token,
#         'provider': provider,
#         'timestamp': ctx.triggered[0]['timestamp']
#     }


def load_pharmacy_info(auth_state):
    """
    Cargar información de la farmacia del usuario autenticado para mostrar en navbar.

    Issue #154: El nombre de farmacia debe ser visible permanentemente en el navbar
    para que el usuario mantenga contexto de qué farmacia está gestionando.

    Args:
        auth_state: Estado de autenticación del usuario

    Returns:
        Dict con pharmacy_name o None si no hay autenticación
    """
    # Verificar autenticación
    if not auth_state or not auth_state.get("authenticated"):
        raise PreventUpdate

    # Obtener info del usuario
    user = auth_state.get("user")
    if not user:
        raise PreventUpdate

    # Extraer pharmacy_id
    pharmacy_id = user.get("pharmacy_id")
    if not pharmacy_id:
        logger.warning("[LOAD_PHARMACY_INFO] User authenticated but no pharmacy_id found")
        return {"pharmacy_name": "Sin farmacia asignada"}

    # Obtener nombre de farmacia desde API
    try:
        from utils.api_client import api_client
        response = api_client.get(f"/api/v1/pharmacy/{pharmacy_id}")

        if response and response.get("success"):
            pharmacy_data = response.get("data", {})
            pharmacy_name = pharmacy_data.get("name", "Farmacia")

            logger.info(f"[LOAD_PHARMACY_INFO] Loaded pharmacy: {pharmacy_name}")
            return {"pharmacy_name": pharmacy_name, "pharmacy_id": pharmacy_id}
        else:
            logger.error(f"[LOAD_PHARMACY_INFO] Failed to load pharmacy info: {response.get('error')}")
            return {"pharmacy_name": "Farmacia", "pharmacy_id": pharmacy_id}

    except Exception as e:
        logger.error(f"[LOAD_PHARMACY_INFO] Exception loading pharmacy: {str(e)}")
        return {"pharmacy_name": "Farmacia", "pharmacy_id": pharmacy_id}


def register_auth_callbacks(app):
    """
    Registrar todos los callbacks de autenticación.

    Args:
        app: Instancia de la aplicación Dash
    """
    # 1. Callback de login tradicional
    app.callback(
        [
            Output("login-alert", "children"),
            Output("url", "pathname", allow_duplicate=True),
            Output("auth-tokens-store", "data", allow_duplicate=True),
            Output("auth-state", "data", allow_duplicate=True),  # Issue #398: Add auth-state Output
            Output("login-button", "disabled"),
            Output("login-button-icon", "className"),
            Output("login-button-text", "children"),
        ],
        Input("login-button", "n_clicks"),
        [State("login-email", "value"), State("login-password", "value"), State("remember-me", "value")],
        prevent_initial_call=True,
    )(handle_login)

    # 2. Callback unificado de registro y validación de password
    app.callback(
        [
            Output("register-alert", "children"),
            Output("register-button", "disabled"),
            Output("url", "pathname", allow_duplicate=True),
            # Password strength outputs
            Output("password-strength", "value"),
            Output("password-strength", "color"),
            Output("password-requirements", "children"),
            # Password match outputs
            Output("register-password-confirm", "valid"),
            Output("register-password-confirm", "invalid"),
        ],
        [
            Input("register-button", "n_clicks"),
            Input("accept-terms", "value"),
            Input("register-password", "value"),
            Input("register-password-confirm", "value"),
        ],
        [
            State("register-email", "value"),
            State("register-username", "value"),
            State("register-fullname", "value"),
            State("register-phone", "value"),
            # Pharmacy fields
            State("register-pharmacy-name", "value"),
            State("register-pharmacy-code", "value"),
            State("register-pharmacy-email", "value"),
            State("register-pharmacy-address", "value"),
            State("register-pharmacy-city", "value"),
            State("register-pharmacy-postal-code", "value"),
            State("register-pharmacy-phone", "value"),
        ],
        prevent_initial_call=True,
    )(handle_register_and_password_validation)

    # 3. Callback de recuperación de contraseña
    app.callback(
        Output("forgot-alert", "children"),
        Input("forgot-button", "n_clicks"),
        State("forgot-email", "value"),
        prevent_initial_call=True,
    )(handle_forgot_password)

    # 4. Callback de OAuth callback con validación CSRF
    app.callback(
        [Output("url", "pathname", allow_duplicate=True), Output("auth-tokens-store", "data", allow_duplicate=True)],
        Input("url", "search"),
        State("url", "pathname"),
        prevent_initial_call=True,
    )(handle_oauth_callback)

    # 5. Callback de logout (Issue #154: Dos botones - navbar y sidebar)
    # FIX Race Condition: Agregar Output para auth-state para limpiar inmediatamente
    app.callback(
        [
            Output("url", "pathname", allow_duplicate=True),
            Output("auth-tokens-store", "clear_data", allow_duplicate=True),
            Output("auth-state", "data", allow_duplicate=True),  # FIX: Limpiar estado inmediatamente
        ],
        [
            Input("navbar-logout-button", "n_clicks"),  # Nuevo: navbar dropdown
            Input("logout-button", "n_clicks"),         # Existente: sidebar footer (backward compatibility)
        ],
        prevent_initial_call=True,
    )(handle_logout)

    # 6. Callback de restauración de tokens desde almacenamiento
    app.callback(
        Output("auth-state", "data", allow_duplicate=True),
        Input("auth-tokens-store", "data"),
        prevent_initial_call=True,
    )(restore_tokens_from_storage)

    # 7. Callback de verificación de autenticación
    app.callback(
        [Output("auth-state", "data"), Output("auth-redirect", "data")],
        [Input("url", "pathname")],
        [State("auth-state", "data"), State("auth-tokens-store", "data")],
        prevent_initial_call=False,
    )(check_authentication)

    # 8. Callback de manejo de redirecciones
    app.callback(
        Output("url", "pathname", allow_duplicate=True),
        Input("auth-redirect", "data"),
        prevent_initial_call=True,
    )(handle_redirect)

    # 9. Callback para cargar información de farmacia (Issue #154)
    app.callback(
        Output("pharmacy-info-store", "data"),
        Input("auth-state", "data"),
        prevent_initial_call=False,  # Ejecutar al cargar para restaurar desde session
    )(load_pharmacy_info)
