"""
Admin Users Callbacks - Issue #348 FASE 3.1
Gestión completa de usuarios y farmacias (CRUD + filtros + storage).

Este módulo implementa todos los callbacks necesarios para el panel de administración de usuarios:
- Load y filtrado de users list con Ag Grid
- Modales de create/edit/delete/restore usuarios
- Storage usage chart con Plotly
- Validaciones y manejo de errores con toasts
"""

import logging
from datetime import datetime
from typing import Any, Dict, List, Optional, Tuple

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
import requests
from dash import Input, Output, State, ctx, dcc, html, no_update, ALL
from dash.exceptions import PreventUpdate

from components.toast_manager import error_toast, success_toast, warning_toast, trigger_toast
from utils.auth import auth_manager
from utils.auth_helpers import is_user_authenticated, should_show_session_expired_toast
from utils.config import BACKEND_URL

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False

# Constants
REQUEST_TIMEOUT = 30  # seconds for API calls
MAX_USERS_PER_PAGE = 500
DEFAULT_USERS_PER_PAGE = 100

# Role and subscription labels for UI
ROLE_LABELS = {"admin": "Admin", "user": "Usuario"}
SUBSCRIPTION_LABELS = {"free": "Free", "pro": "Pro", "max": "Max"}
SUBSCRIPTION_COLORS = {"free": "info", "pro": "success", "max": "warning"}

# Storage limits per plan (in MB) - Synced with backend/app/core/subscription_limits.py
STORAGE_LIMITS = {"free": 100, "pro": 1024, "max": 10240}


def _format_datetime(dt_str: Optional[str]) -> str:
    """
    Formatea datetime ISO string a formato español legible.

    Args:
        dt_str: Datetime en formato ISO (e.g., '2025-11-01T10:30:00Z')

    Returns:
        Fecha formateada (e.g., '01/11/2025 10:30') o '--' si None
    """
    if not dt_str:
        return "--"

    try:
        dt = datetime.fromisoformat(dt_str.replace("Z", "+00:00"))
        return dt.strftime("%d/%m/%Y %H:%M")
    except (ValueError, AttributeError):
        return "--"


def _create_ag_grid_users_table(users_data: List[Dict]) -> html.Div:
    """
    Crea tabla de usuarios con acciones inline usando dbc.Table en lugar de Ag Grid.

    Args:
        users_data: Lista de usuarios con estructura del backend

    Returns:
        Dash component con tabla Bootstrap responsive y botones de acción
    """
    if not users_data:
        return dbc.Alert(
            html.Div([html.I(className="fas fa-info-circle me-2"), "No hay usuarios disponibles."]),
            color="light",
            className="text-center",
        )

    # Crear filas de la tabla
    table_rows = []
    for user in users_data:
        pharmacy = user.get("pharmacy", {})
        is_deleted = user.get("deleted_at") is not None

        # Botones de acción según estado
        action_buttons = []

        if not is_deleted:
            # Usuario activo: botones Edit y Delete
            action_buttons.extend(
                [
                    dbc.Button(
                        html.I(className="fas fa-edit"),
                        id={"type": "edit-user-btn", "index": user.get("id")},
                        color="primary",
                        size="sm",
                        className="me-1",
                        title="Editar usuario",
                    ),
                    dbc.Button(
                        html.I(className="fas fa-trash"),
                        id={"type": "delete-user-btn", "index": user.get("id")},
                        color="danger",
                        size="sm",
                        title="Eliminar usuario",
                    ),
                ]
            )
        else:
            # Usuario eliminado: solo botón Restore
            action_buttons.append(
                dbc.Button(
                    html.I(className="fas fa-undo"),
                    id={"type": "restore-user-btn", "index": user.get("id")},
                    color="success",
                    size="sm",
                    title="Restaurar usuario",
                )
            )

        # Crear fila con estilo condicional para eliminados
        row_style = {"backgroundColor": "#f8d7da"} if is_deleted else {}

        table_rows.append(
            html.Tr(
                [
                    html.Td(user.get("email", "--")),
                    html.Td(user.get("full_name", "--")),
                    html.Td(
                        ROLE_LABELS.get(user.get("role"), user.get("role", "--")),
                        className="text-center",
                    ),
                    html.Td(user.get("subscription_plan", "--").upper(), className="text-center"),
                    html.Td(
                        dbc.Badge(
                            "Activo" if user.get("is_active") else "Inactivo",
                            color="success" if user.get("is_active") else "secondary",
                        ),
                        className="text-center",
                    ),
                    html.Td(pharmacy.get("name", "--")),
                    html.Td(pharmacy.get("city", "--")),
                    html.Td(_format_datetime(user.get("created_at")), className="text-nowrap"),
                    html.Td(
                        html.Div(action_buttons, className="d-flex justify-content-center"),
                        className="text-center",
                    ),
                ],
                style=row_style,
            )
        )

    # Crear tabla Bootstrap responsive
    return html.Div(
        [
            dbc.Table(
                [
                    html.Thead(
                        html.Tr(
                            [
                                html.Th("Email"),
                                html.Th("Nombre Completo"),
                                html.Th("Rol", className="text-center"),
                                html.Th("Plan", className="text-center"),
                                html.Th("Estado", className="text-center"),
                                html.Th("Farmacia"),
                                html.Th("Ciudad"),
                                html.Th("Creado"),
                                html.Th("Acciones", className="text-center"),
                            ]
                        )
                    ),
                    html.Tbody(table_rows),
                ],
                bordered=True,
                hover=True,
                responsive=True,
                striped=True,
                className="align-middle",
            )
        ],
        style={"maxHeight": "600px", "overflowY": "auto"},
    )


def _create_storage_chart(storage_data: List[Dict]) -> dcc.Graph:
    """
    Crea gráfico de barras Plotly para storage usage por farmacia.

    Args:
        storage_data: Lista de storage stats por farmacia

    Returns:
        Plotly Graph component con bar chart
    """
    if not storage_data:
        return dbc.Alert(
            html.Div([html.I(className="fas fa-info-circle me-2"), "No hay datos de almacenamiento disponibles."]),
            color="light",
            className="text-center",
        )

    # Ordenar por uso de almacenamiento descendente
    # Fix: El backend devuelve 'total_size_mb', no 'storage_mb'
    sorted_data = sorted(storage_data, key=lambda x: x.get("total_size_mb", 0), reverse=True)

    # Limitar a top 10 farmacias (para evitar gráfico sobrecargado)
    top_data = sorted_data[:10]

    pharmacy_names = [item.get("pharmacy_name", "Unknown") for item in top_data]
    # Fix: Usar 'total_size_mb' del backend
    storage_mb = [item.get("total_size_mb", 0) for item in top_data]
    subscription_plans = [item.get("subscription_plan", "free").lower() for item in top_data]

    # Calcular porcentajes basados en límites por plan
    storage_percentage = []
    for item in top_data:
        size_mb = item.get("total_size_mb", 0)
        plan = item.get("subscription_plan", "free").lower()
        limit_mb = STORAGE_LIMITS.get(plan)
        if limit_mb is None:
            logger.warning(f"[STORAGE_CHART] Unknown subscription plan '{plan}', defaulting to FREE limits")
            limit_mb = STORAGE_LIMITS["free"]
        pct = (size_mb / limit_mb * 100) if limit_mb > 0 else 0
        storage_percentage.append(min(pct, 100))  # Cap at 100%

    # Colores según plan usando design tokens
    # Import aquí para evitar dependencia circular
    from styles.design_tokens import COLORS as DESIGN_COLORS

    bar_colors = []
    for plan in subscription_plans:
        if plan == "free":
            bar_colors.append(DESIGN_COLORS["info"])  # Teal para FREE
        elif plan == "pro":
            bar_colors.append(DESIGN_COLORS["success"])  # Verde para PRO
        elif plan == "max":
            bar_colors.append(DESIGN_COLORS["warning"])  # Amarillo para MAX
        else:
            bar_colors.append(DESIGN_COLORS["secondary"])  # Gris para desconocido

    # Crear gráfico
    fig = go.Figure()

    # Formatear porcentajes con coma decimal (estilo español: 12,3%)
    text_labels = [f"{pct:.1f}%".replace(".", ",") for pct in storage_percentage]

    # Formatear valores para hover con formato español
    hover_storage = [f"{mb:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".") for mb in storage_mb]

    fig.add_trace(
        go.Bar(
            x=pharmacy_names,
            y=storage_mb,
            text=text_labels,
            textposition="outside",
            marker=dict(color=bar_colors, line=dict(color="rgb(50, 50, 50)", width=1)),
            customdata=hover_storage,
            hovertemplate="<b>%{x}</b><br>Almacenamiento: %{customdata} MB<extra></extra>",
        )
    )

    fig.update_layout(
        # Título removido para evitar duplicación con heading de sección
        xaxis=dict(title="Farmacia", tickangle=-45, tickfont=dict(size=10)),
        yaxis=dict(
            title="Almacenamiento (MB)",
            gridcolor=DESIGN_COLORS.get("border_light", "rgba(200, 200, 200, 0.3)")
        ),
        plot_bgcolor="rgba(0,0,0,0)",
        paper_bgcolor="rgba(0,0,0,0)",
        height=400,
        margin=dict(l=60, r=40, t=40, b=120),  # Reducido margin top sin título
        hovermode="x unified",
    )

    return dcc.Graph(
        figure=fig,
        config={"displayModeBar": False, "responsive": True}
    )


def register_user_management_callbacks(app):
    """
    Registra todos los callbacks de gestión de usuarios (Issue #348 FASE 3.1).

    Callbacks implementados (10/10):
    1. load_users_list - Carga tabla Bootstrap con acciones y storage chart
    2. open_create_user_modal - Abre modal en modo CREATE
    3. populate_user_form_content - Renderiza form dinámico (user/pharmacy tabs)
    4. save_user - CREATE o UPDATE según mode
    5. open_edit_user_modal - Abre modal en modo EDIT (pattern-matching)
    6. open_delete_user_modal - Abre confirmación delete (pattern-matching)
    7. confirm_delete_user - Ejecuta soft delete
    8. open_restore_user_modal - Abre confirmación restore (pattern-matching)
    9. confirm_restore_user - Ejecuta restore
    10. cancel_user_modals - Cierra todos los modales

    Pattern-matching buttons:
    - {"type": "edit-user-btn", "index": user_id}
    - {"type": "delete-user-btn", "index": user_id}
    - {"type": "restore-user-btn", "index": user_id}

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    if _module_callbacks_registered:
        logger.warning("[USERS_CALLBACKS] Callbacks already registered, skipping duplicate registration")
        return

    logger.info("[USERS_CALLBACKS] Registering user management callbacks")

    # ========================================
    # CALLBACK 1: Load Users List + Storage Chart
    # ========================================
    @app.callback(
        [
            Output("admin-users-table-container", "children"),
            Output("admin-users-data-store", "data"),
            Output("admin-storage-usage-chart", "children"),
            Output("toast-trigger-store", "data", allow_duplicate=True),
        ],
        [
            Input("admin-refresh-users-btn", "n_clicks"),
            Input("admin-user-role-filter", "value"),
            Input("admin-user-subscription-filter", "value"),
            Input("admin-user-include-deleted", "value"),
            Input("admin-user-operation-result", "data"),  # Refresh after CRUD operations
            Input("url", "pathname"),  # Initial load on page mount
        ],
        [State("admin-tabs", "value"), State("auth-state", "data"), State("auth-tokens-store", "data")],
        prevent_initial_call='initial_duplicate',  # Permite initial call con allow_duplicate
    )
    def load_users_list(
        refresh_clicks: Optional[int],
        role_filter: str,
        subscription_filter: str,
        include_deleted_list: List[str],
        operation_result: Optional[Dict],
        pathname: str,
        active_tab: str,
        auth_state: Dict,
        tokens_data: Optional[Dict],
    ) -> Tuple[Any, Dict, Any, Dict]:
        """
        Carga lista de usuarios con filtros y storage usage chart.

        Triggers:
        - Refresh button click
        - Filter changes (role, subscription, include_deleted)
        - CRUD operation success (via operation-result store)
        - URL pathname change (initial load)

        Returns:
        - Ag Grid table with users
        - Updated users-data-store
        - Storage usage chart
        - Toast notification (errors only)
        """
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # Guard 1: Verificar pathname SIEMPRE (no solo cuando trigger es url)
        # Fix: Prevenir toast de "sesión expirada" en landing page
        if pathname != "/admin":
            raise PreventUpdate

        # Guard 1b: Verificar tab activo cuando el trigger es url
        if trigger_id == "url":
            if not active_tab or active_tab != "users":
                # Retornar skeleton mientras se navega a otra tab
                return (
                    dbc.Alert("Cargando usuarios...", color="info", className="text-center"),
                    {},
                    dbc.Alert("Cargando almacenamiento...", color="light", className="text-center"),
                    no_update,
                )

        # Guard 2: Verificar autenticación (REGLA #7)
        if not is_user_authenticated(auth_state):
            # Determinar si mostrar toast de sesión expirada
            # Solo mostrar si había sesión previa (tokens en store)
            show_toast = should_show_session_expired_toast(auth_state, tokens_data)

            toast_data = (
                error_toast("Sesión expirada. Por favor, inicie sesión nuevamente.", "Error de Autenticación")
                if show_toast
                else no_update
            )

            return (
                dbc.Alert(
                    html.Div(
                        [html.I(className="fas fa-lock me-2"), "Debe iniciar sesión para ver usuarios."]
                    ),
                    color="warning",
                    className="text-center",
                ),
                {},
                dbc.Alert("No autenticado", color="light", className="text-center"),
                toast_data,  # Condicional: solo si había sesión previa
            )

        # Obtener token de acceso (REGLA #7 - Auth Manager pattern)
        access_token = auth_manager.get_access_token()
        if not access_token:
            logger.error("[USERS_LOAD] No access token available")
            return (
                dbc.Alert("Error de autenticación", color="danger", className="text-center"),
                {},
                dbc.Alert("Error", color="light", className="text-center"),
                error_toast("Token de acceso no disponible", "Error"),
            )

        # Construir query parameters para filtros
        params = {"skip": 0, "limit": DEFAULT_USERS_PER_PAGE}

        # Filtro de rol
        if role_filter and role_filter != "all":
            params["role"] = role_filter

        # Filtro de suscripción
        if subscription_filter and subscription_filter != "all":
            params["subscription"] = subscription_filter

        # Filtro de incluir eliminados
        include_deleted = include_deleted_list and "include_deleted" in include_deleted_list
        params["include_deleted"] = include_deleted

        try:
            # Request 1: GET /api/v1/admin/users (REGLA #9 - API versioning)
            logger.info(f"[USERS_LOAD] Fetching users with params: {params}")
            response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/users",
                headers={"Authorization": f"Bearer {access_token}"},
                params=params,
                timeout=REQUEST_TIMEOUT,
            )

            response.raise_for_status()
            users_response = response.json()
            users = users_response.get("users", [])
            total_users = users_response.get("total", 0)

            logger.info(f"[USERS_LOAD] Loaded {len(users)} users (total: {total_users})")

            # Crear tabla Ag Grid
            if not users:
                users_table = dbc.Alert(
                    html.Div(
                        [
                            html.I(className="fas fa-info-circle me-2"),
                            "No se encontraron usuarios con los filtros aplicados.",
                        ]
                    ),
                    color="light",
                    className="text-center",
                )
            else:
                users_table = _create_ag_grid_users_table(users)

            # Request 2: GET /api/v1/admin/storage/usage (OPCIONAL - no crítico)
            try:
                storage_response = requests.get(
                    f"{BACKEND_URL}/api/v1/admin/storage/usage",
                    headers={"Authorization": f"Bearer {access_token}"},
                    timeout=REQUEST_TIMEOUT,
                )

                storage_response.raise_for_status()
                storage_data = storage_response.json()

                logger.info(f"[USERS_LOAD] Loaded storage data for {len(storage_data)} pharmacies")

                # Crear storage chart
                storage_chart = _create_storage_chart(storage_data)

            except requests.HTTPError as storage_error:
                # Storage endpoint falló - NO crítico, mostrar placeholder
                logger.warning(f"[USERS_LOAD] Storage endpoint failed (non-critical): {storage_error.response.status_code}")
                storage_chart = dbc.Alert(
                    html.Div([
                        html.I(className="fas fa-exclamation-triangle me-2"),
                        "Error al cargar estadísticas de almacenamiento"
                    ]),
                    color="warning",
                    className="text-center"
                )
            except Exception as storage_error:
                logger.warning(f"[USERS_LOAD] Storage request failed (non-critical): {str(storage_error)}")
                storage_chart = dbc.Alert(
                    html.Div([
                        html.I(className="fas fa-exclamation-triangle me-2"),
                        "Error al cargar estadísticas de almacenamiento"
                    ]),
                    color="warning",
                    className="text-center"
                )

            # Actualizar store con datos de usuarios (para otros callbacks)
            users_store_data = {"users": users, "total": total_users, "timestamp": datetime.now().isoformat()}

            return users_table, users_store_data, storage_chart, no_update

        except requests.HTTPError as e:
            logger.error(f"[USERS_LOAD] HTTP error: {e.response.status_code} - {e.response.text}")

            error_message = "Error al cargar usuarios"
            if e.response.status_code == 401:
                error_message = "Sesión expirada"
            elif e.response.status_code == 403:
                error_message = "Sin permisos para ver usuarios"
            elif e.response.status_code == 429:
                error_message = "Demasiadas solicitudes. Intente nuevamente en unos segundos."

            return (
                dbc.Alert(f"{error_message}", color="danger", className="text-center"),
                {},
                dbc.Alert("Error al cargar almacenamiento", color="light", className="text-center"),
                error_toast(error_message, "Error"),
            )

        except requests.Timeout:
            logger.error("[USERS_LOAD] Request timeout")
            return (
                dbc.Alert("Tiempo de espera agotado al cargar usuarios", color="warning", className="text-center"),
                {},
                dbc.Alert("Timeout", color="light", className="text-center"),
                warning_toast("La carga de usuarios está tardando más de lo esperado", "Timeout"),
            )

        except Exception as e:
            logger.exception(f"[USERS_LOAD] Unexpected error: {e}")
            return (
                dbc.Alert("Error inesperado al cargar usuarios", color="danger", className="text-center"),
                {},
                dbc.Alert("Error", color="light", className="text-center"),
                error_toast(f"Error inesperado: {str(e)}", "Error"),
            )

    # ========================================
    # CALLBACK 2: Open Create User Modal
    # ========================================
    @app.callback(
        [
            Output("admin-user-modal", "is_open", allow_duplicate=True),
            Output("admin-user-form-mode-store", "data", allow_duplicate=True),
            Output("admin-selected-user-store", "data", allow_duplicate=True),
            Output("admin-user-modal-title", "children", allow_duplicate=True),
            Output("admin-user-modal-icon", "className", allow_duplicate=True),
            Output("admin-user-form-data-store", "data", allow_duplicate=True),  # NEW: Limpiar store
        ],
        Input("admin-create-user-btn", "n_clicks"),
        State("admin-user-modal", "is_open"),
        prevent_initial_call=True,
    )
    def open_create_user_modal(n_clicks: Optional[int], is_open: bool) -> Tuple[bool, str, None, str, str, None]:
        """
        Abre modal de usuario en modo CREATE.

        Returns:
        - modal is_open = True
        - form_mode = "create"
        - selected_user = None (formulario vacío)
        - modal_title = "Crear Usuario"
        - modal_icon = "fas fa-user-plus"
        - form_data_store = None (limpiar datos previos)
        """
        if not n_clicks:
            raise PreventUpdate

        logger.info("[USERS_CREATE_MODAL] Opening create user modal")

        return (
            True,  # Open modal
            "create",  # Mode
            None,  # No user selected (empty form)
            "Crear Usuario",  # Title
            "fas fa-user-plus me-2 text-primary",  # Icon
            None,  # Limpiar form data store
        )

    # ========================================
    # CALLBACK 3A: Save Form Data Before Tab Switch (NEW - Issue #365 FIX)
    # ========================================
    @app.callback(
        Output("admin-user-form-data-store", "data"),
        [
            Input("admin-user-email-input", "value"),
            Input("admin-user-full-name-input", "value"),
            Input("admin-user-password-input", "value"),
            Input("admin-user-role-select", "value"),
            Input("admin-user-subscription-select", "value"),
            Input("admin-user-is-active-check", "value"),
            Input("admin-pharmacy-name-input", "value"),
            Input("admin-pharmacy-nif-input", "value"),
            Input("admin-pharmacy-address-input", "value"),
            Input("admin-pharmacy-city-input", "value"),
            Input("admin-pharmacy-postal-code-input", "value"),
            Input("admin-pharmacy-phone-input", "value"),
            Input("admin-pharmacy-erp-input", "value"),
        ],
        State("admin-user-form-data-store", "data"),
        prevent_initial_call=True,
    )
    def save_form_data_on_change(
        email, full_name, password, role, subscription_plan, is_active_list,
        pharmacy_name, pharmacy_nif, pharmacy_address, pharmacy_city,
        pharmacy_postal_code, pharmacy_phone, pharmacy_erp_type,
        current_form_data
    ):
        """
        Guarda los datos del formulario en el store cada vez que cambia algún input.
        Esto preserva los valores al cambiar entre tabs.
        """
        # Inicializar store si es None
        if current_form_data is None:
            current_form_data = {}

        # Actualizar con valores actuales (solo si no son None)
        form_data = {
            "email": email if email is not None else current_form_data.get("email"),
            "full_name": full_name if full_name is not None else current_form_data.get("full_name"),
            "password": password if password is not None else current_form_data.get("password"),
            "role": role if role is not None else current_form_data.get("role"),
            "subscription_plan": subscription_plan if subscription_plan is not None else current_form_data.get("subscription_plan"),
            "is_active_list": is_active_list if is_active_list is not None else current_form_data.get("is_active_list"),
            "pharmacy_name": pharmacy_name if pharmacy_name is not None else current_form_data.get("pharmacy_name"),
            "pharmacy_nif": pharmacy_nif if pharmacy_nif is not None else current_form_data.get("pharmacy_nif"),
            "pharmacy_address": pharmacy_address if pharmacy_address is not None else current_form_data.get("pharmacy_address"),
            "pharmacy_city": pharmacy_city if pharmacy_city is not None else current_form_data.get("pharmacy_city"),
            "pharmacy_postal_code": pharmacy_postal_code if pharmacy_postal_code is not None else current_form_data.get("pharmacy_postal_code"),
            "pharmacy_phone": pharmacy_phone if pharmacy_phone is not None else current_form_data.get("pharmacy_phone"),
            "pharmacy_erp_type": pharmacy_erp_type if pharmacy_erp_type is not None else current_form_data.get("pharmacy_erp_type"),
        }

        return form_data

    # ========================================
    # CALLBACK 3: Populate User Form Content (Dynamic)
    # ========================================
    @app.callback(
        Output("admin-user-form-content", "children"),
        [
            Input("admin-user-form-tabs", "active_tab"),
            Input("admin-user-form-mode-store", "data"),
            Input("admin-selected-user-store", "data"),
        ],
        State("admin-user-form-data-store", "data"),
        prevent_initial_call=False,
    )
    def populate_user_form_content(
        active_tab: str, form_mode: Optional[str], selected_user: Optional[Dict],
        saved_form_data: Optional[Dict]
    ) -> html.Div:
        """
        Renderiza contenido dinámico del formulario según tab activo.

        **FIX (Issue #365)**: Renderiza AMBAS tabs pero oculta la no activa con CSS.
        Esto mantiene los inputs en el DOM para que Dash pueda leerlos como States,
        evitando el error "nonexistent object was used in State".

        **NEW (Issue #365 FIX)**: Lee datos guardados del store para preservar valores
        al cambiar entre tabs.

        Tabs:
        - user-tab: Email, full_name, password, role, subscription_plan
        - pharmacy-tab: name, nif, address, city, postal_code, phone

        Mode:
        - create: Todos los campos vacíos, password obligatorio
        - edit: Pre-llenar campos con selected_user, password opcional
        """
        if not active_tab:
            return html.Div()

        is_edit_mode = form_mode == "edit"

        # Prioridad de datos: saved_form_data > selected_user > defaults
        if saved_form_data:
            # Usar datos guardados del store (ya ingresados por usuario)
            user_data = {
                "email": saved_form_data.get("email", ""),
                "full_name": saved_form_data.get("full_name", ""),
                "role": saved_form_data.get("role", "user"),
                "subscription_plan": saved_form_data.get("subscription_plan", "free"),
                "is_active": "is_active" in (saved_form_data.get("is_active_list") or [])
            }
            pharmacy_data = {
                "name": saved_form_data.get("pharmacy_name", ""),
                "nif": saved_form_data.get("pharmacy_nif", ""),
                "address": saved_form_data.get("pharmacy_address", ""),
                "city": saved_form_data.get("pharmacy_city", ""),
                "postal_code": saved_form_data.get("pharmacy_postal_code", ""),
                "phone": saved_form_data.get("pharmacy_phone", ""),
                "erp_type": saved_form_data.get("pharmacy_erp_type")
            }
        elif is_edit_mode and selected_user:
            # Modo edición: usar datos del usuario seleccionado
            user_data = selected_user
            pharmacy_data = selected_user.get("pharmacy", {})
        else:
            # Modo creación: campos vacíos
            user_data = {}
            pharmacy_data = {}

        # FIX: Renderizar AMBAS tabs pero ocultar la que no está activa
        # Esto mantiene los inputs en el DOM para evitar error de Dash con States
        user_tab_style = {} if active_tab == "user-tab" else {"display": "none"}
        pharmacy_tab_style = {} if active_tab == "pharmacy-tab" else {"display": "none"}

        # Tab de usuario
        user_form = html.Div(
            [
                # Email
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Email *", className="form-label"),
                                dbc.Input(
                                    id="admin-user-email-input",
                                    type="email",
                                    placeholder="usuario@farmacia.com",
                                    value=user_data.get("email", ""),
                                    disabled=is_edit_mode,  # No se puede cambiar email en edit
                                    className="mb-3",
                                ),
                            ],
                            width=12,
                        )
                    ]
                ),
                # Full Name
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Nombre Completo", className="form-label"),
                                dbc.Input(
                                    id="admin-user-full-name-input",
                                    type="text",
                                    placeholder="Nombre del usuario o farmacia",
                                    value=user_data.get("full_name", ""),
                                    className="mb-3",
                                ),
                            ],
                            width=12,
                        )
                    ]
                ),
                # Password (solo en create, opcional en edit)
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label(
                                    "Contraseña " + ("*" if not is_edit_mode else "(dejar vacío para mantener)"),
                                    className="form-label",
                                ),
                                dbc.InputGroup(
                                    [
                                        dbc.Input(
                                            id="admin-user-password-input",
                                            type="password",
                                            placeholder="Mínimo 8 caracteres",
                                        ),
                                        dbc.Button(
                                            html.I(className="fas fa-eye", id="admin-password-toggle-icon"),
                                            id="admin-password-toggle-btn",
                                            color="secondary",
                                            outline=True,
                                            size="sm",
                                        ),
                                    ],
                                    className="mb-2",
                                ),
                                dbc.FormText(
                                    "Mínimo 8 caracteres, al menos una mayúscula y un número",
                                    color="muted",
                                ),
                            ],
                            width=12,
                        )
                    ]
                ),
                # Role y Subscription (2 columnas)
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Rol *", className="form-label"),
                                dbc.Select(
                                    id="admin-user-role-select",
                                    options=[
                                        {"label": "Usuario", "value": "user"},
                                        {"label": "Administrador", "value": "admin"},
                                    ],
                                    value=user_data.get("role", "user"),
                                    className="mb-3",
                                ),
                            ],
                            width=12,
                            md=6,
                        ),
                        dbc.Col(
                            [
                                html.Label("Plan de Suscripción *", className="form-label"),
                                dbc.Select(
                                    id="admin-user-subscription-select",
                                    options=[
                                        {"label": "Free (1 GB)", "value": "free"},
                                        {"label": "Pro (10 GB)", "value": "pro"},
                                        {"label": "Max (100 GB)", "value": "max"},
                                    ],
                                    value=user_data.get("subscription_plan", "free"),
                                    className="mb-3",
                                ),
                            ],
                            width=12,
                            md=6,
                        ),
                    ]
                ),
                # Is Active (solo en edit mode)
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                dbc.Checklist(
                                    id="admin-user-is-active-check",
                                    options=[{"label": " Usuario activo", "value": "is_active"}],
                                    value=["is_active"] if user_data.get("is_active", True) else [],
                                    switch=True,
                                    className="mb-3",
                                )
                            ],
                            width=12,
                        )
                    ]
                )
                if is_edit_mode
                else html.Div(),
            ],
            style=user_tab_style,
        )

        # Tab de farmacia
        pharmacy_form = html.Div(
            [
                # Pharmacy Name
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Nombre de Farmacia *", className="form-label"),
                                dbc.Input(
                                    id="admin-pharmacy-name-input",
                                    type="text",
                                    placeholder="Farmacia Central",
                                    value=pharmacy_data.get("name", ""),
                                    className="mb-3",
                                    disabled=False,  # Editable en ambos modos
                                ),
                            ],
                            width=12,
                        )
                    ]
                ),
                # NIF
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("NIF/CIF", className="form-label"),
                                dbc.Input(
                                    id="admin-pharmacy-nif-input",
                                    type="text",
                                    placeholder="B12345678",
                                    value=pharmacy_data.get("nif", ""),
                                    className="mb-3",
                                    disabled=is_edit_mode,  # Editable en CREATE, bloqueado en EDIT (identificador único)
                                ),
                                dbc.FormText("Formato: letra + 8 dígitos (opcional)", color="muted"),
                            ],
                            width=12,
                        )
                    ]
                ),
                # Address
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Dirección", className="form-label"),
                                dbc.Input(
                                    id="admin-pharmacy-address-input",
                                    type="text",
                                    placeholder="Calle Mayor 1",
                                    value=pharmacy_data.get("address", ""),
                                    className="mb-3",
                                    disabled=False,  # Editable
                                ),
                            ],
                            width=12,
                        )
                    ]
                ),
                # City y Postal Code (2 columnas)
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Ciudad", className="form-label"),
                                dbc.Input(
                                    id="admin-pharmacy-city-input",
                                    type="text",
                                    placeholder="Madrid",
                                    value=pharmacy_data.get("city", ""),
                                    className="mb-3",
                                    disabled=False,  # Editable
                                ),
                            ],
                            width=12,
                            md=6,
                        ),
                        dbc.Col(
                            [
                                html.Label("Código Postal", className="form-label"),
                                dbc.Input(
                                    id="admin-pharmacy-postal-code-input",
                                    type="text",
                                    placeholder="28001",
                                    value=pharmacy_data.get("postal_code", ""),
                                    className="mb-3",
                                    disabled=False,  # Editable
                                ),
                            ],
                            width=12,
                            md=6,
                        ),
                    ]
                ),
                # Phone
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Teléfono", className="form-label"),
                                dbc.Input(
                                    id="admin-pharmacy-phone-input",
                                    type="tel",
                                    placeholder="+34912345678",
                                    value=pharmacy_data.get("phone", ""),
                                    className="mb-3",
                                    disabled=False,  # Editable
                                ),
                            ],
                            width=12,
                        )
                    ]
                ),
                # ERP Type (CRÍTICO para parsers - Issue #xxx)
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Label("Tipo de ERP *", className="form-label"),
                                dcc.Dropdown(
                                    id="admin-pharmacy-erp-input",
                                    options=[
                                        {"label": "Farmatic", "value": "farmatic"},
                                        {"label": "Farmanager", "value": "farmanager"},
                                        {"label": "Nixfarma", "value": "nixfarma"},
                                        {"label": "Unycop", "value": "unycop"},
                                        {"label": "IOFWin", "value": "iofwin"},
                                        {"label": "Otro", "value": "other"},
                                    ],
                                    value=pharmacy_data.get("erp_type", None),
                                    placeholder="Seleccionar ERP de la farmacia",
                                    className="mb-3",
                                    disabled=False,  # Editable (puede cambiar ERP)
                                ),
                                dbc.FormText(
                                    "El tipo de ERP es necesario para procesar archivos de ventas correctamente.",
                                    className="text-muted",
                                ),
                            ],
                            width=12,
                        )
                    ]
                ),
                # Info alert sobre edición de farmacia
                dbc.Alert(
                    html.Div(
                        [
                            html.I(className="fas fa-info-circle me-2"),
                            "Los campos de la farmacia son editables. ",
                            "El NIF no puede modificarse por ser identificador único.",
                        ]
                    ),
                    color="info",
                    className="small",
                )
                if is_edit_mode
                else html.Div(),
            ],
            style=pharmacy_tab_style,
        )

        # Renderizar AMBOS formularios (solo uno visible)
        return html.Div([user_form, pharmacy_form])

    # ========================================
    # CALLBACK 4: Save User (CREATE or UPDATE)
    # ========================================
    @app.callback(
        [
            Output("admin-user-operation-result", "data"),
            Output("toast-trigger-store", "data", allow_duplicate=True),
            Output("admin-user-modal", "is_open", allow_duplicate=True),
            Output("admin-user-form-errors", "children"),
            Output("admin-user-form-data-store", "data", allow_duplicate=True),  # NEW: Limpiar store después de guardar
        ],
        Input("admin-user-modal-save", "n_clicks"),
        [
            State("admin-user-form-mode-store", "data"),
            State("admin-selected-user-store", "data"),
            State("admin-user-form-data-store", "data"),  # NEW: Leer desde store en lugar de inputs
        ],
        prevent_initial_call=True,
    )
    def save_user(
        n_clicks: Optional[int],
        form_mode: str,
        selected_user: Optional[Dict],
        form_data: Optional[Dict],  # NEW: Datos del formulario desde store
    ) -> Tuple[Dict, Dict, bool, Any, None]:
        """
        Guarda usuario (CREATE o UPDATE según form_mode).

        **NEW (Issue #365 FIX)**: Lee todos los datos desde form-data-store en lugar de inputs.
        Esto resuelve el problema de pérdida de datos al cambiar entre tabs.

        CREATE mode:
        - POST /api/v1/admin/users con user + pharmacy
        - Valida todos los campos requeridos

        EDIT mode:
        - PUT /api/v1/admin/users/{id} solo con role, subscription_plan, is_active
        - Password opcional (solo si se proporciona)

        Returns:
        - operation-result (triggers list refresh)
        - toast notification
        - modal is_open = False (cierra modal)
        - form errors (validación client-side)
        - form-data-store = None (limpiar después de guardar)
        """
        if not n_clicks:
            raise PreventUpdate

        # Extraer datos del store (o usar valores por defecto)
        if not form_data:
            form_data = {}

        email = form_data.get("email", "")
        full_name = form_data.get("full_name", "")
        password = form_data.get("password", "")
        role = form_data.get("role", "user")
        subscription_plan = form_data.get("subscription_plan", "free")
        is_active_list = form_data.get("is_active_list", [])
        pharmacy_name = form_data.get("pharmacy_name", "")
        pharmacy_nif = form_data.get("pharmacy_nif", "")
        pharmacy_address = form_data.get("pharmacy_address", "")
        pharmacy_city = form_data.get("pharmacy_city", "")
        pharmacy_postal_code = form_data.get("pharmacy_postal_code", "")
        pharmacy_phone = form_data.get("pharmacy_phone", "")
        pharmacy_erp_type = form_data.get("pharmacy_erp_type")

        # Validación client-side básica
        errors = []

        # ✅ SOLO validar en CREATE mode
        if form_mode == "create":
            if not email or "@" not in email:
                errors.append("Email inválido")
            if not password or len(password) < 8:
                errors.append("La contraseña debe tener al menos 8 caracteres")
            if not pharmacy_name:
                errors.append("El nombre de la farmacia es obligatorio")
            if not pharmacy_erp_type:
                errors.append("El tipo de ERP es obligatorio")

        if errors:
            error_alert = dbc.Alert(
                html.Div([html.Strong("Errores de validación:"), html.Ul([html.Li(err) for err in errors])]),
                color="danger",
                dismissable=True,
            )
            return no_update, error_toast("Por favor, corrija los errores del formulario", "Validación"), no_update, error_alert, no_update

        # Obtener token
        access_token = auth_manager.get_access_token()
        if not access_token:
            return (
                no_update,
                error_toast("Token de acceso no disponible", "Error"),
                no_update,
                dbc.Alert("Error de autenticación", color="danger"),
                no_update,  # No limpiar store en error de auth
            )

        try:
            if form_mode == "create":
                # CREATE: POST /api/v1/admin/users
                # Generate username from email if not provided
                username = email.split("@")[0] if email else ""

                request_body = {
                    "user": {
                        "email": email.strip(),
                        "username": username.strip(),  # REQUIRED by backend schema
                        "full_name": full_name.strip() if full_name else username,
                        "password": password,
                        "role": role,
                        "subscription_plan": subscription_plan,
                    },
                    "pharmacy": {
                        "name": pharmacy_name.strip(),
                        "nif": pharmacy_nif.strip() if pharmacy_nif else None,
                        "address": pharmacy_address.strip() if pharmacy_address else None,
                        "city": pharmacy_city.strip() if pharmacy_city else None,
                        "postal_code": pharmacy_postal_code.strip() if pharmacy_postal_code else None,
                        "phone": pharmacy_phone.strip() if pharmacy_phone else None,
                        "erp_type": pharmacy_erp_type,  # NUEVO: ERP type para parsers
                    },
                }

                logger.info(f"[USERS_CREATE] Creating user: {email}")

                response = requests.post(
                    f"{BACKEND_URL}/api/v1/admin/users",
                    headers={"Authorization": f"Bearer {access_token}", "Content-Type": "application/json"},
                    json=request_body,
                    timeout=REQUEST_TIMEOUT,
                )

                response.raise_for_status()
                created_user = response.json()

                logger.info(f"[USERS_CREATE] User created successfully: {created_user.get('id')}")

                return (
                    {"action": "create", "user_id": created_user.get("id"), "timestamp": datetime.now().isoformat()},
                    success_toast(f"Usuario {email} creado exitosamente", "Usuario Creado"),
                    False,  # Close modal
                    html.Div(),  # Clear errors
                    None,  # Limpiar form data store después de guardar exitosamente
                )

            elif form_mode == "edit":
                # UPDATE: PUT /api/v1/admin/users/{id} + PUT /api/v1/pharmacy/{pharmacy_id}
                if not selected_user or not selected_user.get("id"):
                    return (
                        no_update,
                        error_toast("Usuario no seleccionado", "Error"),
                        no_update,
                        dbc.Alert("Error: Usuario no encontrado", color="danger"),
                        no_update,  # No limpiar store en error
                    )

                user_id = selected_user["id"]
                pharmacy_id = selected_user.get("pharmacy_id")

                # NEW: Con store, todos los datos están disponibles sin importar la tab activa
                is_active = "is_active" in is_active_list if is_active_list else False

                # Construir request body con datos del store
                user_request_body = {
                    "role": role,
                    "subscription_plan": subscription_plan,
                    "is_active": is_active
                }

                # FIX: Convertir password vacío a None (evita validación Pydantic)
                if password and password.strip() and len(password) >= 8:
                    user_request_body["password"] = password
                # Si password es "" o None, NO incluirlo (mantener password actual)

                logger.info(f"[USERS_UPDATE] Updating user: {user_id} with data from store")

                response = requests.put(
                    f"{BACKEND_URL}/api/v1/admin/users/{user_id}",
                    headers={"Authorization": f"Bearer {access_token}", "Content-Type": "application/json"},
                    json=user_request_body,
                    timeout=REQUEST_TIMEOUT,
                )

                response.raise_for_status()
                updated_user = response.json()

                # Actualizar farmacia (si hay pharmacy_id)
                pharmacy_update_success = True
                pharmacy_error_msg = None

                if pharmacy_id:
                    pharmacy_request_body = {
                        "name": pharmacy_name.strip() if pharmacy_name else None,
                        "address": pharmacy_address.strip() if pharmacy_address else None,
                        "city": pharmacy_city.strip() if pharmacy_city else None,
                        "postal_code": pharmacy_postal_code.strip() if pharmacy_postal_code else None,
                        "phone": pharmacy_phone.strip() if pharmacy_phone else None,
                        "erp_type": pharmacy_erp_type,
                        "subscription_plan": subscription_plan,  # Sincronizar con user
                    }

                    logger.info(f"[PHARMACY_UPDATE] Updating pharmacy: {pharmacy_id} with ERP type: {pharmacy_erp_type}")
                    logger.debug(f"[PHARMACY_UPDATE] Request body: {pharmacy_request_body}")

                    try:
                        pharmacy_response = requests.put(
                            f"{BACKEND_URL}/api/v1/pharmacy/{pharmacy_id}",
                            headers={"Authorization": f"Bearer {access_token}", "Content-Type": "application/json"},
                            json=pharmacy_request_body,
                            timeout=REQUEST_TIMEOUT,
                        )

                        pharmacy_response.raise_for_status()
                        pharmacy_result = pharmacy_response.json()
                        logger.info(f"[PHARMACY_UPDATE] Pharmacy updated successfully: {pharmacy_id}, updated fields: {pharmacy_result.get('updated_fields', [])}")
                    except requests.HTTPError as pharmacy_error:
                        pharmacy_update_success = False
                        error_detail = "desconocido"
                        try:
                            error_json = pharmacy_error.response.json()
                            error_detail = error_json.get("detail", str(error_json))
                        except Exception:
                            error_detail = pharmacy_error.response.text[:200]  # Limitar longitud

                        pharmacy_error_msg = f"Error HTTP {pharmacy_error.response.status_code}: {error_detail}"
                        logger.error(f"[PHARMACY_UPDATE] HTTP error: {pharmacy_error.response.status_code} - {pharmacy_error.response.text}")
                    except requests.RequestException as pharmacy_error:
                        pharmacy_update_success = False
                        pharmacy_error_msg = f"Error de conexión: {str(pharmacy_error)[:100]}"
                        logger.error(f"[PHARMACY_UPDATE] Request error: {str(pharmacy_error)}")

                logger.info(f"[USERS_UPDATE] User updated successfully: {user_id}")

                # Mensaje de éxito basado en resultado de pharmacy update
                if pharmacy_update_success:
                    success_message = f"Usuario {email} actualizado exitosamente"
                    toast_color = "success"
                else:
                    success_message = f"Usuario {email} actualizado, pero {pharmacy_error_msg}"
                    toast_color = "warning"

                return (
                    {"action": "update", "user_id": user_id, "timestamp": datetime.now().isoformat()},
                    trigger_toast(success_message, toast_color, "Usuario Actualizado"),
                    False,  # Close modal
                    html.Div(),  # Clear errors
                    None,  # Limpiar form data store después de guardar exitosamente
                )

        except requests.HTTPError as e:
            logger.error(f"[USERS_SAVE] HTTP error: {e.response.status_code} - {e.response.text}")

            error_message = "Error al guardar usuario"
            if e.response.status_code == 400:
                error_message = "Datos inválidos: Email duplicado o farmacia ya tiene usuario (REGLA #10)"
            elif e.response.status_code == 403:
                error_message = "Sin permisos para gestionar usuarios"
            elif e.response.status_code == 422:
                # FIX: Mensaje más específico basado en el detalle del error
                try:
                    error_detail = e.response.json().get("detail", [])
                    # Pydantic devuelve lista de errores, extraer el primero
                    if isinstance(error_detail, list) and error_detail:
                        field_error = error_detail[0]
                        field = field_error.get("loc", ["unknown"])[-1]
                        msg = field_error.get("msg", "Error de validación")
                        error_message = f"Error en campo '{field}': {msg}"
                    elif isinstance(error_detail, str):
                        error_message = f"Validación fallida: {error_detail}"
                    else:
                        error_message = "Validación fallida. Verifique los datos del formulario."
                except Exception:
                    error_message = "Validación fallida. Verifique los datos del formulario."
            elif e.response.status_code == 429:
                error_message = "Demasiadas solicitudes. Intente nuevamente en 1 minuto."

            return (
                no_update,
                error_toast(error_message, "Error"),
                no_update,
                dbc.Alert(error_message, color="danger", dismissable=True),
                no_update,  # No limpiar store en error HTTP
            )

        except Exception as e:
            logger.exception(f"[USERS_SAVE] Unexpected error: {e}")
            return (
                no_update,
                error_toast(f"Error inesperado: {str(e)}", "Error"),
                no_update,
                dbc.Alert("Error inesperado al guardar usuario", color="danger", dismissable=True),
                no_update,  # No limpiar store en error inesperado
            )

    # ========================================
    # CALLBACK 5: Open Edit User Modal (Pattern-Matching)
    # ========================================
    @app.callback(
        [
            Output("admin-user-modal", "is_open", allow_duplicate=True),
            Output("admin-user-form-mode-store", "data", allow_duplicate=True),
            Output("admin-selected-user-store", "data", allow_duplicate=True),
            Output("admin-user-modal-title", "children", allow_duplicate=True),
            Output("admin-user-modal-icon", "className", allow_duplicate=True),
        ],
        Input({"type": "edit-user-btn", "index": ALL}, "n_clicks"),
        State("admin-users-data-store", "data"),
        prevent_initial_call=True,
    )
    def open_edit_user_modal(
        n_clicks_list: List[Optional[int]], users_data: Dict
    ) -> Tuple[bool, str, Optional[Dict], str, str]:
        """
        Abre modal en modo EDIT con datos del usuario seleccionado.

        Triggered by: Pattern-matching edit buttons ({"type": "edit-user-btn", "index": user_id})

        Returns:
        - modal is_open = True
        - form_mode = "edit"
        - selected_user = user data dict
        - modal_title = "Editar Usuario"
        - modal_icon = "fas fa-user-edit"
        """
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Obtener ID del botón clickeado
        button_id = ctx.triggered_id
        if not button_id or not isinstance(button_id, dict):
            raise PreventUpdate

        user_id = button_id.get("index")
        logger.info(f"[USERS_EDIT_MODAL] Opening edit modal for user: {user_id}")

        # Buscar usuario en users_data
        selected_user = None
        for user in users_data.get("users", []):
            if user.get("id") == user_id:
                selected_user = user
                break

        if not selected_user:
            logger.warning(f"[USERS_EDIT_MODAL] User not found: {user_id}")
            return no_update, no_update, no_update, no_update, no_update

        return (
            True,  # Open modal
            "edit",  # Mode
            selected_user,  # User data
            "Editar Usuario",  # Title
            "fas fa-user-edit me-2 text-warning",  # Icon
        )

    # ========================================
    # CALLBACK 6: Open Delete User Modal (Pattern-Matching)
    # ========================================
    @app.callback(
        [
            Output("admin-delete-user-modal", "is_open", allow_duplicate=True),
            Output("admin-selected-user-store", "data", allow_duplicate=True),
            Output("admin-delete-user-name", "children"),
        ],
        Input({"type": "delete-user-btn", "index": ALL}, "n_clicks"),
        State("admin-users-data-store", "data"),
        prevent_initial_call=True,
    )
    def open_delete_user_modal(
        n_clicks_list: List[Optional[int]], users_data: Dict
    ) -> Tuple[bool, Optional[Dict], str]:
        """
        Abre modal de confirmación para eliminar usuario.

        Triggered by: Pattern-matching delete buttons ({"type": "delete-user-btn", "index": user_id})

        Returns:
        - delete_modal is_open = True
        - selected_user = user data dict
        - delete_user_name = user's full_name or email
        """
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        button_id = ctx.triggered_id
        if not button_id or not isinstance(button_id, dict):
            raise PreventUpdate

        user_id = button_id.get("index")
        logger.info(f"[USERS_DELETE_MODAL] Opening delete modal for user: {user_id}")

        # Buscar usuario en users_data
        selected_user = None
        for user in users_data.get("users", []):
            if user.get("id") == user_id:
                selected_user = user
                break

        if not selected_user:
            logger.warning(f"[USERS_DELETE_MODAL] User not found: {user_id}")
            return no_update, no_update, no_update

        user_name = selected_user.get("full_name") or selected_user.get("email", "Usuario")

        return True, selected_user, user_name

    # ========================================
    # CALLBACK 7: Confirm Delete User (Soft Delete)
    # ========================================
    @app.callback(
        [
            Output("admin-user-operation-result", "data", allow_duplicate=True),
            Output("toast-trigger-store", "data", allow_duplicate=True),
            Output("admin-delete-user-modal", "is_open", allow_duplicate=True),
        ],
        Input("admin-delete-user-modal-confirm", "n_clicks"),
        State("admin-selected-user-store", "data"),
        prevent_initial_call=True,
    )
    def confirm_delete_user(n_clicks: Optional[int], selected_user: Optional[Dict]) -> Tuple[Dict, Dict, bool]:
        """
        Ejecuta soft delete del usuario seleccionado.

        API: DELETE /api/v1/admin/users/{id}

        Returns:
        - operation-result (triggers list refresh)
        - toast notification
        - modal is_open = False
        """
        if not n_clicks or not selected_user:
            raise PreventUpdate

        user_id = selected_user.get("id")
        user_email = selected_user.get("email", "Usuario")

        access_token = auth_manager.get_access_token()
        if not access_token:
            return (
                no_update,
                error_toast("Token de acceso no disponible", "Error"),
                no_update,
            )

        try:
            logger.info(f"[USERS_DELETE] Soft deleting user: {user_id}")

            response = requests.delete(
                f"{BACKEND_URL}/api/v1/admin/users/{user_id}",
                headers={"Authorization": f"Bearer {access_token}"},
                timeout=REQUEST_TIMEOUT,
            )

            response.raise_for_status()

            logger.info(f"[USERS_DELETE] User deleted successfully: {user_id}")

            return (
                {"action": "delete", "user_id": user_id, "timestamp": datetime.now().isoformat()},
                success_toast(f"Usuario {user_email} eliminado exitosamente (soft delete)", "Usuario Eliminado"),
                False,  # Close modal
            )

        except requests.HTTPError as e:
            logger.error(f"[USERS_DELETE] HTTP error: {e.response.status_code} - {e.response.text}")

            error_message = "Error al eliminar usuario"
            if e.response.status_code == 400:
                error_message = "El usuario ya está eliminado"
            elif e.response.status_code == 403:
                error_message = "Sin permisos para eliminar usuarios"
            elif e.response.status_code == 404:
                error_message = "Usuario no encontrado"

            return (
                no_update,
                error_toast(error_message, "Error"),
                no_update,
            )

        except Exception as e:
            logger.exception(f"[USERS_DELETE] Unexpected error: {e}")
            return (
                no_update,
                error_toast(f"Error inesperado: {str(e)}", "Error"),
                no_update,
            )

    # ========================================
    # CALLBACK 8: Open Restore User Modal (Pattern-Matching)
    # ========================================
    @app.callback(
        [
            Output("admin-restore-user-modal", "is_open", allow_duplicate=True),
            Output("admin-selected-user-store", "data", allow_duplicate=True),
            Output("admin-restore-user-name", "children"),
        ],
        Input({"type": "restore-user-btn", "index": ALL}, "n_clicks"),
        State("admin-users-data-store", "data"),
        prevent_initial_call=True,
    )
    def open_restore_user_modal(
        n_clicks_list: List[Optional[int]], users_data: Dict
    ) -> Tuple[bool, Optional[Dict], str]:
        """
        Abre modal de confirmación para restaurar usuario eliminado.

        Triggered by: Pattern-matching restore buttons ({"type": "restore-user-btn", "index": user_id})

        Returns:
        - restore_modal is_open = True
        - selected_user = user data dict
        - restore_user_name = user's full_name or email
        """
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        button_id = ctx.triggered_id
        if not button_id or not isinstance(button_id, dict):
            raise PreventUpdate

        user_id = button_id.get("index")
        logger.info(f"[USERS_RESTORE_MODAL] Opening restore modal for user: {user_id}")

        # Buscar usuario en users_data
        selected_user = None
        for user in users_data.get("users", []):
            if user.get("id") == user_id:
                selected_user = user
                break

        if not selected_user:
            logger.warning(f"[USERS_RESTORE_MODAL] User not found: {user_id}")
            return no_update, no_update, no_update

        user_name = selected_user.get("full_name") or selected_user.get("email", "Usuario")

        return True, selected_user, user_name

    # ========================================
    # CALLBACK 9: Confirm Restore User
    # ========================================
    @app.callback(
        [
            Output("admin-user-operation-result", "data", allow_duplicate=True),
            Output("toast-trigger-store", "data", allow_duplicate=True),
            Output("admin-restore-user-modal", "is_open", allow_duplicate=True),
        ],
        Input("admin-restore-user-modal-confirm", "n_clicks"),
        State("admin-selected-user-store", "data"),
        prevent_initial_call=True,
    )
    def confirm_restore_user(n_clicks: Optional[int], selected_user: Optional[Dict]) -> Tuple[Dict, Dict, bool]:
        """
        Restaura usuario desde soft delete.

        API: POST /api/v1/admin/users/{id}/restore

        Returns:
        - operation-result (triggers list refresh)
        - toast notification
        - modal is_open = False
        """
        if not n_clicks or not selected_user:
            raise PreventUpdate

        user_id = selected_user.get("id")
        user_email = selected_user.get("email", "Usuario")

        access_token = auth_manager.get_access_token()
        if not access_token:
            return (
                no_update,
                error_toast("Token de acceso no disponible", "Error"),
                no_update,
            )

        try:
            logger.info(f"[USERS_RESTORE] Restoring user: {user_id}")

            response = requests.post(
                f"{BACKEND_URL}/api/v1/admin/users/{user_id}/restore",
                headers={"Authorization": f"Bearer {access_token}"},
                timeout=REQUEST_TIMEOUT,
            )

            response.raise_for_status()

            logger.info(f"[USERS_RESTORE] User restored successfully: {user_id}")

            return (
                {"action": "restore", "user_id": user_id, "timestamp": datetime.now().isoformat()},
                success_toast(f"Usuario {user_email} restaurado exitosamente", "Usuario Restaurado"),
                False,  # Close modal
            )

        except requests.HTTPError as e:
            logger.error(f"[USERS_RESTORE] HTTP error: {e.response.status_code} - {e.response.text}")

            error_message = "Error al restaurar usuario"
            if e.response.status_code == 400:
                error_message = "El usuario no está eliminado"
            elif e.response.status_code == 403:
                error_message = "Sin permisos para restaurar usuarios"
            elif e.response.status_code == 404:
                error_message = "Usuario no encontrado"

            return (
                no_update,
                error_toast(error_message, "Error"),
                no_update,
            )

        except Exception as e:
            logger.exception(f"[USERS_RESTORE] Unexpected error: {e}")
            return (
                no_update,
                error_toast(f"Error inesperado: {str(e)}", "Error"),
                no_update,
            )

    # ========================================
    # CALLBACK 10: Cancel All User Modals
    # ========================================
    @app.callback(
        [
            Output("admin-user-modal", "is_open", allow_duplicate=True),
            Output("admin-delete-user-modal", "is_open", allow_duplicate=True),
            Output("admin-restore-user-modal", "is_open", allow_duplicate=True),
            Output("admin-selected-user-store", "data", allow_duplicate=True),
            Output("admin-user-form-data-store", "data", allow_duplicate=True),  # NEW: Limpiar store
        ],
        [
            Input("admin-user-modal-cancel", "n_clicks"),
            Input("admin-delete-user-modal-cancel", "n_clicks"),
            Input("admin-restore-user-modal-cancel", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def cancel_user_modals(
        modal_cancel_clicks: Optional[int],
        delete_cancel_clicks: Optional[int],
        restore_cancel_clicks: Optional[int],
    ) -> Tuple[bool, bool, bool, None, None]:
        """
        Cierra todos los modales de usuario sin guardar cambios.

        Returns:
        - admin-user-modal is_open = False
        - admin-delete-user-modal is_open = False
        - admin-restore-user-modal is_open = False
        - selected-user-store = None (clear selection)
        - form-data-store = None (clear form data)
        """
        if not ctx.triggered:
            raise PreventUpdate

        logger.info("[USERS_MODAL] Canceling user modal")

        # Cerrar todos los modales, limpiar selección y datos del formulario
        return False, False, False, None, None

    # ========================================
    # CALLBACK 11: Toggle Password Visibility
    # ========================================
    @app.callback(
        [
            Output("admin-user-password-input", "type"),
            Output("admin-password-toggle-icon", "className"),
        ],
        Input("admin-password-toggle-btn", "n_clicks"),
        State("admin-user-password-input", "type"),
        prevent_initial_call=True,
    )
    def toggle_password_visibility(n_clicks: Optional[int], current_type: str) -> Tuple[str, str]:
        """
        Alterna visibilidad de contraseña entre texto plano y oculto.

        UX 2025 standard: Eye icon toggle para passwords.

        Args:
            n_clicks: Número de clicks en botón toggle
            current_type: Tipo actual del input ("password" o "text")

        Returns:
            - input type: "text" (visible) o "password" (oculto)
            - icon className: "fas fa-eye" (mostrar) o "fas fa-eye-slash" (ocultar)
        """
        if not n_clicks:
            raise PreventUpdate

        if current_type == "password":
            # Mostrar contraseña
            return "text", "fas fa-eye-slash"
        else:
            # Ocultar contraseña
            return "password", "fas fa-eye"

    logger.info("[USERS_CALLBACKS] All 11 callbacks registered successfully (Issue #348 FASE 3.1 + Password Toggle)")
    logger.info("[USERS_CALLBACKS] Pattern-matching buttons: edit-user-btn, delete-user-btn, restore-user-btn")

    _module_callbacks_registered = True
