"""
Sync Operations Callbacks Module.
Responsabilidad: Gestión avanzada de sincronizaciones con el catálogo.
"""

import logging
from datetime import datetime

import dash_bootstrap_components as dbc
from components.toast_manager import error_toast, info_toast, success_toast, warning_toast
from dash import Input, Output, State, ctx, html, no_update
from dash.exceptions import PreventUpdate
from utils.callback_helpers import unified_toast_trigger
from utils.config import BACKEND_URL
from utils.request_coordinator import request_coordinator
from utils.auth_helpers import get_auth_headers_from_tokens  # REGLA #7.6: Multi-worker token restoration

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_sync_operations_callbacks(app):
    """
    Register sync operations related callbacks.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Sync operations callbacks already registered, skipping")
        return app

    logger.info("Registering sync operations callbacks")

    @app.callback(
        [
            # CIMA button - actualizar IDs internos individualmente
            Output("sync-cima-icon", "className"),
            Output("sync-cima-text", "children"),
            Output("admin-sync-cima-button", "disabled"),
            Output("admin-sync-cima-button", "color"),
            # Nomenclator button - actualizar IDs internos individualmente
            Output("sync-nomenclator-icon", "className"),
            Output("sync-nomenclator-text", "children"),
            Output("admin-sync-nomenclator-button", "disabled"),
            Output("admin-sync-nomenclator-button", "color"),
            # All button - actualizar IDs internos individualmente
            Output("sync-all-icon", "className"),
            Output("sync-all-text", "children"),
            Output("admin-sync-all-button", "disabled"),
            Output("admin-sync-all-button", "color"),
            # Intermediate store for toast
            Output("sync-operations-toast-store", "data"),
        ],
        [
            Input("admin-sync-cima-button", "n_clicks"),
            Input("admin-sync-nomenclator-button", "n_clicks"),
            Input("admin-sync-all-button", "n_clicks"),
        ],
        [State("auth-tokens-store", "data")],  # REGLA #7.6: Multi-worker token restoration
        prevent_initial_call=True,
    )
    def handle_sync_operations(cima_clicks, nomenclator_clicks, all_clicks, auth_tokens):
        """
        Maneja las operaciones de sincronización con feedback visual mediante toasts.
        Actualiza IDs internos de los botones para evitar conflictos de callbacks.

        Fix para Issue de callback error: Actualiza sync-*-icon y sync-*-text
        en lugar de sobrescribir .children del botón completo.
        """
        # CRÍTICO: Validar que hay un trigger válido
        if not ctx.triggered:
            return get_default_sync_states() + ({},)

        # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
        # Use centralized helper for token extraction (bypasses singleton limitation)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        # CRÍTICO: Validar que el trigger es un click real (no None, no 0)
        trigger_prop = ctx.triggered[0]["prop_id"]
        trigger_value = ctx.triggered[0]["value"]

        # Prevenir ejecución si n_clicks es None o 0 (componente recién creado)
        if trigger_value is None or trigger_value == 0:
            logger.debug(f"[SYNC] Prevented - trigger value invalid: {trigger_value}")
            return get_default_sync_states() + ({},)

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # Estados base
        states = get_default_sync_states()
        toast = {}

        try:
            if trigger_id == "admin-sync-cima-button":
                states = get_sync_in_progress_states("cima")
                toast = execute_sync_operation("cima", auth_headers=auth_headers)

            elif trigger_id == "admin-sync-nomenclator-button":
                states = get_sync_in_progress_states("nomenclator")
                toast = execute_sync_operation("nomenclator", auth_headers=auth_headers)

            elif trigger_id == "admin-sync-all-button":
                states = get_sync_in_progress_states("all")
                toast = execute_sync_operation("all", auth_headers=auth_headers)

        except Exception as e:
            logger.error(f"Error in sync operation: {e}")
            toast = error_toast(f"Error ejecutando sincronización: {str(e)}", title="Error de Sincronización")

        return states + (toast,)

    @app.callback(
        Output("sync-history-display", "children"),
        Input("admin-stats-interval", "n_intervals"),
        [
            State("auth-state", "data"),  # Issue #333: Verificar autenticación
            State("auth-context-sync", "data"),  # Issue #333: JWT timing protection
        ],
        prevent_initial_call=False,
    )
    def update_sync_history(n_intervals, auth_state, auth_context_sync):
        """
        Actualiza el historial de sincronizaciones.

        Issue #333: Protección contra JWT timing race condition.

        Args:
            n_intervals: Número de intervalos transcurridos
            auth_state: Estado de autenticación del usuario
            auth_context_sync: Estado de sincronización del token JWT

        Returns:
            Componente con historial de sincronizaciones
        """
        from utils.auth_helpers import is_user_authenticated

        # Issue #333: Verificar autenticación ANTES de hacer llamadas API
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Issue #333: Verificar que el token JWT esté sincronizado
        # Previene errores HTTP 401 por race condition en auth_context_sync
        if not auth_context_sync or not auth_context_sync.get("synced"):
            logger.debug("[update_sync_history] Token not synced yet (auth-context-sync pending) - showing skeleton")
            return html.Div([html.I(className="fas fa-spinner fa-spin me-2"), "Cargando historial..."])

        # TODO: Endpoint /api/v1/system/catalog/sync-history no implementado aún
        # Por ahora retornamos mensaje placeholder (componente está oculto en UI)
        return html.Div("Historial de sincronizaciones en desarrollo", className="text-muted")

    @app.callback(
        [Output("scheduled-syncs-panel", "children"), Output("scheduled-sync-toast-store", "data")],
        [Input("enable-scheduled-sync-switch", "value"), Input("sync-schedule-dropdown", "value")],
        [
            State("auth-state", "data"),  # Issue #333: Verificar autenticación
            State("auth-context-sync", "data"),  # Issue #333: JWT timing protection
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def handle_scheduled_syncs(enabled, schedule, auth_state, auth_context_sync, auth_tokens):
        """
        Maneja la configuración de sincronizaciones programadas.

        Issue #333: Protección contra JWT timing race condition.

        Args:
            enabled: Switch de activación
            schedule: Frecuencia programada
            auth_state: Estado de autenticación del usuario
            auth_context_sync: Estado de sincronización del token JWT

        Returns:
            Tuple: (panel content, toast data)
        """
        from utils.auth_helpers import is_user_authenticated

        # Issue #333: Verificar autenticación ANTES de hacer llamadas API
        if not is_user_authenticated(auth_state):
            error_toast_data = error_toast("Sesión expirada - Por favor inicie sesión", title="Error de Autenticación")
            return no_update, error_toast_data

        # Issue #333: Verificar que el token JWT esté sincronizado
        # Previene errores HTTP 401 por race condition en auth_context_sync
        if not auth_context_sync or not auth_context_sync.get("synced"):
            logger.debug("[handle_scheduled_syncs] Token not synced yet (auth-context-sync pending) - waiting for sync")
            raise PreventUpdate  # Silent - callback will retry on next interval/change

        # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
        # Use centralized helper for token extraction (bypasses singleton limitation)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        try:
            if enabled:
                # Configurar sincronización programada
                config = {"enabled": True, "schedule": schedule}  # "daily", "weekly", "monthly"
                response_data = request_coordinator.make_request(
                    "/api/v1/system/scheduled-sync", method="POST", data=config, cache_ttl=0, bypass_cache=True,
                    auth_headers=auth_headers  # REGLA #7.6: Explicit headers for multi-worker
                )

                if response_data:
                    toast = success_toast(
                        f"Sincronización {schedule} programada correctamente", title="Configuración Guardada"
                    )
                    return html.Div(), toast
                else:
                    toast = error_toast("Error configurando sincronización programada", title="Error de Configuración")
                    return html.Div(), toast
            else:
                # Desactivar sincronización programada
                response = requests.delete(f"{BACKEND_URL}/api/system/scheduled-sync", timeout=10)
                toast = info_toast("Sincronización programada desactivada", title="Configuración Actualizada")
                return html.Div(), toast
        except Exception as e:
            logger.error(f"Error configuring scheduled sync: {e}")
            toast = error_toast(f"Error: {str(e)}", title="Error del Sistema")
            return html.Div(), toast

    @app.callback(
        [Output("sync-conflicts-resolution", "children"), Output("conflict-resolution-toast-store", "data")],
        Input("resolve-conflicts-button", "n_clicks"),
        [
            State("conflict-resolution-strategy", "value"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def handle_sync_conflicts(n_clicks, strategy, auth_tokens):
        """
        Maneja la resolución de conflictos de sincronización.
        """
        if not n_clicks:
            return html.Div(), {}

        # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
        # Use centralized helper for token extraction (bypasses singleton limitation)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        try:
            # Resolver conflictos según estrategia con autenticación
            response_data = request_coordinator.make_request(
                "/api/v1/system/resolve-sync-conflicts",
                method="POST",
                data={"strategy": strategy},  # "keep_newest", "keep_oldest", "merge"
                cache_ttl=0,
                bypass_cache=True,
                auth_headers=auth_headers,  # REGLA #7.6: Explicit headers for multi-worker
            )

            if response_data:
                result = response_data.get("data", {})
                conflicts_count = result.get("conflicts_resolved", 0)
                toast = success_toast(
                    f"{conflicts_count} productos actualizados correctamente", title="Conflictos Resueltos"
                )
                return html.Div(), toast
            else:
                toast = error_toast("Error resolviendo conflictos de sincronización", title="Error de Resolución")
                return html.Div(), toast
        except Exception as e:
            logger.error(f"Error resolving conflicts: {e}")
            toast = error_toast(f"Error: {str(e)}", title="Error del Sistema")
            return html.Div(), toast

    @app.callback(
        Output("toast-trigger-store", "data", allow_duplicate=True),
        [
            Input("sync-operations-toast-store", "data"),
            Input("scheduled-sync-toast-store", "data"),
            Input("conflict-resolution-toast-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def unified_toast_trigger_callback(sync_toast, scheduled_toast, conflict_toast):
        """
        Callback unificado usando helper reutilizable (utils/callback_helpers.py).
        Solución a DASH002: Un solo callback escribe a toast-trigger-store.
        """
        return unified_toast_trigger(sync_toast, scheduled_toast, conflict_toast)

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("Sync operations callbacks registered successfully")

    return app


def get_default_sync_states():
    """
    Retorna los estados por defecto de los botones de sincronización.
    Actualiza IDs internos en lugar de .children completo.

    Returns:
        tuple: (
            cima_icon_className, cima_text_children, cima_button_disabled, cima_button_color,
            nomenclator_icon_className, nomenclator_text_children, nomenclator_button_disabled, nomenclator_button_color,
            all_icon_className, all_text_children, all_button_disabled, all_button_color
        )
    """
    return (
        # CIMA button
        "fas fa-sync me-2",  # sync-cima-icon className
        "Sync CIMA",  # sync-cima-text children
        False,  # admin-sync-cima-button disabled
        "primary",  # admin-sync-cima-button color
        # Nomenclator button
        "fas fa-sync me-2",  # sync-nomenclator-icon className
        "Sync Nomenclátor",  # sync-nomenclator-text children
        False,  # admin-sync-nomenclator-button disabled
        "info",  # admin-sync-nomenclator-button color
        # All button
        "fas fa-sync-alt me-2",  # sync-all-icon className
        "Sync Completo",  # sync-all-text children
        False,  # admin-sync-all-button disabled
        "success",  # admin-sync-all-button color
    )


def get_sync_in_progress_states(sync_type):
    """
    Retorna los estados de los botones durante una sincronización.
    Actualiza IDs internos en lugar de .children completo.

    Args:
        sync_type: Tipo de sincronización ('cima', 'nomenclator', 'all')

    Returns:
        tuple: (
            cima_icon_className, cima_text_children, cima_button_disabled, cima_button_color,
            nomenclator_icon_className, nomenclator_text_children, nomenclator_button_disabled, nomenclator_button_color,
            all_icon_className, all_text_children, all_button_disabled, all_button_color
        )
    """
    # Estados base
    cima_icon = "fas fa-sync me-2"
    cima_text = "Sync CIMA"
    cima_disabled = False
    cima_color = "primary"

    nomenclator_icon = "fas fa-sync me-2"
    nomenclator_text = "Sync Nomenclátor"
    nomenclator_disabled = False
    nomenclator_color = "info"

    all_icon = "fas fa-sync-alt me-2"
    all_text = "Sync Completo"
    all_disabled = False
    all_color = "success"

    # Actualizar según tipo de sincronización activa
    if sync_type == "cima":
        cima_icon = "fas fa-spinner fa-spin me-2"
        cima_text = "Sincronizando CIMA..."
        cima_disabled = True
        cima_color = "outline-secondary"
        nomenclator_disabled = True
        all_disabled = True

    elif sync_type == "nomenclator":
        nomenclator_icon = "fas fa-spinner fa-spin me-2"
        nomenclator_text = "Sincronizando Nomenclátor..."
        nomenclator_disabled = True
        nomenclator_color = "outline-secondary"
        cima_disabled = True
        all_disabled = True

    elif sync_type == "all":
        all_icon = "fas fa-spinner fa-spin me-2"
        all_text = "Sincronización Completa..."
        all_disabled = True
        all_color = "outline-secondary"
        cima_disabled = True
        nomenclator_disabled = True

    return (
        cima_icon,
        cima_text,
        cima_disabled,
        cima_color,
        nomenclator_icon,
        nomenclator_text,
        nomenclator_disabled,
        nomenclator_color,
        all_icon,
        all_text,
        all_disabled,
        all_color,
    )


def execute_sync_operation(sync_type, auth_headers=None):
    """
    Ejecuta una operación de sincronización en el backend.
    Retorna un toast dict para notificar el resultado.

    Args:
        sync_type: Tipo de sincronización ('cima', 'nomenclator', 'all')
        auth_headers: Headers de autenticación explícitos (REGLA #7.6 multi-worker)
    """
    try:
        # Determinar target según tipo
        target_map = {"cima": "cima", "nomenclator": "nomenclator", "all": "all"}

        target = target_map.get(sync_type, "all")

        # Llamar al endpoint de sincronización con autenticación
        # REGLA #7.6: Pass explicit auth_headers for multi-worker support
        response_data = request_coordinator.make_request(
            "/api/v1/system/sync",
            method="POST",
            data={"target": target, "force": False},
            cache_ttl=0,
            bypass_cache=True,
            auth_headers=auth_headers,
        )

        if response_data:
            result = response_data
            operation_id = result.get("data", {}).get("operation_id", "N/A")

            sync_names = {"cima": "CIMA", "nomenclator": "Nomenclátor", "all": "completa"}
            sync_name = sync_names.get(sync_type, "del catálogo")

            return info_toast(
                f"Sincronización {sync_name} iniciada. La operación se ejecuta en segundo plano. ID: {operation_id}",
                title="Sincronización Iniciada",
            )
        else:
            # Si response_data es None, hubo un error
            return error_toast(
                "Error al comunicarse con el servidor. Verifique la conexión.", title="Error de Sincronización"
            )

    except Exception as e:
        logger.error(f"Error executing sync: {e}")
        error_msg = str(e)

        # Determinar tipo de toast según el error
        if "timeout" in error_msg.lower() or "timed out" in error_msg.lower():
            return warning_toast("La solicitud tardó demasiado. Intente nuevamente.", title="Tiempo de Espera Agotado")
        elif "429" in error_msg:
            return warning_toast(
                "Demasiadas sincronizaciones. Por favor espere antes de intentar nuevamente.",
                title="Límite de Solicitudes",
            )
        else:
            return error_toast(f"Error de conexión: {error_msg}", title="Error de Sincronización")


def create_sync_history_display(history):
    """
    Crea el display del historial de sincronizaciones.
    """
    if not history:
        return html.P("No hay sincronizaciones previas", className="text-muted")

    history_rows = []
    for sync in history[:10]:  # Mostrar últimas 10
        sync_type = sync.get("type", "unknown")
        sync_date = sync.get("date", "")
        sync_status = sync.get("status", "unknown")
        sync_duration = sync.get("duration_minutes", 0)

        status_colors = {"success": "success", "failed": "danger", "partial": "warning"}
        status_color = status_colors.get(sync_status, "secondary")

        history_rows.append(
            html.Tr(
                [
                    html.Td(sync_type.upper()),
                    html.Td(format_datetime(sync_date)),
                    html.Td(dbc.Badge(sync_status.upper(), color=status_color)),
                    html.Td(f"{sync_duration} min"),
                ]
            )
        )

    return dbc.Table(
        [
            html.Thead([html.Tr([html.Th("Tipo"), html.Th("Fecha"), html.Th("Estado"), html.Th("Duración")])]),
            html.Tbody(history_rows),
        ],
        striped=True,
        hover=True,
        size="sm",
    )


def format_datetime(dt_string):
    """
    Formatea una fecha/hora para display.
    """
    try:
        dt = datetime.fromisoformat(dt_string.replace("Z", "+00:00"))
        return dt.strftime("%d/%m/%Y %H:%M")
    except:
        return dt_string
