"""
Admin Prescription Callbacks - Issue #16 Fase 2
Gestión de clasificación de productos de prescripción.

Este módulo implementa todos los callbacks necesarios para el tab de Prescripción:
- Load de stats (total, clasificados, no clasificados, categorías activas)
- Gráfico de distribución de categorías (pie chart)
- Tabla de productos no clasificados ordenados por ventas
- Batch classification automática
- Upload Excel para clasificación masiva
- Export CSV de clasificaciones actuales
- Modal de categorías válidas
"""

import base64
import io
import logging
from datetime import datetime
from typing import Any, Dict, List, Optional, Tuple

import dash_bootstrap_components as dbc
import plotly.express as px
import plotly.graph_objects as go
import requests
from dash import Input, Output, State, callback, ctx, dcc, html, no_update
from dash.exceptions import PreventUpdate

from components.toast_manager import error_toast, info_toast, success_toast, warning_toast
from utils.auth import auth_manager
from utils.auth_helpers import is_user_authenticated
from utils.config import BACKEND_URL
from utils.helpers import format_currency, format_number

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False

# Constants
REQUEST_TIMEOUT = 30  # seconds for API calls
MAX_UNCLASSIFIED_PRODUCTS = 100  # Top 100 productos no clasificados

# Prescription categories (14 total)
PRESCRIPTION_CATEGORIES = [
    "MEDICAMENTOS",
    "FORMULAS_MAGISTRALES",
    "VACUNAS_INDIVIDUALIZADAS",
    "TIRAS_REACTIVAS_GLUCOSA",
    "TIRAS_REACTIVAS_CETONEMIA",
    "AGUJAS_LANCETAS",
    "ABSORBENTES_INCONTINENCIA",
    "SISTEMAS_AUTOADMINISTRACION",
    "DISPOSITIVOS_INHALACION",
    "SUEROS",
    "MEDICAMENTOS_EXTRANJEROS",
    "PRODUCTOS_SANITARIOS",
    "OXIGENOTERAPIA",
    "OTROS",
]

# Category labels for UI (más legibles)
CATEGORY_LABELS = {
    "MEDICAMENTOS": "Medicamentos",
    "FORMULAS_MAGISTRALES": "Fórmulas Magistrales",
    "VACUNAS_INDIVIDUALIZADAS": "Vacunas Individualizadas",
    "TIRAS_REACTIVAS_GLUCOSA": "Tiras Glucosa",
    "TIRAS_REACTIVAS_CETONEMIA": "Tiras Cetonemia",
    "AGUJAS_LANCETAS": "Agujas y Lancetas",
    "ABSORBENTES_INCONTINENCIA": "Absorbentes",
    "SISTEMAS_AUTOADMINISTRACION": "Autoadministración",
    "DISPOSITIVOS_INHALACION": "Inhalación",
    "SUEROS": "Sueros",
    "MEDICAMENTOS_EXTRANJEROS": "Medicamentos Extranjeros",
    "PRODUCTOS_SANITARIOS": "Productos Sanitarios",
    "OXIGENOTERAPIA": "Oxigenoterapia",
    "OTROS": "Otros",
    "NO_CLASIFICADOS": "No Clasificados",  # ✅ Nueva categoría
}


def _format_percentage(value: float, total: float) -> str:
    """
    Formatea porcentaje para UI.

    Args:
        value: Valor parcial
        total: Valor total

    Returns:
        String con porcentaje (e.g., "45,3%") o "--" si total es 0
    """
    if not total or total == 0:
        return "--"
    percentage = (value / total) * 100
    return f"{percentage:.1f}%".replace(".", ",")


def _create_unclassified_products_table(products_data: List[Dict]) -> html.Div:
    """
    Crea tabla de productos no clasificados con columnas:
    - Código Nacional
    - Nombre
    - Ventas (últimos 12 meses)
    - Sugerencia Automática (si disponible)

    Args:
        products_data: Lista de productos no clasificados

    Returns:
        Dash component con tabla Bootstrap responsive
    """
    if not products_data:
        return dbc.Alert(
            html.Div(
                [
                    html.I(className="fas fa-check-circle me-2 text-success"),
                    "¡Excelente! Todos los productos de prescripción están clasificados.",
                ]
            ),
            color="success",
            className="text-center",
        )

    # Construir filas de tabla
    table_rows = []
    for idx, product in enumerate(products_data[:MAX_UNCLASSIFIED_PRODUCTS], 1):
        national_code = product.get("codigo_nacional", "--")
        name = product.get("nombre", "--")
        sales = product.get("ventas_12_meses", 0)
        suggested_category = product.get("categoria_sugerida")  # Puede ser None

        # Badge de sugerencia
        suggested_badge = html.Span("--", className="text-muted small")
        if suggested_category:
            category_label = CATEGORY_LABELS.get(suggested_category, suggested_category)
            suggested_badge = dbc.Badge(
                category_label,
                color="info",
                className="me-1",
                pill=True,
            )

        table_rows.append(
            html.Tr(
                [
                    html.Td(idx, className="text-center"),
                    html.Td(html.Code(national_code, className="text-primary")),
                    html.Td(name, className="text-truncate", style={"maxWidth": "300px"}),
                    html.Td(format_currency(sales), className="text-end"),
                    html.Td(suggested_badge, className="text-center"),
                ]
            )
        )

    return dbc.Table(
        [
            html.Thead(
                html.Tr(
                    [
                        html.Th("#", className="text-center", style={"width": "50px"}),
                        html.Th("Código", style={"width": "120px"}),
                        html.Th("Nombre"),
                        html.Th("Ventas (12m)", className="text-end", style={"width": "130px"}),
                        html.Th("Sugerencia", className="text-center", style={"width": "150px"}),
                    ],
                    className="table-light",
                )
            ),
            html.Tbody(table_rows),
        ],
        bordered=True,
        hover=True,
        responsive=True,
        striped=True,
        className="mb-0",
    )


def _create_category_distribution_chart(stats_data: Dict) -> go.Figure:
    """
    Crea gráfico de barras horizontales con distribución de categorías (clickable).

    Args:
        stats_data: Diccionario con stats de clasificación (/stats endpoint format)

    Returns:
        Plotly Figure con barras horizontales interactivas
    """
    # El endpoint /stats retorna category_distribution como lista de objetos
    category_distribution = stats_data.get("category_distribution", [])

    if not category_distribution:
        # Placeholder si no hay datos
        return go.Figure().add_annotation(
            text="No hay datos de clasificación disponibles",
            xref="paper",
            yref="paper",
            x=0.5,
            y=0.5,
            showarrow=False,
            font=dict(size=14, color="gray"),
        )

    # Preparar datos para gráfico
    # Formato: [{"category": "MEDICAMENTOS", "count": 600, "percentage": 75.0}, ...]
    categories = []
    counts = []
    category_keys = []  # Guardar keys originales para clickData
    for item in category_distribution:
        category_key = item.get("category")
        count = item.get("count", 0)
        if count > 0:  # Solo categorías con productos
            label = CATEGORY_LABELS.get(category_key, category_key)
            categories.append(label)
            counts.append(count)
            category_keys.append(category_key)

    # ✅ NUEVO: Agregar "No clasificados" como categoría adicional
    unclassified_count = stats_data.get("unclassified_count", 0)
    if unclassified_count > 0:
        categories.append("No Clasificados")
        counts.append(unclassified_count)
        category_keys.append("NO_CLASIFICADOS")  # Key especial para filtro

    # Crear barras horizontales (ordenadas por count descendente)
    fig = go.Figure(data=[
        go.Bar(
            y=categories,
            x=counts,
            orientation='h',
            text=[f"{count:,}" for count in counts],  # Mostrar número en barras
            textposition='auto',
            hovertemplate="<b>%{y}</b><br>Productos: %{x}<extra></extra>",
            marker=dict(
                color=counts,
                colorscale='Viridis',
                showscale=False
            ),
            customdata=[[key] for key in category_keys]  # Array 2D para Plotly
        )
    ])

    # Personalización
    fig.update_layout(
        xaxis_title="Número de Productos",
        yaxis_title="",
        showlegend=False,
        height=max(400, len(categories) * 40),  # Altura dinámica según categorías
        margin=dict(t=20, b=40, l=150, r=40),
        xaxis=dict(gridcolor='lightgray'),
        yaxis=dict(categoryorder='total ascending'),  # Ordenar por valor
        hovermode='closest',
        plot_bgcolor='white'
    )

    return fig


def _create_category_products_table(products: List[Dict], category_key: str) -> dbc.Table:
    """
    Crea tabla de productos de una categoría específica.

    Args:
        products: Lista de productos desde backend
        category_key: Key de la categoría (ej: "MEDICAMENTOS")

    Returns:
        Dash component con tabla Bootstrap responsive
    """
    if not products:
        category_label = CATEGORY_LABELS.get(category_key, category_key)
        # ✅ Mensaje sutil sin Alert (elimina "toast" al entrar en tab)
        return html.Div(
            [
                html.P(
                    [
                        html.I(className="fas fa-check text-muted me-2"),
                        f"No hay productos en {category_label}",
                    ],
                    className="text-center text-muted py-4",
                    style={"fontSize": "0.9rem"}
                )
            ]
        )

    # Construir filas de tabla (limitado a 100 productos)
    table_rows = []
    for idx, product in enumerate(products[:100], 1):
        national_code = product.get("national_code", "--")
        name = product.get("product_name") or product.get("nomen_nombre", "--")
        laboratory = product.get("laboratory") or product.get("nomen_laboratorio", "--")

        table_rows.append(
            html.Tr(
                [
                    html.Td(idx, className="text-center"),
                    html.Td(html.Code(national_code, className="text-primary")),
                    html.Td(name, className="text-truncate", style={"maxWidth": "400px"}),
                    html.Td(laboratory, className="text-truncate", style={"maxWidth": "200px"}),
                ]
            )
        )

    return dbc.Table(
        [
            html.Thead(
                html.Tr(
                    [
                        html.Th("#", className="text-center", style={"width": "50px"}),
                        html.Th("Código", style={"width": "120px"}),
                        html.Th("Nombre"),
                        html.Th("Laboratorio", style={"width": "200px"}),
                    ],
                    className="table-light",
                )
            ),
            html.Tbody(table_rows),
        ],
        bordered=True,
        hover=True,
        responsive=True,
        striped=True,
        className="mb-0",
        style={"fontSize": "0.9rem"},  # Texto ligeramente más pequeño
    )


def register_prescription_callbacks(app):
    """
    Register prescription classification callbacks.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Prescription callbacks already registered, skipping")
        return app

    logger.info("Registering prescription callbacks")

    # Callback 1: Load stats and populate stores on tab activation
    @app.callback(
        [
            Output("prescription-stats-store", "data"),
            Output("prescription-unclassified-data-store", "data"),
            Output("prescription-all-products-store", "data"),  # ✅ NUEVO: Todos los productos
            Output("prescription-selected-category-store", "data", allow_duplicate=True),  # ✅ Auto-seleccionar NO_CLASIFICADOS (duplicado con Callback 5)
            Output("prescription-reference-stats-trigger", "data"),  # Fix REGLA #11: Trigger separado para reference list stats
        ],
        [Input("prescription-tab-activated-trigger", "data")],  # ✅ Fix REGLA #11: Leer de Store trigger
        [State("auth-state", "data")],
        prevent_initial_call=True,  # ✅ OBLIGATORIO con allow_duplicate
    )
    def load_prescription_data_on_tab_change(trigger_data, auth_state):
        """
        Carga datos de prescription cuando se activa el tab (via Store trigger).
        Auto-selecciona "NO_CLASIFICADOS" para mostrar tabla por defecto.

        Fix REGLA #11: Lee de prescription-tab-activated-trigger en lugar de admin-tabs
        para evitar Input duplicado.

        Returns:
            Tuple[Dict, List, List, str]: (stats_data, unclassified_products, all_products, selected_category)
        """
        # Solo ejecutar si trigger es para prescription tab
        if not trigger_data or trigger_data.get("tab") != "prescription":
            raise PreventUpdate

        # Verificar autenticación (REGLA #7)
        if not is_user_authenticated(auth_state):
            logger.debug("[load_prescription_data] User not authenticated - skipping")
            raise PreventUpdate

        try:
            # Obtener JWT token
            access_token = auth_manager.get_access_token()
            if not access_token:
                logger.warning("[load_prescription_data] No access token available")
                raise PreventUpdate

            headers = {"Authorization": f"Bearer {access_token}"}

            # Llamada a /api/v1/admin/prescription/stats
            stats_response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/prescription/stats",
                headers=headers,
                timeout=REQUEST_TIMEOUT,
            )
            stats_response.raise_for_status()
            stats_data = stats_response.json()

            logger.info(f"[load_prescription_data] Stats loaded: {stats_data}")

            # Extraer top productos no clasificados de unclassified_summary
            unclassified_products = []
            unclassified_summary = stats_data.get("unclassified_summary")
            if unclassified_summary and "top_unclassified_by_sales" in unclassified_summary:
                # Formato: [{"national_code": "123", "product_name": "X", "suggested_category": "...", "total_sales_last_month": 1000}, ...]
                for item in unclassified_summary["top_unclassified_by_sales"]:
                    unclassified_products.append({
                        "codigo_nacional": item.get("national_code"),
                        "nombre": item.get("product_name"),
                        "ventas_12_meses": item.get("total_sales_last_month", 0),  # Solo tenemos last_month del backend
                        "categoria_sugerida": item.get("suggested_category"),
                    })

            # ✅ NUEVO: Cargar TODOS los productos para filtro por categoría
            products_response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/prescription/products",
                headers=headers,
                timeout=REQUEST_TIMEOUT,
            )
            products_response.raise_for_status()
            all_products = products_response.json()

            logger.info(
                f"[load_prescription_data] Data loaded successfully - "
                f"Classification rate: {stats_data.get('classification_rate')}%, "
                f"Top unclassified: {len(unclassified_products)}, "
                f"All products: {len(all_products)}"
            )

            # ✅ Auto-seleccionar "NO_CLASIFICADOS" para mostrar tabla por defecto
            # Fix REGLA #11: Trigger separado para reference list stats
            reference_trigger = {"loaded": True}
            return stats_data, unclassified_products, all_products, "NO_CLASIFICADOS", reference_trigger

        except requests.RequestException as e:
            logger.error(f"[load_prescription_data] API error: {str(e)}")
            return None, [], [], None, None

    # Callback 2: Update KPI cards AND chart from stats store (Fix REGLA #11: Consolidado)
    @app.callback(
        [
            Output("prescription-total-count", "children"),
            Output("prescription-classified-count", "children"),
            Output("prescription-classified-percentage", "children"),
            Output("prescription-unclassified-count", "children"),
            Output("prescription-unclassified-percentage", "children"),
            Output("prescription-active-categories-count", "children"),
            Output("prescription-category-distribution-chart", "children"),  # ✅ Consolidado
        ],
        [Input("prescription-stats-store", "data")],
        prevent_initial_call=True,
    )
    def update_kpis_and_chart(stats_data):
        """
        Actualiza KPI cards Y gráfico de distribución desde stats store.

        Fix REGLA #11: Consolidados callbacks 2 y 3 en uno solo para evitar
        Input duplicado en prescription-stats-store.

        Args:
            stats_data: Stats del backend (/stats endpoint format)

        Returns:
            Tuple: (total, clasificados, classified_pct, no_clasificados,
                   unclassified_pct, categorias_activas, chart)
        """
        # Default values para KPIs y chart vacío
        if not stats_data:
            empty_chart = dbc.Alert(
                html.Div(
                    [
                        html.I(className="fas fa-info-circle me-2"),
                        "No hay datos de clasificación disponibles",
                    ]
                ),
                color="light",
                className="text-center",
            )
            return "--", "--", "--", "--", "--", "--", empty_chart

        # ✅ KPI values
        total = stats_data.get("prescription_products", 0)
        clasificados = stats_data.get("classified_count", 0)
        no_clasificados = stats_data.get("unclassified_count", 0)
        category_dist = stats_data.get("category_distribution", [])
        categorias_activas = len(category_dist)

        # Formatear porcentajes
        classified_pct = _format_percentage(clasificados, total)
        unclassified_pct = _format_percentage(no_clasificados, total)

        # ✅ Chart
        fig = _create_category_distribution_chart(stats_data)
        chart = dcc.Graph(
            id="prescription-distribution-graph",  # ID para capturar clickData
            figure=fig,
            config={"displayModeBar": False, "responsive": True},
            style={"height": "400px"},
        )

        return (
            format_number(total),
            format_number(clasificados),
            classified_pct,
            format_number(no_clasificados),
            unclassified_pct,
            str(categorias_activas),
            chart,
        )

    # ✅ Fix REGLA #11: Callback 4 eliminado (huérfano)
    # Era: update_unclassified_table → prescription-unclassified-table-container
    # Razón: Componente eliminado del layout (tabla duplicada)
    # Funcionalidad cubierta por: Callback 6 (filter_category_products)

    # Callback 5: Capturar clic en gráfico (categoría seleccionada)
    @app.callback(
        Output("prescription-selected-category-store", "data"),
        Input("prescription-distribution-graph", "clickData"),
        prevent_initial_call=True,
    )
    def on_category_click(click_data):
        """
        Captura clic en barra del gráfico y guarda categoría seleccionada.

        Args:
            click_data: Datos del clic en Plotly

        Returns:
            str: Categoría seleccionada (key original como "MEDICAMENTOS")
        """
        if not click_data or "points" not in click_data:
            raise PreventUpdate

        # Extraer customdata (category_key) del punto clickeado
        point = click_data["points"][0]
        customdata = point.get("customdata")

        if not customdata or len(customdata) == 0:
            raise PreventUpdate

        category_key = customdata[0]  # customdata es array 2D: [[key1], [key2], ...]

        logger.info(f"[on_category_click] Categoría seleccionada: {category_key}")
        return category_key

    # Callback 6: Filtrar productos por categoría seleccionada (LOCAL - sin API call)
    @app.callback(
        [
            Output("prescription-category-table-container", "children"),
            Output("prescription-category-table-title", "children"),
        ],
        Input("prescription-selected-category-store", "data"),
        State("prescription-all-products-store", "data"),  # ✅ Leer del store local
        prevent_initial_call=True,
    )
    def filter_category_products(category_key, all_products):
        """
        Filtra productos por categoría seleccionada LOCALMENTE (sin API call).

        Args:
            category_key: Categoría seleccionada (ej: "MEDICAMENTOS", "NO_CLASIFICADOS")
            all_products: Todos los productos cargados en el store

        Returns:
            Tuple[Component, str]: (tabla de productos filtrados, título)
        """
        if not category_key:
            raise PreventUpdate

        if not all_products:
            return (
                html.Div([dbc.Alert("No hay productos cargados", color="warning")]),
                "Productos por Categoría",
            )

        # ✅ Filtrar productos por categoría (LOCAL - instantáneo)
        if category_key == "NO_CLASIFICADOS":
            # Productos sin categoría (category=None o null)
            filtered_products = [p for p in all_products if p.get("category") is None]
        else:
            # Productos con categoría específica
            filtered_products = [p for p in all_products if p.get("category") == category_key]

        # Formatear título
        category_label = CATEGORY_LABELS.get(category_key, category_key)
        title = f"Productos: {category_label} ({len(filtered_products)} productos)"

        # Crear tabla (limitado a primeros 100 para performance)
        table = _create_category_products_table(filtered_products[:100], category_key)
        table_container = html.Div([table])

        logger.info(f"[filter_category_products] Filtered {len(filtered_products)} products for category {category_key}")

        return table_container, title

    # Callback 7: Toggle modal de categorías
    @app.callback(
        Output("prescription-categories-modal", "is_open"),
        [
            Input("prescription-show-categories-link", "n_clicks"),
            Input("prescription-categories-modal-close", "n_clicks"),
        ],
        [State("prescription-categories-modal", "is_open")],
        prevent_initial_call=True,
    )
    def toggle_categories_modal(link_clicks, close_clicks, is_open):
        """
        Abre/cierra modal de categorías válidas.

        Fix Issue #408: Guards contra clicks fantasma que abrían modal automáticamente.

        Returns:
            bool: Nuevo estado del modal
        """
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # ✅ Fix Issue #408: Guard contra clicks fantasma
        if trigger_id == "prescription-show-categories-link":
            if link_clicks is None or link_clicks == 0:
                logger.debug("[toggle_categories_modal] Ignoring phantom click (n_clicks=None or 0)")
                raise PreventUpdate
            return True
        elif trigger_id == "prescription-categories-modal-close":
            return False

        return is_open

    # Callback 8: Auto-scroll ELIMINADO (Issue #408 fix)
    # Razón: Molesto y no aporta valor en este contexto
    # La tabla es visible en pantalla sin necesidad de scroll

    # =========================================================================
    # Callback 9: Batch Classify Products (Issue #408)
    # =========================================================================
    @app.callback(
        [
            Output("prescription-batch-status", "children"),
            Output("prescription-batch-classify-btn", "disabled"),
            Output("prescription-stats-store", "data", allow_duplicate=True),
        ],
        Input("prescription-batch-classify-btn", "n_clicks"),
        State("auth-state", "data"),
        prevent_initial_call=True,
    )
    def batch_classify_products(n_clicks, auth_state):
        """
        Ejecuta clasificación batch de productos de prescripción.

        Llama al endpoint POST /api/v1/admin/prescription/classify con dry_run=False
        para clasificar automáticamente productos no clasificados.

        Args:
            n_clicks: Número de clics en el botón
            auth_state: Estado de autenticación

        Returns:
            Tuple[Component, bool, Dict]: (status badge, disabled state, new stats)
        """
        if not n_clicks:
            raise PreventUpdate

        # Verificar autenticación (REGLA #7)
        if not is_user_authenticated(auth_state):
            logger.debug("[batch_classify_products] User not authenticated - skipping")
            raise PreventUpdate

        try:
            access_token = auth_manager.get_access_token()
            if not access_token:
                logger.warning("[batch_classify_products] No access token available")
                return (
                    dbc.Badge("Error: No autenticado", color="danger", className="mt-2"),
                    False,
                    no_update,
                )

            headers = {"Authorization": f"Bearer {access_token}"}

            # Llamar endpoint de clasificación batch
            # Limit alto (40000) para cubrir todos los productos de prescripción (~39k)
            response = requests.post(
                f"{BACKEND_URL}/api/v1/admin/prescription/classify",
                json={"dry_run": False, "limit": 50000},
                headers=headers,
                timeout=180,  # Timeout más largo para operación batch (~40k productos)
            )
            response.raise_for_status()
            result = response.json()

            classified = result.get("classified_count", 0)
            execution_time = result.get("execution_time_seconds", 0)

            logger.info(
                f"[batch_classify_products] Classified {classified} products in {execution_time:.2f}s"
            )

            # Refrescar stats
            stats_response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/prescription/stats",
                headers=headers,
                timeout=REQUEST_TIMEOUT,
            )
            new_stats = stats_response.json() if stats_response.ok else None

            # Badge de éxito
            status_badge = dbc.Badge(
                f"✓ Clasificados: {classified} ({execution_time:.1f}s)",
                color="success",
                className="mt-2",
            )

            return status_badge, False, new_stats

        except requests.RequestException as e:
            logger.error(f"[batch_classify_products] API error: {str(e)}")
            return (
                dbc.Badge("Error en clasificación", color="danger", className="mt-2"),
                False,
                no_update,
            )

    # =========================================================================
    # Callback 10: Upload Classifications from Excel (Issue #408)
    # =========================================================================
    @app.callback(
        [
            Output("prescription-upload-status", "children"),
            Output("prescription-upload-result-store", "data"),
            Output("prescription-stats-store", "data", allow_duplicate=True),
        ],
        Input("prescription-upload-excel", "contents"),
        [
            State("prescription-upload-excel", "filename"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def upload_classifications(contents, filename, auth_state):
        """
        Procesa archivo Excel con clasificaciones manuales.

        Valida el archivo y llama al endpoint bulk-update para persistir cambios.

        Args:
            contents: Contenido del archivo en base64
            filename: Nombre del archivo
            auth_state: Estado de autenticación

        Returns:
            Tuple[Component, Dict, Dict]: (status alert, result data, new stats)
        """
        if not contents:
            raise PreventUpdate

        # Verificar autenticación (REGLA #7)
        if not is_user_authenticated(auth_state):
            logger.debug("[upload_classifications] User not authenticated - skipping")
            raise PreventUpdate

        try:
            import pandas as pd

            # Decodificar contenido base64
            content_type, content_string = contents.split(",")
            decoded = base64.b64decode(content_string)

            # Leer Excel
            df = pd.read_excel(io.BytesIO(decoded))

            # Validar columnas requeridas
            required_cols = ["codigo_nacional", "categoria"]
            missing = [c for c in required_cols if c not in df.columns]
            if missing:
                return (
                    dbc.Alert(
                        f"Columnas faltantes: {', '.join(missing)}",
                        color="warning",
                        className="mt-2",
                    ),
                    {"error": "missing_columns", "missing": missing},
                    no_update,
                )

            # Validar categorías
            valid_cats = set(PRESCRIPTION_CATEGORIES)
            df["categoria"] = df["categoria"].astype(str).str.upper().str.strip()
            invalid_mask = ~df["categoria"].isin(valid_cats)
            invalid_count = invalid_mask.sum()

            if invalid_count > 0:
                invalid_cats = df.loc[invalid_mask, "categoria"].unique().tolist()[:5]
                return (
                    dbc.Alert(
                        [
                            html.Strong(f"{invalid_count} filas con categorías inválidas"),
                            html.Br(),
                            html.Small(f"Categorías: {', '.join(invalid_cats)}"),
                        ],
                        color="warning",
                        className="mt-2",
                    ),
                    {"error": "invalid_categories", "count": invalid_count},
                    no_update,
                )

            # Preparar datos para backend
            classifications = [
                {
                    "codigo_nacional": str(row["codigo_nacional"]),
                    "categoria": row["categoria"],
                }
                for _, row in df.iterrows()
            ]

            # Llamar endpoint bulk-update
            access_token = auth_manager.get_access_token()
            if not access_token:
                return (
                    dbc.Alert("Error: No autenticado", color="danger", className="mt-2"),
                    {"error": "no_auth"},
                    no_update,
                )

            headers = {"Authorization": f"Bearer {access_token}"}
            response = requests.post(
                f"{BACKEND_URL}/api/v1/admin/prescription/bulk-update",
                json={"classifications": classifications},
                headers=headers,
                timeout=60,
            )
            response.raise_for_status()
            result = response.json()

            updated = result.get("updated_count", 0)
            failed = result.get("failed_count", 0)

            logger.info(
                f"[upload_classifications] Updated {updated}, failed {failed} from {filename}"
            )

            # Refrescar stats
            stats_response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/prescription/stats",
                headers=headers,
                timeout=REQUEST_TIMEOUT,
            )
            new_stats = stats_response.json() if stats_response.ok else None

            # Mostrar resultado
            if failed > 0:
                status = dbc.Alert(
                    [
                        html.I(className="fas fa-exclamation-triangle me-2"),
                        f"Actualizados: {updated} | Errores: {failed}",
                    ],
                    color="warning",
                    className="mt-2",
                )
            else:
                status = dbc.Alert(
                    [
                        html.I(className="fas fa-check-circle me-2"),
                        f"✓ {updated} clasificaciones actualizadas",
                    ],
                    color="success",
                    className="mt-2",
                )

            return status, {"updated": updated, "failed": failed}, new_stats

        except Exception as e:
            logger.error(f"[upload_classifications] Error: {str(e)}")
            return (
                dbc.Alert(f"Error: {str(e)}", color="danger", className="mt-2"),
                {"error": str(e)},
                no_update,
            )

    # =========================================================================
    # Callback 11: Export Classifications to CSV (Issue #408)
    # =========================================================================
    @app.callback(
        Output("prescription-download-csv", "data"),
        Input("prescription-export-csv-btn", "n_clicks"),
        [
            State("prescription-all-products-store", "data"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def export_classifications(n_clicks, all_products, auth_state):
        """
        Exporta clasificaciones actuales a archivo CSV.

        Usa datos del store local o hace fetch si está vacío.

        Args:
            n_clicks: Número de clics en el botón
            all_products: Productos cargados en el store
            auth_state: Estado de autenticación

        Returns:
            dcc.send_data_frame: Descarga del archivo CSV
        """
        if not n_clicks:
            raise PreventUpdate

        # Verificar autenticación (REGLA #7)
        if not is_user_authenticated(auth_state):
            logger.debug("[export_classifications] User not authenticated - skipping")
            raise PreventUpdate

        try:
            import pandas as pd

            # Usar datos del store o fetch si vacío
            if not all_products:
                access_token = auth_manager.get_access_token()
                if not access_token:
                    logger.warning("[export_classifications] No access token")
                    raise PreventUpdate

                response = requests.get(
                    f"{BACKEND_URL}/api/v1/admin/prescription/products",
                    headers={"Authorization": f"Bearer {access_token}"},
                    timeout=REQUEST_TIMEOUT,
                )
                response.raise_for_status()
                all_products = response.json()

            if not all_products:
                logger.warning("[export_classifications] No products to export")
                raise PreventUpdate

            # Convertir a DataFrame
            df = pd.DataFrame(all_products)

            # Renombrar columnas para español
            column_map = {
                "national_code": "codigo_nacional",
                "product_name": "nombre",
                "category": "categoria",
                "laboratory": "laboratorio",
                "suggested_category": "categoria_sugerida",
            }
            df = df.rename(columns=column_map)

            # Seleccionar solo columnas relevantes (por si hay más)
            export_cols = ["codigo_nacional", "nombre", "categoria", "laboratorio", "categoria_sugerida"]
            df = df[[c for c in export_cols if c in df.columns]]

            # Generar nombre de archivo con timestamp
            timestamp = datetime.now().strftime("%Y%m%d_%H%M")
            filename = f"clasificaciones_prescripcion_{timestamp}.csv"

            logger.info(f"[export_classifications] Exporting {len(df)} products to {filename}")

            return dcc.send_data_frame(df.to_csv, filename, index=False)

        except Exception as e:
            logger.error(f"[export_classifications] Error: {str(e)}")
            raise PreventUpdate

    # =========================================================================
    # Callback 12: Load Reference List Stats (Issue #445)
    # =========================================================================
    @app.callback(
        Output("reference-list-stats-mini", "children"),
        Input("prescription-reference-stats-trigger", "data"),  # Fix REGLA #11: Trigger separado (no duplicar Input con update_kpis_and_chart)
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # Fix REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def load_reference_list_stats(trigger_data, auth_state, auth_tokens):
        """
        Carga estadísticas de listados de referencia cuando se activa el tab prescription.

        Fix REGLA #11: Se activa desde prescription-reference-stats-trigger
        (trigger separado para no duplicar Input con update_kpis_and_chart).
        Fix REGLA #7.6: Restaura tokens antes de API call (multi-worker Render).

        Args:
            trigger_data: Trigger {"loaded": True} cuando se cargaron stats
            auth_state: Estado de autenticación
            auth_tokens: Tokens encriptados para restaurar

        Returns:
            Component: Mini stats de listados de referencia
        """
        if not trigger_data or not trigger_data.get("loaded"):
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        try:
            # Fix REGLA #7.6: Restaurar tokens en este worker antes de API call
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

            access_token = auth_manager.get_access_token()
            if not access_token:
                return html.Small("Error de autenticación", className="text-danger")

            headers = {"Authorization": f"Bearer {access_token}"}
            response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/prescription/reference-list/stats",
                headers=headers,
                timeout=REQUEST_TIMEOUT,
            )

            if response.ok:
                stats = response.json()
                total = stats.get("total_entries", 0)
                last_loaded = stats.get("last_loaded_at")

                if total > 0:
                    # Formatear fecha si existe
                    date_str = ""
                    if last_loaded:
                        try:
                            from datetime import datetime
                            dt = datetime.fromisoformat(last_loaded.replace("Z", "+00:00"))
                            date_str = f" ({dt.strftime('%d/%m/%Y')})"
                        except Exception:
                            pass

                    return html.Small(
                        [
                            html.I(className="fas fa-check-circle text-success me-1"),
                            f"{format_number(total)} productos{date_str}",
                        ],
                        className="text-success",
                    )
                else:
                    return html.Small(
                        [
                            html.I(className="fas fa-exclamation-circle text-warning me-1"),
                            "Sin datos cargados",
                        ],
                        className="text-warning",
                    )
            else:
                return html.Small("Error del servidor", className="text-danger")

        except Exception as e:
            logger.error(f"[load_reference_list_stats] Error: {e}")
            return html.Small("Error de conexión", className="text-danger")

    # =========================================================================
    # Callback 13: Upload Reference List Excel (Issue #445)
    # =========================================================================
    @app.callback(
        [
            Output("reference-list-upload-status", "children"),
            Output("reference-list-stats-mini", "children", allow_duplicate=True),
        ],
        Input("reference-list-upload-excel", "contents"),
        [
            State("reference-list-upload-excel", "filename"),
            State("reference-list-truncate-checkbox", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # Fix REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def upload_reference_list(contents, filename, truncate, auth_state, auth_tokens):
        """
        Sube archivo Excel de listados de referencia oficiales.

        Llama al endpoint POST /api/v1/admin/prescription/reference-list/upload
        para cargar dietas, tiras reactivas y efectos/accesorios.

        Fix REGLA #7.6: Restaura tokens antes de API call (multi-worker Render).

        Args:
            contents: Contenido del archivo en base64
            filename: Nombre del archivo
            truncate: Si True, reemplaza datos existentes
            auth_state: Estado de autenticación
            auth_tokens: Tokens encriptados para restaurar

        Returns:
            Tuple[Component, Component]: (status alert, updated stats)
        """
        if not contents:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            logger.debug("[upload_reference_list] User not authenticated")
            raise PreventUpdate

        # Validación frontend del tipo de archivo
        if not filename or not filename.lower().endswith(('.xlsx', '.xls')):
            return (
                dbc.Alert("Archivo inválido. Solo .xlsx o .xls permitidos.", color="warning", className="mt-2 small"),
                no_update,
            )

        try:
            # Decodificar contenido base64
            content_type, content_string = contents.split(",")
            decoded = base64.b64decode(content_string)

            # Fix REGLA #7.6: Restaurar tokens en este worker antes de API call
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

            # Obtener token
            access_token = auth_manager.get_access_token()
            if not access_token:
                return (
                    dbc.Alert("Error: No autenticado", color="danger", className="mt-2 small"),
                    no_update,
                )

            headers = {"Authorization": f"Bearer {access_token}"}

            # Llamar endpoint de upload
            files = {"file": (filename, decoded, "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet")}
            params = {"truncate": "true" if truncate else "false"}

            response = requests.post(
                f"{BACKEND_URL}/api/v1/admin/prescription/reference-list/upload",
                files=files,
                params=params,
                headers=headers,
                timeout=120,  # Timeout largo para archivos grandes
            )

            if response.ok:
                result = response.json()
                total_loaded = result.get("total_loaded", 0)
                execution_time = result.get("execution_time_seconds", 0)
                warnings = result.get("warnings", [])

                # Desglose
                dietas = result.get("dietas_count", 0)
                tiras = result.get("tiras_count", 0)
                incontinencia = result.get("incontinencia_count", 0)
                ortopedia = result.get("ortopedia_count", 0)
                efectos = result.get("efectos_count", 0)

                logger.info(
                    f"[upload_reference_list] Loaded {total_loaded} products in {execution_time:.1f}s"
                    + (f" (warnings: {warnings})" if warnings else "")
                )

                # Alert de éxito (con warnings si los hay)
                alert_children = [
                    html.I(className="fas fa-check-circle me-2"),
                    html.Strong(f"{format_number(total_loaded)} productos"),
                    html.Br(),
                    html.Small(
                        f"Dietas: {dietas} | Tiras: {tiras} | Inc: {incontinencia} | "
                        f"Ort: {ortopedia} | Ef: {efectos}",
                        className="text-muted",
                    ),
                ]

                # Añadir warnings si los hay
                if warnings:
                    alert_children.append(html.Br())
                    alert_children.append(
                        html.Small(
                            f"⚠️ Advertencias: {', '.join(warnings[:3])}",
                            className="text-warning",
                        )
                    )

                status = dbc.Alert(
                    alert_children,
                    color="success" if not warnings else "warning",
                    className="mt-2 small py-2",
                )

                # Stats actualizados
                updated_stats = html.Small(
                    [
                        html.I(className="fas fa-check-circle text-success me-1"),
                        f"{format_number(total_loaded)} productos",
                    ],
                    className="text-success",
                )

                return status, updated_stats

            else:
                error_detail = response.json().get("detail", response.text)
                logger.error(f"[upload_reference_list] API error: {error_detail}")
                return (
                    dbc.Alert(f"Error: {error_detail}", color="danger", className="mt-2 small"),
                    no_update,
                )

        except Exception as e:
            logger.error(f"[upload_reference_list] Error: {str(e)}")
            return (
                dbc.Alert(f"Error: {str(e)}", color="danger", className="mt-2 small"),
                no_update,
            )

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("Prescription callbacks registered successfully")

    return app
