"""
ML Monitoring Callbacks Module (Issue #458 M6).

Responsabilidad: Carga de datos y gestión del panel ML Monitoring en Admin.
Monitorea salud del clasificador NECESIDAD para productos venta libre.
"""

import logging
from datetime import datetime

import dash
import dash_bootstrap_components as dbc
import plotly.express as px
import plotly.graph_objects as go
from dash import Input, Output, State, ctx, html, no_update
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_ml_monitoring_callbacks(app):
    """
    Register ML monitoring callbacks for admin panel.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("ML monitoring callbacks already registered, skipping")
        return app

    logger.info("Registering ML monitoring callbacks")

    # =========================================================================
    # CALLBACK 1: Load ML Monitoring Data on Tab Activation or Auto-refresh
    # =========================================================================
    @app.callback(
        [
            Output("ml-monitoring-entropy", "children"),
            Output("ml-monitoring-outlier-rate", "children"),
            Output("ml-monitoring-confidence", "children"),
            Output("ml-monitoring-pending", "children"),
            Output("ml-monitoring-coverage", "children"),  # Issue #458 F7: Coverage %
            Output("ml-monitoring-coverage-classified", "children"),  # Issue #458 F7
            Output("ml-monitoring-coverage-total", "children"),  # Issue #458 F7
            Output("ml-monitoring-last-update", "children"),
            Output("ml-monitoring-alerts-container", "children"),
            Output("ml-monitoring-distribution-chart", "figure"),
            Output("ml-monitoring-high-risk-table", "children"),
            Output("ml-monitoring-trend-chart", "figure"),
            Output("ml-monitoring-metrics-store", "data"),
            # Issue #465/#466: Precision and Drift KPIs
            Output("ml-monitoring-precision", "children"),
            Output("ml-monitoring-precision", "className"),
            Output("ml-monitoring-precision-status", "children"),
            Output("ml-monitoring-precision-status", "color"),
            Output("ml-monitoring-drift", "children"),
            Output("ml-monitoring-drift", "className"),
            Output("ml-monitoring-drift-status", "children"),
            Output("ml-monitoring-drift-status", "color"),
        ],
        [
            Input("ml-monitoring-tab-activated-trigger", "data"),
            Input("ml-monitoring-refresh-btn", "n_clicks"),
            Input("ml-monitoring-refresh-interval", "n_intervals"),  # Issue #458: Auto-refresh cada 60s
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_ml_monitoring_data(trigger_data, n_clicks, n_intervals, auth_state, auth_tokens):
        """
        Load all ML monitoring data when:
        - Tab is activated (trigger_data)
        - Refresh button clicked (n_clicks)
        - Auto-refresh interval fires (n_intervals, every 60s)

        REGLA #7.6: Multi-worker token restoration before API calls.
        """
        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[ML_MONITORING] User not authenticated - skipping")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[ML_MONITORING] No auth headers available - skipping API calls")
            raise PreventUpdate

        # Default outputs
        empty_fig = go.Figure()
        empty_fig.update_layout(
            annotations=[{"text": "Sin datos", "showarrow": False, "font": {"size": 14}}],
            margin=dict(t=20, b=20, l=20, r=20),
        )

        try:
            # Fetch classifier metrics
            metrics_response = request_coordinator.make_request(
                endpoint="/api/v1/admin/classifier/metrics",
                method="GET",
                timeout=30,
                auth_headers=auth_headers,
            )

            if not metrics_response.get("success"):
                logger.warning(f"[ML_MONITORING] Metrics fetch failed: {metrics_response.get('error')}")
                return (
                    "--", "--", "--", "--",
                    "--", "--", "--",  # Coverage outputs
                    "Error cargando datos",
                    _create_error_alert("No se pudieron cargar las métricas"),
                    empty_fig, _create_error_alert("Error"), empty_fig,
                    None,
                    # Issue #465/#466: Precision and Drift defaults
                    "--", "text-muted mb-1", "--", "secondary",
                    "--", "text-muted mb-1", "--", "secondary",
                )

            metrics = metrics_response.get("metrics", {})

            # Format KPIs
            entropy = f"{metrics.get('cluster_entropy', 0):.2f}"
            outlier_rate = f"{metrics.get('outlier_rate', 0):.1%}"
            confidence = f"{metrics.get('avg_ml_confidence', 0):.1%}"
            pending = str(metrics.get('validation_pending', 0))

            # Issue #458 F7: Coverage KPI
            coverage_pct = metrics.get('coverage', 0) * 100
            coverage = f"{coverage_pct:.1f}"
            coverage_classified = f"{metrics.get('classified_count', 0):,}".replace(",", ".")
            coverage_total = f"{metrics.get('total_count', 0):,}".replace(",", ".")

            last_update = f"Actualizado: {datetime.now().strftime('%H:%M:%S')}"

            # Fetch alerts
            alerts_container = _load_alerts(auth_headers)

            # Fetch distribution
            distribution_fig = _load_distribution_chart(auth_headers)

            # Fetch high-risk products
            high_risk_table = _load_high_risk_table(auth_headers)

            # Fetch trend chart
            trend_fig = _load_trend_chart(auth_headers)

            # Issue #465: Fetch Precision KPI
            precision_data = _load_precision_kpi(auth_headers)

            # Issue #466: Fetch Drift KPI
            drift_data = _load_drift_kpi(auth_headers)

            return (
                entropy, outlier_rate, confidence, pending,
                coverage, coverage_classified, coverage_total,  # Issue #458 F7
                last_update, alerts_container,
                distribution_fig, high_risk_table, trend_fig,
                metrics,
                # Issue #465/#466: Precision and Drift KPIs
                precision_data["value"], precision_data["className"],
                precision_data["status"], precision_data["color"],
                drift_data["value"], drift_data["className"],
                drift_data["status"], drift_data["color"],
            )

        except Exception as e:
            logger.error(f"[ML_MONITORING] Error loading data: {e}")
            return (
                "--", "--", "--", "--",
                "--", "--", "--",  # Coverage outputs
                f"Error: {str(e)[:50]}",
                _create_error_alert("Error de conexión"),
                empty_fig, _create_error_alert("Error"), empty_fig,
                None,
                # Issue #465/#466: Precision and Drift defaults
                "--", "text-muted mb-1", "--", "secondary",
                "--", "text-muted mb-1", "--", "secondary",
            )

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("ML monitoring callbacks registered successfully")

    return app


# =============================================================================
# HELPER FUNCTIONS
# =============================================================================

def _create_error_alert(message: str) -> dbc.Alert:
    """Create an error alert component."""
    return dbc.Alert(
        html.Div([
            html.I(className="fas fa-exclamation-circle me-2"),
            message
        ]),
        color="warning",
        className="mb-0",
    )


def _load_alerts(auth_headers: dict = None) -> html.Div:
    """Load and render active alerts."""
    try:
        response = request_coordinator.make_request(
            endpoint="/api/v1/admin/classifier/alerts",
            method="GET",
            params={"include_resolved": False},
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response.get("success"):
            return _create_error_alert("No se pudieron cargar alertas")

        alerts = response.get("data", {}).get("alerts", [])

        if not alerts:
            return dbc.Alert(
                html.Div([
                    html.I(className="fas fa-check-circle me-2 text-success"),
                    "No hay alertas activas - el clasificador está saludable"
                ]),
                color="success",
                className="mb-0",
            )

        # Create alert cards
        alert_cards = []
        for alert in alerts:
            severity = alert.get("severity", "info")
            color_map = {"critical": "danger", "warning": "warning", "info": "info"}
            color = color_map.get(severity, "secondary")

            icon_map = {"critical": "fa-skull-crossbones", "warning": "fa-exclamation-triangle", "info": "fa-info-circle"}
            icon = icon_map.get(severity, "fa-bell")

            alert_cards.append(
                dbc.Alert(
                    [
                        html.Div(
                            [
                                html.I(className=f"fas {icon} me-2"),
                                html.Strong(alert.get("title", "Alerta")),
                                dbc.Badge(
                                    severity.upper(),
                                    color=color,
                                    className="ms-2",
                                ),
                            ],
                            className="d-flex align-items-center mb-2",
                        ),
                        html.P(alert.get("message", ""), className="mb-0 small"),
                    ],
                    color=color,
                    className="mb-2",
                )
            )

        return html.Div(alert_cards)

    except Exception as e:
        logger.error(f"[ML_MONITORING] Error loading alerts: {e}")
        return _create_error_alert("Error cargando alertas")


def _load_distribution_chart(auth_headers: dict = None) -> go.Figure:
    """Load and create category distribution chart."""
    try:
        response = request_coordinator.make_request(
            endpoint="/api/v1/admin/classifier/distribution",
            method="GET",
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response.get("success"):
            return _empty_figure("Error cargando distribución")

        distribution = response.get("data", {}).get("distribution", [])

        if not distribution:
            return _empty_figure("Sin datos de distribución")

        # Prepare data
        categories = [d.get("category", "N/A")[:20] for d in distribution[:15]]  # Top 15
        counts = [d.get("count", 0) for d in distribution[:15]]
        confidences = [d.get("avg_confidence", 0) for d in distribution[:15]]

        # Create bar chart
        fig = go.Figure()
        fig.add_trace(
            go.Bar(
                x=categories,
                y=counts,
                marker_color=[_confidence_color(c) for c in confidences],
                text=counts,
                textposition="outside",
                hovertemplate="<b>%{x}</b><br>Productos: %{y}<br>Confianza: %{customdata:.1%}<extra></extra>",
                customdata=confidences,
            )
        )

        fig.update_layout(
            margin=dict(t=20, b=80, l=40, r=20),
            xaxis_tickangle=-45,
            xaxis_title="",
            yaxis_title="Productos",
            showlegend=False,
            height=300,
        )

        return fig

    except Exception as e:
        logger.error(f"[ML_MONITORING] Error loading distribution: {e}")
        return _empty_figure("Error de conexión")


def _load_high_risk_table(auth_headers: dict = None) -> html.Div:
    """Load and render high-risk products table."""
    try:
        response = request_coordinator.make_request(
            endpoint="/api/v1/admin/classifier/high-risk",
            method="GET",
            params={"limit": 10},
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response.get("success"):
            return _create_error_alert("Error cargando productos")

        products = response.get("data", {}).get("products", [])

        if not products:
            return dbc.Alert(
                html.Div([
                    html.I(className="fas fa-check-circle me-2"),
                    "No hay productos de alto riesgo"
                ]),
                color="success",
                className="mb-0",
            )

        # Create compact table
        rows = []
        for p in products[:10]:
            product_name = p.get("product_name", "N/A")[:35]
            category = p.get("ml_category", "N/A")
            z_score = p.get("z_score", 0)
            confidence = p.get("ml_confidence", 0)

            # Z-score badge color
            if z_score > 3.0:
                z_color = "danger"
            elif z_score > 2.5:
                z_color = "warning"
            else:
                z_color = "secondary"

            rows.append(
                html.Tr([
                    html.Td(product_name, className="small"),
                    html.Td(dbc.Badge(category[:15], color="info", className="small")),
                    html.Td(dbc.Badge(f"z={z_score:.1f}", color=z_color, className="small")),
                    html.Td(f"{confidence:.0%}", className="small text-muted"),
                ])
            )

        return dbc.Table(
            [
                html.Thead(
                    html.Tr([
                        html.Th("Producto", className="small"),
                        html.Th("Categoría", className="small"),
                        html.Th("Z-Score", className="small"),
                        html.Th("Conf.", className="small"),
                    ])
                ),
                html.Tbody(rows),
            ],
            bordered=True,
            hover=True,
            size="sm",
            className="mb-0",
        )

    except Exception as e:
        logger.error(f"[ML_MONITORING] Error loading high-risk products: {e}")
        return _create_error_alert("Error de conexión")


def _load_trend_chart(auth_headers: dict = None) -> go.Figure:
    """Load and create metrics trend chart."""
    try:
        response = request_coordinator.make_request(
            endpoint="/api/v1/admin/classifier/metrics/history",
            method="GET",
            params={"days": 30},
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response.get("success"):
            return _empty_figure("Error cargando tendencias")

        history = response.get("data", {}).get("history", {})

        if not history:
            return _empty_figure("Sin datos históricos")

        fig = go.Figure()

        # Add traces for each metric
        metric_config = {
            "cluster_entropy": {"name": "Entropía", "color": "#3498db"},
            "outlier_rate": {"name": "Outliers", "color": "#e74c3c"},
            "avg_ml_confidence": {"name": "Confianza", "color": "#2ecc71"},
        }

        for metric_name, config in metric_config.items():
            if metric_name in history:
                data_points = history[metric_name]
                timestamps = [d.get("timestamp") for d in data_points]
                values = [d.get("value", 0) for d in data_points]

                fig.add_trace(
                    go.Scatter(
                        x=timestamps,
                        y=values,
                        mode="lines+markers",
                        name=config["name"],
                        line=dict(color=config["color"], width=2),
                        marker=dict(size=4),
                    )
                )

        fig.update_layout(
            margin=dict(t=20, b=40, l=40, r=20),
            xaxis_title="",
            yaxis_title="Valor",
            legend=dict(orientation="h", yanchor="bottom", y=1.02, xanchor="right", x=1),
            height=250,
        )

        return fig

    except Exception as e:
        logger.error(f"[ML_MONITORING] Error loading trends: {e}")
        return _empty_figure("Error de conexión")


def _empty_figure(message: str) -> go.Figure:
    """Create an empty figure with message."""
    fig = go.Figure()
    fig.update_layout(
        annotations=[{"text": message, "showarrow": False, "font": {"size": 14}}],
        margin=dict(t=20, b=20, l=20, r=20),
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
    )
    return fig


def _confidence_color(confidence: float) -> str:
    """Return color based on confidence level."""
    if confidence >= 0.85:
        return "#2ecc71"  # Green
    elif confidence >= 0.70:
        return "#f39c12"  # Orange
    else:
        return "#e74c3c"  # Red


def _load_precision_kpi(auth_headers: dict = None) -> dict:
    """
    Load Precision KPI from API (Issue #465).
    Precision = (APPROVE count) / (APPROVE + CORRECT count)
    """
    default = {
        "value": "--",
        "className": "text-muted mb-1",
        "status": "N/A",
        "color": "secondary",
    }

    try:
        response = request_coordinator.make_request(
            endpoint="/api/v1/admin/classifier/precision",
            method="GET",
            params={"days": 30},
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response.get("success"):
            return default

        data = response.get("data", {})
        precision = data.get("precision", 0)
        status = data.get("status", "no_data")
        evaluated_count = data.get("evaluated_count", 0)

        if evaluated_count == 0:
            return {
                "value": "N/A",
                "className": "text-muted mb-1",
                "status": "Sin datos",
                "color": "secondary",
            }

        # Format precision as percentage
        value = f"{precision:.1%}"

        # Color based on threshold
        if precision >= 0.85:
            className = "text-success mb-1"
            color = "success"
            status_text = "Saludable"
        elif precision >= 0.70:
            className = "text-warning mb-1"
            color = "warning"
            status_text = "Atención"
        else:
            className = "text-danger mb-1"
            color = "danger"
            status_text = "Crítico"

        return {
            "value": value,
            "className": className,
            "status": status_text,
            "color": color,
        }

    except Exception as e:
        logger.error(f"[ML_MONITORING] Error loading precision KPI: {e}")
        return default


def _load_drift_kpi(auth_headers: dict = None) -> dict:
    """
    Load Distribution Drift KPI from API (Issue #466).
    Compares category distribution week-over-week.
    """
    default = {
        "value": "--",
        "className": "text-muted mb-1",
        "status": "N/A",
        "color": "secondary",
    }

    try:
        response = request_coordinator.make_request(
            endpoint="/api/v1/admin/classifier/drift",
            method="GET",
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response.get("success"):
            return default

        data = response.get("data", {})
        max_drift = data.get("max_drift", 0)
        has_drift = data.get("has_drift", False)
        drift_count = data.get("drift_count", 0)

        # Format drift as percentage
        value = f"{max_drift:.1%}"

        # Color based on drift level
        if max_drift > 0.20:
            className = "text-danger mb-1"
            color = "danger"
            status_text = f"Crítico ({drift_count})"
        elif has_drift:
            className = "text-warning mb-1"
            color = "warning"
            status_text = f"Drift ({drift_count})"
        else:
            className = "text-success mb-1"
            color = "success"
            status_text = "Estable"

        return {
            "value": value,
            "className": className,
            "status": status_text,
            "color": color,
        }

    except Exception as e:
        logger.error(f"[ML_MONITORING] Error loading drift KPI: {e}")
        return default
