"""
Callbacks para exportación de productos en manual_review.

Issue #447: Exportar productos sin enriquecer a CSV para análisis offline.

Clasificación de códigos mejorada:
- CN_CATALOGED: CN válido, existe en CIMA/Nomenclator
- CN_OTC: CN válido checksum, NO en catálogos (parafarmacia)
- EAN_CATALOGED: EAN válido, existe en catálogos
- EAN_OTC: EAN válido checksum, NO en catálogos
- CN_UNVERIFIED: 6 dígitos sin validar (puede ser CN legacy o código interno)
- INTERNAL: Código interno farmacia
"""

import logging
from datetime import datetime, timezone

import dash_bootstrap_components as dbc
import requests
from dash import Input, Output, State, callback, ctx, html, no_update
from dash.exceptions import PreventUpdate

from utils.auth import auth_manager
from utils.config import BACKEND_URL
from utils.helpers import format_currency, format_number

logger = logging.getLogger(__name__)

# Guard to prevent duplicate registration
_callbacks_registered = False


def register_manual_review_callbacks(app):
    """
    Register manual review export callbacks.

    Issue #447: Admin panel for exporting products pending manual review.
    """
    global _callbacks_registered
    if _callbacks_registered:
        logger.debug("Manual review callbacks already registered, skipping")
        return
    _callbacks_registered = True

    # Issue #448: Trigger store intermedio para separar stats de tabla (REGLA #11)
    @app.callback(
        [
            Output("manual-review-products-count", "children"),
            Output("manual-review-sales-count", "children"),
            Output("manual-review-total-amount", "children"),
            Output("manual-review-cn-count", "children"),
            Output("manual-review-ean-count", "children"),
            Output("manual-review-internal-count", "children"),
            Output("manual-review-stats-loaded-trigger", "data"),  # Trigger para tabla
        ],
        [Input("manual-review-tab-activated-trigger", "data")],  # Issue #448: Trigger cuando se activa manual-review tab
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Token restoration
        ],
        prevent_initial_call=True,
    )
    def load_manual_review_stats(tools_trigger, auth_state, auth_tokens):
        """
        Load manual review statistics when Manual Review tab is activated.

        Issue #448: Stats loaded first, then triggers table load via intermediate store.
        """
        # Validar que el trigger es para el manual-review tab
        if not tools_trigger or tools_trigger.get("tab") != "manual-review":
            raise PreventUpdate

        # Auth check (REGLA #7)
        from utils.auth_helpers import is_user_authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[manual_review] User not authenticated - skipping API call")
            raise PreventUpdate

        # Default values
        default_stats = ("--", "--", "--", "--", "--", "--")
        no_trigger = no_update

        try:
            # REGLA #7.6: Restore tokens for multi-worker (Render)
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

            access_token = auth_manager.get_access_token()
            if not access_token:
                logger.warning("[manual_review] No access token available")
                return (*default_stats, no_trigger)

            # --- STATS REQUEST ---
            stats_response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/manual-review/stats",
                headers={"Authorization": f"Bearer {access_token}"},
                timeout=30,
            )

            if stats_response.status_code != 200:
                logger.error(
                    "[manual_review] Failed to fetch stats",
                    extra={"status_code": stats_response.status_code},
                )
                return (*default_stats, no_trigger)

            data = stats_response.json()
            products_count = format_number(data.get("total_products", 0))
            sales_count = format_number(data.get("total_sales", 0))
            total_amount = format_currency(float(data.get("total_amount", 0)))
            by_code_type = data.get("by_code_type", {})
            cn_count = format_number(by_code_type.get("CN", 0))
            ean_count = format_number(by_code_type.get("EAN", 0))
            internal_count = format_number(by_code_type.get("INTERNAL", 0))

            logger.info(
                "[manual_review] Stats loaded successfully",
                extra={"total_products": data.get("total_products")},
            )

            # Trigger table load
            table_trigger = {"loaded": True}

            return (products_count, sales_count, total_amount, cn_count, ean_count, internal_count, table_trigger)

        except requests.RequestException as e:
            logger.error("[manual_review] Request failed", extra={"error": str(e)})
            return (*default_stats, no_trigger)
        except Exception as e:
            logger.error("[manual_review] Unexpected error", extra={"error": str(e)})
            return (*default_stats, no_trigger)

    @app.callback(
        Output("manual-review-download", "data"),
        [Input("manual-review-export-btn", "n_clicks")],
        [
            State("manual-review-min-sales", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Token restoration
        ],
        prevent_initial_call=True,
    )
    def export_manual_review_csv(n_clicks, min_sales, auth_state, auth_tokens):
        """
        Export manual review products to CSV when download button is clicked.
        """
        if not n_clicks:
            raise PreventUpdate

        # Auth check (REGLA #7)
        from utils.auth_helpers import is_user_authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[manual_review] Export attempted without auth")
            raise PreventUpdate

        try:
            # REGLA #7.6: Restore tokens for multi-worker (Render)
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

            access_token = auth_manager.get_access_token()
            if not access_token:
                logger.warning("[manual_review] No access token for export")
                return no_update

            # Build params
            params = {"format": "csv"}
            if min_sales and min_sales > 0:
                params["min_sales"] = min_sales

            response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/manual-review/export",
                params=params,
                headers={"Authorization": f"Bearer {access_token}"},
                timeout=60,  # Longer timeout for export
            )

            if response.status_code != 200:
                logger.error(
                    "[manual_review] Export failed",
                    extra={"status_code": response.status_code},
                )
                return no_update

            # Generate filename with timestamp (UTC per CLAUDE.md)
            timestamp = datetime.now(timezone.utc).strftime("%Y%m%d_%H%M%S")
            filename = f"manual_review_products_{timestamp}.csv"

            logger.info(
                f"[manual_review] CSV export successful: {filename} ({len(response.content)} bytes)"
            )

            return {
                "content": response.text,
                "filename": filename,
                "type": "text/csv",
            }

        except requests.RequestException as e:
            logger.error(f"[manual_review] Export request failed: {e}")
            return no_update
        except Exception as e:
            import traceback
            logger.error(f"[manual_review] Unexpected export error: {e}\n{traceback.format_exc()}")
            return no_update

    @app.callback(
        [
            Output("manual-review-table-container", "children"),
            Output("manual-review-pagination-info", "children"),
            Output("manual-review-current-page", "children"),
            Output("manual-review-prev-page", "disabled"),
            Output("manual-review-next-page", "disabled"),
            Output("manual-review-current-page-store", "data"),  # Update page store
        ],
        [
            Input("manual-review-stats-loaded-trigger", "data"),  # Initial load after stats
            Input("manual-review-code-type-filter", "value"),
            Input("manual-review-search", "value"),
            Input("manual-review-order-by", "value"),
            Input("manual-review-page-size", "value"),
            Input("manual-review-prev-page", "n_clicks"),
            Input("manual-review-next-page", "n_clicks"),
        ],
        [
            State("manual-review-current-page-store", "data"),  # Current page state
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_manual_review_table(
        stats_loaded_trigger,
        code_type,
        search,
        order_by,
        page_size,
        prev_clicks,
        next_clicks,
        current_page_store,
        auth_state,
        auth_tokens,
    ):
        """
        Load manual review products table with filters and pagination.

        Issue #448: Interactive table with filtering, search, sorting, and pagination.
        Uses manual-review-current-page-store to maintain pagination state.
        Triggered by stats load or filter changes (REGLA #11 compliant).
        """
        # Auth check (REGLA #7)
        from utils.auth_helpers import is_user_authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[manual_review] User not authenticated - skipping table load")
            raise PreventUpdate

        # Determine current page based on trigger
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # Determine current page based on trigger and store
        current_page = current_page_store if current_page_store else 1

        if trigger_id == "manual-review-stats-loaded-trigger":
            if not stats_loaded_trigger:
                raise PreventUpdate
            current_page = 1  # Reset to page 1 on initial load
        elif trigger_id in ["manual-review-code-type-filter", "manual-review-search", "manual-review-order-by", "manual-review-page-size"]:
            current_page = 1  # Reset to page 1 on filter/sort/page_size change
        elif trigger_id == "manual-review-prev-page":
            current_page = max(1, current_page - 1)  # Decrement page, min 1
        elif trigger_id == "manual-review-next-page":
            current_page = current_page + 1  # Increment page (will be capped by API)

        # Default outputs
        default_table = html.P("Cargando productos...", className="text-muted")
        default_info = "Mostrando 0-0 de 0 productos"
        default_page_display = "Página 1 de 1"
        disabled_prev = True
        disabled_next = True

        try:
            # REGLA #7.6: Restore tokens for multi-worker (Render)
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

            access_token = auth_manager.get_access_token()
            if not access_token:
                logger.warning("[manual_review] No access token for table")
                return (
                    html.P("Sin token de acceso", className="text-warning"),
                    default_info,
                    default_page_display,
                    disabled_prev,
                    disabled_next,
                    current_page,  # Return current page to store
                )

            # Build API params
            params = {
                "page": current_page,
                "page_size": int(page_size) if page_size else 50,
            }

            if code_type and code_type != "all":
                params["code_type"] = code_type

            if search:
                params["search"] = search

            if order_by:
                params["order_by"] = order_by

            # --- TABLE REQUEST ---
            table_response = requests.get(
                f"{BACKEND_URL}/api/v1/admin/manual-review/products",
                params=params,
                headers={"Authorization": f"Bearer {access_token}"},
                timeout=30,
            )

            if table_response.status_code != 200:
                logger.error(f"[manual_review] Products API returned {table_response.status_code}")
                return (
                    html.P(f"Error cargando productos: {table_response.status_code}", className="text-danger"),
                    default_info,
                    default_page_display,
                    disabled_prev,
                    disabled_next,
                    current_page,
                )

            data = table_response.json()
            products = data.get("products", [])
            total_products = data.get("total", 0)
            page_num = data.get("page", 1)
            page_size_val = data.get("page_size", 50)
            total_pages = data.get("total_pages", 1)

            if not products:
                table = html.P("No hay productos que coincidan con los filtros", className="text-muted")
            else:
                # Determinar si hay múltiples farmacias
                unique_pharmacies = set(p.get("pharmacy_id") for p in products if p.get("pharmacy_id"))
                show_pharmacy_column = len(unique_pharmacies) > 1

                # Crear tabla header
                header_cells = []
                if show_pharmacy_column:
                    header_cells.append(html.Th("Farmacia", style={"width": "120px"}))
                header_cells.extend([
                    html.Th("CN", style={"width": "80px"}),
                    html.Th("EAN-13", style={"width": "120px"}),
                    html.Th("Nombre"),
                    html.Th("Tipo", style={"width": "100px"}),
                    html.Th("Ventas", className="text-end", style={"width": "80px"}),
                    html.Th("Importe", className="text-end", style={"width": "100px"}),
                ])
                table_header = html.Thead(html.Tr(header_cells))

                table_rows = []
                for p in products:
                    code_badge_color = {
                        "CN_CATALOGED": "primary",
                        "CN_OTC": "info",
                        "CN_UNVERIFIED": "secondary",
                        "EAN_CATALOGED": "success",
                        "EAN_OTC": "warning",
                        "INTERNAL": "dark",
                    }.get(p.get("code_type", ""), "secondary")

                    # Construir celdas de la fila
                    row_cells = []
                    if show_pharmacy_column:
                        pharmacy_name = p.get("pharmacy_name", "")
                        display_pharmacy = pharmacy_name[:15] + "..." if len(pharmacy_name) > 15 else pharmacy_name
                        row_cells.append(html.Td(display_pharmacy, className="small"))

                    row_cells.extend([
                        html.Td(
                            html.Code(p.get("codigo_nacional") or "-", className="small"),
                            className="text-muted" if not p.get("codigo_nacional") else ""
                        ),
                        html.Td(
                            html.Code(p.get("ean13") or "-", className="small"),
                            className="text-muted" if not p.get("ean13") else ""
                        ),
                        html.Td(p.get("product_name", "")[:50] + ("..." if len(p.get("product_name", "")) > 50 else "")),
                        html.Td(dbc.Badge(p.get("code_type", ""), color=code_badge_color, className="small")),
                        html.Td(format_number(p.get("sale_count", 0)), className="text-end"),
                        html.Td(format_currency(float(p.get("total_amount", 0))), className="text-end"),
                    ])

                    table_rows.append(html.Tr(row_cells))

                table_body = html.Tbody(table_rows)
                table = dbc.Table(
                    [table_header, table_body],
                    striped=True,
                    bordered=True,
                    hover=True,
                    responsive=True,
                    size="sm",
                    className="mb-0",
                )

            # Pagination info
            start_idx = (page_num - 1) * page_size_val + 1
            end_idx = min(page_num * page_size_val, total_products)
            pagination_info = f"Mostrando {start_idx}-{end_idx} de {format_number(total_products)} productos"
            page_display = f"Página {page_num} de {total_pages}"

            # Pagination buttons state
            disabled_prev = page_num <= 1
            disabled_next = page_num >= total_pages

            logger.info(
                "[manual_review] Table loaded successfully",
                extra={"page": page_num, "total": total_products},
            )

            return table, pagination_info, page_display, disabled_prev, disabled_next, page_num

        except requests.RequestException as e:
            logger.error("[manual_review] Table request failed", extra={"error": str(e)})
            return (
                html.P(f"Error de conexión: {str(e)}", className="text-danger"),
                default_info,
                default_page_display,
                disabled_prev,
                disabled_next,
                current_page,
            )
        except Exception as e:
            logger.error("[manual_review] Unexpected table error", extra={"error": str(e)})
            return (
                html.P(f"Error inesperado: {str(e)}", className="text-danger"),
                default_info,
                default_page_display,
                disabled_prev,
                disabled_next,
                current_page,
            )

    logger.info("[manual_review] Callbacks registered successfully")
