"""
Keywords Management Callbacks Module (Issue #449).

Responsabilidad: UI para gestión dinámica de keywords del clasificador NECESIDAD.
Permite crear, editar, eliminar y previsualizar impacto de keywords.
"""

import logging
from datetime import datetime

import dash_bootstrap_components as dbc
from dash import Input, Output, State, ctx, html, ALL, no_update
from dash.exceptions import PreventUpdate

from utils.auth import auth_manager
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Module-level flags to prevent duplicate callback registration
_module_callbacks_registered = False
_apply_callbacks_registered = False


def register_keywords_callbacks(app):
    """
    Register keywords management callbacks for admin panel.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Keywords callbacks already registered, skipping")
        return app

    logger.info("Registering keywords callbacks")

    # =========================================================================
    # CALLBACK 1: Load Keywords Data on Tab Activation or Refresh
    # =========================================================================
    @app.callback(
        [
            Output("keywords-stats-container", "children"),
            Output("keywords-table-container", "children"),
            Output("keywords-pagination-info", "children"),
            Output("keywords-prev-page", "disabled"),
            Output("keywords-next-page", "disabled"),
            Output("keywords-data-store", "data"),
            Output("keywords-categories-store", "data"),
            Output("keywords-form-category", "options"),
        ],
        [
            Input("keywords-tab-activated-trigger", "data"),
            Input("keywords-refresh-btn", "n_clicks"),
            Input("keywords-type-filter", "value"),
            Input("keywords-category-filter", "value"),
            Input("keywords-search-input", "value"),
            Input("keywords-show-inactive", "value"),
            Input("keywords-page-store", "data"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_keywords_data(
        trigger_data, n_clicks, type_filter, category_filter, search_text,
        show_inactive, current_page, auth_state, auth_tokens
    ):
        """
        Load keywords data when:
        - Tab is activated (trigger_data)
        - Refresh button clicked (n_clicks)
        - Filters changed
        - Page changed

        REGLA #7.6: Multi-worker token restoration before API calls.
        """
        from utils.auth_helpers import is_user_authenticated

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[KEYWORDS] User not authenticated - skipping")
            raise PreventUpdate

        # REGLA #7.6: Restore tokens for multi-worker
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        # Default values
        page = current_page or 1
        limit = 20
        offset = (page - 1) * limit

        # Build query params
        params = {"limit": limit, "offset": offset}
        if type_filter and type_filter != "all":
            params["keyword_type"] = type_filter
        if category_filter and category_filter != "all":
            params["category"] = category_filter
        if search_text:
            params["search"] = search_text
        if show_inactive and "show_inactive" in show_inactive:
            params["is_active"] = None  # Show all
        else:
            params["is_active"] = True  # Only active

        # Load keywords
        keywords_data = {"items": [], "total": 0, "stats": {}}
        try:
            response = request_coordinator.make_request(
                "/api/v1/admin/keywords",
                method="GET",
                params=params,
            )
            if response and isinstance(response, dict):
                keywords_data = response
        except Exception as e:
            logger.error(f"[KEYWORDS] Error loading keywords: {e}")

        # Load categories (for dropdown)
        categories = []
        try:
            cats_response = request_coordinator.make_request(
                "/api/v1/admin/keywords/categories",
                method="GET",
            )
            if cats_response and isinstance(cats_response, list):
                categories = cats_response
        except Exception as e:
            logger.error(f"[KEYWORDS] Error loading categories: {e}")

        # Build stats badges
        stats = keywords_data.get("stats", {})
        by_type = stats.get("by_type", {})
        stats_content = dbc.Row([
            dbc.Col([
                dbc.Badge(
                    f"Keywords: {by_type.get('keyword', 0)}",
                    color="primary",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Marcas: {by_type.get('brand', 0)}",
                    color="info",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Blacklist: {by_type.get('blacklist', 0)}",
                    color="danger",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Total: {keywords_data.get('total', 0)}",
                    color="secondary",
                ),
            ])
        ])

        # Build table
        items = keywords_data.get("items", [])
        total_items = keywords_data.get("total", 0)
        total_pages = (total_items + limit - 1) // limit if total_items > 0 else 1

        if items:
            table_content = _build_keywords_table(items)
        else:
            table_content = dbc.Alert(
                html.Span([
                    html.I(className="fas fa-info-circle me-2 text-info"),
                    "No hay keywords registrados. Usa 'Añadir Keyword' para crear uno.",
                ]),
                color="info",
                className="text-center py-4",
            )

        # Pagination info
        pagination_info = f"Mostrando {len(items)} de {total_items}"
        prev_disabled = page <= 1
        next_disabled = page >= total_pages

        # Category options for dropdown
        category_options = [{"label": "Todas", "value": "all"}]
        category_options.extend([
            {"label": cat.get("label", cat.get("value", "")), "value": cat.get("value", "")}
            for cat in categories
        ])

        return (
            stats_content,
            table_content,
            pagination_info,
            prev_disabled,
            next_disabled,
            keywords_data,
            categories,
            category_options,
        )

    # =========================================================================
    # CALLBACK 2: Pagination
    # =========================================================================
    @app.callback(
        Output("keywords-page-store", "data"),
        [
            Input("keywords-prev-page", "n_clicks"),
            Input("keywords-next-page", "n_clicks"),
        ],
        [
            State("keywords-page-store", "data"),
            State("keywords-data-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_pagination(prev_clicks, next_clicks, current_page, keywords_data):
        """Handle pagination button clicks."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
        page = current_page or 1

        total = keywords_data.get("total", 0) if keywords_data else 0
        limit = 20
        total_pages = (total + limit - 1) // limit if total > 0 else 1

        if trigger_id == "keywords-prev-page" and page > 1:
            return page - 1
        elif trigger_id == "keywords-next-page" and page < total_pages:
            return page + 1

        raise PreventUpdate

    # =========================================================================
    # CALLBACK 3: Open Add Modal
    # =========================================================================
    @app.callback(
        [
            Output("keywords-modal", "is_open", allow_duplicate=True),
            Output("keywords-modal-title", "children", allow_duplicate=True),
            Output("keywords-form-keyword", "value", allow_duplicate=True),
            Output("keywords-form-type", "value", allow_duplicate=True),
            Output("keywords-form-category", "value", allow_duplicate=True),
            Output("keywords-form-priority", "value", allow_duplicate=True),
            Output("keywords-form-notes", "value", allow_duplicate=True),
            Output("keywords-edit-id-store", "data", allow_duplicate=True),
            Output("keywords-preview-container", "children", allow_duplicate=True),
        ],
        [Input("keywords-add-btn", "n_clicks")],
        prevent_initial_call=True,
    )
    def open_add_modal(n_clicks):
        """Open modal for adding new keyword."""
        if not n_clicks:
            raise PreventUpdate

        return (
            True,  # is_open
            "Añadir Keyword",  # title
            "",  # keyword
            "keyword",  # type
            "",  # category (empty)
            100,  # priority default
            "",  # notes
            None,  # edit_id (None for new)
            "",  # preview container (empty)
        )

    # =========================================================================
    # CALLBACK 4: Open Edit Modal (via pattern matching)
    # =========================================================================
    @app.callback(
        [
            Output("keywords-modal", "is_open", allow_duplicate=True),
            Output("keywords-modal-title", "children", allow_duplicate=True),
            Output("keywords-form-keyword", "value", allow_duplicate=True),
            Output("keywords-form-type", "value", allow_duplicate=True),
            Output("keywords-form-category", "value", allow_duplicate=True),
            Output("keywords-form-priority", "value", allow_duplicate=True),
            Output("keywords-form-notes", "value", allow_duplicate=True),
            Output("keywords-edit-id-store", "data", allow_duplicate=True),
            Output("keywords-preview-container", "children", allow_duplicate=True),
        ],
        [Input({"type": "keywords-edit-btn", "index": ALL}, "n_clicks")],
        [State("keywords-data-store", "data")],
        prevent_initial_call=True,
    )
    def open_edit_modal(edit_clicks, keywords_data):
        """Open modal for editing existing keyword."""
        import json

        if not ctx.triggered or not any(edit_clicks):
            raise PreventUpdate

        # Find which button was clicked
        trigger = ctx.triggered[0]
        if trigger["value"] is None:
            raise PreventUpdate

        try:
            id_part = trigger["prop_id"].rsplit(".", 1)[0]
            id_dict = json.loads(id_part)
            keyword_id = id_dict.get("index")
        except (json.JSONDecodeError, KeyError):
            raise PreventUpdate

        if not keyword_id:
            raise PreventUpdate

        # Find keyword in data store
        items = keywords_data.get("items", []) if keywords_data else []
        keyword_item = next((item for item in items if item.get("id") == keyword_id), None)

        if not keyword_item:
            raise PreventUpdate

        return (
            True,  # is_open
            f"Editar Keyword: {keyword_item.get('keyword', '')}",  # title
            keyword_item.get("keyword", ""),
            keyword_item.get("keyword_type", "keyword"),
            keyword_item.get("category", ""),
            keyword_item.get("priority", 100),
            keyword_item.get("notes", "") or "",
            keyword_id,  # edit_id
            "",  # preview container (empty)
        )

    # =========================================================================
    # CALLBACK 5: Close Modal
    # =========================================================================
    @app.callback(
        Output("keywords-modal", "is_open", allow_duplicate=True),
        [Input("keywords-modal-cancel-btn", "n_clicks")],
        prevent_initial_call=True,
    )
    def close_modal(n_clicks):
        """Close the keyword modal."""
        if not n_clicks:
            raise PreventUpdate
        return False

    # =========================================================================
    # CALLBACK 6: Preview Keyword Impact
    # =========================================================================
    @app.callback(
        Output("keywords-preview-container", "children"),
        [Input("keywords-preview-btn", "n_clicks")],
        [
            State("keywords-form-keyword", "value"),
            State("keywords-form-category", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def preview_keyword_impact(n_clicks, keyword, category, auth_state, auth_tokens):
        """Preview impact of keyword before saving."""
        from utils.auth_helpers import is_user_authenticated

        if not n_clicks or not keyword or not category:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Restore tokens
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        try:
            response = request_coordinator.make_request(
                "/api/v1/admin/keywords/preview",
                method="POST",
                data={
                    "keyword": keyword,
                    "category": category,
                    "limit": 5,
                },
            )

            if response and isinstance(response, dict):
                affected_count = response.get("affected_count", 0)
                sample_products = response.get("sample_products", [])

                if affected_count == 0:
                    return dbc.Alert(
                        "No se encontraron productos que se reclasificarían con este keyword.",
                        color="info",
                        className="mt-3 mb-0",
                    )

                # Build preview table
                rows = []
                for p in sample_products:
                    rows.append(html.Tr([
                        html.Td(p.get("product_name", "")[:40] + "..."),
                        html.Td(dbc.Badge(p.get("current_category", "--"), color="secondary")),
                        html.Td(html.I(className="fas fa-arrow-right text-muted")),
                        html.Td(dbc.Badge(p.get("new_category", "--"), color="success")),
                    ]))

                return html.Div([
                    dbc.Alert([
                        html.I(className="fas fa-info-circle me-2"),
                        f"Este keyword afectaría a {affected_count} productos.",
                    ], color="warning", className="mt-3 mb-2"),
                    dbc.Table([
                        html.Thead(html.Tr([
                            html.Th("Producto"),
                            html.Th("Actual"),
                            html.Th(""),
                            html.Th("Nueva"),
                        ])),
                        html.Tbody(rows),
                    ], size="sm", bordered=True, className="mb-0"),
                ])

        except Exception as e:
            logger.error(f"[KEYWORDS] Error previewing: {e}")
            return dbc.Alert(f"Error al previsualizar: {str(e)}", color="danger", className="mt-3 mb-0")

        raise PreventUpdate

    # =========================================================================
    # CALLBACK 7: Save Keyword (Create or Update)
    # =========================================================================
    @app.callback(
        [
            Output("keywords-modal", "is_open", allow_duplicate=True),
            Output("keywords-refresh-btn", "n_clicks", allow_duplicate=True),
        ],
        [Input("keywords-modal-save-btn", "n_clicks")],
        [
            State("keywords-form-keyword", "value"),
            State("keywords-form-type", "value"),
            State("keywords-form-category", "value"),
            State("keywords-form-priority", "value"),
            State("keywords-form-notes", "value"),
            State("keywords-edit-id-store", "data"),
            State("keywords-refresh-btn", "n_clicks"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def save_keyword(
        save_clicks, keyword, keyword_type, category, priority, notes,
        edit_id, current_refresh, auth_state, auth_tokens
    ):
        """Save keyword (create new or update existing)."""
        from utils.auth_helpers import is_user_authenticated

        if not save_clicks:
            raise PreventUpdate

        if not keyword or not category:
            logger.warning("[KEYWORDS] Missing required fields")
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Restore tokens
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        try:
            data = {
                "keyword": keyword.lower().strip(),
                "keyword_type": keyword_type or "keyword",
                "category": category,
                "priority": int(priority) if priority else 100,
                "notes": notes or None,
            }

            if edit_id:
                # Update existing
                response = request_coordinator.make_request(
                    f"/api/v1/admin/keywords/{edit_id}",
                    method="PUT",
                    data=data,
                )
                logger.info(f"[KEYWORDS] Updated keyword {edit_id}")
            else:
                # Create new
                data["is_active"] = True
                response = request_coordinator.make_request(
                    "/api/v1/admin/keywords",
                    method="POST",
                    data=data,
                )
                logger.info(f"[KEYWORDS] Created keyword: {keyword}")

            # Close modal and trigger refresh
            return False, (current_refresh or 0) + 1

        except Exception as e:
            logger.error(f"[KEYWORDS] Error saving: {e}")
            # Don't close modal on error
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 8: Delete Keyword (via pattern matching)
    # =========================================================================
    @app.callback(
        Output("keywords-refresh-btn", "n_clicks", allow_duplicate=True),
        [Input({"type": "keywords-delete-btn", "index": ALL}, "n_clicks")],
        [
            State("keywords-refresh-btn", "n_clicks"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def delete_keyword(delete_clicks, current_refresh, auth_state, auth_tokens):
        """Delete (deactivate) a keyword."""
        import json
        from utils.auth_helpers import is_user_authenticated

        if not ctx.triggered or not any(delete_clicks):
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Find which button was clicked
        trigger = ctx.triggered[0]
        if trigger["value"] is None:
            raise PreventUpdate

        try:
            id_part = trigger["prop_id"].rsplit(".", 1)[0]
            id_dict = json.loads(id_part)
            keyword_id = id_dict.get("index")
        except (json.JSONDecodeError, KeyError):
            raise PreventUpdate

        if not keyword_id:
            raise PreventUpdate

        # REGLA #7.6: Restore tokens
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        try:
            response = request_coordinator.make_request(
                f"/api/v1/admin/keywords/{keyword_id}",
                method="DELETE",
                params={"hard_delete": False},  # Soft delete
            )
            logger.info(f"[KEYWORDS] Deleted keyword {keyword_id}")

            # Trigger refresh
            return (current_refresh or 0) + 1

        except Exception as e:
            logger.error(f"[KEYWORDS] Error deleting: {e}")
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 9: Close Preview Modal
    # =========================================================================
    @app.callback(
        Output("keywords-preview-modal", "is_open"),
        [Input("keywords-preview-modal-close", "n_clicks")],
        prevent_initial_call=True,
    )
    def close_preview_modal(n_clicks):
        """Close the preview modal."""
        if not n_clicks:
            raise PreventUpdate
        return False

    _module_callbacks_registered = True
    logger.info("Keywords callbacks registered successfully")
    return app


def _build_keywords_table(items: list) -> dbc.Table:
    """
    Build the keywords table with action buttons.

    Args:
        items: List of keywords from API

    Returns:
        dbc.Table containing keywords
    """
    rows = []
    for item in items:
        keyword_id = item.get("id", 0)
        keyword_text = item.get("keyword", "")
        keyword_type = item.get("keyword_type", "keyword")
        category = item.get("category", "--")
        priority = item.get("priority", 100)
        is_active = item.get("is_active", True)
        source = item.get("source", "manual")
        creator_email = item.get("creator_email", "")

        # Type badge color
        type_color = {
            "keyword": "primary",
            "brand": "info",
            "blacklist": "danger",
        }.get(keyword_type, "secondary")

        # Active/inactive badge
        status_badge = dbc.Badge(
            "Activo" if is_active else "Inactivo",
            color="success" if is_active else "secondary",
            className="ms-1",
        )

        row = html.Tr([
            # Keyword
            html.Td([
                html.Strong(keyword_text),
                html.Br() if source != "manual" else None,
                dbc.Badge(source, color="light", text_color="muted", className="small") if source != "manual" else None,
            ]),
            # Type
            html.Td(dbc.Badge(keyword_type.capitalize(), color=type_color)),
            # Category
            html.Td(
                dbc.Badge(category.replace("_", " ").title(), color="warning"),
                style={"maxWidth": "150px"},
            ),
            # Priority
            html.Td(str(priority), className="text-center"),
            # Status
            html.Td(status_badge, className="text-center"),
            # Actions
            html.Td(
                dbc.ButtonGroup([
                    dbc.Button(
                        html.I(className="fas fa-edit"),
                        id={"type": "keywords-edit-btn", "index": keyword_id},
                        color="primary",
                        size="sm",
                        outline=True,
                        title="Editar",
                    ),
                    dbc.Button(
                        html.I(className="fas fa-trash"),
                        id={"type": "keywords-delete-btn", "index": keyword_id},
                        color="danger",
                        size="sm",
                        outline=True,
                        title="Eliminar",
                    ),
                ], size="sm"),
                className="text-center",
            ),
        ], className="" if is_active else "table-secondary")
        rows.append(row)

    return dbc.Table(
        [
            html.Thead(
                html.Tr([
                    html.Th("Keyword", style={"width": "25%"}),
                    html.Th("Tipo", style={"width": "12%"}),
                    html.Th("Categoría", style={"width": "20%"}),
                    html.Th("Prioridad", style={"width": "10%"}, className="text-center"),
                    html.Th("Estado", style={"width": "10%"}, className="text-center"),
                    html.Th("Acciones", style={"width": "15%"}, className="text-center"),
                ])
            ),
            html.Tbody(rows),
        ],
        bordered=True,
        hover=True,
        responsive=True,
        size="sm",
        className="mb-0",
    )


# =============================================================================
# Apply Keywords Modal Callbacks (Issue #449 Phase 3)
# =============================================================================


def register_apply_keywords_callbacks(app):
    """
    Register callbacks for applying keywords to products.

    Separated from main keywords callbacks for clarity.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _apply_callbacks_registered

    # Guard against duplicate registration
    if _apply_callbacks_registered:
        logger.warning("Apply keywords callbacks already registered, skipping")
        return app

    logger.info("Registering apply keywords callbacks")

    @app.callback(
        Output("keywords-apply-modal", "is_open"),
        [
            Input("keywords-apply-btn", "n_clicks"),
            Input("keywords-apply-close-btn", "n_clicks"),
        ],
        State("keywords-apply-modal", "is_open"),
        prevent_initial_call=True,
    )
    def toggle_apply_modal(open_clicks, close_clicks, is_open):
        """Open/close the apply keywords modal."""
        if not ctx.triggered:
            raise PreventUpdate
        return not is_open

    @app.callback(
        [
            Output("keywords-apply-preview-container", "children"),
            Output("keywords-apply-results-store", "data"),
            Output("keywords-apply-confirm-btn", "disabled"),
        ],
        Input("keywords-apply-preview-btn", "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def preview_apply_keywords(n_clicks, auth_state, auth_tokens):
        """Preview which products would be reclassified."""
        from utils.auth_helpers import is_user_authenticated

        if not n_clicks:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            return (
                dbc.Alert("No autenticado", color="danger"),
                None,
                True,
            )

        # REGLA #7.6: Restore tokens for multi-worker
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        try:
            response = request_coordinator.make_request(
                endpoint="/api/v1/admin/keywords/apply",
                method="POST",
                data={"dry_run": True},
            )

            if not response:
                return (
                    dbc.Alert("Error al conectar con el servidor", color="danger"),
                    None,
                    True,
                )
            if isinstance(response, dict) and response.get("error"):
                return (
                    dbc.Alert(f"Error: {response.get('message', response.get('error'))}", color="danger"),
                    None,
                    True,
                )

            data = response
            products_count = data.get("products_reclassified", 0)
            total_scanned = data.get("total_products_scanned", 0)
            sample = data.get("sample_changes", [])

            if products_count == 0:
                content = dbc.Alert(
                    [
                        html.I(className="fas fa-check-circle me-2"),
                        f"No hay cambios pendientes. {total_scanned} productos escaneados.",
                    ],
                    color="success",
                )
                return content, None, True

            # Build preview table
            rows = []
            for item in sample[:20]:
                rows.append(
                    html.Tr([
                        html.Td(item.get("product_name", "")[:50]),
                        html.Td(item.get("old_category", "-") or "-"),
                        html.Td(html.I(className="fas fa-arrow-right text-muted")),
                        html.Td(
                            dbc.Badge(
                                item.get("new_category", ""),
                                color="success",
                            )
                        ),
                        html.Td(
                            dbc.Badge(
                                item.get("matched_keyword", ""),
                                color="info",
                                className="small",
                            )
                        ),
                    ])
                )

            content = html.Div([
                dbc.Alert(
                    [
                        html.Strong(f"{products_count} productos"),
                        f" serían reclasificados de {total_scanned} escaneados.",
                    ],
                    color="warning",
                    className="mb-3",
                ),
                dbc.Table(
                    [
                        html.Thead(
                            html.Tr([
                                html.Th("Producto"),
                                html.Th("Anterior"),
                                html.Th("", style={"width": "30px"}),
                                html.Th("Nueva"),
                                html.Th("Keyword"),
                            ])
                        ),
                        html.Tbody(rows),
                    ],
                    bordered=True,
                    hover=True,
                    responsive=True,
                    size="sm",
                ),
                html.P(
                    f"Mostrando {len(rows)} de {products_count} cambios.",
                    className="text-muted small",
                ) if products_count > 20 else None,
            ])

            return content, data, False  # Enable confirm button

        except Exception as e:
            logger.exception(f"Error previewing keywords apply: {e}")
            return (
                dbc.Alert(f"Error: {str(e)}", color="danger"),
                None,
                True,
            )

    @app.callback(
        [
            Output("keywords-apply-preview-container", "children", allow_duplicate=True),
            Output("keywords-apply-confirm-btn", "disabled", allow_duplicate=True),
        ],
        Input("keywords-apply-confirm-btn", "n_clicks"),
        [
            State("keywords-apply-results-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def apply_keywords_confirm(n_clicks, preview_data, auth_state, auth_tokens):
        """Actually apply keywords after confirmation."""
        from utils.auth_helpers import is_user_authenticated

        if not n_clicks or not preview_data:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            return dbc.Alert("No autenticado", color="danger"), True

        # REGLA #7.6: Restore tokens for multi-worker
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        try:
            response = request_coordinator.make_request(
                endpoint="/api/v1/admin/keywords/apply",
                method="POST",
                data={"dry_run": False},
            )

            if not response:
                return dbc.Alert("Error al conectar con el servidor", color="danger"), True
            if isinstance(response, dict) and response.get("error"):
                return dbc.Alert(f"Error: {response.get('message', response.get('error'))}", color="danger"), True

            products_count = response.get("products_reclassified", 0)

            return (
                dbc.Alert(
                    [
                        html.I(className="fas fa-check-circle me-2"),
                        html.Strong(f"¡Éxito! "),
                        f"{products_count} productos fueron reclasificados.",
                    ],
                    color="success",
                ),
                True,  # Disable button after apply
            )

        except Exception as e:
            logger.exception(f"Error applying keywords: {e}")
            return dbc.Alert(f"Error: {str(e)}", color="danger"), True

    # Mark as registered
    _apply_callbacks_registered = True
    logger.info("Apply keywords callbacks registered successfully")

    return app
