"""
Health Monitoring Callbacks Module.
Responsabilidad: Monitoreo del sistema y métricas en tiempo real.
"""

import logging

import dash_bootstrap_components as dbc
from dash import Input, Output, State, html
from utils.request_coordinator import request_coordinator
from utils.auth_helpers import get_auth_headers_from_tokens  # REGLA #7.6: Multi-worker token restoration

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_health_monitoring_callbacks(app):
    """
    Register health monitoring related callbacks.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Health monitoring callbacks already registered, skipping")
        return app

    logger.info("Registering health monitoring callbacks")

    @app.callback(
        [
            # Métricas del sistema
            Output("health-cpu-usage", "children"),
            Output("health-memory-usage", "children"),
            Output("health-connections", "children"),
            # Estado de servicios
            Output("health-backend-status", "children"),
            Output("health-backend-status", "color"),
            Output("health-backend-icon", "className"),
            Output("health-postgres-status", "children"),
            Output("health-postgres-status", "color"),
            Output("health-postgres-icon", "className"),
            Output("health-uptime", "children"),
            # Render metrics
            Output("health-render-service", "children"),
            Output("health-render-region", "children"),
            Output("health-last-deploy", "children"),
            # Alertas
            Output("health-alerts-count", "children"),
            Output("health-alerts-list", "children"),
        ],
        [
            Input("health-refresh-interval", "n_intervals"),
            Input("health-manual-refresh-btn", "n_clicks"),  # Refresh manual
        ],
        [
            State("auth-state", "data"),  # Issue #302: Verificar autenticación
            State("auth-context-sync", "data"),  # Issue #333: JWT timing protection
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=False,
    )
    def update_health_dashboard(n_intervals, n_clicks, auth_state, auth_context_sync, auth_tokens):
        """
        Actualiza todo el dashboard de salud del sistema.

        Triggers:
        - Automático: cada 30s via health-refresh-interval (solo en /admin)
        - Manual: botón "Actualizar" en header del dashboard

        Issue #302: Verificación auth proactiva antes de API calls.
        Issue #333: Protección contra JWT timing race condition.

        Args:
            n_intervals: Número de intervalos transcurridos (automático)
            n_clicks: Número de clicks en botón refresh (manual)
            auth_state: Estado de autenticación del usuario
            auth_context_sync: Estado de sincronización del token JWT

        Returns:
            Tupla con valores para todos los outputs del dashboard
        """
        from utils.auth_helpers import is_user_authenticated

        # Guard: Verificar auth antes de llamadas API (Issue #302)
        if not is_user_authenticated(auth_state):
            logger.debug("[update_health_dashboard] User not authenticated - using placeholder values")
            return get_placeholder_values()

        # Issue #333: Verificar que el token JWT esté sincronizado
        if not auth_context_sync or not auth_context_sync.get("synced"):
            logger.debug("[update_health_dashboard] Token not synced yet - showing skeleton")
            return get_skeleton_values()

        # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        try:
            # Llamar al endpoint unificado de health summary
            response = request_coordinator.make_request(
                "/api/v1/system-unified/health-summary",
                method="GET",
                params={"level": "detailed"},
                timeout=10,
                auth_headers=auth_headers,  # REGLA #7.6: Explicit headers for multi-worker
            )

            if response and response.status_code == 200:
                data = response.json()
                return process_health_data(data)
            else:
                logger.warning(f"[update_health_dashboard] API returned status {response.status_code if response else 'None'}")
                return get_error_values()

        except Exception as e:
            logger.error(f"[update_health_dashboard] Error: {str(e)}")
            return get_error_values()

    @app.callback(
        Output("health-refresh-interval", "disabled"),
        Input("url", "pathname"),
        prevent_initial_call=True,
    )
    def toggle_health_interval(pathname):
        """
        Activa/desactiva el interval de health monitoring según la página actual.

        Solo debe estar activo en /admin para ahorrar recursos del navegador.
        Fix para REGLA #0.5: Interval definido en skeleton global (app.py),
        activado/desactivado via callback.

        Args:
            pathname: Ruta actual de la página

        Returns:
            bool: True (disabled) si no está en /admin, False (enabled) si está en /admin
        """
        if pathname == "/admin":
            logger.debug("[toggle_health_interval] Enabling health monitoring on /admin")
            return False  # False = enabled
        else:
            logger.debug(f"[toggle_health_interval] Disabling health monitoring on {pathname}")
            return True  # True = disabled

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("Health monitoring callbacks registered successfully")

    return app


def create_render_metrics_unavailable():
    """
    Muestra mensaje informativo cuando las métricas de Render no están disponibles.

    Endpoint específico en desarrollo (ver Issue #214).
    Mientras tanto, se usa /api/v1/system-unified/health-summary como workaround.

    Returns:
        dbc.Alert: Alert informativo con clase 'info' indicando que el endpoint
                   dedicado está en desarrollo.

    Related:
        - Issue #214: feat: Endpoint específico /api/v1/system/render-metrics
        - PR #213: Migración de rutas legacy a /api/v1/
    """
    return dbc.Alert(
        [
            html.I(className="fas fa-info-circle me-2"),
            html.Div(
                [
                    html.Strong("Métricas Render no disponibles"),
                    html.Br(),
                    html.Small(
                        ["Endpoint específico en desarrollo. ", "Ver Issue #214 para más información."],
                        className="text-muted",
                    ),
                ]
            ),
        ],
        color="info",
        className="mt-3",
    )


def format_uptime(seconds):
    """
    Formatea los segundos de uptime en un string legible.
    """
    if seconds < 60:
        return f"{int(seconds)}s"
    elif seconds < 3600:
        return f"{int(seconds / 60)}m"
    elif seconds < 86400:
        return f"{int(seconds / 3600)}h {int((seconds % 3600) / 60)}m"
    else:
        days = int(seconds / 86400)
        hours = int((seconds % 86400) / 3600)
        return f"{days}d {hours}h"


def get_placeholder_values():
    """
    Retorna valores placeholder cuando el usuario no está autenticado.
    """
    return (
        "--",  # CPU
        "--",  # Memory
        "--",  # Connections
        "--",  # Backend status text
        "secondary",  # Backend status color
        "fas fa-circle me-2 text-secondary",  # Backend icon
        "--",  # Postgres status text
        "secondary",  # Postgres status color
        "fas fa-circle me-2 text-secondary",  # Postgres icon
        "--",  # Uptime
        "--",  # Render service
        "--",  # Render region
        "--",  # Last deploy
        "0",  # Alerts count
        dbc.Alert(
            html.Div([html.I(className="fas fa-info-circle me-2"), "Autenticación requerida"]),
            color="info",
            className="mb-0 mt-2",
        ),  # Alerts list
    )


def get_skeleton_values():
    """
    Retorna skeleton loaders mientras sincroniza el token JWT.
    """
    skeleton_text = html.Span([html.I(className="fas fa-spinner fa-spin me-2"), "Cargando..."])
    return (
        skeleton_text,  # CPU
        skeleton_text,  # Memory
        skeleton_text,  # Connections
        "Cargando",  # Backend status text
        "secondary",  # Backend status color
        "fas fa-circle me-2 text-secondary",  # Backend icon
        "Cargando",  # Postgres status text
        "secondary",  # Postgres status color
        "fas fa-circle me-2 text-secondary",  # Postgres icon
        skeleton_text,  # Uptime
        skeleton_text,  # Render service
        skeleton_text,  # Render region
        skeleton_text,  # Last deploy
        "0",  # Alerts count
        dbc.Alert(
            html.Div([html.I(className="fas fa-spinner fa-spin me-2"), "Cargando alertas..."]),
            color="info",
            className="mb-0 mt-2",
        ),  # Alerts list
    )


def get_error_values():
    """
    Retorna valores de error cuando falla la llamada API.
    """
    error_text = "Error"
    return (
        error_text,  # CPU
        error_text,  # Memory
        error_text,  # Connections
        "Error",  # Backend status text
        "danger",  # Backend status color
        "fas fa-circle me-2 text-danger",  # Backend icon
        "Error",  # Postgres status text
        "danger",  # Postgres status color
        "fas fa-circle me-2 text-danger",  # Postgres icon
        error_text,  # Uptime
        error_text,  # Render service
        error_text,  # Render region
        error_text,  # Last deploy
        "0",  # Alerts count
        dbc.Alert(
            html.Div([html.I(className="fas fa-exclamation-triangle me-2"), "Error al cargar alertas"]),
            color="danger",
            className="mb-0 mt-2",
        ),  # Alerts list
    )


def process_health_data(data):
    """
    Procesa los datos del endpoint /api/v1/system-unified/health-summary
    y los formatea para el dashboard.

    Args:
        data: Diccionario con los datos del endpoint

    Returns:
        Tupla con valores formateados para todos los outputs
    """
    try:
        # Extraer métricas del sistema
        cpu_percent = data.get("cpu_percent", 0)
        memory_percent = data.get("memory_percent", 0)
        memory_used_mb = data.get("memory_used_mb", 0)
        memory_total_mb = data.get("memory_total_mb", 512)
        active_connections = data.get("active_connections", 0)

        # Formatear CPU y Memoria
        cpu_usage = f"{cpu_percent:.1f}%"
        memory_usage = f"{memory_percent:.1f}% ({memory_used_mb:.0f}/{memory_total_mb:.0f} MB)"

        # Estado del backend
        backend_status = data.get("status", "unknown")
        if backend_status == "healthy":
            backend_badge_text = "Online"
            backend_badge_color = "success"
            backend_icon_class = "fas fa-circle me-2 text-success"
        elif backend_status == "degraded":
            backend_badge_text = "Degradado"
            backend_badge_color = "warning"
            backend_icon_class = "fas fa-circle me-2 text-warning"
        else:
            backend_badge_text = "Offline"
            backend_badge_color = "danger"
            backend_icon_class = "fas fa-circle me-2 text-danger"

        # Estado de PostgreSQL
        db_status = data.get("database", "unknown")
        if db_status == "healthy":
            postgres_badge_text = "Conectado"
            postgres_badge_color = "success"
            postgres_icon_class = "fas fa-circle me-2 text-success"
        else:
            postgres_badge_text = "Error"
            postgres_badge_color = "danger"
            postgres_icon_class = "fas fa-circle me-2 text-danger"

        # Uptime
        uptime_seconds = data.get("uptime_seconds", 0)
        uptime_str = format_uptime(uptime_seconds)

        # Render metrics (si está disponible)
        render_service = data.get("render_service", "Local")
        render_region = data.get("render_region", "N/A")
        last_deploy = data.get("last_deploy", "Desconocido")

        # Alertas
        alerts = data.get("alerts", [])
        alerts_count = str(len(alerts))

        # Crear lista de alertas
        if alerts:
            alerts_components = []
            for alert in alerts[:5]:  # Máximo 5 alertas
                severity = alert.get("severity", "info")
                severity_colors = {"critical": "danger", "warning": "warning", "info": "info"}
                color = severity_colors.get(severity, "info")

                alerts_components.append(
                    dbc.Alert(
                        [
                            html.Strong(alert.get("title", "Alerta")),
                            html.Br(),
                            html.Small(alert.get("message", "")),
                        ],
                        color=color,
                        className="mb-2 mt-2",
                    )
                )
            alerts_list = html.Div(alerts_components)
        else:
            alerts_list = dbc.Alert(
                html.Div([html.I(className="fas fa-check-circle me-2"), "No hay alertas activas"]),
                color="success",
                className="mb-0 mt-2",
            )

        return (
            cpu_usage,
            memory_usage,
            str(active_connections),
            backend_badge_text,
            backend_badge_color,
            backend_icon_class,
            postgres_badge_text,
            postgres_badge_color,
            postgres_icon_class,
            uptime_str,
            render_service,
            render_region,
            last_deploy,
            alerts_count,
            alerts_list,
        )

    except Exception as e:
        logger.error(f"[process_health_data] Error processing data: {str(e)}")
        return get_error_values()
