"""
Admin Duplicates Callbacks Module.
Issue #477: Admin UI para revisión de duplicados ProductCatalogVentaLibre

Responsabilidad: Callbacks para gestión de duplicados en parafarmacia.
- Carga de estadísticas y lista de grupos
- Acciones: merge, diferentes, skip
- Paginación
"""

import logging
from datetime import datetime

import dash
import dash_bootstrap_components as dbc
from dash import Input, Output, State, callback, ctx, html, no_update
from dash.exceptions import PreventUpdate

from utils.auth import auth_manager
from utils.auth_helpers import get_auth_headers_from_tokens
# BACKEND_URL no necesario - request_coordinator ya lo añade
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def create_duplicate_group_card(group: dict, index: int) -> dbc.Card:
    """
    Crea una tarjeta para un grupo de duplicados potenciales.

    Muestra comparación lado a lado de los productos del grupo.
    """
    products = group.get("products", [])
    suggested_primary_id = group.get("suggested_primary_id")
    group_id = group.get("group_id")
    tokens = group.get("tokens", "")
    similarity = group.get("similarity_score", 0)

    # Crear filas de productos para comparación
    product_rows = []
    for i, product in enumerate(products):
        is_suggested_primary = str(product.get("id")) == str(suggested_primary_id)
        cn_codes = product.get("cn_codes", []) or []
        ean = product.get("ean13") or "-"

        product_rows.append(
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardHeader(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className=f"fas fa-{'star text-warning' if is_suggested_primary else 'box text-secondary'} me-2"
                                            ),
                                            html.Span(
                                                "Primario sugerido" if is_suggested_primary else f"Producto {i+1}",
                                                className="fw-bold" if is_suggested_primary else "",
                                            ),
                                        ]
                                    )
                                ],
                                className="bg-light" if is_suggested_primary else "",
                            ),
                            dbc.CardBody(
                                [
                                    html.H6(
                                        product.get("product_name_display", "Sin nombre")[:60],
                                        className="card-title text-truncate",
                                        title=product.get("product_name_display", ""),
                                    ),
                                    html.Hr(className="my-2"),
                                    html.Div(
                                        [
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-barcode me-1"),
                                                    f"CN: {', '.join(cn_codes) if cn_codes else '-'}",
                                                ],
                                                className="text-muted d-block",
                                            ),
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-qrcode me-1"),
                                                    f"EAN: {ean}",
                                                ],
                                                className="text-muted d-block",
                                            ),
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-chart-line me-1"),
                                                    f"Ventas: {product.get('total_sales_count', 0):,}",
                                                ],
                                                className="text-muted d-block",
                                            ),
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-store me-1"),
                                                    f"Farmacias: {product.get('pharmacies_count', 0)}",
                                                ],
                                                className="text-muted d-block",
                                            ),
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-tag me-1"),
                                                    f"Categoría: {product.get('ml_category', '-')}",
                                                ],
                                                className="text-muted d-block",
                                            ),
                                            html.Div(
                                                [
                                                    dbc.Badge(
                                                        "Verificado",
                                                        color="success",
                                                        className="me-1",
                                                    )
                                                    if product.get("human_verified")
                                                    else None,
                                                ],
                                                className="mt-2",
                                            ),
                                        ]
                                    ),
                                ]
                            ),
                        ],
                        className=f"h-100 {'border-warning border-2' if is_suggested_primary else ''}",
                    )
                ],
                width=12,
                md=6 if len(products) == 2 else 4,
                className="mb-2",
            )
        )

    return dbc.Card(
        [
            dbc.CardHeader(
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-clone me-2"),
                                        html.Span(f"Grupo #{index + 1}"),
                                        dbc.Badge(
                                            f"{similarity*100:.0f}% similitud",
                                            color="info",
                                            className="ms-2",
                                        ),
                                        dbc.Badge(
                                            f"{len(products)} productos",
                                            color="secondary",
                                            className="ms-1",
                                        ),
                                    ]
                                ),
                                html.Small(
                                    f"Tokens: {tokens[:50]}..." if len(tokens) > 50 else f"Tokens: {tokens}",
                                    className="text-muted d-block mt-1",
                                ),
                            ],
                            width=12,
                            md=8,
                        ),
                        dbc.Col(
                            [
                                dbc.ButtonGroup(
                                    [
                                        dbc.Button(
                                            html.Span([html.I(className="fas fa-compress-arrows-alt me-1"), "Merge"]),
                                            id={"type": "duplicate-merge-btn", "index": str(group_id)},
                                            color="success",
                                            size="sm",
                                            outline=True,
                                        ),
                                        dbc.Button(
                                            html.Span([html.I(className="fas fa-not-equal me-1"), "Diferentes"]),
                                            id={"type": "duplicate-reject-btn", "index": str(group_id)},
                                            color="secondary",
                                            size="sm",
                                            outline=True,
                                        ),
                                        dbc.Button(
                                            html.Span([html.I(className="fas fa-forward me-1"), "Skip"]),
                                            id={"type": "duplicate-skip-btn", "index": str(group_id)},
                                            color="warning",
                                            size="sm",
                                            outline=True,
                                        ),
                                    ],
                                    className="float-end",
                                )
                            ],
                            width=12,
                            md=4,
                            className="text-end",
                        ),
                    ],
                    className="align-items-center",
                )
            ),
            dbc.CardBody(dbc.Row(product_rows)),
        ],
        className="mb-3 shadow-sm",
    )


def register_duplicates_callbacks(app):
    """
    Register admin duplicates panel callbacks.
    """
    global _module_callbacks_registered

    if _module_callbacks_registered:
        logger.warning("Duplicates callbacks already registered, skipping")
        return app

    logger.info("Registering duplicates callbacks")

    @app.callback(
        [
            Output("duplicates-brand-filter", "value"),
            Output("duplicates-sort-by", "value"),
            Output("duplicates-sort-order", "value"),
            Output("duplicates-date-from", "value"),
        ],
        Input("duplicates-clear-filters-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def clear_filters(n_clicks):
        """
        Limpia todos los filtros (Issue #480).
        """
        return "", "similarity", "desc", None

    @app.callback(
        [
            Output("duplicates-pending-count", "children"),
            Output("duplicates-merged-count", "children"),
            Output("duplicates-different-count", "children"),
            Output("duplicates-coverage-pct", "children"),
            Output("duplicates-groups-container", "children"),
            Output("duplicates-pagination-info", "children"),
            Output("duplicates-current-page", "children"),
            Output("duplicates-prev-page", "disabled"),
            Output("duplicates-next-page", "disabled"),
            Output("duplicates-last-update", "children"),
            Output("duplicates-page-store", "data"),
            Output("duplicates-brand-filter", "options"),  # Issue #480: Consolidado aquí (REGLA #11)
        ],
        [
            Input("duplicates-tab-activated-trigger", "data"),
            Input("duplicates-refresh-btn", "n_clicks"),
            Input("duplicates-prev-page", "n_clicks"),
            Input("duplicates-next-page", "n_clicks"),
            Input("duplicates-apply-filters-btn", "n_clicks"),  # Issue #480
        ],
        [
            State("duplicates-page-store", "data"),
            State("duplicates-brand-filter", "value"),  # Issue #480
            State("duplicates-sort-by", "value"),  # Issue #480
            State("duplicates-sort-order", "value"),  # Issue #480
            State("duplicates-date-from", "value"),  # Issue #480
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_duplicates_data(
        trigger, refresh_clicks, prev_clicks, next_clicks, apply_clicks,
        current_page, brand_filter, sort_by, sort_order, date_from,
        auth_state, auth_tokens
    ):
        """
        Carga estadísticas, opciones de filtro y lista de grupos de duplicados.
        Soporta filtros y ordenación (Issue #480).
        REGLA #11: Consolidado con load_filter_options para evitar Input duplicado.
        """
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Restaurar tokens antes de API calls
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        triggered_id = ctx.triggered_id if ctx.triggered else None

        # Calcular página actual
        page = current_page or 1
        if triggered_id == "duplicates-prev-page" and page > 1:
            page -= 1
        elif triggered_id == "duplicates-next-page":
            page += 1
        elif triggered_id in ["duplicates-tab-activated-trigger", "duplicates-refresh-btn", "duplicates-apply-filters-btn"]:
            page = 1  # Reset to page 1 on refresh or filter change

        # Default brand options (fallback)
        brand_options = [{"label": "Todas las marcas", "value": ""}]

        try:
            # Obtener estadísticas
            stats_response = request_coordinator.make_request(
                "/api/v1/admin/duplicates/stats",
            )

            if not stats_response or "error" in stats_response:
                return (
                    "--", "--", "--", "--",
                    dbc.Alert("Error cargando estadísticas", color="danger"),
                    "", f"Página {page}", True, True, "", page, brand_options
                )

            pending_count = stats_response.get("pending_review", 0)
            merged_count = stats_response.get("merged", 0)
            different_count = stats_response.get("confirmed_different", 0)
            coverage_pct = stats_response.get("coverage_pct", 0)

            # Issue #480: Cargar opciones de filtro (solo en tab activation o refresh)
            if triggered_id in ["duplicates-tab-activated-trigger", "duplicates-refresh-btn"]:
                try:
                    filter_response = request_coordinator.make_request(
                        "/api/v1/admin/duplicates/filter-options",
                    )
                    if filter_response and "brands" in filter_response:
                        brand_options = [{"label": "Todas las marcas", "value": ""}]
                        brand_options.extend([
                            {"label": b, "value": b}
                            for b in filter_response["brands"]
                        ])
                except Exception as e:
                    logger.warning(f"Could not load filter options: {e}")

            # Construir parámetros con filtros (Issue #480)
            params = {
                "page": page,
                "page_size": 10,
                "sort_by": sort_by or "similarity",
                "sort_order": sort_order or "desc",
            }
            if brand_filter:
                params["brand"] = brand_filter
            if date_from:
                params["date_from"] = date_from

            # Obtener lista de grupos
            groups_response = request_coordinator.make_request(
                "/api/v1/admin/duplicates/pending",
                params=params,
            )

            if not groups_response or "error" in groups_response:
                return (
                    str(pending_count), str(merged_count), str(different_count), f"{coverage_pct:.1f}",
                    dbc.Alert("Error cargando grupos", color="danger"),
                    "", f"Página {page}", True, True,
                    f"Actualizado: {datetime.now().strftime('%H:%M:%S')}", page,
                    no_update  # Don't update brand options on error
                )

            groups = groups_response.get("groups", [])
            has_more = groups_response.get("has_more", False)
            total_groups = groups_response.get("total_groups", 0)

            # Crear tarjetas de grupos
            if not groups:
                # Mensaje diferente si hay filtros activos
                if brand_filter or date_from:
                    msg = "No se encontraron grupos con los filtros seleccionados."
                else:
                    msg = "No hay grupos de duplicados pendientes de revisión."
                groups_content = dbc.Alert(
                    html.Div([
                        html.I(className="fas fa-check-circle me-2"),
                        msg,
                    ]),
                    color="success" if not (brand_filter or date_from) else "info",
                )
            else:
                groups_content = html.Div([
                    create_duplicate_group_card(group, i + (page - 1) * 10)
                    for i, group in enumerate(groups)
                ])

            # Info de paginación
            if groups:
                start_idx = (page - 1) * 10 + 1
                end_idx = start_idx + len(groups) - 1
                pagination_info = f"Mostrando {start_idx}-{end_idx} de {total_groups} grupos"
            else:
                pagination_info = ""

            # Determine if we should update brand options
            # Only update on initial load or refresh, not on pagination/filter apply
            brand_options_output = brand_options if triggered_id in ["duplicates-tab-activated-trigger", "duplicates-refresh-btn"] else no_update

            return (
                str(pending_count),
                str(merged_count),
                str(different_count),
                f"{coverage_pct:.1f}",
                groups_content,
                pagination_info,
                f"Página {page}",
                page <= 1,  # prev disabled
                not has_more,  # next disabled
                f"Actualizado: {datetime.now().strftime('%H:%M:%S')}",
                page,
                brand_options_output,  # Issue #480: Brand filter options
            )

        except Exception as e:
            logger.exception(f"Error loading duplicates data: {e}")
            return (
                "--", "--", "--", "--",
                dbc.Alert("Error cargando datos. Por favor intente de nuevo.", color="danger"),
                "", f"Página {page}", True, True, "", page, no_update
            )

    # NOTA: update_page_store() eliminado - consolidado en load_duplicates_data() (REGLA #11)

    @app.callback(
        [
            Output("duplicates-merge-modal", "is_open"),
            Output("duplicates-merge-preview", "children"),
            Output("duplicates-merge-data-store", "data"),
        ],
        [
            Input({"type": "duplicate-merge-btn", "index": dash.ALL}, "n_clicks"),
            Input("duplicates-merge-cancel", "n_clicks"),
            Input("duplicates-merge-confirm", "n_clicks"),
        ],
        [
            State("duplicates-merge-modal", "is_open"),
            State("duplicates-merge-data-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_merge_modal(
        merge_clicks, cancel_clicks, confirm_clicks,
        is_open, merge_data, auth_state, auth_tokens
    ):
        """
        Maneja el modal de confirmación de merge.
        """
        from utils.auth_helpers import is_user_authenticated

        if not ctx.triggered:
            raise PreventUpdate

        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        # Cancelar
        if "duplicates-merge-cancel" in triggered_id:
            return False, no_update, None

        # Confirmar merge
        if "duplicates-merge-confirm" in triggered_id:
            if not merge_data:
                raise PreventUpdate

            if not is_user_authenticated(auth_state):
                raise PreventUpdate

            # REGLA #7.6: Restaurar tokens
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

            try:
                response = request_coordinator.make_request(
                    "/api/v1/admin/duplicates/merge",
                    method="POST",
                    data={
                        "primary_id": merge_data["primary_id"],
                        "secondary_ids": merge_data["secondary_ids"],
                    },
                )

                if response and response.get("success"):
                    # Cerrar modal - el refresh se hará automáticamente
                    return False, no_update, None
                else:
                    error_msg = response.get("detail", "Error desconocido") if response else "Error de conexión"
                    return True, dbc.Alert(f"Error: {error_msg}", color="danger"), merge_data

            except Exception as e:
                logger.exception(f"Error merging products: {e}")
                return True, dbc.Alert(f"Error: {str(e)}", color="danger"), merge_data

        # Abrir modal (click en merge button)
        if "duplicate-merge-btn" in triggered_id:
            # Extraer group_id del botón clickeado
            try:
                import json
                prop_id = triggered_id.replace(".n_clicks", "")
                button_info = json.loads(prop_id)
                group_id = button_info.get("index")

                if not group_id:
                    raise PreventUpdate

                if not is_user_authenticated(auth_state):
                    raise PreventUpdate

                # REGLA #7.6: Restaurar tokens
                if auth_tokens and "tokens" in auth_tokens:
                    auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

                # Obtener datos del grupo
                # TODO: Optimización futura - crear endpoint GET /groups/{group_id}
                # para evitar fetch de todos los grupos cuando solo necesitamos uno
                response = request_coordinator.make_request(
                    "/api/v1/admin/duplicates/pending",
                    params={"page": 1, "page_size": 50},
                )

                if not response or "error" in response:
                    raise PreventUpdate

                # Buscar el grupo por ID
                groups = response.get("groups", [])
                target_group = None
                for group in groups:
                    if str(group.get("group_id")) == str(group_id):
                        target_group = group
                        break

                if not target_group:
                    raise PreventUpdate

                products = target_group.get("products", [])
                primary_id = str(target_group.get("suggested_primary_id"))
                secondary_ids = [str(p["id"]) for p in products if str(p["id"]) != primary_id]

                # Crear preview
                primary_product = next((p for p in products if str(p["id"]) == primary_id), None)

                preview = html.Div([
                    html.H6("Producto primario (se conservará):", className="text-success"),
                    html.P(primary_product.get("product_name_display") if primary_product else "-"),
                    html.H6(f"Productos a fusionar ({len(secondary_ids)}):", className="text-warning mt-3"),
                    html.Ul([
                        html.Li(p.get("product_name_display"))
                        for p in products if str(p["id"]) != primary_id
                    ]),
                ])

                merge_data = {
                    "group_id": group_id,
                    "primary_id": primary_id,
                    "secondary_ids": secondary_ids,
                }

                return True, preview, merge_data

            except Exception as e:
                logger.exception(f"Error preparing merge: {e}")
                raise PreventUpdate

        raise PreventUpdate

    @app.callback(
        Output("duplicates-groups-container", "children", allow_duplicate=True),
        [
            Input({"type": "duplicate-reject-btn", "index": dash.ALL}, "n_clicks"),
            Input({"type": "duplicate-skip-btn", "index": dash.ALL}, "n_clicks"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_reject_skip(reject_clicks, skip_clicks, auth_state, auth_tokens):
        """
        Maneja acciones de rechazar (diferentes) y skip (posponer).
        """
        from utils.auth_helpers import is_user_authenticated

        if not ctx.triggered or (not any(reject_clicks) and not any(skip_clicks)):
            raise PreventUpdate

        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Restaurar tokens
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        try:
            import json
            prop_id = triggered_id.replace(".n_clicks", "")
            button_info = json.loads(prop_id)
            group_id = button_info.get("index")
            action_type = button_info.get("type")

            if not group_id:
                raise PreventUpdate

            if action_type == "duplicate-reject-btn":
                # Obtener productos del grupo para rechazarlos
                response = request_coordinator.make_request(
                    "/api/v1/admin/duplicates/pending",
                    params={"page": 1, "page_size": 50},
                )

                if not response:
                    raise PreventUpdate

                groups = response.get("groups", [])
                target_group = next((g for g in groups if str(g.get("group_id")) == str(group_id)), None)

                if not target_group:
                    raise PreventUpdate

                product_ids = [str(p["id"]) for p in target_group.get("products", [])]

                reject_response = request_coordinator.make_request(
                    "/api/v1/admin/duplicates/reject",
                    method="POST",
                    data={"product_ids": product_ids, "group_id": group_id},
                )

                if reject_response and reject_response.get("success"):
                    return dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-check me-2"),
                            f"Marcados {len(product_ids)} productos como diferentes. Actualizando...",
                        ]),
                        color="success",
                    )

            elif action_type == "duplicate-skip-btn":
                skip_response = request_coordinator.make_request(
                    "/api/v1/admin/duplicates/skip",
                    method="POST",
                    data={"group_id": group_id},
                )

                if skip_response and skip_response.get("success"):
                    return dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-check me-2"),
                            "Grupo pospuesto. Actualizando...",
                        ]),
                        color="info",
                    )

        except Exception as e:
            logger.exception(f"Error handling action: {e}")
            return dbc.Alert(f"Error: {str(e)}", color="danger")

        raise PreventUpdate

    # === BÚSQUEDA MANUAL (Issue #475) ===
    @app.callback(
        Output("duplicates-search-results", "children"),
        [
            Input("duplicates-search-btn", "n_clicks"),
            Input("duplicates-search-input", "n_submit"),
        ],
        [
            State("duplicates-search-input", "value"),
            State("duplicates-search-threshold", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def search_similar_products(
        search_clicks, search_submit,
        query, threshold, auth_state, auth_tokens
    ):
        """
        Búsqueda manual de productos similares (Issue #475).
        Usa el endpoint POST /api/v1/admin/duplicates/search
        """
        from utils.auth_helpers import is_user_authenticated

        if not query or len(query) < 3:
            return dbc.Alert(
                html.Span([
                    html.I(className="fas fa-info-circle me-2"),
                    "Ingrese al menos 3 caracteres para buscar.",
                ]),
                color="info",
            )

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Obtener auth headers para multi-worker compatibility
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[SEARCH] No auth headers available")
            return dbc.Alert(
                html.Span([
                    html.I(className="fas fa-exclamation-triangle me-2"),
                    "Sesión expirada. Por favor, recargue la página.",
                ]),
                color="warning",
            )

        try:
            response = request_coordinator.make_request(
                "/api/v1/admin/duplicates/search",
                method="POST",
                data={
                    "query": query,
                    "threshold": float(threshold or 0.6),
                    "limit": 10,
                },
                auth_headers=auth_headers,
            )

            if not response or "error" in response:
                error_msg = response.get("detail", "Error desconocido") if response else "Error de conexión"
                return dbc.Alert(f"Error: {error_msg}", color="danger")

            results = response.get("results", [])
            total_found = response.get("total_found", 0)
            threshold_used = response.get("threshold_used", 0.6)

            if not results:
                return dbc.Alert(
                    html.Div([
                        html.I(className="fas fa-search me-2"),
                        f"No se encontraron productos similares a '{query}' con umbral {threshold_used*100:.0f}%.",
                    ]),
                    color="info",
                )

            # Crear tabla de resultados
            table_rows = []
            for result in results:
                similarity_color = (
                    "success" if result["similarity_score"] >= 0.85
                    else "warning" if result["similarity_score"] >= 0.70
                    else "secondary"
                )
                table_rows.append(
                    html.Tr([
                        html.Td(
                            html.Div([
                                html.Div(
                                    result["product_name_display"][:60],
                                    className="fw-bold text-truncate",
                                    title=result["product_name_display"],
                                    style={"maxWidth": "300px"},
                                ),
                                html.Small(
                                    f"ID: {str(result['id'])[:8]}...",
                                    className="text-muted",
                                ),
                            ])
                        ),
                        html.Td(
                            dbc.Badge(
                                f"{result['similarity_score']*100:.0f}%",
                                color=similarity_color,
                            )
                        ),
                        html.Td(result.get("match_method", "-")),
                        html.Td(result.get("ml_category") or "-"),
                        html.Td(f"{result.get('total_sales_count', 0):,}"),
                        html.Td(result.get("ean13") or "-"),
                        html.Td(", ".join(result.get("cn_codes", [])) or "-"),
                    ])
                )

            return html.Div([
                dbc.Alert(
                    html.Span([
                        html.I(className="fas fa-check-circle me-2"),
                        f"Se encontraron {total_found} producto(s) similar(es) a '{query}'.",
                    ]),
                    color="success",
                    className="mb-3",
                ),
                dbc.Table(
                    [
                        html.Thead(
                            html.Tr([
                                html.Th("Producto"),
                                html.Th("Similitud"),
                                html.Th("Match"),
                                html.Th("Categoría"),
                                html.Th("Ventas"),
                                html.Th("EAN"),
                                html.Th("CN"),
                            ])
                        ),
                        html.Tbody(table_rows),
                    ],
                    bordered=True,
                    hover=True,
                    responsive=True,
                    size="sm",
                    className="mb-0",
                ),
            ])

        except Exception as e:
            logger.exception(f"Error searching similar products: {e}")
            return dbc.Alert(f"Error: {str(e)}", color="danger")

    _module_callbacks_registered = True
    logger.info("Duplicates callbacks registered successfully")

    return app
