"""
Database Stats Callbacks Module.
Responsabilidad: Cargar estadísticas de base de datos en Tab Sistema.

Issue #485: Tab Sistema reorganizado con sub-tab "Base de Datos".
"""

import logging

from dash import Input, Output, State
from dash.exceptions import PreventUpdate
from utils.request_coordinator import request_coordinator
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_database_stats_callbacks(app):
    """
    Register database stats related callbacks.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Database stats callbacks already registered, skipping")
        return app

    logger.info("Registering database stats callbacks")

    @app.callback(
        [
            Output("admin-db-total-size", "children"),
            Output("admin-db-total-records", "children"),
            Output("admin-db-active-connections", "children"),
        ],
        [
            Input("system-sub-tabs", "active_tab"),  # Trigger cuando se selecciona sub-tab (dbc.Tabs)
            Input("url", "pathname"),  # Trigger inicial
        ],
        [
            State("admin-tabs", "value"),  # Tab principal activo (REGLA #11: State, no Input)
            State("auth-state", "data"),
            State("auth-context-sync", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=False,
    )
    def update_database_stats(sub_tab, pathname, admin_tab, auth_state, auth_context_sync, auth_tokens):
        """
        Actualiza las estadísticas de base de datos cuando se selecciona el sub-tab.

        Triggers:
        - sub_tab: Cuando usuario selecciona sub-tab "database"
        - pathname: Carga inicial en /admin

        States (REGLA #11 - no pueden ser Inputs duplicados):
        - admin_tab: Tab principal activo (verificar es "system")

        Guards:
        - Solo ejecuta si pathname es /admin
        - Solo ejecuta si admin_tab es "system"
        - Solo ejecuta si sub_tab es "database"
        - Verifica autenticación antes de API calls
        """
        # Guard: Solo en /admin
        if pathname != "/admin":
            raise PreventUpdate

        # Guard: Solo en Tab Sistema
        if admin_tab != "system":
            return "--", "--", "--"

        # Guard: Solo en sub-tab Base de Datos
        if sub_tab != "database":
            return "--", "--", "--"

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[DATABASE_STATS] User not authenticated - showing placeholder")
            return "--", "--", "--"

        # Guard: Verificar token sincronizado
        if not auth_context_sync or not auth_context_sync.get("synced"):
            logger.debug("[DATABASE_STATS] Token not synced - showing placeholder")
            return "Cargando...", "Cargando...", "Cargando..."

        # REGLA #7.6: Multi-worker token restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        try:
            # Obtener tamaño de BD (router prefix: /api/v1/admin)
            size_response = request_coordinator.make_request(
                "/api/v1/admin/size",
                method="GET",
                timeout=10,
                auth_headers=auth_headers,
            )

            # Obtener conexiones activas del health summary
            health_response = request_coordinator.make_request(
                "/api/v1/system-unified/health-summary",
                method="GET",
                params={"level": "basic"},
                timeout=10,
                auth_headers=auth_headers,
            )

            # Procesar tamaño
            if size_response:
                db_size_pretty = size_response.get("database_size_pretty", "--")

                # Calcular registros totales sumando todas las tablas
                top_tables = size_response.get("top_tables", [])
                total_records = sum(t.get("row_count", 0) for t in top_tables)
                total_records_formatted = f"{total_records:,}".replace(",", ".")
            else:
                db_size_pretty = "Error"
                total_records_formatted = "Error"

            # Procesar conexiones
            if health_response:
                active_connections = health_response.get("active_connections", 0)
                connections_str = str(active_connections)
            else:
                connections_str = "Error"

            logger.info(f"[DATABASE_STATS] Loaded: size={db_size_pretty}, records={total_records_formatted}, connections={connections_str}")

            return db_size_pretty, total_records_formatted, connections_str

        except Exception as e:
            logger.error(f"[DATABASE_STATS] Error loading stats: {e}")
            return "Error", "Error", "Error"

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("Database stats callbacks registered successfully")

    return app
