"""
Dangerous Tools Callbacks Module.
Responsabilidad: Operaciones destructivas con confirmación doble y rate limiting.
"""

import logging
from datetime import datetime, timezone

import dash_bootstrap_components as dbc
from dash import Input, Output, State, ctx, html, no_update
from utils.request_coordinator import request_coordinator
from utils.auth import auth_manager  # REGLA #7.6: Multi-worker token restoration

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_dangerous_tools_callbacks(app):
    """
    Register dangerous tools related callbacks with double confirmation.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Dangerous tools callbacks already registered, skipping")
        return app

    logger.info("Registering dangerous tools callbacks")

    @app.callback(
        [
            Output("dangerous-modal-state-store", "data"),
            Output("dangerous-operation-description", "children"),
            Output("selected-dangerous-operation", "data"),
            Output("dangerous-operation-confirmation-instruction", "children"),
        ],
        [
            Input("admin-delete-sales-button", "n_clicks"),
            Input("admin-clean-catalog-button", "n_clicks"),
            Input("admin-vacuum-db-button", "n_clicks"),
            Input("admin-reindex-button", "n_clicks"),
            Input("dangerous-operation-cancel", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def handle_dangerous_operation_modal(
        delete_sales_clicks, clean_catalog_clicks, vacuum_clicks, reindex_clicks, cancel_clicks
    ):
        """
        Maneja la apertura/cierre del modal de operaciones peligrosas con doble confirmación.
        Escribe al store intermedio dangerous-modal-state-store (REGLA #11 - DASH002 fix).
        """
        if not ctx.triggered:
            return None, "", {}, ""

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # Issue #198: Validación adicional - verificar que hay un click real (not None)
        if trigger_id == "admin-delete-sales-button" and not delete_sales_clicks:
            return None, "", {}, ""
        elif trigger_id == "admin-clean-catalog-button" and not clean_catalog_clicks:
            return None, "", {}, ""
        elif trigger_id == "admin-vacuum-db-button" and not vacuum_clicks:
            return None, "", {}, ""
        elif trigger_id == "admin-reindex-button" and not reindex_clicks:
            return None, "", {}, ""

        if trigger_id == "dangerous-operation-cancel":
            # Escribir al store: cerrar modal
            return {"is_open": False}, "", {}, ""

        elif trigger_id == "admin-delete-sales-button":
            description = html.Div(
                [
                    html.H5("⚠️ Borrar TODOS los Datos de Ventas", className="text-danger"),
                    html.P("Esta operación eliminará permanentemente:"),
                    html.Ul(
                        [
                            html.Li("Todos los archivos de ventas subidos"),
                            html.Li("Todos los datos de ventas procesados"),
                            html.Li("Todos los enriquecimientos de datos"),
                            html.Li("TODO el historial de análisis"),
                        ]
                    ),
                    html.P(
                        [html.Strong("NO SE ELIMINA: "), "Catálogo de productos, configuración de farmacia, usuarios."],
                        className="text-info",
                    ),
                    create_audit_warning(),
                ]
            )
            operation_data = {
                "operation_id": "delete_all_sales_data",
                "confirmation_text": "ELIMINAR VENTAS",
                "danger_level": "high",
            }
            confirmation_instruction = create_confirmation_instruction("ELIMINAR VENTAS")
            # Escribir al store: abrir modal
            return {"is_open": True}, description, operation_data, confirmation_instruction

        elif trigger_id == "admin-clean-catalog-button":
            description = html.Div(
                [
                    html.H5("☠️ Limpiar Catálogo Completo", className="text-danger"),
                    html.P("Esta operación eliminará permanentemente:"),
                    html.Ul(
                        [
                            html.Li("TODOS los productos del catálogo"),
                            html.Li("Datos CIMA y Nomenclátor"),
                            html.Li("Enriquecimientos automáticos"),
                            html.Li("Configuraciones de laboratorios"),
                        ]
                    ),
                    html.P(
                        [html.Strong("REQUIERE: "), "Re-sincronización completa desde cero (varias horas)."],
                        className="text-warning",
                    ),
                    create_audit_warning(),
                ]
            )
            operation_data = {
                "operation_id": "clean_catalog_full",
                "confirmation_text": "LIMPIAR CATALOGO",
                "danger_level": "critical",
            }
            confirmation_instruction = create_confirmation_instruction("LIMPIAR CATALOGO")
            # Escribir al store: abrir modal
            return {"is_open": True}, description, operation_data, confirmation_instruction

        elif trigger_id == "admin-vacuum-db-button":
            description = html.Div(
                [
                    html.H5("🔧 VACUUM Database", className="text-warning"),
                    html.P("Esta operación optimizará la base de datos:"),
                    html.Ul(
                        [
                            html.Li("Recupera espacio de registros eliminados"),
                            html.Li("Actualiza estadísticas para el query planner"),
                            html.Li("Puede tardar varios minutos"),
                            html.Li("La BD estará bloqueada durante el proceso"),
                        ]
                    ),
                    html.P(
                        [html.Strong("IMPACTO: "), "Sistema no disponible durante la operación."],
                        className="text-warning",
                    ),
                    create_audit_warning(),
                ]
            )
            operation_data = {
                "operation_id": "vacuum_database",
                "confirmation_text": "VACUUM DB",
                "danger_level": "medium",
            }
            confirmation_instruction = create_confirmation_instruction("VACUUM DB")
            # Escribir al store: abrir modal
            return {"is_open": True}, description, operation_data, confirmation_instruction

        elif trigger_id == "admin-reindex-button":
            description = html.Div(
                [
                    html.H5("📊 Reindexar Tablas", className="text-warning"),
                    html.P("Esta operación reconstruirá todos los índices:"),
                    html.Ul(
                        [
                            html.Li("Mejora performance de queries"),
                            html.Li("Corrige índices corruptos"),
                            html.Li("Proceso intensivo en I/O"),
                            html.Li("Puede tardar hasta 30 minutos"),
                        ]
                    ),
                    html.P([html.Strong("IMPACTO: "), "Queries más lentas durante el proceso."], className="text-info"),
                    create_audit_warning(),
                ]
            )
            operation_data = {
                "operation_id": "reindex_tables",
                "confirmation_text": "REINDEX TABLES",
                "danger_level": "medium",
            }
            confirmation_instruction = create_confirmation_instruction("REINDEX TABLES")
            # Escribir al store: abrir modal
            return {"is_open": True}, description, operation_data, confirmation_instruction

        return None, "", {}, ""

    @app.callback(
        Output("dangerous-operation-execute", "disabled"),
        [Input("dangerous-operation-confirmation", "value")],
        [State("selected-dangerous-operation", "data")],
        prevent_initial_call=True,
    )
    def validate_dangerous_operation_confirmation(confirmation_text, operation_data):
        """
        Valida la confirmación para operaciones peligrosas con múltiples niveles de seguridad.
        """
        if not operation_data or not confirmation_text:
            return True

        expected_text = operation_data.get("confirmation_text", "")
        danger_level = operation_data.get("danger_level", "high")

        # Para operaciones críticas, requerir texto exacto
        if danger_level == "critical":
            return confirmation_text.strip() != expected_text.strip()
        else:
            # Para otras operaciones, permitir mayúsculas/minúsculas
            return confirmation_text.strip().upper() != expected_text.strip().upper()

    @app.callback(
        [Output("dangerous-confirmation-state-store", "data"), Output("admin-dangerous-operations-result", "children")],
        [Input("dangerous-operation-execute", "n_clicks")],
        [
            State("dangerous-operation-confirmation", "value"),
            State("dangerous-operations-modal", "is_open"),
            State("selected-dangerous-operation", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def execute_dangerous_operation(execute_clicks, confirmation_text, modal_is_open, operation_data, auth_tokens):
        """
        Ejecuta la operación peligrosa confirmada con audit logging y rate limiting.
        Escribe al store intermedio dangerous-confirmation-state-store (REGLA #11 - DASH002 fix).
        """
        # Validación defensiva
        if not execute_clicks or not modal_is_open or not operation_data:
            return None, html.Div()

        if not ctx.triggered:
            return None, html.Div()

        # REGLA #7.6: Multi-worker token restoration (Render)
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])
            logger.debug("[execute_dangerous_operation] Restored tokens from auth-tokens-store")

        # Validación adicional
        try:
            if not isinstance(operation_data, dict):
                return {"is_open": False, "confirmation_value": ""}, create_error_alert("Datos de operación inválidos")
        except Exception:
            return {"is_open": False, "confirmation_value": ""}, create_error_alert("Estado de operación corrupto")

        expected_text = operation_data.get("confirmation_text", "")
        if confirmation_text.strip().upper() != expected_text.strip().upper():
            return {"is_open": True, "confirmation_value": confirmation_text}, create_error_alert(
                "Confirmación incorrecta"
            )

        try:
            operation_id = operation_data.get("operation_id")

            # Audit log de la operación
            log_dangerous_operation(operation_id)

            # Ejecutar operación con rate limiting aplicado en backend
            result = execute_backend_dangerous_operation(operation_id, operation_data)

            if result["success"]:
                return {"is_open": False, "confirmation_value": ""}, create_success_alert(operation_id, result)
            else:
                return {"is_open": False, "confirmation_value": ""}, create_error_alert(
                    result.get("message", "Error desconocido")
                )

        except Exception as e:
            logger.error(f"Excepción en operación peligrosa {operation_data.get('operation_id', 'unknown')}: {str(e)}")
            return {"is_open": False, "confirmation_value": ""}, create_error_alert(f"Error de conexión: {str(e)}")

    @app.callback(
        [Output("dangerous-operations-modal", "is_open"), Output("dangerous-operation-confirmation", "value")],
        [Input("dangerous-modal-state-store", "data"), Input("dangerous-confirmation-state-store", "data")],
        prevent_initial_call=True,
    )
    def unified_dangerous_modal_controller(modal_state, confirmation_state):
        """
        Callback unificado que lee stores intermedios y controla outputs finales del modal.
        Solución a DASH002: Un solo callback escribe a dangerous-operations-modal y dangerous-operation-confirmation.
        (REGLA #11 - Intermediate Store Decoupling pattern)
        """
        if not ctx.triggered:
            return no_update, no_update

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # Determinar fuente del trigger
        if trigger_id == "dangerous-modal-state-store" and modal_state:
            # Actualización desde handle_dangerous_operation_modal
            is_open = modal_state.get("is_open", False)
            return is_open, "" if is_open else no_update

        elif trigger_id == "dangerous-confirmation-state-store" and confirmation_state:
            # Actualización desde execute_dangerous_operation
            is_open = confirmation_state.get("is_open", False)
            confirmation_value = confirmation_state.get("confirmation_value", "")
            return is_open, confirmation_value

        return no_update, no_update

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("Dangerous tools callbacks registered successfully")

    return app


def create_audit_warning():
    """
    Crea un mensaje de advertencia sobre audit logging.
    """
    return html.Div(
        [
            html.Hr(),
            html.Small(
                [
                    html.I(className="fas fa-info-circle me-1"),
                    "Esta operación quedará registrada en el audit log con timestamp, usuario y IP.",
                ],
                className="text-muted",
            ),
        ]
    )


def create_confirmation_instruction(confirmation_text):
    """
    Crea las instrucciones de confirmación con múltiples pasos.
    """
    return html.Div(
        [
            html.P([html.Strong("Paso 1: "), "Confirme que entiende el impacto de esta operación"]),
            html.P(
                [
                    html.Strong("Paso 2: "),
                    "Escriba exactamente: ",
                    html.Code(confirmation_text, className="text-danger"),
                ]
            ),
            html.P([html.Strong("Paso 3: "), "Pulse EJECUTAR para proceder (no se puede deshacer)"], className="mb-0"),
        ]
    )


def log_dangerous_operation(operation_id):
    """
    Registra la operación peligrosa en el audit log con timeout apropiado.
    """
    try:
        audit_data = {
            "operation": operation_id,
            "timestamp": datetime.now(timezone.utc).isoformat(),
            "user": "admin",  # En producción, obtener del contexto de sesión
            "ip_address": "127.0.0.1",  # En producción, obtener de la request
        }

        # Fix Problema #4: Corregir ruta del endpoint (falta /v1)
        response_data = request_coordinator.make_request(
            "/api/v1/admin/audit-log",
            method="POST",
            data=audit_data,
            cache_ttl=0,
            bypass_cache=True,
            timeout=10,  # Audit log no requiere timeout extendido
        )

        if not response_data:
            logger.warning(f"Failed to log audit for operation {operation_id}")
    except Exception as e:
        logger.error(f"Error logging dangerous operation: {e}")


def execute_backend_dangerous_operation(operation_id, operation_data):
    """
    Ejecuta la operación peligrosa en el backend con rate limiting y timeouts específicos.

    Timeouts configurados:
    - vacuum_database, reindex_tables: 60s (operaciones pesadas)
    - delete_all_sales_data, clean_catalog_full: 30s (operaciones de eliminación)
    """
    try:
        # Issue #330: Endpoint unificado admin-tools
        # Todas las operaciones peligrosas ahora usan el mismo patrón REST
        endpoint = f"/api/v1/admin-tools/dangerous-operations/{operation_id}/execute"

        # Timeouts específicos por tipo de operación
        timeout_map = {
            "delete_all_sales_data": 30,
            "clean_catalog_full": 30,
            "vacuum_database": 60,
            "reindex_tables": 60,
        }

        timeout = timeout_map.get(operation_id, 30)
        confirmation_text = operation_data.get("confirmation_text", "")

        # Usar request_coordinator para todas las operaciones con error handling mejorado
        response_data = request_coordinator.make_request(
            endpoint,
            method="POST",  # Ahora todos usan POST
            params={"confirmation_text": confirmation_text},  # Texto de confirmación en query params
            cache_ttl=0,
            bypass_cache=True,
            timeout=timeout,
            return_error_details=True,
        )

        # Manejar respuesta exitosa
        if response_data and not response_data.get("error"):
            logger.info(f"Operación exitosa: {operation_id}, resultado: {response_data}")
            return {
                "success": True,
                "data": response_data.get("data", {}),
                "message": response_data.get("message", "Operación completada"),
            }

        # Manejar errores estructurados
        if response_data and response_data.get("error"):
            error_type = response_data.get("error_type", "unknown")
            error_message = response_data.get("message", "Error desconocido")

            # Mensajes específicos por tipo de error
            if error_type == "timeout":
                user_message = f"Timeout: La operación está tardando más de {timeout} segundos. Por favor, verifique el estado del sistema."
            elif error_type == "unauthorized":
                user_message = "Sesión expirada. Por favor, inicie sesión nuevamente."
            elif error_type == "forbidden":
                user_message = "No tiene permisos suficientes para ejecutar esta operación."
            elif error_type == "rate_limit":
                retry_after = response_data.get("retry_after", 60)
                user_message = (
                    f"Rate limit excedido. Por favor espere {retry_after} segundos antes de intentar nuevamente."
                )
            elif error_type == "server":
                user_message = "Error del servidor. Por favor, intente nuevamente en unos momentos."
            elif error_type == "connection":
                user_message = "Error de conexión con el servidor. Verifique su conexión a internet."
            else:
                user_message = error_message

            return {"success": False, "message": user_message, "error_type": error_type}

        # Fallback para respuesta None (no debería ocurrir con return_error_details=True)
        return {"success": False, "message": "Error desconocido al procesar la operación"}

    except Exception as e:
        logger.error(f"Excepción no capturada en operación {operation_id}: {str(e)}")
        return {"success": False, "message": f"Error inesperado: {str(e)}"}


def create_success_alert(operation_id, result):
    """
    Crea un alert de éxito con detalles de la operación.
    """
    operation_names = {
        "delete_all_sales_data": "Eliminación de datos de ventas",
        "clean_catalog_full": "Limpieza del catálogo",
        "vacuum_database": "VACUUM de base de datos",
        "reindex_tables": "Reindexación de tablas",
    }

    operation_name = operation_names.get(operation_id, "Operación")

    return dbc.Alert(
        [
            html.I(className="fas fa-check-circle me-2"),
            html.Strong(f"{operation_name} completada. "),
            html.Br(),
            html.Small(
                ["ID de operación: ", result.get("data", {}).get("operation_id", "N/A")], className="text-muted"
            ),
            html.Br(),
            html.Small(
                ["Registrado en audit log: ", datetime.now().strftime("%Y-%m-%d %H:%M:%S")], className="text-muted"
            ),
        ],
        color="success",
        dismissable=True,
    )


def create_error_alert(message):
    """
    Crea un alert de error con el mensaje proporcionado.
    """
    return dbc.Alert(
        [html.I(className="fas fa-exclamation-triangle me-2"), html.Strong("Error: "), str(message)],
        color="danger",
        dismissable=True,
    )
