"""
Curated Interchangeable Groups Management Callbacks Module (Issue #521).

Responsabilidad: UI para gestión de grupos intercambiables curados.
Permite crear, editar, eliminar grupos y gestionar sus miembros.

ADR-004: Simplificación del Sistema de Clasificación
- Los grupos curados reemplazan el clustering dinámico
- Gestión manual desde Admin UI
"""

import logging
from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
from dash import ALL, Input, Output, State, ctx, dcc, html, no_update
from dash.exceptions import PreventUpdate

from components.toast_manager import success_toast, error_toast
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False

# L1 categories for filter dropdown
L1_CATEGORIES = [
    {"label": "Todas", "value": ""},
    {"label": "Dermofarmacia", "value": "dermofarmacia"},
    {"label": "Nutrición", "value": "nutricion"},
    {"label": "Higiene Bucal", "value": "higiene_bucal"},
    {"label": "Salud Sexual", "value": "salud_sexual"},
    {"label": "Control de Peso", "value": "control_peso"},
    {"label": "Vitaminas", "value": "vitaminas"},
    {"label": "Digestivo", "value": "digestivo"},
    {"label": "Dolor/Fiebre", "value": "dolor_fiebre"},
    {"label": "Sistema Inmune", "value": "sistema_inmune"},
    {"label": "Caída Cabello", "value": "caida_cabello"},
    {"label": "Infantil", "value": "infantil"},
]


def register_curated_groups_callbacks(app):
    """
    Register curated groups management callbacks for admin panel.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Curated groups callbacks already registered, skipping")
        return app

    logger.info("Registering curated groups callbacks")

    # =========================================================================
    # CALLBACK 1: Load Curated Groups Data
    # =========================================================================
    @app.callback(
        [
            Output("curated-groups-stats-container", "children"),
            Output("curated-groups-table-container", "children"),
            Output("curated-groups-pagination-info", "children"),
            Output("curated-groups-prev-page", "disabled"),
            Output("curated-groups-next-page", "disabled"),
            Output("curated-groups-l1-filter", "options"),
        ],
        [
            Input("curated-groups-tab-activated-trigger", "data"),
            Input("curated-groups-refresh-btn", "n_clicks"),
            Input("curated-groups-search-input", "value"),
            Input("curated-groups-l1-filter", "value"),
            Input("curated-groups-source-filter", "value"),
            Input("curated-groups-show-inactive", "value"),
            Input("curated-groups-pagination-store", "data"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_curated_groups_data(
        trigger_data, n_clicks, search_text, l1_filter, source_filter,
        show_inactive, pagination_data, auth_state, auth_tokens
    ):
        """
        Load curated groups data when:
        - Tab is activated (trigger_data)
        - Refresh button clicked (n_clicks)
        - Filters changed
        - Page changed

        REGLA #7.6: Multi-worker token restoration before API calls.
        """
        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[CURATED_GROUPS] User not authenticated - skipping")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[CURATED_GROUPS] No auth headers available - skipping API calls")
            raise PreventUpdate

        # Default pagination
        page = pagination_data.get("page", 1) if pagination_data else 1
        page_size = pagination_data.get("page_size", 20) if pagination_data else 20

        # Build query params
        params = {"page": page, "page_size": page_size}
        if search_text:
            params["search"] = search_text
        if l1_filter:
            params["necesidad_l1"] = l1_filter
        if source_filter:
            params["source"] = source_filter
        if not (show_inactive and "show_inactive" in show_inactive):
            params["is_active"] = True

        # Load groups
        groups_data = {"groups": [], "total": 0}
        stats_data = {}
        try:
            response = request_coordinator.make_request(
                "/api/v1/admin/curated-groups",
                method="GET",
                params=params,
                auth_headers=auth_headers,
            )
            if response and isinstance(response, dict):
                groups_data = response

            # Load stats
            stats_response = request_coordinator.make_request(
                "/api/v1/admin/curated-groups/stats",
                method="GET",
                auth_headers=auth_headers,
            )
            if stats_response and isinstance(stats_response, dict):
                stats_data = stats_response
        except Exception as e:
            logger.error(f"[CURATED_GROUPS] Error loading data: {e}")

        # Build stats badges
        stats_content = _build_stats_badges(stats_data)

        # Build table
        groups = groups_data.get("groups", [])
        table_content = _build_groups_table(groups)

        # Pagination info
        total = groups_data.get("total", 0)
        total_pages = (total + page_size - 1) // page_size if total > 0 else 1
        pagination_info = f"Página {page} de {total_pages} ({total} grupos)"

        # Disable pagination buttons
        prev_disabled = page <= 1
        next_disabled = page >= total_pages

        return (
            stats_content,
            table_content,
            pagination_info,
            prev_disabled,
            next_disabled,
            L1_CATEGORIES,
        )

    # =========================================================================
    # CALLBACK 2: Handle Pagination
    # =========================================================================
    @app.callback(
        Output("curated-groups-pagination-store", "data"),
        [
            Input("curated-groups-prev-page", "n_clicks"),
            Input("curated-groups-next-page", "n_clicks"),
        ],
        [State("curated-groups-pagination-store", "data")],
        prevent_initial_call=True,
    )
    def handle_curated_groups_pagination(prev_clicks, next_clicks, current_data):
        """Handle pagination button clicks."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
        page = current_data.get("page", 1) if current_data else 1
        page_size = current_data.get("page_size", 20) if current_data else 20

        if trigger_id == "curated-groups-prev-page" and page > 1:
            page -= 1
        elif trigger_id == "curated-groups-next-page":
            page += 1

        return {"page": page, "page_size": page_size}

    # =========================================================================
    # CALLBACK 3: Open Create Group Modal
    # =========================================================================
    @app.callback(
        [
            Output("curated-groups-modal", "is_open"),
            Output("curated-groups-modal-title", "children"),
            Output("curated-groups-form-name", "value"),
            Output("curated-groups-form-description", "value"),
            Output("curated-groups-form-l1", "value"),
            Output("curated-groups-form-l1", "options"),
            Output("curated-groups-form-l2", "value"),
            Output("curated-groups-edit-id", "data"),
        ],
        [
            Input("curated-groups-add-btn", "n_clicks"),
            Input("curated-groups-modal-cancel-btn", "n_clicks"),
            Input({"type": "curated-groups-edit-btn", "index": ALL}, "n_clicks"),
        ],
        [
            State("curated-groups-modal", "is_open"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def toggle_group_modal(
        add_clicks, cancel_clicks, edit_clicks,
        is_open, auth_state, auth_tokens
    ):
        """Open/close create/edit group modal."""
        from utils.auth_helpers import is_user_authenticated

        if not ctx.triggered:
            raise PreventUpdate

        trigger = ctx.triggered[0]
        trigger_id = trigger["prop_id"]

        # Close modal
        if "cancel" in trigger_id:
            return False, "Nuevo Grupo", "", "", "", L1_CATEGORIES[1:], "", None

        # Open for create
        if "add-btn" in trigger_id:
            return True, "Nuevo Grupo", "", "", "", L1_CATEGORIES[1:], "", None

        # Open for edit
        if "edit-btn" in trigger_id and trigger["value"]:
            # Extract group ID from pattern match
            import json
            prop_id = json.loads(trigger_id.replace(".n_clicks", ""))
            group_id = prop_id.get("index")

            if not group_id:
                raise PreventUpdate

            if not is_user_authenticated(auth_state):
                raise PreventUpdate

            # REGLA #7.6: Multi-Worker Token Restoration
            auth_headers = get_auth_headers_from_tokens(auth_tokens)
            if not auth_headers:
                raise PreventUpdate

            # Load group data
            try:
                response = request_coordinator.make_request(
                    f"/api/v1/admin/curated-groups/{group_id}",
                    method="GET",
                    auth_headers=auth_headers,
                )
                if response:
                    return (
                        True,
                        f"Editar: {response.get('group_name', '')}",
                        response.get("group_name", ""),
                        response.get("description", "") or "",
                        response.get("necesidad_l1", "") or "",
                        L1_CATEGORIES[1:],
                        response.get("subcategory_l2", "") or "",
                        group_id,
                    )
            except Exception as e:
                logger.error(f"[CURATED_GROUPS] Error loading group: {e}")

        raise PreventUpdate

    # =========================================================================
    # CALLBACK 4: Save Group (Create/Update)
    # =========================================================================
    @app.callback(
        [
            Output("curated-groups-modal", "is_open", allow_duplicate=True),
            Output("curated-groups-refresh-btn", "n_clicks"),
            Output("toast-container", "children", allow_duplicate=True),
        ],
        [Input("curated-groups-modal-save-btn", "n_clicks")],
        [
            State("curated-groups-form-name", "value"),
            State("curated-groups-form-description", "value"),
            State("curated-groups-form-l1", "value"),
            State("curated-groups-form-l2", "value"),
            State("curated-groups-edit-id", "data"),
            State("curated-groups-refresh-btn", "n_clicks"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def save_curated_group(
        save_clicks, name, description, l1, l2, edit_id,
        current_refresh, auth_state, auth_tokens
    ):
        """Save (create or update) a curated group."""
        if not save_clicks or not name:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            return no_update, no_update, error_toast("Error de autenticación")

        payload = {
            "group_name": name,
            "description": description or None,
            "necesidad_l1": l1 or None,
            "subcategory_l2": l2 or None,
        }

        try:
            if edit_id:
                # Update
                request_coordinator.make_request(
                    f"/api/v1/admin/curated-groups/{edit_id}",
                    method="PUT",
                    json=payload,
                    auth_headers=auth_headers,
                )
                toast = success_toast("Grupo actualizado correctamente")
            else:
                # Create
                request_coordinator.make_request(
                    "/api/v1/admin/curated-groups",
                    method="POST",
                    json=payload,
                    auth_headers=auth_headers,
                )
                toast = success_toast(f"Grupo '{name}' creado correctamente")

            # Close modal, refresh, and show toast
            return False, (current_refresh or 0) + 1, toast
        except Exception as e:
            logger.error(f"[CURATED_GROUPS] Error saving group: {e}")
            return no_update, no_update, error_toast(f"Error al guardar: {str(e)}")

    # =========================================================================
    # CALLBACK 5: Open Delete Confirmation
    # =========================================================================
    @app.callback(
        [
            Output("curated-groups-delete-confirm", "displayed"),
            Output("curated-groups-delete-group-id", "data"),
        ],
        [Input({"type": "curated-groups-delete-btn", "index": ALL}, "n_clicks")],
        prevent_initial_call=True,
    )
    def open_delete_confirmation(delete_clicks):
        """Open confirmation dialog before deleting."""
        if not ctx.triggered or not any(delete_clicks):
            raise PreventUpdate

        trigger = ctx.triggered[0]
        if not trigger["value"]:
            raise PreventUpdate

        import json
        prop_id = json.loads(trigger["prop_id"].replace(".n_clicks", ""))
        group_id = prop_id.get("index")

        if not group_id:
            raise PreventUpdate

        return True, group_id

    # =========================================================================
    # CALLBACK 5b: Execute Delete After Confirmation
    # =========================================================================
    @app.callback(
        [
            Output("curated-groups-refresh-btn", "n_clicks", allow_duplicate=True),
            Output("toast-container", "children", allow_duplicate=True),
        ],
        [Input("curated-groups-delete-confirm", "submit_n_clicks")],
        [
            State("curated-groups-delete-group-id", "data"),
            State("curated-groups-refresh-btn", "n_clicks"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def delete_curated_group_confirmed(
        submit_clicks, group_id, current_refresh, auth_state, auth_tokens
    ):
        """Delete a curated group after confirmation."""
        if not submit_clicks or not group_id:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            return no_update, error_toast("Error de autenticación")

        try:
            request_coordinator.make_request(
                f"/api/v1/admin/curated-groups/{group_id}",
                method="DELETE",
                auth_headers=auth_headers,
            )
            return (current_refresh or 0) + 1, success_toast("Grupo eliminado correctamente")
        except Exception as e:
            logger.error(f"[CURATED_GROUPS] Error deleting group: {e}")
            return no_update, error_toast(f"Error al eliminar: {str(e)}")

    # =========================================================================
    # CALLBACK 6: Toggle Group Active State
    # =========================================================================
    @app.callback(
        [
            Output("curated-groups-refresh-btn", "n_clicks", allow_duplicate=True),
            Output("toast-container", "children", allow_duplicate=True),
        ],
        [Input({"type": "curated-groups-toggle-btn", "index": ALL}, "n_clicks")],
        [
            State("curated-groups-refresh-btn", "n_clicks"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def toggle_curated_group(toggle_clicks, current_refresh, auth_state, auth_tokens):
        """Toggle active/inactive state of a curated group."""
        if not ctx.triggered or not any(toggle_clicks):
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        trigger = ctx.triggered[0]
        if not trigger["value"]:
            raise PreventUpdate

        import json
        prop_id = json.loads(trigger["prop_id"].replace(".n_clicks", ""))
        group_id = prop_id.get("index")

        if not group_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            return no_update, error_toast("Error de autenticación")

        try:
            response = request_coordinator.make_request(
                f"/api/v1/admin/curated-groups/{group_id}/toggle",
                method="POST",
                auth_headers=auth_headers,
            )
            new_state = "activado" if response.get("is_active") else "desactivado"
            return (current_refresh or 0) + 1, success_toast(f"Grupo {new_state}")
        except Exception as e:
            logger.error(f"[CURATED_GROUPS] Error toggling group: {e}")
            return no_update, error_toast(f"Error al cambiar estado: {str(e)}")

    # =========================================================================
    # CALLBACK 7: Open Members Modal
    # =========================================================================
    @app.callback(
        [
            Output("curated-members-modal", "is_open"),
            Output("curated-members-modal-title", "children"),
            Output("curated-members-group-info", "children"),
            Output("curated-members-list-container", "children"),
            Output("curated-groups-selected-group-store", "data"),
        ],
        [
            Input({"type": "curated-groups-members-btn", "index": ALL}, "n_clicks"),
            Input("curated-members-modal-close-btn", "n_clicks"),
        ],
        [
            State("curated-groups-selected-group-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def toggle_members_modal(
        members_clicks, close_clicks,
        selected_group, auth_state, auth_tokens
    ):
        """Open/close members management modal."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger = ctx.triggered[0]
        trigger_id = trigger["prop_id"]

        # Close modal
        if "close-btn" in trigger_id:
            return False, "Miembros del Grupo", "", [], None

        # Open modal
        if "members-btn" in trigger_id and trigger["value"]:
            import json
            prop_id = json.loads(trigger_id.replace(".n_clicks", ""))
            group_id = prop_id.get("index")

            if not group_id:
                raise PreventUpdate

            if not is_user_authenticated(auth_state):
                raise PreventUpdate

            # REGLA #7.6: Multi-Worker Token Restoration
            auth_headers = get_auth_headers_from_tokens(auth_tokens)
            if not auth_headers:
                raise PreventUpdate

            try:
                response = request_coordinator.make_request(
                    f"/api/v1/admin/curated-groups/{group_id}",
                    method="GET",
                    auth_headers=auth_headers,
                )
                if response:
                    group_info = _build_group_info(response)
                    members_list = _build_members_list(response.get("members", []), group_id)
                    return (
                        True,
                        f"Miembros: {response.get('group_name', '')}",
                        group_info,
                        members_list,
                        group_id,
                    )
            except Exception as e:
                logger.error(f"[CURATED_GROUPS] Error loading members: {e}")

        raise PreventUpdate

    # =========================================================================
    # CALLBACK 8: Add Member
    # =========================================================================
    @app.callback(
        [
            Output("curated-members-list-container", "children", allow_duplicate=True),
            Output("curated-members-form-ean", "value"),
            Output("curated-members-form-cn", "value"),
            Output("curated-members-form-name", "value"),
        ],
        [Input("curated-members-add-btn", "n_clicks")],
        [
            State("curated-members-form-ean", "value"),
            State("curated-members-form-cn", "value"),
            State("curated-members-form-name", "value"),
            State("curated-groups-selected-group-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def add_member(
        add_clicks, ean, cn, name,
        group_id, auth_state, auth_tokens
    ):
        """Add a member to the selected group."""
        if not add_clicks or not group_id:
            raise PreventUpdate

        if not ean and not cn:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            raise PreventUpdate

        payload = {
            "ean13": ean or None,
            "codigo_nacional": cn or None,
            "product_name": name or None,
        }

        try:
            request_coordinator.make_request(
                f"/api/v1/admin/curated-groups/{group_id}/members",
                method="POST",
                json=payload,
                auth_headers=auth_headers,
            )

            # Reload members
            response = request_coordinator.make_request(
                f"/api/v1/admin/curated-groups/{group_id}",
                method="GET",
                auth_headers=auth_headers,
            )
            if response:
                members_list = _build_members_list(response.get("members", []), group_id)
                return members_list, "", "", ""

        except Exception as e:
            logger.error(f"[CURATED_GROUPS] Error adding member: {e}")

        raise PreventUpdate

    # =========================================================================
    # CALLBACK 9: Remove Member
    # =========================================================================
    @app.callback(
        Output("curated-members-list-container", "children", allow_duplicate=True),
        [Input({"type": "curated-members-remove-btn", "index": ALL}, "n_clicks")],
        [
            State("curated-groups-selected-group-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def remove_member(remove_clicks, group_id, auth_state, auth_tokens):
        """Remove a member from the selected group."""
        if not ctx.triggered or not any(remove_clicks) or not group_id:
            raise PreventUpdate

        trigger = ctx.triggered[0]
        if not trigger["value"]:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        import json
        prop_id = json.loads(trigger["prop_id"].replace(".n_clicks", ""))
        member_id = prop_id.get("index")

        if not member_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            raise PreventUpdate

        try:
            request_coordinator.make_request(
                f"/api/v1/admin/curated-groups/{group_id}/members/{member_id}",
                method="DELETE",
                auth_headers=auth_headers,
            )

            # Reload members
            response = request_coordinator.make_request(
                f"/api/v1/admin/curated-groups/{group_id}",
                method="GET",
                auth_headers=auth_headers,
            )
            if response:
                return _build_members_list(response.get("members", []), group_id)

        except Exception as e:
            logger.error(f"[CURATED_GROUPS] Error removing member: {e}")

        raise PreventUpdate

    # Mark as registered
    _module_callbacks_registered = True
    logger.info("Curated groups callbacks registered successfully")

    return app


# =============================================================================
# HELPER FUNCTIONS
# =============================================================================

def _build_stats_badges(stats: Dict[str, Any]) -> dbc.Row:
    """Build stats badges row."""
    return dbc.Row([
        dbc.Col([
            dbc.Badge(
                f"Total: {stats.get('total_groups', 0)}",
                color="primary",
                className="me-2",
            ),
            dbc.Badge(
                f"Activos: {stats.get('active_groups', 0)}",
                color="success",
                className="me-2",
            ),
            dbc.Badge(
                f"Legacy: {stats.get('legacy_groups', 0)}",
                color="secondary",
                className="me-2",
            ),
            dbc.Badge(
                f"Manual: {stats.get('manual_groups', 0)}",
                color="info",
                className="me-2",
            ),
            dbc.Badge(
                f"Productos: {stats.get('total_products', 0):,}",
                color="warning",
                className="me-2",
            ),
            dbc.Badge(
                f"Marcas: {stats.get('total_brands', 0)}",
                color="dark",
            ),
        ], width=12),
    ], className="mb-3")


def _build_groups_table(groups: List[Dict[str, Any]]) -> html.Div:
    """Build groups table."""
    if not groups:
        return dbc.Alert(
            "No hay grupos configurados. Haz clic en 'Nuevo Grupo' para crear uno.",
            color="info",
        )

    rows = []
    for group in groups:
        status_badge = dbc.Badge(
            "Activo" if group.get("is_active") else "Inactivo",
            color="success" if group.get("is_active") else "secondary",
        )
        source_badge = dbc.Badge(
            "Manual" if group.get("source") == "manual_curated" else "Legacy",
            color="info" if group.get("source") == "manual_curated" else "secondary",
        )

        rows.append(
            html.Tr([
                html.Td([
                    html.Strong(group.get("group_name", "")),
                    html.Br(),
                    html.Small(group.get("description", "") or "", className="text-muted"),
                ]),
                html.Td(group.get("necesidad_l1", "-") or "-"),
                html.Td(f"{group.get('product_count', 0)}"),
                html.Td(f"{group.get('brand_count', 0)}"),
                html.Td(source_badge),
                html.Td(status_badge),
                html.Td([
                    dbc.Button(
                        html.I(className="fas fa-users"),
                        color="primary",
                        size="sm",
                        className="me-1",
                        id={"type": "curated-groups-members-btn", "index": str(group.get("id"))},
                        title="Gestionar miembros",
                    ),
                    dbc.Button(
                        html.I(className="fas fa-edit"),
                        color="warning",
                        size="sm",
                        className="me-1",
                        id={"type": "curated-groups-edit-btn", "index": str(group.get("id"))},
                        title="Editar",
                    ),
                    dbc.Button(
                        html.I(className="fas fa-toggle-on" if group.get("is_active") else "fas fa-toggle-off"),
                        color="info" if group.get("is_active") else "secondary",
                        size="sm",
                        className="me-1",
                        id={"type": "curated-groups-toggle-btn", "index": str(group.get("id"))},
                        title="Toggle activo/inactivo",
                    ),
                    dbc.Button(
                        html.I(className="fas fa-trash"),
                        color="danger",
                        size="sm",
                        id={"type": "curated-groups-delete-btn", "index": str(group.get("id"))},
                        title="Eliminar",
                    ),
                ], className="d-flex"),
            ])
        )

    return dbc.Table(
        [
            html.Thead(
                html.Tr([
                    html.Th("Grupo"),
                    html.Th("Categoría L1"),
                    html.Th("Productos"),
                    html.Th("Marcas"),
                    html.Th("Origen"),
                    html.Th("Estado"),
                    html.Th("Acciones"),
                ])
            ),
            html.Tbody(rows),
        ],
        bordered=True,
        hover=True,
        responsive=True,
        size="sm",
    )


def _build_group_info(group: Dict[str, Any]) -> html.Div:
    """Build group info section for members modal."""
    return dbc.Alert(
        [
            html.Strong(group.get("group_name", "")),
            html.Span(" - "),
            html.Span(f"Categoría: {group.get('necesidad_l1', '-') or '-'}"),
            html.Span(" | "),
            html.Span(f"{group.get('product_count', 0)} productos"),
            html.Span(" | "),
            html.Span(f"{group.get('brand_count', 0)} marcas"),
        ],
        color="info",
        className="mb-3",
    )


def _build_members_list(members: List[Dict[str, Any]], group_id: str) -> html.Div:
    """Build members list for modal."""
    if not members:
        return dbc.Alert(
            "Este grupo no tiene miembros. Añade productos usando el formulario de arriba.",
            color="warning",
        )

    rows = []
    for member in members:
        rows.append(
            html.Tr([
                html.Td(member.get("ean13", "-") or "-"),
                html.Td(member.get("codigo_nacional", "-") or "-"),
                html.Td(member.get("product_name", "-") or "-"),
                html.Td(member.get("detected_brand", "-") or "-"),
                html.Td([
                    dbc.Button(
                        html.I(className="fas fa-trash"),
                        color="danger",
                        size="sm",
                        id={"type": "curated-members-remove-btn", "index": str(member.get("id"))},
                        title="Eliminar miembro",
                    ),
                ]),
            ])
        )

    return dbc.Table(
        [
            html.Thead(
                html.Tr([
                    html.Th("EAN13"),
                    html.Th("Cód. Nacional"),
                    html.Th("Nombre"),
                    html.Th("Marca"),
                    html.Th("Acciones"),
                ])
            ),
            html.Tbody(rows),
        ],
        bordered=True,
        hover=True,
        responsive=True,
        size="sm",
    )
