"""
Classification Callbacks Module (Issue #449).

Responsabilidad: UI para Human-in-the-Loop del clasificador NECESIDAD.
Permite aprobar, corregir, marcar outlier o saltar clasificaciones.
"""

import logging
from datetime import datetime

import dash_bootstrap_components as dbc
from dash import Input, Output, State, ctx, html, ALL
from dash.exceptions import PreventUpdate

from utils.auth import auth_manager
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_classification_callbacks(app):
    """
    Register classification callbacks for admin panel.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Classification callbacks already registered, skipping")
        return app

    logger.info("Registering classification callbacks")

    # =========================================================================
    # CALLBACK 1: Load Classification Data on Tab Activation or Refresh
    # =========================================================================
    @app.callback(
        [
            Output("classification-coverage-pct", "children"),
            Output("classification-p1-count", "children"),
            Output("classification-p2-count", "children"),
            Output("classification-p3-count", "children", allow_duplicate=True),
            Output("classification-accuracy-pct", "children"),
            Output("classification-last-update", "children"),
            Output("classification-table-container", "children"),
            Output("classification-pagination-info", "children"),
            Output("classification-current-page", "children"),
            Output("classification-prev-page", "disabled"),
            Output("classification-next-page", "disabled"),
            Output("classification-queue-store", "data"),
            Output("classification-categories-store", "data"),
        ],
        [
            Input("classification-tab-activated-trigger", "data"),
            Input("classification-refresh-btn", "n_clicks"),
            Input("classification-priority-filter", "value"),
            Input("classification-page-store", "data"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_classification_data(trigger_data, n_clicks, priority_filter, current_page, auth_state, auth_tokens):
        """
        Load classification queue and stats when:
        - Tab is activated (trigger_data)
        - Refresh button clicked (n_clicks)
        - Priority filter changed
        - Page changed

        REGLA #7.6: Multi-worker token restoration before API calls.
        """
        from utils.auth_helpers import is_user_authenticated

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[CLASSIFICATION] User not authenticated - skipping")
            raise PreventUpdate

        # REGLA #7.6: Restore tokens for multi-worker
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        # Default values
        page = current_page or 1
        limit = 10
        offset = (page - 1) * limit
        priority = priority_filter if priority_filter and priority_filter != "all" else None

        # Load stats
        stats_data = {}
        try:
            stats_response = request_coordinator.make_request(
                "/api/v1/feedback/stats",
                method="GET",
            )
            # API returns CorrectionStats model directly, not wrapped
            if stats_response and isinstance(stats_response, dict):
                stats_data = stats_response
        except Exception as e:
            logger.error(f"[CLASSIFICATION] Error loading stats: {e}")

        # Load categories (only once, cache in store)
        categories_data = []
        try:
            cats_response = request_coordinator.make_request(
                "/api/v1/feedback/categories",
                method="GET",
            )
            # API returns list directly, not wrapped in {success, data}
            if cats_response and isinstance(cats_response, list):
                categories_data = cats_response
            elif cats_response and isinstance(cats_response, dict):
                categories_data = cats_response.get("data", [])
        except Exception as e:
            logger.error(f"[CLASSIFICATION] Error loading categories: {e}")

        # Load review queue
        queue_data = {"items": [], "total": 0, "breakdown": {}}
        try:
            params = {"limit": limit, "offset": offset}
            if priority:
                params["priority"] = priority.upper()

            queue_response = request_coordinator.make_request(
                "/api/v1/feedback/review-queue",
                method="GET",
                params=params,
            )
            # API returns ReviewQueueResponse model directly, not wrapped
            if queue_response and isinstance(queue_response, dict):
                queue_data = queue_response
        except Exception as e:
            logger.error(f"[CLASSIFICATION] Error loading queue: {e}")

        # Calculate KPIs
        breakdown = queue_data.get("breakdown", {})
        total_products = stats_data.get("total_products", 0)
        total_classified = stats_data.get("total_classified", 0)
        coverage_pct = f"{(total_classified / total_products * 100):.1f}%" if total_products > 0 else "--"

        p1_count = breakdown.get("p1_otros", 0)
        p2_count = breakdown.get("p2_low_confidence", 0)
        p3_count = breakdown.get("p3_pending", 0)

        # Accuracy from stats
        total_corrections = stats_data.get("total_corrections", 0)
        approved_count = stats_data.get("approved", 0)
        accuracy_pct = f"{(approved_count / total_corrections * 100):.1f}%" if total_corrections > 0 else "--"

        # Build table
        items = queue_data.get("items", [])
        total_items = queue_data.get("total", 0)
        total_pages = (total_items + limit - 1) // limit if total_items > 0 else 1

        if items:
            table_content = _build_classification_table(items, categories_data)
        else:
            table_content = dbc.Alert(
                html.Span([
                    html.I(className="fas fa-check-circle me-2 text-success"),
                    "No hay productos pendientes de clasificar.",
                ]),
                color="success",
                className="text-center py-4",
            )

        # Pagination info
        pagination_info = f"Mostrando {len(items)} de {total_items} productos"
        prev_disabled = page <= 1
        next_disabled = page >= total_pages

        # Last update timestamp
        last_update = f"Actualizado: {datetime.now().strftime('%H:%M:%S')}"

        return (
            coverage_pct,
            str(p1_count),
            str(p2_count),
            str(p3_count),
            accuracy_pct,
            last_update,
            table_content,
            pagination_info,
            str(page),
            prev_disabled,
            next_disabled,
            queue_data,
            categories_data,
        )

    # =========================================================================
    # CALLBACK 2: Pagination
    # =========================================================================
    @app.callback(
        Output("classification-page-store", "data"),
        [
            Input("classification-prev-page", "n_clicks"),
            Input("classification-next-page", "n_clicks"),
        ],
        [
            State("classification-page-store", "data"),
            State("classification-queue-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_pagination(prev_clicks, next_clicks, current_page, queue_data):
        """Handle pagination button clicks."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
        page = current_page or 1

        total = queue_data.get("total", 0) if queue_data else 0
        limit = 10
        total_pages = (total + limit - 1) // limit if total > 0 else 1

        if trigger_id == "classification-prev-page" and page > 1:
            return page - 1
        elif trigger_id == "classification-next-page" and page < total_pages:
            return page + 1

        raise PreventUpdate

    # =========================================================================
    # CALLBACK 3: Handle Actions (Approve, Correct, Outlier, Skip)
    # =========================================================================
    @app.callback(
        Output("classification-refresh-btn", "n_clicks"),
        [
            Input({"type": "classification-approve-btn", "index": ALL}, "n_clicks"),
            Input({"type": "classification-outlier-btn", "index": ALL}, "n_clicks"),
            Input({"type": "classification-skip-btn", "index": ALL}, "n_clicks"),
            Input({"type": "classification-category-dropdown", "index": ALL}, "value"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("classification-refresh-btn", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def handle_classification_actions(
        approve_clicks, outlier_clicks, skip_clicks, category_values,
        auth_state, auth_tokens, current_refresh_clicks
    ):
        """
        Handle classification actions via pattern matching callbacks.

        Actions:
        - Approve: Confirm ML classification is correct
        - Outlier: Mark product as outlier (doesn't fit categories)
        - Skip: Skip for later review
        - Correct: When dropdown value changes, correct classification
        """
        from utils.auth_helpers import is_user_authenticated
        from components.toast_manager import success_toast, error_toast

        if not ctx.triggered:
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Restore tokens for multi-worker
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        trigger = ctx.triggered[0]
        trigger_id = trigger["prop_id"]
        trigger_value = trigger["value"]

        # Skip if no actual click/value
        if trigger_value is None:
            raise PreventUpdate

        # Parse the trigger to get action type and index
        try:
            import json
            # Remove the ".n_clicks" or ".value" suffix
            id_part = trigger_id.rsplit(".", 1)[0]
            id_dict = json.loads(id_part)
            action_type = id_dict.get("type", "")
            enrichment_id = id_dict.get("index", "")
        except (json.JSONDecodeError, KeyError):
            raise PreventUpdate

        if not enrichment_id:
            raise PreventUpdate

        # Determine action and make API call
        success = False
        message = ""

        try:
            if action_type == "classification-approve-btn":
                response = request_coordinator.make_request(
                    "/api/v1/feedback/approve",
                    method="POST",
                    data={"sales_enrichment_id": enrichment_id},
                )
                if response and response.get("success"):
                    success = True
                    message = "Clasificación aprobada"

            elif action_type == "classification-outlier-btn":
                response = request_coordinator.make_request(
                    "/api/v1/feedback/outlier",
                    method="POST",
                    data={
                        "sales_enrichment_id": enrichment_id,
                        "outlier_reason": "ambiguo",
                    },
                )
                if response and response.get("success"):
                    success = True
                    message = "Marcado como outlier"

            elif action_type == "classification-skip-btn":
                response = request_coordinator.make_request(
                    "/api/v1/feedback/skip",
                    method="POST",
                    data={"sales_enrichment_id": enrichment_id},
                )
                if response and response.get("success"):
                    success = True
                    message = "Producto saltado"

            elif action_type == "classification-category-dropdown":
                # Correct classification with new category
                if trigger_value:
                    response = request_coordinator.make_request(
                        "/api/v1/feedback/correct",
                        method="POST",
                        data={
                            "sales_enrichment_id": enrichment_id,
                            "corrected_category": trigger_value,
                        },
                    )
                    if response and response.get("success"):
                        success = True
                        message = f"Corregido a: {trigger_value}"

        except Exception as e:
            logger.error(f"[CLASSIFICATION] Action error: {e}")
            message = f"Error: {str(e)}"

        # Trigger refresh by incrementing n_clicks
        new_refresh = (current_refresh_clicks or 0) + 1

        return new_refresh

    _module_callbacks_registered = True
    logger.info("Classification callbacks registered successfully")
    return app


def _build_classification_table(items: list, categories: list) -> html.Div:
    """
    Build the classification table with action buttons.

    Args:
        items: List of products from review queue
        categories: List of available categories for dropdown

    Returns:
        html.Div containing the table
    """
    # Build category options for dropdown
    category_options = [{"label": cat.get("name", cat.get("slug", "")), "value": cat.get("slug", "")} for cat in categories]

    rows = []
    for item in items:
        enrichment_id = item.get("sales_enrichment_id", item.get("id", ""))
        product_name = item.get("product_name", item.get("descripcion", "Sin nombre"))
        current_category = item.get("predicted_category", item.get("ml_category", "--"))
        confidence = item.get("confidence_score", item.get("ml_confidence", 0))
        priority = item.get("priority", "P3")

        # Priority badge color
        priority_color = {
            "P1": "danger",
            "P2": "warning",
            "P3": "secondary",
        }.get(priority, "secondary")

        # Confidence color
        conf_value = float(confidence) if confidence else 0
        conf_color = "success" if conf_value >= 0.7 else "warning" if conf_value >= 0.5 else "danger"

        row = html.Tr([
            # Product name + priority
            html.Td([
                dbc.Badge(priority, color=priority_color, className="me-2"),
                html.Span(product_name[:50] + "..." if len(product_name) > 50 else product_name),
            ]),
            # Current category
            html.Td(
                dbc.Badge(current_category or "--", color="info", className="text-truncate"),
                style={"maxWidth": "120px"},
            ),
            # Category dropdown for correction
            html.Td(
                dbc.Select(
                    id={"type": "classification-category-dropdown", "index": str(enrichment_id)},
                    options=[{"label": "-- Seleccionar --", "value": ""}] + category_options,
                    value="",
                    size="sm",
                    style={"minWidth": "150px"},
                ),
            ),
            # Confidence
            html.Td(
                dbc.Badge(
                    f"{conf_value:.0%}" if conf_value else "--",
                    color=conf_color,
                ),
                className="text-center",
            ),
            # Action buttons
            html.Td(
                dbc.ButtonGroup([
                    dbc.Button(
                        html.I(className="fas fa-check"),
                        id={"type": "classification-approve-btn", "index": str(enrichment_id)},
                        color="success",
                        size="sm",
                        outline=True,
                        title="Aprobar clasificación actual",
                    ),
                    dbc.Button(
                        html.I(className="fas fa-times"),
                        id={"type": "classification-outlier-btn", "index": str(enrichment_id)},
                        color="danger",
                        size="sm",
                        outline=True,
                        title="Marcar como outlier",
                    ),
                    dbc.Button(
                        html.I(className="fas fa-forward"),
                        id={"type": "classification-skip-btn", "index": str(enrichment_id)},
                        color="warning",
                        size="sm",
                        outline=True,
                        title="Saltar para después",
                    ),
                ], size="sm"),
                className="text-center",
            ),
        ])
        rows.append(row)

    return dbc.Table(
        [
            html.Thead(
                html.Tr([
                    html.Th("Producto", style={"width": "35%"}),
                    html.Th("Categoría Actual", style={"width": "15%"}),
                    html.Th("Nueva Categoría", style={"width": "20%"}),
                    html.Th("Confianza", style={"width": "10%"}, className="text-center"),
                    html.Th("Acciones", style={"width": "20%"}, className="text-center"),
                ])
            ),
            html.Tbody(rows),
        ],
        bordered=True,
        hover=True,
        responsive=True,
        size="sm",
        className="mb-0",
    )
