"""
Catalog Management Callbacks Module.
Responsabilidad: Gestión completa de catálogo CIMA/Nomenclator.
"""

import logging
from datetime import datetime, timezone
from typing import Any, Optional, Tuple

import dash_bootstrap_components as dbc
from components.catalog_sync_progress import create_catalog_sync_progress_card
from dash import Input, Output, State, ctx, html
from dash.exceptions import PreventUpdate
from utils.config import BACKEND_URL
from utils.helpers import format_number
from utils.request_coordinator import request_coordinator
from utils.auth_helpers import get_auth_headers_from_tokens  # REGLA #7.6: Multi-worker token restoration

logger = logging.getLogger(__name__)

# Log prefix para mensajes estructurados
LOG_PREFIX = "[CATALOG_STATS]"

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


# Polling intervals (Issue #237, Issue #277, Issue #283 - Phase 2)
POLLING_INTERVAL_ACTIVE = 5000  # ms - fast polling during active sync
POLLING_INTERVAL_IDLE = 300000  # ms - Issue #283: 5 minutes when idle (reduce load 60x)
COUNTDOWN_INTERVAL = 1000  # ms - countdown ticker update frequency

# Request timeouts (Issue #292 - PR Review)
CATALOG_STATS_TIMEOUT_SECONDS = 10  # Timeout para requests de catalog stats

# Sync type labels for UI
SYNC_LABELS = {"cima": "CIMA", "nomenclator": "Nomenclátor", "all": "Completo"}


def fetch_initial_catalog_stats():
    """
    Carga inicial síncrona de estadísticas de catálogo (Issue #283 - Phase 1).
    Llamada al inicio del layout para pre-poblar el store y evitar espera de 5-10s.

    Returns:
        dict: Estadísticas del catálogo o None si falla
    """
    try:
        import requests
        from utils.auth import auth_manager

        token = auth_manager.get_access_token()

        if not token:
            logger.warning("No auth token available for initial catalog stats fetch")
            return None

        response = requests.get(
            f"{BACKEND_URL}/api/v1/admin/catalog/stats",
            headers={"Authorization": f"Bearer {token}"},
            timeout=CATALOG_STATS_TIMEOUT_SECONDS,
        )

        if response.status_code == 200:
            logger.info("Initial catalog stats fetched successfully")
            return response.json()
        else:
            logger.error(f"Error fetching initial catalog stats: {response.status_code}")
            return None

    except Exception as e:
        logger.error(f"Exception fetching initial catalog stats: {e}")
        return None


# Constantes para mensajes de estado (Issue #200 - Review suggestion #5)
class CatalogStatusMessages:
    """
    Mensajes de estado centralizados para el catálogo.
    Facilita mantenimiento y futura internacionalización.
    """

    # Mensajes de error
    ERROR_WITH_DETAILS = "Error: {message}"
    ERROR_NO_DETAILS = "Error: Sincronización fallida (ver logs para detalles)"

    # Mensajes de NEVER_RUN
    NEVER_RUN_WITH_PRODUCTS = "Datos cargados (sin registro de sync)"
    NEVER_RUN_NO_PRODUCTS = "Nunca sincronizado"

    # Mensajes de estado activo
    READY = "Sincronizado"
    UPDATING = "Sincronizando..."
    INITIALIZING = "Inicializando..."

    # Fallback
    UNKNOWN_STATUS = "Estado desconocido"

    # Timestamps
    TIMESTAMP_INITIAL_LOAD = "Carga inicial (sin timestamp)"
    TIMESTAMP_NEVER = "Nunca"
    TIMESTAMP_ERROR = "Error"
    TIMESTAMP_INVALID_DATE = "Fecha inválida"


def _should_process_catalog_stats(
    trigger_id: str, pathname: str, active_tab: str, auth_state: dict
) -> Tuple[bool, str]:
    """
    Centraliza la lógica de guards para el callback de catalog stats.

    Args:
        trigger_id: ID del componente que disparó el callback
        pathname: Pathname actual de la URL
        active_tab: Tab activo en el panel de admin
        auth_state: Estado de autenticación del usuario

    Returns:
        Tuple[bool, str]: (should_process, reason)
            - should_process: True si debe continuar, False si debe PreventUpdate
            - reason: Razón de la decisión ("proceed", "not_admin_page", "skeleton_needed",
                     "wrong_tab", "not_authenticated")
    """
    # Guard 1: Verificar trigger desde URL pathname
    if trigger_id == "url":
        if pathname != "/admin":
            return False, "not_admin_page"
        # En /admin pero sin tab activo → Retornar skeleton
        # Issue #485: Tab renombrado de "database" a "catalogs"
        if not active_tab or active_tab != "catalogs":
            return True, "skeleton_needed"

    # Guard 2: Verificar trigger desde interval (incluye reset automático al cambiar de tab)
    elif trigger_id == "admin-stats-interval":
        # Issue #485: Tab renombrado de "database" a "catalogs"
        if not active_tab or active_tab != "catalogs":
            return False, "wrong_tab"

    # Guard 3: Verificar autenticación (Issue #302)
    from utils.auth_helpers import is_user_authenticated

    if not is_user_authenticated(auth_state):
        return True, "not_authenticated"

    # ✅ TODO OK: Proceder con request API
    return True, "proceed"


def register_catalog_management_callbacks(app):
    """
    Register catalog management related callbacks.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Catalog management callbacks already registered, skipping")
        return app

    logger.info("Registering catalog management callbacks")

    @app.callback(
        [
            # Sección 1: Catálogos Externos
            Output("admin-cima-count", "children"),
            Output("admin-cima-last-sync", "children"),
            Output("admin-cima-status", "children"),
            Output("admin-cima-status", "className"),
            Output("admin-nomenclator-count", "children"),
            Output("admin-nomenclator-last-sync", "children"),
            Output("admin-nomenclator-status", "children"),
            Output("admin-nomenclator-status", "className"),
            # Sección 2: Enriquecimiento de Ventas (condicional)
            Output("admin-sales-enrichment-section", "children"),
            # Sección 3: Última Actividad
            Output("admin-activity-cima-sync", "children"),
            Output("admin-activity-nomenclator-sync", "children"),
            Output("admin-activity-catalog-update", "children"),
            # Issue #237 - Fase 1: Feedback Visual
            Output("admin-last-update-timestamp", "children"),
            Output("admin-sync-progress-bar-container", "children"),
            # Issue #237 - Fase 2: Polling Inteligente
            Output("admin-stats-interval", "interval"),
            Output("admin-last-poll-time", "data"),
        ],
        [
            Input("url", "pathname"),  # Trigger inicial (carga inmediata)
            Input("admin-stats-interval", "n_intervals")  # Trigger recurrente (polling + reset al cambiar tab)
        ],
        [
            State("admin-last-poll-time", "data"),
            State("admin-tabs", "value"),
            State("auth-state", "data"),  # Auth guard (Issue #302)
            State("auth-context-sync", "data"),  # Token sync (Issue #187) - CRÍTICO para evitar 401s
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=False,  # Permitir ejecución inicial
    )
    def update_catalog_stats_display(pathname, n_intervals, last_poll_time, active_tab, auth_state, auth_context_sync, auth_tokens):
        """
        Actualiza las estadísticas del catálogo (Issue #200, #283, #302, #301).
        Patrón Dual Trigger: carga inmediata + polling con reset automático.

        Triggers:
        1. pathname: Carga inicial cuando usuario navega a /admin
        2. n_intervals: Polling cada 30s + reset automático al cambiar de tab (via admin_tabs.py)

        Mecánica de carga inmediata (Issue #301 - Fase 3):
        - admin_tabs.py resetea n_intervals=0 cuando se cambia al tab "catalogs"
        - Este reset dispara este callback inmediatamente (no espera 30s)
        - Resultado: datos cargan < 1s al cambiar de tab

        Guards:
        - Solo ejecuta si tab 'catalogs' está activo (Issue #485: tabs reorganizados)
        - Valida autenticación ANTES de hacer request API (Issue #302)
        """
        # Guard: Verificar qué trigger disparó el callback
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
        logger.debug(f"{LOG_PREFIX} Callback triggered by: {trigger_id} (active_tab: {active_tab})")

        # Guards centralizados: Validar condiciones de ejecución
        should_process, reason = _should_process_catalog_stats(
            trigger_id, pathname, active_tab, auth_state
        )

        if not should_process:
            logger.debug(f"{LOG_PREFIX} Skipping callback - reason: {reason}")
            raise PreventUpdate

        if reason in ["skeleton_needed", "not_authenticated"]:
            logger.debug(f"{LOG_PREFIX} Showing skeleton - reason: {reason}")
            return _get_skeleton_defaults()

        # CRÍTICO: Verificar que el token está sincronizado antes de hacer request API
        # Issue #187: Timing issue - callback se ejecuta antes de que auth_context reciba el token
        # Si auth_context_sync no está listo → mostrar skeleton y reintentar en próximo interval
        if not auth_context_sync or not auth_context_sync.get("synced"):
            logger.debug(f"{LOG_PREFIX} Token not synced yet - showing skeleton (will retry on next interval)")
            # Use ACTIVE interval (5s) to retry quickly instead of default (30s)
            return _get_skeleton_defaults(interval_override=POLLING_INTERVAL_ACTIVE)

        # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
        # Use centralized helper for token extraction (bypasses singleton limitation)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        # Calculate timestamp once at the start for consistent time reference
        now = datetime.now(timezone.utc)

        # Fetch stats using request_coordinator (centralized error handling)
        # Note: request_coordinator.make_request() returns the parsed JSON dict directly (not Response object)
        stats = None

        try:
            logger.info(f"{LOG_PREFIX} Fetching stats (trigger: {trigger_id})")

            stats = request_coordinator.make_request(
                "/api/v1/admin/catalog/stats",
                method="GET",
                cache_ttl=0,  # Disable frontend cache - backend has 5min Redis cache to prevent DB overload
                              # Fresh data needed for immediate load on navigation (dual trigger pattern)
                timeout=CATALOG_STATS_TIMEOUT_SECONDS,
                auth_headers=auth_headers,  # REGLA #7.6: Explicit headers for multi-worker
            )

            # request_coordinator.make_request() returns:
            # - Dict with data if success (response.json() already parsed)
            # - None if error (with logging already handled)
            if stats:
                logger.info(f"{LOG_PREFIX} Stats loaded successfully (trigger: {trigger_id})")
            else:
                logger.error(f"{LOG_PREFIX} Error: request_coordinator returned None (trigger: {trigger_id})")

        except Exception as e:
            logger.error(f"{LOG_PREFIX} Exception: {e} (trigger: {trigger_id})")
            stats = None

        # Procesar datos si están disponibles
        if stats:
            # SECCIÓN 1: CATÁLOGOS EXTERNOS
            external_catalogs = stats.get("external_catalogs", {})
            cima = external_catalogs.get("cima", {})
            nomenclator = external_catalogs.get("nomenclator", {})

            # CIMA
            cima_products_count = cima.get("products_count", 0)
            cima_count = format_number(cima_products_count)
            cima_last_sync = _format_timestamp(cima.get("last_sync"), has_products=(cima_products_count > 0))
            cima_status_text = _format_status_message(
                cima.get("status"), cima.get("message", "Desconocido"), has_products=(cima_products_count > 0)
            )
            cima_status_class = _get_status_class(cima.get("status"))

            # Nomenclátor
            nomenclator_products_count = nomenclator.get("products_count", 0)
            nomenclator_count = format_number(nomenclator_products_count)
            nomenclator_last_sync = _format_timestamp(
                nomenclator.get("last_sync"), has_products=(nomenclator_products_count > 0)
            )
            nomenclator_status_text = _format_status_message(
                nomenclator.get("status"),
                nomenclator.get("message", "Desconocido"),
                has_products=(nomenclator_products_count > 0),
            )
            nomenclator_status_class = _get_status_class(nomenclator.get("status"))

            # SECCIÓN 2: ENRIQUECIMIENTO DE VENTAS
            sales_enrichment = stats.get("sales_enrichment")
            enrichment_section = _create_enrichment_section(sales_enrichment)

            # SECCIÓN 3: ÚLTIMA ACTIVIDAD
            last_activity = stats.get("last_activity", {})
            activity_cima = _format_timestamp(last_activity.get("cima_sync"))
            activity_nomenclator = _format_timestamp(last_activity.get("nomenclator_sync"))
            activity_catalog = _format_timestamp(last_activity.get("catalog_update"))

            # Issue #237 - Fase 1: Timestamp de última actualización
            last_update_text = now.strftime("%H:%M:%S")

            # Issue #237 - Fase 1: Progress bar contextual (solo durante sync)
            sync_in_progress = stats.get("sync_in_progress", False)
            sync_info = stats.get("sync_info", {})
            progress_bar = _create_sync_progress_bar(sync_in_progress, sync_info)

            # Issue #237 - Fase 2: Polling inteligente
            # Use constants for polling intervals
            new_interval = POLLING_INTERVAL_ACTIVE if sync_in_progress else POLLING_INTERVAL_IDLE

            return (
                # Sección 1: Catálogos Externos
                cima_count,
                cima_last_sync,
                cima_status_text,
                cima_status_class,
                nomenclator_count,
                nomenclator_last_sync,
                nomenclator_status_text,
                nomenclator_status_class,
                # Sección 2: Enriquecimiento
                enrichment_section,
                # Sección 3: Última Actividad
                activity_cima,
                activity_nomenclator,
                activity_catalog,
                # Issue #237 - Fase 1: Feedback Visual
                last_update_text,
                progress_bar,
                # Issue #237 - Fase 2: Polling Inteligente
                new_interval,
                now.isoformat(),
            )
        else:
            # Issue #292 PR Review: Manejo mejorado cuando stats es None
            # Proporcionar feedback claro al usuario y reintentar pronto
            logger.warning("No se pudieron cargar estadísticas del catálogo, reintentando...")

            # Mensaje user-friendly en lugar de "Error"
            error_message = "No disponible"
            error_class = "text-muted"

            # Alert informativo para la sección de enriquecimiento
            error_enrichment_section = dbc.Alert(
                [
                    html.I(className="fas fa-exclamation-triangle me-2"),
                    html.Strong("No se pudieron cargar las estadísticas del catálogo."),
                    html.Br(),
                    html.Small("Reintentando automáticamente en unos segundos...", className="text-muted"),
                ],
                color="warning",
                className="mb-0 mt-4",
            )

            return (
                # CIMA stats
                error_message,
                error_message,
                error_message,
                error_class,
                # Nomenclátor stats
                error_message,
                error_message,
                error_message,
                error_class,
                # Enrichment section con mensaje amigable
                error_enrichment_section,
                # Last activity
                error_message,
                error_message,
                error_message,
                # Feedback visual
                now.strftime("%H:%M:%S"),  # Last update timestamp
                html.Div(),  # No progress bar
                # Reintentar rápido (5s) cuando hay error
                POLLING_INTERVAL_ACTIVE,  # Usar intervalo activo para reintentar pronto
                now.isoformat(),
            )

    # Issue #237 - Fase 1: Callback para actualizar countdown cada segundo
    @app.callback(
        Output("admin-next-update-countdown", "children"),
        [Input("admin-countdown-interval", "n_intervals")],
        [
            State("admin-last-poll-time", "data"),
            State("admin-stats-interval", "interval"),
            State("admin-tabs", "value"),
        ],
        prevent_initial_call=True,  # CRITICAL: Prevenir ejecución antes de que existan los componentes
    )
    def update_next_poll_countdown(n_intervals, last_poll_time, current_interval_ms, active_tab):
        """
        Actualiza el countdown "Próxima actualización en: X segundos" cada segundo.

        CRITICAL: Solo actualiza cuando el tab 'catalogs' está activo (Issue #485)
        para evitar errores de IDs inexistentes en otros tabs.
        """
        # CRITICAL: Validación defensiva - solo ejecutar si tab correcto está activo
        # Issue #485: Tab renombrado de "database" a "catalogs"
        if not active_tab or active_tab != "catalogs":
            raise PreventUpdate  # Abortar si tab no seleccionado o incorrecto

        if not last_poll_time:
            return "--"

        try:
            from datetime import datetime, timezone

            # Parsear último poll time
            last_poll_dt = datetime.fromisoformat(last_poll_time)
            now = datetime.now(timezone.utc)

            # Calcular tiempo transcurrido desde último poll
            elapsed_seconds = (now - last_poll_dt).total_seconds()

            # Obtener intervalo actual en segundos
            interval_seconds = current_interval_ms / 1000

            # Calcular segundos restantes hasta próximo poll
            remaining_seconds = max(0, int(interval_seconds - elapsed_seconds))

            if remaining_seconds == 0:
                return "actualizando..."

            return f"{remaining_seconds}s"

        except Exception as e:
            logger.error(f"Error calculating countdown: {e}")
            return "--"

    # Issue #250: Callback control_catalog_intervals() REMOVIDO - duplicado con admin_tabs.py
    # El callback render_admin_tab_content() en admin_tabs.py ya controla estos intervals
    # mediante los outputs:
    # - Output('admin-stats-interval', 'disabled')
    # - Output('admin-countdown-interval', 'disabled')
    #
    # Mantener ambos callbacks violaba REGLA #11 (ONE INPUT ONE CALLBACK) causando
    # advertencias del validador. El callback de admin_tabs.py tiene prioridad porque:
    # 1. Controla el renderizado del contenido del tab (responsabilidad principal)
    # 2. Se registra ANTES en admin/__init__.py
    # 3. Tiene prevent_initial_call=False para establecer estado inicial correcto

    # Issue #236: Callback de sincronización REMOVIDO - duplicado con sync_operations.py
    # El callback handle_sync_actions llamaba a endpoints inexistentes:
    # - /api/v1/admin/catalog/sync-cima (NO EXISTE)
    # - /api/v1/admin/catalog/sync-nomenclator (NO EXISTE)
    # - /api/v1/admin/catalog/sync-all (NO EXISTE)
    #
    # El callback correcto está en callbacks/admin/sync_operations.py que usa:
    # - /api/v1/system/sync (SÍ EXISTE)
    # - Se registra DESPUÉS de este módulo en admin/__init__.py
    # - Maneja los mismos botones con endpoints correctos

    @app.callback(
        Output("admin-analysis-results", "children"),
        [
            Input("admin-coverage-report-button", "n_clicks"),
            Input("admin-duplicates-button", "n_clicks"),
            Input("admin-orphans-button", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def handle_catalog_analysis(coverage_clicks, duplicates_clicks, orphans_clicks):
        """
        Maneja las operaciones de análisis del catálogo.
        """
        if not ctx.triggered:
            return html.Div()

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        try:
            if trigger_id == "admin-coverage-report-button":
                return generate_coverage_report()
            elif trigger_id == "admin-duplicates-button":
                return detect_duplicate_products()
            elif trigger_id == "admin-orphans-button":
                return find_orphan_products()
        except Exception as e:
            logger.error(f"Error in catalog analysis: {e}")
            return dbc.Alert(f"Error ejecutando análisis: {str(e)}", color="danger", dismissable=True)

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("Catalog management callbacks registered successfully")

    return app


def create_sync_progress_display(stats):
    """
    Crea el componente de visualización de progreso de sincronización.
    """
    sync_type = stats.get("sync_type", "unknown")
    progress = stats.get("sync_progress", 0)
    current_chunk = stats.get("current_chunk", 0)
    total_chunks = stats.get("total_chunks", 0)

    return dbc.Card(
        [
            dbc.CardBody(
                [
                    html.H6(f"Sincronización {sync_type.upper()} en progreso", className="mb-3"),
                    dbc.Progress(
                        value=progress,
                        label=f"{progress}%",
                        color="primary" if progress < 100 else "success",
                        animated=True,
                        striped=True,
                        className="mb-2",
                    ),
                    html.Small(f"Chunk {current_chunk}/{total_chunks}", className="text-muted"),
                ]
            )
        ],
        className="mt-3",
    )


def generate_coverage_report():
    """
    Genera un reporte detallado de cobertura del catálogo.
    """
    try:
        response_data = request_coordinator.make_request("/api/v1/admin/catalog/coverage-report", cache_ttl=60)
        if response_data:
            report = response_data.get("data", {})

            return dbc.Card(
                [
                    dbc.CardHeader("Reporte de Cobertura"),
                    dbc.CardBody(
                        [
                            html.H6("Resumen", className="mb-3"),
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [
                                            html.P(
                                                [
                                                    html.Strong("Total Productos: "),
                                                    f"{report.get('total_products', 0):,}",
                                                ]
                                            )
                                        ],
                                        width=6,
                                    ),
                                    dbc.Col(
                                        [
                                            html.P(
                                                [
                                                    html.Strong("Con Datos Completos: "),
                                                    f"{report.get('complete_products', 0):,} ({report.get('complete_percentage', 0):.1f}%)",
                                                ]
                                            )
                                        ],
                                        width=6,
                                    ),
                                ]
                            ),
                            html.Hr(),
                            html.H6("Cobertura por Fuente", className="mb-3"),
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [html.P([html.Strong("Solo CIMA: "), f"{report.get('only_cima', 0):,}"])],
                                        width=4,
                                    ),
                                    dbc.Col(
                                        [
                                            html.P(
                                                [
                                                    html.Strong("Solo Nomenclátor: "),
                                                    f"{report.get('only_nomenclator', 0):,}",
                                                ]
                                            )
                                        ],
                                        width=4,
                                    ),
                                    dbc.Col(
                                        [
                                            html.P(
                                                [html.Strong("Ambas Fuentes: "), f"{report.get('both_sources', 0):,}"]
                                            )
                                        ],
                                        width=4,
                                    ),
                                ]
                            ),
                            html.Hr(),
                            html.H6("Productos Problemáticos", className="mb-3"),
                            html.P([html.Strong("Sin Laboratorio: "), f"{report.get('without_laboratory', 0):,}"]),
                            html.P([html.Strong("Sin Categoría: "), f"{report.get('without_category', 0):,}"]),
                        ]
                    ),
                ]
            )
        else:
            return dbc.Alert("Error generando reporte", color="danger")
    except Exception as e:
        return dbc.Alert(f"Error: {str(e)}", color="danger")


def detect_duplicate_products():
    """
    Detecta productos duplicados en el catálogo.
    """
    try:
        response_data = request_coordinator.make_request("/api/v1/admin/catalog/duplicates", cache_ttl=60)
        if response_data:
            duplicates = response_data.get("data", {})

            if not duplicates.get("duplicates_found"):
                return dbc.Alert(
                    html.Div([html.I(className="fas fa-check-circle me-2"), "No se encontraron productos duplicados"]),
                    color="success",
                )

            duplicate_list = duplicates.get("duplicate_groups", [])

            return dbc.Card(
                [
                    dbc.CardHeader(f"Duplicados Encontrados: {len(duplicate_list)} grupos"),
                    dbc.CardBody(
                        [
                            html.Div(
                                [
                                    dbc.Alert(
                                        [
                                            html.Strong(f"Código Nacional: {group.get('national_code')}"),
                                            html.Br(),
                                            html.Small(f"Repeticiones: {group.get('count')}"),
                                        ],
                                        color="warning",
                                        className="mb-2",
                                    )
                                    for group in duplicate_list[:10]  # Mostrar solo primeros 10
                                ]
                            ),
                            (
                                html.Small(
                                    f"Mostrando primeros 10 de {len(duplicate_list)} grupos", className="text-muted"
                                )
                                if len(duplicate_list) > 10
                                else None
                            ),
                        ]
                    ),
                ]
            )
        else:
            return dbc.Alert("Error detectando duplicados", color="danger")
    except Exception as e:
        return dbc.Alert(f"Error: {str(e)}", color="danger")


def find_orphan_products():
    """
    Encuentra productos huérfanos sin referencias en ventas.
    """
    try:
        response_data = request_coordinator.make_request("/api/v1/admin/catalog/orphans", cache_ttl=60)
        if response_data:
            orphans = response_data.get("data", {})

            orphan_count = orphans.get("orphan_count", 0)

            if orphan_count == 0:
                return dbc.Alert(
                    html.Div([html.I(className="fas fa-check-circle me-2"), "No se encontraron productos huérfanos"]),
                    color="success",
                )

            return dbc.Card(
                [
                    dbc.CardHeader(f"Productos Huérfanos: {orphan_count:,}"),
                    dbc.CardBody(
                        [
                            html.P(["Productos en catálogo sin ventas asociadas: ", html.Strong(f"{orphan_count:,}")]),
                            html.Hr(),
                            dbc.Button(
                                html.Span([html.I(className="fas fa-broom me-2"), "Limpiar Huérfanos"]),
                                id="admin-clean-orphans-button",
                                color="warning",
                                size="sm",
                                disabled=True,  # Por seguridad, requiere confirmación adicional
                            ),
                            html.Small(
                                "Esta operación está deshabilitada por seguridad", className="text-muted d-block mt-2"
                            ),
                        ]
                    ),
                ]
            )
        else:
            return dbc.Alert("Error buscando huérfanos", color="danger")
    except Exception as e:
        return dbc.Alert(f"Error: {str(e)}", color="danger")


# Helper functions para Issue #200


def _create_sync_progress_bar(sync_in_progress, sync_info):
    """
    Crea una progress bar contextual durante sincronización (Issue #283 - Phase 3).
    Delegado al componente especializado create_catalog_sync_progress_card.

    Args:
        sync_in_progress: Boolean indicando si hay sync activa
        sync_info: Dict con información de la sync (tipo, progreso, registros)

    Returns:
        Componente contextual de progreso o div vacío si no hay sync
    """
    if not sync_in_progress or not sync_info:
        # Retornar div vacío si no hay sincronización activa
        return create_catalog_sync_progress_card(None)

    # Delegar al componente especializado con información detallada
    return create_catalog_sync_progress_card(sync_info)


def _format_timestamp(timestamp_str, has_products=False):
    """
    Formatea un timestamp ISO a formato legible en español.
    Si es None pero hay productos, indica carga inicial sin timestamp.
    Si es None y no hay productos, retorna 'Nunca'.
    """
    if not timestamp_str:
        if has_products:
            return CatalogStatusMessages.TIMESTAMP_INITIAL_LOAD
        return CatalogStatusMessages.TIMESTAMP_NEVER

    try:
        ts = datetime.fromisoformat(timestamp_str.replace("Z", "+00:00"))
        now = datetime.now(ts.tzinfo) if ts.tzinfo else datetime.now()

        # Calcular diferencia
        time_diff = now - ts
        hours = int(time_diff.total_seconds() / 3600)

        if hours < 1:
            minutes = int(time_diff.total_seconds() / 60)
            return f"Hace {minutes} min"
        elif hours < 24:
            return f"Hace {hours}h"
        else:
            days = hours // 24
            return f"Hace {days}d"
    except (ValueError, TypeError) as e:
        logger.error(f"Invalid timestamp format: {timestamp_str}, error: {e}")
        return CatalogStatusMessages.TIMESTAMP_INVALID_DATE
    except Exception as e:
        logger.error(f"Unexpected error formatting timestamp: {e}", exc_info=True)
        return CatalogStatusMessages.TIMESTAMP_ERROR


def _get_status_class(status):
    """
    Retorna la clase CSS apropiada según el estado.
    """
    status_map = {
        "READY": "text-success",
        "UPDATING": "text-info",
        "ERROR": "text-danger",
        "NEVER_RUN": "text-warning",
        "INITIALIZING": "text-info",
    }
    return status_map.get(status, "text-muted")


def _format_status_message(status, message, has_products=False):
    """
    Formatea el mensaje de estado de forma más informativa.

    Args:
        status: Estado del sistema (READY, ERROR, NEVER_RUN, etc.)
        message: Mensaje original del backend
        has_products: Si hay productos en el catálogo

    Returns:
        Mensaje formateado para mostrar al usuario
    """
    if status == "ERROR":
        # Issue #211: Detectar tipo de error y agregar contexto útil
        if message and message != "No sincronizado":
            msg_lower = message.lower()

            # Detectar tipo de error por palabras clave
            if "conexión" in msg_lower or "connection" in msg_lower:
                return f"⚠️ {message} - Verificar conexión con API externa"
            elif "timeout" in msg_lower or "timed out" in msg_lower:
                return f"⏱️ {message} - La operación tardó demasiado"
            elif "validación" in msg_lower or "validation" in msg_lower:
                return f"ℹ️ {message} - Revisar formato de datos"
            elif "sincronización" in msg_lower:
                return f"🔄 {message}"
            else:
                return f"❌ {message} - Revisar logs del sistema para más detalles"
        return CatalogStatusMessages.ERROR_NO_DETAILS

    elif status == "NEVER_RUN":
        if has_products:
            return CatalogStatusMessages.NEVER_RUN_WITH_PRODUCTS
        return CatalogStatusMessages.NEVER_RUN_NO_PRODUCTS

    elif status == "READY":
        return CatalogStatusMessages.READY

    elif status == "UPDATING":
        return CatalogStatusMessages.UPDATING

    elif status == "INITIALIZING":
        return CatalogStatusMessages.INITIALIZING

    else:
        # Fallback: mostrar mensaje original si existe
        return message if message else CatalogStatusMessages.UNKNOWN_STATUS


def _create_enrichment_section(sales_enrichment):
    """
    Crea la sección de enriquecimiento de ventas si hay datos.
    Si no hay ventas (sales_enrichment es None), retorna mensaje informativo.
    """
    if not sales_enrichment:
        return html.Div(
            [
                html.H5(
                    [html.I(className="fas fa-chart-line me-2 text-secondary"), "Enriquecimiento de Ventas"],
                    className="mb-3 mt-4",
                ),
                dbc.Alert(
                    [
                        html.I(className="fas fa-info-circle me-2"),
                        html.Strong("No hay datos de ventas cargados aún."),
                        html.Br(),
                        html.Small(
                            "Las métricas de enriquecimiento aparecerán después de subir archivos ERP.",
                            className="text-muted",
                        ),
                    ],
                    color="info",
                    className="mb-0",
                ),
            ]
        )

    # Si hay datos, mostrar métricas
    total_sales = sales_enrichment.get("total_sales", 0)
    enriched_sales = sales_enrichment.get("enriched_sales", 0)
    enrichment_rate = sales_enrichment.get("enrichment_rate", 0)
    not_enriched = sales_enrichment.get("not_enriched", 0)

    return html.Div(
        [
            html.H5(
                [html.I(className="fas fa-chart-line me-2 text-success"), "Enriquecimiento de Ventas"],
                className="mb-3 mt-4",
            ),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.Strong("Ventas totales: "),
                                                    html.Span(f"{total_sales:,}", className="fs-5 text-primary"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.Div(
                                                [
                                                    html.Strong("Enriquecidas: "),
                                                    html.Span(
                                                        f"{enriched_sales:,} ({enrichment_rate}%)",
                                                        className="text-success",
                                                    ),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.Div(
                                                [
                                                    html.Strong("Sin enriquecer: "),
                                                    html.Span(f"{not_enriched:,}", className="text-warning"),
                                                ]
                                            ),
                                        ]
                                    )
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                        className="mb-3",
                    )
                ]
            ),
        ],
        className="mb-4",
    )


# Issue #236: Funciones de sincronización REMOVIDAS - usaban endpoints inexistentes
# Estas funciones han sido eliminadas porque llamaban a endpoints que NO existen en el backend:
# - trigger_cima_sync() → /api/v1/admin/catalog/sync-cima (404 Not Found)
# - trigger_nomenclator_sync() → /api/v1/admin/catalog/sync-nomenclator (404 Not Found)
# - trigger_full_sync() → /api/v1/admin/catalog/sync-all (404 Not Found)
#
# La funcionalidad correcta está implementada en:
# - callbacks/admin/sync_operations.py → execute_sync_operation()
# - Usa endpoint correcto: POST /api/v1/system/sync (SÍ existe)
# - Parámetros: {"target": "cima"|"nomenclator"|"all", "force": false}


def _get_skeleton_defaults(interval_override: Optional[int] = None) -> Tuple[Any, ...]:
    """
    Retorna valores por defecto (skeleton) cuando no hay datos disponibles.
    Usado durante carga inicial o cuando auth no está lista.

    Args:
        interval_override: Intervalo personalizado en ms (ej: 5000 para retry rápido).
                          Si None, usa default de 30000ms.

    Returns:
        Tuple[Any, ...]: Valores por defecto que coinciden con todos los outputs
            del callback update_catalog_stats_display:
            - 4 outputs para CIMA (count, last_sync, status text, status className)
            - 4 outputs para Nomenclator (count, last_sync, status text, status className)
            - 1 output para sales enrichment section (html.Div skeleton)
            - 3 outputs para last activity (cima, nomenclator, catalog update)
            - 2 outputs para feedback visual (timestamp, progress bar)
            - 2 outputs para polling inteligente (interval, last_poll_time)
    """
    # Use interval_override if provided, otherwise use default 30s
    polling_interval = interval_override if interval_override is not None else 30000

    return (
        # Sección 1: CIMA
        "--",  # admin-cima-count
        "Cargando...",  # admin-cima-last-sync
        "Verificando estado...",  # admin-cima-status text
        "badge bg-secondary",  # admin-cima-status className
        # Sección 1: Nomenclator
        "--",  # admin-nomenclator-count
        "Cargando...",  # admin-nomenclator-last-sync
        "Verificando estado...",  # admin-nomenclator-status text
        "badge bg-secondary",  # admin-nomenclator-status className
        # Sección 2: Sales enrichment skeleton
        _create_enrichment_section_skeleton(),  # admin-sales-enrichment-section
        # Sección 3: Last activity
        "Cargando...",  # admin-activity-cima-sync
        "Cargando...",  # admin-activity-nomenclator-sync
        "Cargando...",  # admin-activity-catalog-update
        # Issue #237 - Feedback Visual
        "Cargando...",  # admin-last-update-timestamp
        html.Div(),  # admin-sync-progress-bar-container
        # Issue #237 - Polling Inteligente
        polling_interval,  # admin-stats-interval (usa override si está esperando auth_sync)
        None,  # admin-last-poll-time
    )


def _create_enrichment_section_skeleton() -> html.Div:
    """
    Skeleton para sección de enriquecimiento mientras se cargan datos.

    Returns:
        html.Div: Componente skeleton con spinner y mensaje de carga.
            Usado durante carga inicial para dar feedback visual al usuario
            mientras se obtienen las estadísticas del catálogo desde el backend.
    """
    return html.Div([
        html.H5(
            [html.I(className="fas fa-chart-line me-2 text-secondary"), "Enriquecimiento de Ventas"],
            className="mb-3 mt-4",
        ),
        dbc.Alert(
            [
                html.I(className="fas fa-spinner fa-spin me-2"),
                html.Strong("Cargando estadísticas del catálogo..."),
                html.Br(),
                html.Small("Un momento por favor.", className="text-muted"),
            ],
            color="info",
            className="mb-0",
        ),
    ])
