"""
Brand Aliases Management Callbacks Module.

Responsabilidad: UI para gestión dinámica de aliases de marcas.
Permite crear, editar, eliminar y toggle aliases que corrigen outputs de BrandDetectionService.

Soporta dos acciones:
    - ALIAS: Fusionar marcas (ej: "oralkin" → "kin")
    - EXCLUDE: Excluir falsos positivos (ej: "clorhexidina" no es marca)

Diferente de CategoryAlias:
    - CategoryAlias: Normaliza OUTPUT del clasificador NECESIDAD
    - BrandAlias: Normaliza OUTPUT del BrandDetectionService
"""

import json
import logging
from datetime import datetime

import dash_bootstrap_components as dbc
from dash import Input, Output, State, ctx, html, ALL
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_brand_aliases_callbacks(app):
    """
    Register brand aliases management callbacks for admin panel.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Brand aliases callbacks already registered, skipping")
        return app

    logger.info("Registering brand aliases callbacks")

    # =========================================================================
    # CALLBACK 1: Load Brand Aliases Data on Tab Activation or Refresh
    # =========================================================================
    @app.callback(
        [
            Output("brand-aliases-stats-container", "children"),
            Output("brand-aliases-table-container", "children"),
            Output("brand-aliases-pagination-info", "children"),
            Output("brand-aliases-prev-page", "disabled"),
            Output("brand-aliases-next-page", "disabled"),
            Output("brand-aliases-last-update", "children"),
        ],
        [
            Input("brand-aliases-refresh-btn", "n_clicks"),
            Input("brand-aliases-search-input", "value"),
            Input("brand-aliases-action-filter", "value"),
            Input("brand-aliases-show-inactive", "value"),
            Input("brand-aliases-current-page", "data"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=False,
    )
    def load_brand_aliases_data(
        n_clicks, search_text, action_filter, show_inactive, current_page,
        auth_state, auth_tokens
    ):
        """
        Load brand aliases data when:
        - Page loads (prevent_initial_call=False)
        - Refresh button clicked
        - Filters changed
        - Page changed

        REGLA #7.6: Multi-worker token restoration before API calls.
        """
        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[BRAND-ALIASES] User not authenticated - returning empty")
            return (
                html.Div(),
                dbc.Alert("Inicia sesión para ver los aliases.", color="warning"),
                "",
                True,
                True,
                "",
            )

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[BRAND-ALIASES] No auth headers available")
            return (
                html.Div(),
                dbc.Alert("Error de autenticación.", color="danger"),
                "",
                True,
                True,
                "",
            )

        # Default values
        page = current_page or 0
        limit = 20
        offset = page * limit

        # Build query params
        params = {"limit": limit, "offset": offset}
        if search_text:
            params["search"] = search_text
        if action_filter:
            params["action"] = action_filter
        if show_inactive and "show_inactive" in show_inactive:
            params["is_active"] = None  # Show all
        else:
            params["is_active"] = True  # Only active

        # Load aliases
        aliases_data = {"items": [], "total": 0, "stats": {}}
        try:
            response = request_coordinator.make_request(
                "/api/v1/admin/brand-aliases",
                method="GET",
                params=params,
                auth_headers=auth_headers,
            )
            if response and isinstance(response, dict):
                aliases_data = response
        except Exception as e:
            logger.error(f"[BRAND-ALIASES] Error loading aliases: {e}")

        # Build stats badges
        stats = aliases_data.get("stats", {})
        stats_content = dbc.Row([
            dbc.Col([
                dbc.Badge(
                    f"Total: {stats.get('total', 0)}",
                    color="primary",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Activos: {stats.get('active', 0)}",
                    color="success",
                    className="me-2",
                ),
                dbc.Badge(
                    [html.I(className="fas fa-link me-1"), f"Alias: {stats.get('aliases', 0)}"],
                    color="info",
                    className="me-2",
                ),
                dbc.Badge(
                    [html.I(className="fas fa-ban me-1"), f"Excluidos: {stats.get('excludes', 0)}"],
                    color="warning",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Usos totales: {stats.get('total_usage', 0):,}",
                    color="secondary",
                ),
            ], width=12),
        ], className="mb-3")

        # Build table
        items = aliases_data.get("items", [])
        if not items:
            table_content = dbc.Alert(
                "No hay brand aliases configurados. Haz clic en 'Añadir' para crear uno.",
                color="info",
            )
        else:
            rows = []
            for alias in items:
                # Action badge
                action = alias.get("action", "alias")
                if action == "exclude":
                    action_badge = dbc.Badge(html.Span([html.I(className="fas fa-ban me-1"), "Excluir"]), color="warning")
                    target_display = html.Span("(se elimina)", className="text-muted fst-italic")
                else:
                    action_badge = dbc.Badge(html.Span([html.I(className="fas fa-link me-1"), "Alias"]), color="info")
                    target_display = html.Strong(alias.get("target_brand", ""))

                # Status badge
                status_badge = dbc.Badge(
                    "Activo" if alias.get("is_active") else "Inactivo",
                    color="success" if alias.get("is_active") else "secondary",
                )

                # Affected count with badge color based on count
                affected = alias.get("affected_count", 0)
                if affected > 100:
                    affected_badge = dbc.Badge(f"{affected:,}", color="danger", title="Muchos productos afectados")
                elif affected > 0:
                    affected_badge = dbc.Badge(f"{affected:,}", color="warning", title="Productos afectados")
                else:
                    affected_badge = dbc.Badge("0", color="secondary")

                rows.append(
                    html.Tr([
                        html.Td(html.Strong(alias.get("source_brand", ""))),
                        html.Td(target_display),
                        html.Td(action_badge),
                        html.Td(alias.get("reason", "-") or "-", className="text-muted small"),
                        html.Td(f"{alias.get('usage_count', 0):,}"),
                        html.Td(affected_badge),
                        html.Td(status_badge),
                        html.Td([
                            dbc.Button(
                                html.I(className="fas fa-edit"),
                                color="warning",
                                size="sm",
                                className="me-1",
                                id={"type": "brand-aliases-edit-btn", "index": alias.get("id")},
                                title="Editar",
                            ),
                            dbc.Button(
                                html.I(className="fas fa-toggle-on" if alias.get("is_active") else "fas fa-toggle-off"),
                                color="info" if alias.get("is_active") else "secondary",
                                size="sm",
                                className="me-1",
                                id={"type": "brand-aliases-toggle-btn", "index": alias.get("id")},
                                title="Toggle activo/inactivo",
                            ),
                            dbc.Button(
                                html.I(className="fas fa-sync"),
                                color="success",
                                size="sm",
                                className="me-1",
                                id={"type": "brand-aliases-reprocess-btn", "index": alias.get("id")},
                                title=f"Re-procesar {affected:,} productos",
                                disabled=affected == 0 or not alias.get("is_active"),
                            ),
                            dbc.Button(
                                html.I(className="fas fa-trash"),
                                color="danger",
                                size="sm",
                                id={"type": "brand-aliases-delete-btn", "index": alias.get("id")},
                                title="Eliminar",
                            ),
                        ], className="d-flex"),
                    ])
                )

            table_content = dbc.Table(
                [
                    html.Thead(
                        html.Tr([
                            html.Th("Marca Origen"),
                            html.Th("Marca Destino"),
                            html.Th("Acción"),
                            html.Th("Razón"),
                            html.Th("Usos"),
                            html.Th("Afectados"),
                            html.Th("Estado"),
                            html.Th("Acciones"),
                        ])
                    ),
                    html.Tbody(rows),
                ],
                bordered=True,
                hover=True,
                responsive=True,
                size="sm",
            )

        # Pagination info
        total = aliases_data.get("total", 0)
        total_pages = (total + limit - 1) // limit if total > 0 else 1
        current_page_display = page + 1  # 0-indexed to 1-indexed
        pagination_info = f"Página {current_page_display} de {total_pages} ({total} aliases)"

        # Disable pagination buttons
        prev_disabled = page <= 0
        next_disabled = current_page_display >= total_pages

        # Last update timestamp
        last_update = f"Actualizado: {datetime.now().strftime('%H:%M:%S')}"

        return (
            stats_content,
            table_content,
            pagination_info,
            prev_disabled,
            next_disabled,
            last_update,
        )

    # =========================================================================
    # CALLBACK 2: Handle Pagination
    # =========================================================================
    @app.callback(
        Output("brand-aliases-current-page", "data"),
        [
            Input("brand-aliases-prev-page", "n_clicks"),
            Input("brand-aliases-next-page", "n_clicks"),
        ],
        [State("brand-aliases-current-page", "data")],
        prevent_initial_call=True,
    )
    def handle_brand_aliases_pagination(prev_clicks, next_clicks, current_page):
        """Handle pagination button clicks."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
        page = current_page or 0

        if trigger_id == "brand-aliases-prev-page" and page > 0:
            return page - 1
        elif trigger_id == "brand-aliases-next-page":
            return page + 1

        return page

    # =========================================================================
    # CALLBACK 3: Toggle Target Container Visibility
    # =========================================================================
    @app.callback(
        Output("brand-aliases-target-container", "style"),
        Input("brand-aliases-form-action", "value"),
        prevent_initial_call=False,
    )
    def toggle_target_visibility(action):
        """
        Show/hide target input based on action type.
        - alias: Show target input
        - exclude: Hide target input
        """
        if action == "exclude":
            return {"display": "none"}
        return {"display": "block"}

    # =========================================================================
    # CALLBACK 4: Open Add Modal
    # =========================================================================
    @app.callback(
        [
            Output("brand-aliases-modal", "is_open", allow_duplicate=True),
            Output("brand-aliases-edit-id", "data", allow_duplicate=True),
            Output("brand-aliases-modal-title", "children", allow_duplicate=True),
            Output("brand-aliases-form-source", "value", allow_duplicate=True),
            Output("brand-aliases-form-action", "value", allow_duplicate=True),
            Output("brand-aliases-form-target", "value", allow_duplicate=True),
            Output("brand-aliases-form-reason", "value", allow_duplicate=True),
        ],
        Input("brand-aliases-add-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def open_add_brand_alias_modal(n_clicks):
        """Open modal for adding new brand alias with empty form."""
        if not n_clicks:
            raise PreventUpdate

        return (
            True,      # Open modal
            None,      # No edit ID (creating new)
            "Añadir Brand Alias",
            "",        # Empty source
            "alias",   # Default action
            "",        # Empty target
            "",        # Empty reason
        )

    # =========================================================================
    # CALLBACK 5: Open Edit Modal
    # =========================================================================
    @app.callback(
        [
            Output("brand-aliases-modal", "is_open", allow_duplicate=True),
            Output("brand-aliases-edit-id", "data", allow_duplicate=True),
            Output("brand-aliases-modal-title", "children", allow_duplicate=True),
            Output("brand-aliases-form-source", "value", allow_duplicate=True),
            Output("brand-aliases-form-action", "value", allow_duplicate=True),
            Output("brand-aliases-form-target", "value", allow_duplicate=True),
            Output("brand-aliases-form-reason", "value", allow_duplicate=True),
        ],
        Input({"type": "brand-aliases-edit-btn", "index": ALL}, "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def open_edit_brand_alias_modal(n_clicks_list, auth_state, auth_tokens):
        """Open modal for editing existing brand alias."""
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Get clicked button's index (alias ID)
        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        # Parse pattern-matched ID
        btn_info = json.loads(triggered_id.split(".")[0])
        alias_id = btn_info.get("index")

        if not alias_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            raise PreventUpdate

        # Fetch single alias by ID (uses dedicated GET /{id} endpoint)
        try:
            alias = request_coordinator.make_request(
                f"/api/v1/admin/brand-aliases/{alias_id}",
                method="GET",
                auth_headers=auth_headers,
            )

            if not alias:
                raise PreventUpdate

            return (
                True,  # Open modal
                alias_id,  # Set edit ID
                "Editar Brand Alias",
                alias.get("source_brand", ""),
                alias.get("action", "alias"),
                alias.get("target_brand", "") or "",
                alias.get("reason", "") or "",
            )
        except Exception as e:
            logger.error(f"[BRAND-ALIASES] Error loading alias for edit: {e}")
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 6: Close Modal
    # =========================================================================
    @app.callback(
        Output("brand-aliases-modal", "is_open", allow_duplicate=True),
        Input("brand-aliases-modal-cancel-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def close_brand_alias_modal(n_clicks):
        """Close modal on cancel."""
        if not n_clicks:
            raise PreventUpdate
        return False

    # =========================================================================
    # CALLBACK 7: Save Brand Alias (Create/Update)
    # =========================================================================
    @app.callback(
        [
            Output("brand-aliases-modal", "is_open", allow_duplicate=True),
            Output("brand-aliases-refresh-btn", "n_clicks", allow_duplicate=True),
        ],
        Input("brand-aliases-modal-save-btn", "n_clicks"),
        [
            State("brand-aliases-edit-id", "data"),
            State("brand-aliases-form-source", "value"),
            State("brand-aliases-form-action", "value"),
            State("brand-aliases-form-target", "value"),
            State("brand-aliases-form-reason", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("brand-aliases-refresh-btn", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def save_brand_alias(
        n_clicks, edit_id, source, action, target, reason,
        auth_state, auth_tokens, current_refresh_clicks
    ):
        """Save brand alias (create new or update existing)."""
        if not n_clicks:
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[BRAND-ALIASES] User not authenticated - cannot save")
            raise PreventUpdate

        if not source:
            logger.warning("[BRAND-ALIASES] Source is required")
            raise PreventUpdate

        if action == "alias" and not target:
            logger.warning("[BRAND-ALIASES] Target is required for alias action")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[BRAND-ALIASES] No auth headers available - cannot save")
            raise PreventUpdate

        # Build payload
        payload = {
            "source_brand": source.strip().lower(),
            "action": action,
            "reason": reason.strip() if reason else None,
        }

        # Only include target_brand for alias action
        if action == "alias":
            payload["target_brand"] = target.strip().lower()
        else:
            payload["target_brand"] = None

        try:
            if edit_id:
                # Update existing
                request_coordinator.make_request(
                    f"/api/v1/admin/brand-aliases/{edit_id}",
                    method="PUT",
                    json=payload,
                    auth_headers=auth_headers,
                )
                logger.info(f"[BRAND-ALIASES] Updated alias id={edit_id}")
            else:
                # Create new
                request_coordinator.make_request(
                    "/api/v1/admin/brand-aliases",
                    method="POST",
                    json=payload,
                    auth_headers=auth_headers,
                )
                logger.info(f"[BRAND-ALIASES] Created new alias: {source} → {action}")

            # Close modal and trigger refresh
            return False, (current_refresh_clicks or 0) + 1

        except Exception as e:
            logger.error(f"[BRAND-ALIASES] Error saving alias: {e}")
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 8: Toggle Brand Alias Active State
    # =========================================================================
    @app.callback(
        Output("brand-aliases-refresh-btn", "n_clicks", allow_duplicate=True),
        Input({"type": "brand-aliases-toggle-btn", "index": ALL}, "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("brand-aliases-refresh-btn", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def toggle_brand_alias_active(n_clicks_list, auth_state, auth_tokens, current_refresh):
        """Toggle brand alias active/inactive state."""
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[BRAND-ALIASES] User not authenticated - cannot toggle")
            raise PreventUpdate

        # Get clicked button's index (alias ID)
        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        btn_info = json.loads(triggered_id.split(".")[0])
        alias_id = btn_info.get("index")

        if not alias_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[BRAND-ALIASES] No auth headers available - cannot toggle")
            raise PreventUpdate

        try:
            request_coordinator.make_request(
                f"/api/v1/admin/brand-aliases/{alias_id}/toggle",
                method="POST",
                auth_headers=auth_headers,
            )
            logger.info(f"[BRAND-ALIASES] Toggled alias id={alias_id}")
            return (current_refresh or 0) + 1

        except Exception as e:
            logger.error(f"[BRAND-ALIASES] Error toggling alias: {e}")
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 9: Reprocess Products for Brand Alias
    # =========================================================================
    @app.callback(
        Output("brand-aliases-refresh-btn", "n_clicks", allow_duplicate=True),
        Input({"type": "brand-aliases-reprocess-btn", "index": ALL}, "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("brand-aliases-refresh-btn", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def reprocess_brand_alias(n_clicks_list, auth_state, auth_tokens, current_refresh):
        """Reprocess products affected by a brand alias."""
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[BRAND-ALIASES] User not authenticated - cannot reprocess")
            raise PreventUpdate

        # Get clicked button's index (alias ID)
        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        btn_info = json.loads(triggered_id.split(".")[0])
        alias_id = btn_info.get("index")

        if not alias_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[BRAND-ALIASES] No auth headers available - cannot reprocess")
            raise PreventUpdate

        try:
            response = request_coordinator.make_request(
                f"/api/v1/admin/brand-aliases/{alias_id}/reprocess",
                method="POST",
                auth_headers=auth_headers,
            )
            updated_count = response.get("updated_count", 0) if response else 0
            logger.info(f"[BRAND-ALIASES] Reprocessed alias id={alias_id}, updated {updated_count} products")
            return (current_refresh or 0) + 1

        except Exception as e:
            logger.error(f"[BRAND-ALIASES] Error reprocessing alias: {e}")
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 10: Delete Brand Alias
    # =========================================================================
    @app.callback(
        Output("brand-aliases-refresh-btn", "n_clicks", allow_duplicate=True),
        Input({"type": "brand-aliases-delete-btn", "index": ALL}, "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("brand-aliases-refresh-btn", "n_clicks"),
        ],
        prevent_initial_call=True,
    )
    def delete_brand_alias(n_clicks_list, auth_state, auth_tokens, current_refresh):
        """Delete brand alias (soft delete by default)."""
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[BRAND-ALIASES] User not authenticated - cannot delete")
            raise PreventUpdate

        # Get clicked button's index (alias ID)
        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        btn_info = json.loads(triggered_id.split(".")[0])
        alias_id = btn_info.get("index")

        if not alias_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[BRAND-ALIASES] No auth headers available - cannot delete")
            raise PreventUpdate

        try:
            request_coordinator.make_request(
                f"/api/v1/admin/brand-aliases/{alias_id}",
                method="DELETE",
                auth_headers=auth_headers,
            )
            logger.info(f"[BRAND-ALIASES] Deleted alias id={alias_id}")
            return (current_refresh or 0) + 1

        except Exception as e:
            logger.error(f"[BRAND-ALIASES] Error deleting alias: {e}")
            raise PreventUpdate

    # Mark module as registered
    _module_callbacks_registered = True
    logger.info("Brand aliases callbacks registered successfully")

    return app
