"""
Category Aliases Management Callbacks Module (Issue #459).

Responsabilidad: UI para gestión dinámica de aliases de categorías.
Permite crear, editar, eliminar y toggle aliases que normalizan outputs del clasificador.

Diferente de Keywords:
    - Keywords: Pattern matching en INPUT (nombre producto)
    - Aliases: Normalización de OUTPUT (categoría clasificada)
"""

import json
import logging
from datetime import datetime

import dash_bootstrap_components as dbc
from dash import Input, Output, State, ctx, html, no_update, ALL
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_aliases_callbacks(app):
    """
    Register category aliases management callbacks for admin panel.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Aliases callbacks already registered, skipping")
        return app

    logger.info("Registering aliases callbacks")

    # =========================================================================
    # CALLBACK 1: Load Aliases Data on Tab Activation or Refresh
    # =========================================================================
    @app.callback(
        [
            Output("aliases-stats-container", "children"),
            Output("aliases-table-container", "children"),
            Output("aliases-pagination-info", "children"),
            Output("aliases-prev-page", "disabled"),
            Output("aliases-next-page", "disabled"),
            Output("aliases-data-store", "data"),
        ],
        [
            Input("aliases-tab-activated-trigger", "data"),
            Input("aliases-refresh-btn", "n_clicks"),
            Input("aliases-search-input", "value"),
            Input("aliases-show-inactive", "value"),
            Input("aliases-page-store", "data"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_aliases_data(
        trigger_data, n_clicks, search_text, show_inactive, current_page,
        auth_state, auth_tokens
    ):
        """
        Load aliases data when:
        - Tab is activated (trigger_data)
        - Refresh button clicked (n_clicks)
        - Filters changed
        - Page changed

        REGLA #7.6: Multi-worker token restoration before API calls.
        """
        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.debug("[ALIASES] User not authenticated - skipping")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[ALIASES] No auth headers available - skipping API calls")
            raise PreventUpdate

        # Default values
        page = current_page or 1
        limit = 20
        offset = (page - 1) * limit

        # Build query params
        params = {"limit": limit, "offset": offset}
        if search_text:
            params["search"] = search_text
        if show_inactive and "show_inactive" in show_inactive:
            params["is_active"] = None  # Show all
        else:
            params["is_active"] = True  # Only active

        # Load aliases
        aliases_data = {"items": [], "total": 0, "stats": {}}
        try:
            response = request_coordinator.make_request(
                "/api/v1/admin/category-aliases",
                method="GET",
                params=params,
                auth_headers=auth_headers,
            )
            if response and isinstance(response, dict):
                aliases_data = response
        except Exception as e:
            logger.error(f"[ALIASES] Error loading aliases: {e}")

        # Build stats badges
        stats = aliases_data.get("stats", {})
        stats_content = dbc.Row([
            dbc.Col([
                dbc.Badge(
                    f"Total: {stats.get('total', 0)}",
                    color="primary",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Activos: {stats.get('active', 0)}",
                    color="success",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Inactivos: {stats.get('inactive', 0)}",
                    color="secondary",
                    className="me-2",
                ),
                dbc.Badge(
                    f"Usos totales: {stats.get('total_usage', 0):,}",
                    color="info",
                ),
            ], width=12),
        ], className="mb-3")

        # Build table
        items = aliases_data.get("items", [])
        if not items:
            table_content = dbc.Alert(
                "No hay aliases configurados. Haz clic en 'Añadir Alias' para crear uno.",
                color="info",
            )
        else:
            rows = []
            for alias in items:
                status_badge = dbc.Badge(
                    "Activo" if alias.get("is_active") else "Inactivo",
                    color="success" if alias.get("is_active") else "secondary",
                )
                rows.append(
                    html.Tr([
                        html.Td(alias.get("source_category", "")),
                        html.Td(alias.get("target_category", "")),
                        html.Td(alias.get("reason", "-") or "-"),
                        html.Td(f"{alias.get('usage_count', 0):,}"),
                        html.Td(status_badge),
                        html.Td([
                            dbc.Button(
                                html.I(className="fas fa-edit"),
                                color="warning",
                                size="sm",
                                className="me-1",
                                id={"type": "aliases-edit-btn", "index": alias.get("id")},
                                title="Editar",
                            ),
                            dbc.Button(
                                html.I(className="fas fa-toggle-on" if alias.get("is_active") else "fas fa-toggle-off"),
                                color="info" if alias.get("is_active") else "secondary",
                                size="sm",
                                className="me-1",
                                id={"type": "aliases-toggle-btn", "index": alias.get("id")},
                                title="Toggle activo/inactivo",
                            ),
                            dbc.Button(
                                html.I(className="fas fa-trash"),
                                color="danger",
                                size="sm",
                                id={"type": "aliases-delete-btn", "index": alias.get("id")},
                                title="Eliminar",
                            ),
                        ], className="d-flex"),
                    ])
                )

            table_content = dbc.Table(
                [
                    html.Thead(
                        html.Tr([
                            html.Th("Origen (Clasificador)"),
                            html.Th("Destino (Base de Datos)"),
                            html.Th("Razón"),
                            html.Th("Usos"),
                            html.Th("Estado"),
                            html.Th("Acciones"),
                        ])
                    ),
                    html.Tbody(rows),
                ],
                bordered=True,
                hover=True,
                responsive=True,
                size="sm",
            )

        # Pagination info
        total = aliases_data.get("total", 0)
        total_pages = (total + limit - 1) // limit if total > 0 else 1
        pagination_info = f"Página {page} de {total_pages} ({total} aliases)"

        # Disable pagination buttons
        prev_disabled = page <= 1
        next_disabled = page >= total_pages

        return (
            stats_content,
            table_content,
            pagination_info,
            prev_disabled,
            next_disabled,
            aliases_data,
        )

    # =========================================================================
    # CALLBACK 2: Handle Pagination
    # =========================================================================
    @app.callback(
        Output("aliases-page-store", "data"),
        [
            Input("aliases-prev-page", "n_clicks"),
            Input("aliases-next-page", "n_clicks"),
        ],
        [State("aliases-page-store", "data")],
        prevent_initial_call=True,
    )
    def handle_aliases_pagination(prev_clicks, next_clicks, current_page):
        """Handle pagination button clicks."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
        page = current_page or 1

        if trigger_id == "aliases-prev-page" and page > 1:
            return page - 1
        elif trigger_id == "aliases-next-page":
            return page + 1

        return page

    # =========================================================================
    # CALLBACK 3: Open Add Modal
    # =========================================================================
    @app.callback(
        [
            Output("aliases-modal", "is_open", allow_duplicate=True),
            Output("aliases-edit-id-store", "data", allow_duplicate=True),
            Output("aliases-modal-title", "children", allow_duplicate=True),
            Output("aliases-form-source", "value", allow_duplicate=True),
            Output("aliases-form-target", "value", allow_duplicate=True),
            Output("aliases-form-reason", "value", allow_duplicate=True),
        ],
        Input("aliases-add-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def open_add_alias_modal(n_clicks):
        """Open modal for adding new alias with empty form."""
        if not n_clicks:
            raise PreventUpdate

        return (
            True,  # Open modal
            None,  # No edit ID (creating new)
            "Añadir Alias",
            "",    # Empty source
            "",    # Empty target
            "",    # Empty reason
        )

    # =========================================================================
    # CALLBACK 4: Open Edit Modal
    # =========================================================================
    @app.callback(
        [
            Output("aliases-modal", "is_open", allow_duplicate=True),
            Output("aliases-edit-id-store", "data", allow_duplicate=True),
            Output("aliases-modal-title", "children", allow_duplicate=True),
            Output("aliases-form-source", "value", allow_duplicate=True),
            Output("aliases-form-target", "value", allow_duplicate=True),
            Output("aliases-form-reason", "value", allow_duplicate=True),
        ],
        Input({"type": "aliases-edit-btn", "index": ALL}, "n_clicks"),
        State("aliases-data-store", "data"),
        prevent_initial_call=True,
    )
    def open_edit_alias_modal(n_clicks_list, aliases_data):
        """Open modal for editing existing alias."""
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Get clicked button's index (alias ID)
        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        # Parse pattern-matched ID
        btn_info = json.loads(triggered_id.split(".")[0])
        alias_id = btn_info.get("index")

        if not alias_id:
            raise PreventUpdate

        # Find alias in stored data
        items = aliases_data.get("items", []) if aliases_data else []
        alias = next((a for a in items if a.get("id") == alias_id), None)

        if not alias:
            raise PreventUpdate

        return (
            True,  # Open modal
            alias_id,  # Set edit ID
            "Editar Alias",
            alias.get("source_category", ""),
            alias.get("target_category", ""),
            alias.get("reason", "") or "",
        )

    # =========================================================================
    # CALLBACK 5: Close Modal
    # =========================================================================
    @app.callback(
        Output("aliases-modal", "is_open", allow_duplicate=True),
        Input("aliases-modal-cancel-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def close_alias_modal(n_clicks):
        """Close modal on cancel."""
        if not n_clicks:
            raise PreventUpdate
        return False

    # =========================================================================
    # CALLBACK 6: Save Alias (Create/Update)
    # =========================================================================
    @app.callback(
        [
            Output("aliases-modal", "is_open", allow_duplicate=True),
            Output("aliases-tab-activated-trigger", "data", allow_duplicate=True),
        ],
        Input("aliases-modal-save-btn", "n_clicks"),
        [
            State("aliases-edit-id-store", "data"),
            State("aliases-form-source", "value"),
            State("aliases-form-target", "value"),
            State("aliases-form-reason", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def save_alias(n_clicks, edit_id, source, target, reason, auth_state, auth_tokens):
        """Save alias (create new or update existing)."""
        if not n_clicks:
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[ALIASES] User not authenticated - cannot save")
            raise PreventUpdate

        if not source or not target:
            logger.warning("[ALIASES] Source and target are required")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[ALIASES] No auth headers available - cannot save")
            raise PreventUpdate

        # Build payload
        payload = {
            "source_category": source.strip().lower(),
            "target_category": target.strip().lower(),
            "reason": reason.strip() if reason else None,
        }

        try:
            if edit_id:
                # Update existing
                response = request_coordinator.make_request(
                    f"/api/v1/admin/category-aliases/{edit_id}",
                    method="PUT",
                    json=payload,
                    auth_headers=auth_headers,
                )
                logger.info(f"[ALIASES] Updated alias id={edit_id}")
            else:
                # Create new
                response = request_coordinator.make_request(
                    "/api/v1/admin/category-aliases",
                    method="POST",
                    json=payload,
                    auth_headers=auth_headers,
                )
                logger.info(f"[ALIASES] Created new alias: {source} → {target}")

            # Close modal and refresh data
            return False, datetime.now().isoformat()

        except Exception as e:
            logger.error(f"[ALIASES] Error saving alias: {e}")
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 7: Toggle Alias Active State
    # =========================================================================
    @app.callback(
        Output("aliases-tab-activated-trigger", "data", allow_duplicate=True),
        Input({"type": "aliases-toggle-btn", "index": ALL}, "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def toggle_alias_active(n_clicks_list, auth_state, auth_tokens):
        """Toggle alias active/inactive state."""
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[ALIASES] User not authenticated - cannot toggle")
            raise PreventUpdate

        # Get clicked button's index (alias ID)
        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        btn_info = json.loads(triggered_id.split(".")[0])
        alias_id = btn_info.get("index")

        if not alias_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[ALIASES] No auth headers available - cannot toggle")
            raise PreventUpdate

        try:
            response = request_coordinator.make_request(
                f"/api/v1/admin/category-aliases/{alias_id}/toggle",
                method="POST",
                auth_headers=auth_headers,
            )
            logger.info(f"[ALIASES] Toggled alias id={alias_id}")
            return datetime.now().isoformat()

        except Exception as e:
            logger.error(f"[ALIASES] Error toggling alias: {e}")
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 8: Delete Alias
    # =========================================================================
    @app.callback(
        Output("aliases-tab-activated-trigger", "data", allow_duplicate=True),
        Input({"type": "aliases-delete-btn", "index": ALL}, "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def delete_alias(n_clicks_list, auth_state, auth_tokens):
        """Delete alias (soft delete by default)."""
        if not ctx.triggered or not any(n_clicks_list):
            raise PreventUpdate

        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            logger.warning("[ALIASES] User not authenticated - cannot delete")
            raise PreventUpdate

        # Get clicked button's index (alias ID)
        triggered = ctx.triggered[0]
        triggered_id = triggered["prop_id"]

        btn_info = json.loads(triggered_id.split(".")[0])
        alias_id = btn_info.get("index")

        if not alias_id:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[ALIASES] No auth headers available - cannot delete")
            raise PreventUpdate

        try:
            response = request_coordinator.make_request(
                f"/api/v1/admin/category-aliases/{alias_id}",
                method="DELETE",
                auth_headers=auth_headers,
            )
            logger.info(f"[ALIASES] Deleted alias id={alias_id}")
            return datetime.now().isoformat()

        except Exception as e:
            logger.error(f"[ALIASES] Error deleting alias: {e}")
            raise PreventUpdate

    # Mark module as registered
    _module_callbacks_registered = True
    logger.info("Aliases callbacks registered successfully")

    return app
