"""
Admin Tabs Callbacks Module.
Responsabilidad: Manejo de navegación entre tabs del panel admin.

Issue #485: Reorganización de 7→4 tabs principales:
- Usuarios: Gestión de usuarios y farmacias
- Catálogos Externos: CIMA/Nomenclator + Prescripción (datos IMPORTADOS)
- Venta Libre: Manual Review + Duplicados + ML Monitoring (datos ENRIQUECIDOS)
- Sistema: Herramientas + Base de Datos (infraestructura)
"""

import logging

import dash
from dash import Input, Output, html

logger = logging.getLogger(__name__)

# Module-level flag to prevent duplicate callback registration
_module_callbacks_registered = False


def register_admin_tabs_callbacks(app):
    """
    Register admin tabs navigation callbacks.
    Implements guard pattern to prevent duplicate registration in multi-worker environments.

    Args:
        app: Dash application instance
    """
    global _module_callbacks_registered

    # Guard against duplicate registration at module level
    if _module_callbacks_registered:
        logger.warning("Admin tabs callbacks already registered, skipping")
        return app

    logger.info("Registering admin tabs callbacks")

    @app.callback(
        [
            Output("admin-tab-content", "children"),
            Output("admin-stats-interval", "disabled"),
            Output("admin-stats-interval", "n_intervals"),
            Output("admin-countdown-interval", "disabled"),
            Output("prescription-tab-activated-trigger", "data"),
            Output("manual-review-tab-activated-trigger", "data"),
            Output("ml-monitoring-tab-activated-trigger", "data"),
            Output("ml-monitoring-refresh-interval", "disabled"),
            Output("duplicates-tab-activated-trigger", "data"),
            Output("classification-tab-activated-trigger", "data"),  # Issue #449
            Output("keywords-tab-activated-trigger", "data"),  # Issue #449 Keywords
            Output("aliases-tab-activated-trigger", "data"),  # Issue #459 Category Aliases
            Output("curated-groups-tab-activated-trigger", "data"),  # Issue #521 Curated Groups
        ],
        [Input("admin-tabs", "value")],
        prevent_initial_call=False,
    )
    def render_admin_tab_content(active_tab):
        """
        Renderiza el contenido según el tab activo y controla los intervals.

        Issue #485: Reorganización 7→4 tabs:
        - users: Panel de usuarios (sin cambios)
        - catalogs: CIMA/Nomenclator + Prescripción (sub-tabs)
        - venta-libre: Manual Review + Duplicados + ML Monitoring (sub-tabs)
        - system: Herramientas + Base de Datos (sub-tabs)

        REGLA #11: Un solo callback para admin-tabs Input que maneja:
        1. Renderizado del contenido del tab principal
        2. Control de intervals (activos en catalogs para sync)
        3. Triggers para carga de datos en sub-tabs
        """
        # Import here to avoid circular imports
        from layouts.admin import (
            create_users_panel,
            create_catalogs_panel,
            create_venta_libre_panel,
            create_system_panel,
        )

        # Determinar contenido según tab (Issue #485: 4 tabs principales)
        if active_tab == "users":
            content = create_users_panel()
        elif active_tab == "catalogs":
            content = create_catalogs_panel()
        elif active_tab == "venta-libre":
            content = create_venta_libre_panel()
        elif active_tab == "system":
            content = create_system_panel()
        else:
            content = html.Div("Contenido no encontrado", className="text-muted p-4")

        # Controlar intervals: habilitados en catalogs tab (CIMA/Nomenclator sync)
        is_catalog_tab = active_tab == "catalogs"
        intervals_disabled = not is_catalog_tab

        # Reset n_intervals cuando se activa el tab de catálogos → trigger inmediato
        n_intervals_reset = 0 if is_catalog_tab else dash.no_update

        # Triggers para sub-tabs (se activan cuando se muestra el panel padre)
        # Los sub-tabs están renderizados estáticamente, los callbacks escuchan estos triggers
        # IMPORTANTE: Los valores deben coincidir con lo que esperan los callbacks receptores

        # Prescription trigger: Enviar "prescription" (esperado por prescription.py:370)
        prescription_trigger = {"tab": "prescription"} if active_tab == "catalogs" else dash.no_update

        # Manual Review trigger: Enviar "manual-review" (esperado por manual_review.py:70)
        manual_review_trigger = {"tab": "manual-review"} if active_tab == "venta-libre" else dash.no_update

        # ML Monitoring trigger: Enviar "ml-monitoring" (esperado por ml_monitoring.py)
        ml_monitoring_trigger = {"tab": "ml-monitoring"} if active_tab == "venta-libre" else dash.no_update

        # ML Monitoring interval: Solo activo cuando está en venta-libre tab
        ml_monitoring_interval_disabled = active_tab != "venta-libre"

        # Duplicates trigger: Enviar "duplicates" por consistencia (duplicates.py solo usa triggered_id)
        duplicates_trigger = {"tab": "duplicates"} if active_tab == "venta-libre" else dash.no_update

        # Classification trigger: Issue #449 - Human-in-the-Loop classification UI
        classification_trigger = {"tab": "classification"} if active_tab == "venta-libre" else dash.no_update

        # Keywords trigger: Issue #449 - Keywords management UI
        keywords_trigger = {"tab": "keywords"} if active_tab == "venta-libre" else dash.no_update

        # Aliases trigger: Issue #459 - Category aliases management UI
        aliases_trigger = {"tab": "aliases"} if active_tab == "venta-libre" else dash.no_update

        # Curated Groups trigger: Issue #521 - Curated interchangeable groups UI
        curated_groups_trigger = {"tab": "curated-groups"} if active_tab == "venta-libre" else dash.no_update

        return (
            content,
            intervals_disabled,
            n_intervals_reset,
            intervals_disabled,
            prescription_trigger,
            manual_review_trigger,
            ml_monitoring_trigger,
            ml_monitoring_interval_disabled,
            duplicates_trigger,
            classification_trigger,
            keywords_trigger,
            aliases_trigger,
            curated_groups_trigger,
        )

    # Mark module callbacks as registered
    _module_callbacks_registered = True
    logger.info("Admin tabs callbacks registered successfully")

    return app
