"""
Admin callbacks module registry.
Provides modular organization for admin panel callbacks.
"""

import logging

from components.admin_danger_zone import register_danger_zone_callbacks

from .admin_tabs import register_admin_tabs_callbacks
from .catalog_management import register_catalog_management_callbacks
from .dangerous_tools import register_dangerous_tools_callbacks
from .health_monitoring import register_health_monitoring_callbacks
from .manual_review import register_manual_review_callbacks  # Issue #447
from .ml_monitoring import register_ml_monitoring_callbacks  # Issue #458 M6
from .prescription import register_prescription_callbacks  # Issue #16 Fase 2
from .sync_operations import register_sync_operations_callbacks
from .users import register_user_management_callbacks  # Issue #348 FASE 3.1
from .duplicates import register_duplicates_callbacks  # Issue #477
from .database_stats import register_database_stats_callbacks  # Issue #485
from .classification import register_classification_callbacks  # Issue #449
from .keywords import register_keywords_callbacks, register_apply_keywords_callbacks  # Issue #449 Keywords
from .aliases import register_aliases_callbacks  # Issue #459 Category Aliases
from .brand_aliases import register_brand_aliases_callbacks  # Brand alias management
from .curated_groups import register_curated_groups_callbacks  # Issue #521 Curated Groups

logger = logging.getLogger(__name__)


def register_admin_callbacks(app):
    """
    Register all admin callbacks with the Dash app.
    Each module implements its own guard pattern to prevent duplicate registration.

    Args:
        app: Dash application instance
    """
    logger.info("Registering admin callbacks")

    # Register each module's callbacks
    # Each module has its own guard to prevent duplicate registration
    register_admin_tabs_callbacks(app)  # Tab navigation must be first
    register_catalog_management_callbacks(app)
    register_dangerous_tools_callbacks(app)  # Legacy dangerous tools
    register_danger_zone_callbacks(app)  # Modern AdminDangerZone component
    register_health_monitoring_callbacks(app)
    register_sync_operations_callbacks(app)

    # Issue #348 FASE 3: New unified admin panel callbacks
    register_user_management_callbacks(app)  # User + Pharmacy management (CRUD, filters, storage)

    # Issue #16 Fase 2: Prescription classification management
    register_prescription_callbacks(app)

    # Issue #447: Manual review export for offline analysis
    register_manual_review_callbacks(app)

    # Issue #458 M6: ML Monitoring for classifier health
    register_ml_monitoring_callbacks(app)

    # Issue #477: Duplicates VentaLibre management
    register_duplicates_callbacks(app)

    # Issue #485: Database stats in Tab Sistema
    register_database_stats_callbacks(app)

    # Issue #449: Classification NECESIDAD UI
    register_classification_callbacks(app)

    # Issue #449: Keywords management for NECESIDAD classifier
    register_keywords_callbacks(app)

    # Issue #449 Phase 3: Apply keywords to products
    register_apply_keywords_callbacks(app)

    # Issue #459: Category aliases management for NECESIDAD classifier
    register_aliases_callbacks(app)

    # Brand aliases management for BrandDetectionService
    register_brand_aliases_callbacks(app)

    # Issue #521: Curated interchangeable groups management
    register_curated_groups_callbacks(app)

    logger.info("Admin callbacks registration completed")

    return app
