# frontend/callbacks/__init__.py
"""
Callbacks para manejo de interacciones en la aplicación xfarma.
Organiza la lógica de negocio separada de los layouts.

Pivot 2026: Imports condicionales para modo local (sin jwt dependency).
"""

from components.enrichment_progress import register_enrichment_progress_callbacks
from components.feature_flags import register_feature_flags_callbacks
from components.system_status_unified import register_unified_status_callbacks
from components.toast_manager import register_toast_callbacks
from layouts.homepage import register_homepage_callbacks

from .admin import register_admin_callbacks
from .auth_callbacks import register_auth_callbacks
from .common import register_common_callbacks
from .dashboard import register_dashboard_callbacks, register_enrichment_callbacks
from . import generics  # Expose module for test patching
from .generics import register_generics_callbacks
from .routing import register_routing_callbacks

# from .config import register_config_callbacks  # DEPRECADO: Reemplazado por settings.py (Issue #155)
from .settings import register_settings_callbacks  # AGREGAR: Issue #155
from .upload import register_upload_callbacks
from .prescription import register_prescription_callbacks, register_prescription_new_callbacks  # AGREGAR: Issue #400
from .storage_usage import register_storage_usage_callbacks  # AGREGAR: Issue #420
from .clustering import register_umap_callbacks  # AGREGAR: Issue #458
from .inventory import register_inventory_callbacks  # AGREGAR: Issue #471
from .ventalibre import register_ventalibre_callbacks  # AGREGAR: Issue #461
from .insights import register_insights_callbacks  # AGREGAR: Issue #506
from .encargos import register_encargos_callbacks  # Pivot 2026: Cazador de Encargos

# Pivot 2026: Conditional imports for auth mode
# Cloud mode requires jwt for token refresh; local mode uses PIN-based auth
from utils.auth import IS_LOCAL_MODE

if IS_LOCAL_MODE:
    # Local mode: Use PIN-based security callbacks (no jwt dependency)
    from .security_local import register_security_local_callbacks
    register_auth_context_callbacks = None  # Not needed in local mode
    register_auth_guard_callbacks = None  # Not needed in local mode
else:
    # Cloud mode: Use JWT token-based auth callbacks
    from .auth_context_callbacks import register_auth_context_callbacks  # Issue #97
    from .auth_guard import register_auth_guard_callbacks  # Issue #187
    register_security_local_callbacks = None  # Not needed in cloud mode


def register_all_callbacks(app):
    """
    Registrar todos los callbacks de la aplicación.
    Incluye callbacks para componentes unificados (FASE 4).

    Pivot 2026: Registra callbacks de auth según el modo (local vs cloud).

    Args:
        app: Instancia de la aplicación Dash
    """
    # Auth callbacks: Different for local vs cloud mode
    if IS_LOCAL_MODE:
        # Local mode: PIN-based authentication
        register_security_local_callbacks(app)
    else:
        # Cloud mode: JWT token-based authentication
        register_auth_context_callbacks(app)  # PRIMERO: Sincronizar token al contexto Python
        register_auth_guard_callbacks(app)  # SEGUNDO: Auth guard para 401s

    register_auth_callbacks(app)  # Callbacks de autenticación (login UI, etc.)
    register_routing_callbacks(app)
    register_common_callbacks(app)
    register_homepage_callbacks(app)  # Callbacks para homepage (panel de estado)
    register_enrichment_progress_callbacks(app)  # Callbacks para progreso de enriquecimiento
    register_dashboard_callbacks(app)
    register_enrichment_callbacks(app)  # Callbacks para enriquecimiento integrados en dashboard único
    register_generics_callbacks(app)  # Callbacks para análisis de genéricos
    register_prescription_callbacks(app)  # Callbacks para dashboard de prescripción (Issue #400)
    register_prescription_new_callbacks(app)  # Nuevos callbacks: tabla productos + evolución temporal
    register_upload_callbacks(app)
    # register_config_callbacks(app)  # DEPRECADO: Reemplazado por register_settings_callbacks (Issue #155)
    register_settings_callbacks(app)  # AGREGAR: Callbacks para ajustes (Issue #155)
    register_storage_usage_callbacks(app)  # AGREGAR: Issue #420 - FREE tier storage indicator
    register_admin_callbacks(app)  # Callbacks modulares para panel admin

    # FASE 4: Callbacks para componentes unificados
    register_unified_status_callbacks(app)  # Sistema de estado unificado
    register_feature_flags_callbacks(app)  # Sistema de feature flags
    register_toast_callbacks(app)  # Sistema de notificaciones toast (Issue #137)

    # Issue #458: Clustering visualization
    register_umap_callbacks(app)  # UMAP 2D visualization callbacks

    # Issue #461: Dashboard Venta Libre
    register_ventalibre_callbacks(app)  # Análisis OTC por NECESIDAD

    # Issue #471: Pestañas de Inventario (reutilizable en prescription y ventalibre)
    register_inventory_callbacks(app)  # KPIs, ABC, tabla de rotación

    # Issue #506: Motor de Insights Automáticos
    register_insights_callbacks(app)  # Data loading + feedback

    # Pivot 2026: Cazador de Encargos (customer order tracking)
    register_encargos_callbacks(app)  # Summary, orders list, actions


__all__ = [
    "register_all_callbacks",
    "register_routing_callbacks",
    "register_dashboard_callbacks",
    "register_upload_callbacks",
    # 'register_config_callbacks',  # DEPRECADO: Issue #155
    "register_settings_callbacks",  # Reemplazo de config_callbacks
    "register_common_callbacks",
]
