/**
 * Keyboard Shortcuts for VentaLibre Validation Queue (Issue #458)
 *
 * Shortcuts:
 * - A: Approve current product (click approve button)
 * - M: Move current product (click move button)
 * - S: Skip current product (click skip button)
 *
 * Only active when validation queue is visible and no input is focused.
 */

document.addEventListener('DOMContentLoaded', function() {
    // Store for tracking shortcut state
    window.dash_clientside = window.dash_clientside || {};
    window.dash_clientside.keyboard = window.dash_clientside.keyboard || {
        lastKey: null,
        timestamp: 0
    };

    // Debounce to prevent rapid-fire key presses causing race conditions
    const DEBOUNCE_MS = 300;
    let lastKeyTime = 0;

    document.addEventListener('keydown', function(event) {
        // Ignore if user is typing in an input, textarea, or select
        const activeElement = document.activeElement;
        const isTyping = activeElement && (
            activeElement.tagName === 'INPUT' ||
            activeElement.tagName === 'TEXTAREA' ||
            activeElement.tagName === 'SELECT' ||
            activeElement.isContentEditable
        );

        if (isTyping) {
            return;
        }

        // Guard: Some special keys (dead keys, IME composition) may not have a key property
        if (!event.key) {
            return;
        }

        // Only handle A, M, S keys (case insensitive)
        const key = event.key.toUpperCase();

        if (key === 'A' || key === 'M' || key === 'S') {
            // Check if validation queue is visible
            const approveBtn = document.getElementById('ventalibre-validation-approve-btn');
            const moveBtn = document.getElementById('ventalibre-validation-move-btn');
            const skipBtn = document.getElementById('ventalibre-validation-skip-btn');

            // Only proceed if buttons exist and are visible
            if (!approveBtn || !moveBtn || !skipBtn) {
                return;
            }

            // Check if buttons are in the viewport (Tab 2 Gestion is active)
            const isVisible = approveBtn.offsetParent !== null;
            if (!isVisible) {
                return;
            }

            // Debounce: prevent rapid-fire key presses
            const now = Date.now();
            if (now - lastKeyTime < DEBOUNCE_MS) {
                return;
            }
            lastKeyTime = now;

            event.preventDefault();

            // Store the key press for Dash callback consumption
            window.dash_clientside.keyboard.lastKey = key;
            window.dash_clientside.keyboard.timestamp = Date.now();

            // Trigger the appropriate button
            switch (key) {
                case 'A':
                    if (!approveBtn.disabled) {
                        approveBtn.click();
                        // Visual feedback
                        approveBtn.classList.add('btn-pulse');
                        setTimeout(() => approveBtn.classList.remove('btn-pulse'), 200);
                    }
                    break;
                case 'M':
                    if (!moveBtn.disabled) {
                        moveBtn.click();
                        moveBtn.classList.add('btn-pulse');
                        setTimeout(() => moveBtn.classList.remove('btn-pulse'), 200);
                    }
                    break;
                case 'S':
                    if (!skipBtn.disabled) {
                        skipBtn.click();
                        skipBtn.classList.add('btn-pulse');
                        setTimeout(() => skipBtn.classList.remove('btn-pulse'), 200);
                    }
                    break;
            }

        }
    });
});
