/**
 * Context Menu for VentaLibre Treemap (Issue #512)
 *
 * Captures right-click on treemap and stores click data for Dash callback.
 * Also supports long-press for mobile devices.
 *
 * Flow:
 * 1. User right-clicks on treemap category
 * 2. JavaScript captures event and extracts Plotly hover data
 * 3. Data stored in hidden dcc.Store via setProps simulation
 * 4. Dash callback detects change and opens context menu modal
 */

(function() {
    'use strict';

    const LONG_PRESS_DURATION = 500; // ms for mobile long-press
    let longPressTimer = null;
    let isAttached = false;

    /**
     * Wait for Plotly graph to be rendered, then attach event listeners.
     */
    function init() {
        // Use MutationObserver to detect when treemap is added to DOM
        const observer = new MutationObserver(function(mutations) {
            const treemapGraph = document.getElementById('ventalibre-treemap');
            if (treemapGraph && !isAttached) {
                attachContextMenu(treemapGraph);
                isAttached = true;
            }
        });

        observer.observe(document.body, { childList: true, subtree: true });

        // Also check immediately in case it's already rendered
        const treemapGraph = document.getElementById('ventalibre-treemap');
        if (treemapGraph && !isAttached) {
            attachContextMenu(treemapGraph);
            isAttached = true;
        }
    }

    /**
     * Attach context menu event listeners to the treemap graph.
     */
    function attachContextMenu(graphElement) {
        // Right-click handler (desktop)
        graphElement.addEventListener('contextmenu', function(event) {
            handleContextClick(event, graphElement);
        });

        // Long-press handlers (mobile)
        graphElement.addEventListener('touchstart', function(event) {
            const touch = event.touches[0];
            longPressTimer = setTimeout(function() {
                handleContextClick({
                    preventDefault: function() {},
                    clientX: touch.clientX,
                    clientY: touch.clientY
                }, graphElement);
            }, LONG_PRESS_DURATION);
        }, { passive: true });

        graphElement.addEventListener('touchend', function() {
            clearTimeout(longPressTimer);
        });

        graphElement.addEventListener('touchmove', function() {
            clearTimeout(longPressTimer);
        });

        console.log('[ContextMenu] Attached to ventalibre-treemap');
    }

    /**
     * Handle context click (right-click or long-press).
     * Extracts Plotly hover data and updates the hidden store.
     */
    function handleContextClick(event, graphElement) {
        event.preventDefault();

        // Get the Plotly container
        const plotlyDiv = graphElement.querySelector('.js-plotly-plot');
        if (!plotlyDiv) {
            console.warn('[ContextMenu] Plotly div not found');
            return;
        }

        // Get hover data from Plotly's internal state
        const hoverData = plotlyDiv._hoverdata;

        if (!hoverData || hoverData.length === 0) {
            console.log('[ContextMenu] No hover data available');
            return;
        }

        const point = hoverData[0];
        const customdata = point.customdata || [];

        // Skip if no valid category (e.g., clicked on root or empty space)
        if (!customdata[0]) {
            console.log('[ContextMenu] No category in customdata');
            return;
        }

        // Skip if clicked on aggregated "Otras Necesidades"
        if (customdata[3] === true) {
            console.log('[ContextMenu] Skipping aggregated category');
            return;
        }

        // Build context data object
        const contextData = {
            category_id: customdata[0],           // L1 category code
            category_label: point.label || '',     // Display name
            count: customdata[1] || 0,             // Product count
            percentage: customdata[2] || 0,        // Sales percentage
            is_aggregated: customdata[3] || false, // Is "Otras Necesidades"
            click_x: event.clientX,
            click_y: event.clientY,
            timestamp: Date.now()
        };

        console.log('[ContextMenu] Context data:', contextData);

        // Update the hidden store to trigger Dash callback
        updateContextStore(contextData);
    }

    /**
     * Update the hidden dcc.Store to trigger Dash callback.
     * Uses Dash's internal mechanism for store updates.
     */
    function updateContextStore(data) {
        // Method 1: Direct store update via Dash's internal API
        if (window.dash_clientside && window.dash_clientside.set_props) {
            window.dash_clientside.set_props('ventalibre-context-click-store', { data: data });
            return;
        }

        // Method 2: Dispatch custom event for clientside callback
        const store = document.getElementById('ventalibre-context-click-store');
        if (store) {
            // Store data in dataset for clientside callback to read
            store.dataset.pendingContext = JSON.stringify(data);

            // Trigger a custom event
            store.dispatchEvent(new CustomEvent('contextclick', {
                detail: data,
                bubbles: true
            }));
        }

        // Method 3: Use window variable as fallback
        window._ventalibreContextData = data;
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Re-attach on page navigation (SPA behavior)
    window.addEventListener('popstate', function() {
        isAttached = false;
        setTimeout(init, 100);
    });

})();
