# frontend/app.py
"""
Aplicación principal de kaiFarma - Dashboard farmacéutico.
Arquitectura modular con routing, componentes reutilizables y callbacks organizados.

kaiFarma: El valor del momento exacto.
"""

import logging
import os
import sys

import dash
import dash_bootstrap_components as dbc
from callbacks import register_all_callbacks

# ===================================================================
# CONFIGURACIÓN DE LOGGING (CRÍTICO - Issue auth context sync)
# Configurar logging básico ANTES de imports que usen logger
# Sin esto, los logs de callbacks NO se imprimen
# ===================================================================
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
    handlers=[
        logging.StreamHandler(sys.stdout)  # Imprimir a stdout (visible en Docker logs)
    ],
)
logger = logging.getLogger(__name__)
logger.info("[APP_INIT] Configuración de logging completada")

# from components.system_status_banner import create_system_status_banner  # DESHABILITADO
from components.catalog_integration import (
    add_catalog_ux_components,
    register_catalog_callbacks,
)
from components.common import create_loading_spinner

# Pivot 2026: Local PIN-based lock screen (only in KAIFARMA_LOCAL mode)
IS_LOCAL_MODE = os.getenv("KAIFARMA_LOCAL", "").lower() == "true"
if IS_LOCAL_MODE:
    from components.lock_screen import create_lock_screen_modal, create_lock_screen_stores

# Imports de módulos locales
from components.navigation import create_sidebar
from constants import PUBLIC_ROUTES
from dash import ctx, dcc, html
from dotenv import load_dotenv
from utils.auth_helpers import is_user_authenticated
from utils.constants import COLORS
from utils.interval_optimizer import get_optimized_interval, should_enable_component

# Cargar variables de entorno
load_dotenv()

# Configuración de la aplicación
# En producción (Render), el backend está en el mismo servicio (RENDER_EXTERNAL_URL)
# En desarrollo local, el backend está en puerto 8000
# Prioridad: BACKEND_URL > RENDER_EXTERNAL_URL > localhost:8000
BACKEND_URL = os.getenv("BACKEND_URL", os.getenv("RENDER_EXTERNAL_URL", "http://localhost:8000"))

# Inicializar aplicación Dash con configuración completa
app = dash.Dash(
    __name__,
    external_stylesheets=[
        dbc.themes.FLATLY,  # Tema Bootstrap moderno
        "https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css",  # FontAwesome
    ],
    suppress_callback_exceptions=True,  # Necesario para routing multi-página
    title="xfarma - Dashboard Farmacéutico",
    meta_tags=[
        {"name": "viewport", "content": "width=device-width, initial-scale=1.0"},
        {"name": "description", "content": "Dashboard de análisis farmacéutico"},
        {"name": "author", "content": "xfarma"},
    ],
)

# Configurar servidor Flask subyacente
server = app.server
server.secret_key = os.getenv("SECRET_KEY", "xfarma-dev-secret-2024")

# Layout principal de la aplicación con sidebar
base_layout = html.Div(
    [
        # URL para routing
        dcc.Location(id="url", refresh=False),
        # Store global para datos compartidos
        dcc.Store(id="app-data"),
        dcc.Store(id="current-page-store"),
        dcc.Store(id="sidebar-progress-store", data={}),
        dcc.Store(id="catalog-status-store", data={}),
        # Issue #440: Add catalog-progress-store to skeleton (REGLA #0.5)
        # Required by update_navigation callback to avoid 500 error on page load
        dcc.Store(id="catalog-progress-store", data={}),
        dcc.Store(id="viewport-store", data={"width": 1440, "height": 900}),
        dcc.Store(id="sidebar-state-store", data={"open": False, "device": "desktop"}),
        # ===================================================================
        # INTERMEDIATE STORES - REGLA #11 Catalog Callbacks Decoupling
        # Stores intermedios para desacoplar lógica de negocio de outputs finales
        # Refactorización: 2025-10-20 - Issue DASH002
        # ===================================================================
        dcc.Store(id="catalog-modal-action-store", storage_type="memory", data=None),
        dcc.Store(id="catalog-toast-action-store", storage_type="memory", data=None),
        dcc.Store(id="catalog-progress-action-store", storage_type="memory", data=None),
        # Authentication stores
        # Issue #344: Changed from memory to session to prevent data loss on F5 refresh
        # PIVOT 2026: In local mode, pre-authenticate to bypass JWT checks
        # The lock screen modal handles real security via PIN
        dcc.Store(
            id="auth-state",
            storage_type="session",
            data={"authenticated": True, "user": {"role": "operativo", "is_local": True}} if IS_LOCAL_MODE
                 else {"authenticated": False, "user": None}
        ),
        dcc.Store(id="auth-redirect", data={"redirect": False, "path": "/"}),
        # Persistent token storage (uses localStorage for 'local', sessionStorage for 'session')
        dcc.Store(id="auth-tokens-store", storage_type="local", data=None),
        # Issue #344: Auth-ready signal to coordinate callback execution order
        # PIVOT 2026: In local mode, always ready (no JWT sync needed)
        dcc.Store(id="auth-ready", data=True if IS_LOCAL_MODE else False),
        # AGREGAR: Store para sincronización de auth context (resuelve Issue #97)
        # Este store sincroniza el token del navegador al contexto Python global
        dcc.Store(id="auth-context-sync", storage_type="memory", data=None),
        # OAuth state store for CSRF protection
        dcc.Store(id="oauth-state-store", storage_type="session", data=None),
        # ===================================================================
        # PIVOT 2026: Local Lock Screen Components (only in KAIFARMA_LOCAL mode)
        # PIN-based terminal lock for desktop installation
        # ===================================================================
        *(create_lock_screen_stores() if IS_LOCAL_MODE else []),
        create_lock_screen_modal() if IS_LOCAL_MODE else html.Div(style={"display": "none"}),
        # Laboratory cache store (Rate limiting prevention - Render production)
        # Cache para códigos de laboratorio (TTL: 7 días - ver laboratory_cache_store.py línea 22)
        # Datos estáticos de catálogo que NO cambian frecuentemente
        # Reduce llamadas de 100+/sesión a ~3-5/sesión (97% reducción)
        #
        # ✅ FIX Rate Limiting + Timeouts (Issue #412): storage_type="local"
        # PROBLEMA: session storage se borra al cerrar pestaña → cache vacío → 50+ API calls → rate limit → timeout 180s
        # SOLUCIÓN: local storage persiste en navegador (sobrevive a F5, cierre pestaña, cambio de worker)
        # - Datos son ESTÁTICOS de CIMA (no cambian con acciones del usuario)
        # - TTL de 7 días se encarga de invalidar si hay cambios en CIMA
        # - Thread-safe para multi-worker (estado almacenado en cliente)
        # - Límite ~5-10MB (cache de laboratorios < 10KB)
        dcc.Store(id="laboratory-cache-store", storage_type="local", data={}),
        # ===================================================================
        # PHARMACY INFO STORE (Issue #154 - Navbar Pharmacy Name)
        # Almacena información de la farmacia del usuario para mostrar en navbar
        # - Se carga después de autenticación exitosa
        # - storage_type="session" para persistir con F5 pero limpiar al logout
        # ===================================================================
        dcc.Store(id="pharmacy-info-store", storage_type="session", data=None),
        # ===================================================================
        # STORAGE USAGE STORE (Issue #420 - FREE tier UX)
        # Almacena uso de almacenamiento para mostrar indicador en sidebar
        # - Solo se llena para usuarios FREE
        # - storage_type="session" para persistir con F5 pero limpiar al logout
        # ===================================================================
        dcc.Store(id="storage-usage-store", storage_type="session", data=None),
        # ===================================================================
        # NAVBAR COMPONENTS SKELETON (Issue #154 - User Menu Dropdown)
        # Global placeholders para componentes del navbar y sidebar
        # ===================================================================
        html.Button(
            id="navbar-logout-button",
            n_clicks=0,
            style={"display": "none"}  # Oculto en skeleton, visible en navbar funcional
        ),
        # Sidebar logout button (existía en sidebar pero causaba error en páginas públicas)
        html.Button(
            id="logout-button",
            n_clicks=0,
            style={"display": "none"}  # Oculto en skeleton, visible en sidebar funcional
        ),
        # ===================================================================
        # UPLOAD PAGE COMPONENTS (Global placeholders to avoid callback errors)
        # These are hidden and overridden by /upload layout when active
        # ===================================================================
        # OPTIMIZATION: storage_type="session" para persistencia con F5 (Plan Standard)
        dcc.Store(id="upload-state", storage_type="session", data=None),
        dcc.Store(id="current-upload-id", storage_type="session", data=None),
        dcc.Store(id="upload-trigger-store", storage_type="session", data=None),  # Issue #330: Feedback inmediato
        dcc.Interval(id="upload-progress-interval", interval=2000, n_intervals=0, disabled=True),
        html.Div(id="upload-progress-container", style={"display": "none"}),
        html.Div(id="upload-history-container", style={"display": "none"}),
        html.Div(id="file-preview-container", style={"display": "none"}),
        html.Div(id="upload-immediate-feedback", style={"display": "none"}),  # Issue #330
        html.Div(id="upload-system-status", style={"display": "none"}),
        dcc.Upload(id="upload-data", style={"display": "none"}),
        # Issue #429: Accordion ERP Help placeholder
        html.Div(id="erp-help-accordion-container", style={"display": "none"}),
        # Issue #429: Interval for initial load trigger (REGLA #0.5 - debe estar en skeleton)
        dcc.Interval(id="app-load", interval=1000, n_intervals=0, max_intervals=1, disabled=True),
        # Delete upload modal components (REGLA #0.5 - skeleton placeholders)
        dbc.Modal(id="delete-upload-modal", is_open=False, style={"display": "none"}),
        dcc.Store(id="delete-upload-target", storage_type="memory", data=None),
        dcc.Store(id="delete-in-progress", storage_type="memory", data=None),  # Patrón 2-fases
        html.Div(id="delete-upload-preview", style={"display": "none"}),
        dbc.Button(id="delete-upload-cancel", style={"display": "none"}),
        dbc.Button(id="delete-upload-confirm", style={"display": "none"}),
        # Issue #476: Inventory upload skeleton placeholders
        dbc.RadioItems(id="upload-type-selector", value="sales", options=[], style={"display": "none"}),
        html.Div(id="inventory-options-container", style={"display": "none"}),
        dcc.DatePickerSingle(id="inventory-snapshot-date", style={"display": "none"}),
        # ===================================================================
        # HOMEPAGE COMPONENTS (Global placeholders - REGLA #0.5 - Issue #152)
        # Componentes del nuevo dashboard rediseñado
        # ===================================================================
        # Stores para datos del dashboard (Issue #152 BLOCKER FIX)
        # OPTIMIZATION: storage_type="session" para persistencia con F5 (Plan Standard)
        dcc.Store(id="dashboard-data-store", storage_type="session", data=None),
        dcc.Store(id="homepage-data-store", storage_type="session", data=None),
        dcc.Store(id="homepage-feature-flags-store", storage_type="session", data=None),
        dcc.Store(id="erp-sync-status-store", storage_type="memory", data={}),  # Pivot 2026: ERP sync status for banner
        dcc.Store(id="global-alerts-store", storage_type="memory", data=None),  # Pivot 2026: Aggregated alerts for ProactiveAlerts (#602)
        # Intervals para carga de datos (Issue #152 + Issue #315 FIX)
        # CRITICAL FIX: disabled=True en skeleton para prevenir ejecución en páginas no-dashboard
        # El callback enable_dashboard_intervals() en homepage.py habilita estos intervals
        # dinámicamente cuando el usuario navega a la homepage (/, /homepage, /home)
        dcc.Interval(
            id="dashboard-update-interval",
            interval=30000,  # 30 segundos
            n_intervals=0,
            disabled=True  # Deshabilitado por defecto - habilitado por enable_dashboard_intervals()
        ),
        dcc.Interval(
            id="catalog-initial-load-interval",
            interval=500,  # 500ms - carga inicial rápida (Issue: Homepage slow load)
            n_intervals=0,
            max_intervals=1,  # Solo se ejecuta una vez
            disabled=True  # Deshabilitado por defecto - habilitado por enable_dashboard_intervals()
        ),
        # Saludo y KPIs (Issue #152)
        html.Span(id="user-greeting-name", style={"display": "none"}),
        html.Div(id="kpis-row", style={"display": "none"}),
        # KPIs Year-over-Year (Issue #152 - reemplaza gráfico de tendencia)
        html.Div(id="yoy-kpis-container", style={"display": "none"}),
        # Trend chart placeholder (REGLA #0.5 - usado en homepage callbacks)
        dcc.Graph(id="sales-trend-chart", style={"display": "none"}),
        # Feed de actividad (Issue #152)
        html.Div(id="activity-feed-content", style={"display": "none"}),
        # ===================================================================
        # GENERICS PAGE COMPONENTS (Global placeholders - REGLA #0.5)
        # Fix dependency cycle Issue: context-store → partners-dropdown → ...
        # Estos componentes están ocultos y son reemplazados por /generics
        # ===================================================================
        # Stores para contexto y partners
        # CRITICAL FIX: storage_type="session" para persistencia multi-worker en Render
        # Issue: Memory stores no persisten entre workers → datos incoherentes después de Ctrl+F5
        dcc.Store(id="context-store", storage_type="session", data=None),
        dcc.Store(id="partners-selection-store", storage_type="session", data=None),
        dcc.Store(id="partners-load-result-store", storage_type="session", data=None),
        dcc.Store(id="analysis-store", storage_type="session", data=None),
        # Store para conjunto homogéneo seleccionado (Issue #346)
        dcc.Store(id="selected-homogeneous-store", storage_type="session", data=None),
        # ✅ FIX Rate Limiting (2025-12-01): Cambiado de session → local
        # Problema: storage_type="session" se perdía con F5/cambio pestaña → rate limiting
        # Solución: storage_type="local" persiste en localStorage (7 días como laboratory-cache-store)
        # Cache de códigos laboratorio (usado por callbacks con get_codes_with_cache)
        dcc.Store(id="codes-cache-store", data={}, storage_type="local"),
        # Interval para contexto (cada 15 min)
        dcc.Interval(
            id="context-refresh-interval",
            interval=15 * 60 * 1000,  # 15 minutos
            n_intervals=0,
            disabled=True,  # Deshabilitado por defecto, habilitado en /generics
        ),
        # ===================================================================
        # FASE 4: FILTER COMPONENTS PLACEHOLDERS (Issue #415)
        # Placeholders para componentes de filtros reutilizables
        # ===================================================================
        dcc.DatePickerRange(id="generics-date-range", style={"display": "none"}),
        # Store para rango de fechas de generics (como prescription)
        dcc.Store(id="generics-date-range-store", storage_type="session", data=None),
        # Slider visual de fechas para generics (sin style - RangeSlider no lo soporta)
        dcc.RangeSlider(id="generics-date-slider", min=0, max=100, value=[0, 100], className="d-none"),
        # Issue #420: FREE tier banner - ELIMINADO del skeleton
        # Los banners están dentro de cada layout (generics.py, prescription.py)
        # Tener IDs duplicados causaba que el placeholder del skeleton se mostrara
        # fuera del flujo y creara un gap entre sidebar y contenido
        dcc.Dropdown(id="generics-employee-filter", options=[], value=[], style={"display": "none"}),
        html.Button(id="generics-apply-btn", style={"display": "none"}),
        html.Div(id="generics-employee-tooltip", style={"display": "none"}),  # Tooltip PRO
        html.Span(id="generics-employee-filter-badge", style={"display": "none"}),  # Badge PRO tier
        # Issue #415: Context Treemap placeholder
        dcc.Graph(id="generics-context-treemap", style={"display": "none"}),
        # ===================================================================
        # END FILTER COMPONENTS PLACEHOLDERS
        # ===================================================================
        # Componentes de UI (placeholders ocultos)
        html.Div(id="context-total-substitutable", style={"display": "none"}),
        html.Div(id="context-total-percentage", style={"display": "none"}),
        html.Div(id="context-analyzable-amount", style={"display": "none"}),
        html.Div(id="context-analyzable-percentage", style={"display": "none"}),
        html.Div(id="context-current-partners", style={"display": "none"}),
        html.Div(id="context-current-percentage", style={"display": "none"}),
        html.Div(id="context-opportunity", style={"display": "none"}),
        html.Div(id="context-opportunity-percentage", style={"display": "none"}),
        html.Div(id="selected-partners-count", style={"display": "none"}),
        dcc.Dropdown(id="partners-dropdown", style={"display": "none"}),
        html.Div(id="partners-dropdown-loading", style={"display": "none"}),
        html.Button(id="refresh-partners-list-btn", style={"display": "none"}),
        html.Div(id="analysis-loading-container", style={"display": "none"}),
        html.Div(id="analysis-results-container", style={"display": "none"}),
        html.Div(id="analysis-kpis-container", style={"display": "none"}),
        html.Div(id="analysis-charts-container", style={"display": "none"}),
        html.Div(id="analysis-tables-container", style={"display": "none"}),
        # Stores adicionales para análisis
        # OPTIMIZATION: storage_type="session" para persistencia con F5 (Plan Standard)
        dcc.Store(id="matrix-sort-store", storage_type="session", data=None),
        # CRITICAL FIX: storage_type="session" para persistencia multi-worker en Render
        dcc.Store(id="homogeneous-expansion-state", storage_type="session", data=None),
        dcc.Store(id="temporal-drill-store", storage_type="session", data=None),
        dcc.Store(id="homogeneous-drill-store", storage_type="session", data=None),  # Issue #346: Drill-down independiente para gráfico de conjunto
        # Componentes de análisis homogéneo (placeholders)
        html.Div(id="homogeneous-matrix-container", style={"display": "none"}),
        html.Div(id="homogeneous-matrix-summary", style={"display": "none"}),
        dcc.Dropdown(id="homogeneous-search-input", options=[], value=None, style={"display": "none"}),  # ✅ Dropdown con value
        html.Button(id="expand-all-homogeneous-btn", style={"display": "none"}),
        html.Button(id="export-homogeneous-btn", style={"display": "none"}),  # Issue #484: Export button
        html.Button(id="drill-up-btn", style={"display": "none"}),
        html.Div(id="drill-level-indicator", style={"display": "none"}),
        dcc.Graph(id="homogeneous-detail-chart", style={"display": "none"}),  # Issue #346: Placeholder para drill-down
        html.Button(id="homogeneous-drill-up-btn", style={"display": "none"}),  # Issue #346: Botón drill-up para conjunto
        dcc.Download(id="download-homogeneous-csv"),
        # Componentes de drill-down de ventas (FIX: dcc.Graph con figura vacía)
        dcc.Graph(id="sales-drill-down-chart", figure={}, style={"display": "none"}),  # ✅ Graph con clickData/figure
        dcc.Graph(id="units-drill-down-chart", figure={}, style={"display": "none"}),  # ✅ Graph con clickData/figure
        # Componentes de referencias de partners
        html.Div(id="partner-references-container", style={"display": "none"}),
        html.Div(id="partner-refs-count", style={"display": "none"}),
        html.Div(id="partner-refs-sales", style={"display": "none"}),
        html.Div(id="discount-additional-savings", style={"display": "none"}),
        dcc.Input(id="partner-refs-search", style={"display": "none"}),
        # CRITICAL: Slider envuelto en Div para ocultar (dcc.Slider no acepta style)
        html.Div(
            dcc.Slider(id="partner-discount-slider", min=0, max=100, value=0),
            style={"display": "none"}
        ),
        # Issue #484: Optimal partners and penetration gauge placeholders
        # These components are only in /generics layout but callbacks fire globally
        dcc.Graph(id="partner-penetration-gauge-chart", figure={}, style={"display": "none"}),
        html.Div(id="penetration-gauge-legend", style={"display": "none"}),
        html.Div(id="optimal-partners-compact-list", style={"display": "none"}),
        dcc.Dropdown(id="optimal-partners-count-selector", value=5, style={"display": "none"}),
        # ===================================================================
        # ADMIN PAGE SKELETON (Global placeholders para callbacks multi-tab)
        # Fix timing issues en navegación /admin (Issue #330 - Render producción)
        # Estos componentes están ocultos y son reemplazados por layouts/admin.py
        # ===================================================================
        # Stores para admin
        dcc.Store(id="admin-last-poll-time", data=None),
        dcc.Store(id="dangerous-modal-state-store", data=None),
        dcc.Store(id="selected-dangerous-operation", data=None),
        dcc.Store(id="dangerous-confirmation-state-store", data=None),
        dcc.Store(id="sync-operations-toast-store", data=None),
        dcc.Store(id="scheduled-sync-toast-store", data=None),
        dcc.Store(id="conflict-resolution-toast-store", data=None),
        # Issue #448: Manual review stores
        dcc.Store(id="manual-review-current-page-store", storage_type="memory", data=1),
        dcc.Store(id="manual-review-stats-loaded-trigger", storage_type="memory", data=None),  # REGLA #11: Intermediate trigger
        # Issue #448: Manual review placeholders (REGLA #0.5 Skeleton Pattern)
        # Inputs
        dcc.Dropdown(id="manual-review-code-type-filter", style={"display": "none"}),
        dbc.Input(id="manual-review-search", style={"display": "none"}),
        dcc.Dropdown(id="manual-review-order-by", style={"display": "none"}),
        dcc.Dropdown(id="manual-review-page-size", style={"display": "none"}),
        dbc.Button(id="manual-review-prev-page", style={"display": "none"}),
        dbc.Button(id="manual-review-next-page", style={"display": "none"}),
        dbc.Input(id="manual-review-min-sales", style={"display": "none"}),
        dbc.Button(id="manual-review-export-btn", style={"display": "none"}),
        # Outputs
        html.Span(id="manual-review-products-count", style={"display": "none"}),
        html.Span(id="manual-review-sales-count", style={"display": "none"}),
        html.Span(id="manual-review-total-amount", style={"display": "none"}),
        html.Span(id="manual-review-cn-count", style={"display": "none"}),
        html.Span(id="manual-review-ean-count", style={"display": "none"}),
        html.Span(id="manual-review-internal-count", style={"display": "none"}),
        html.Div(id="manual-review-table-container", style={"display": "none"}),
        html.Span(id="manual-review-pagination-info", style={"display": "none"}),
        html.Span(id="manual-review-current-page", style={"display": "none"}),
        dcc.Download(id="manual-review-download"),
        # Intervals para admin (deshabilitados por defecto)
        dcc.Interval(id="admin-stats-interval", interval=30000, n_intervals=0, disabled=True),
        dcc.Interval(id="admin-countdown-interval", interval=1000, n_intervals=0, disabled=True),
        dcc.Interval(id="health-refresh-interval", interval=30000, n_intervals=0, disabled=True),
        # Issue #458 M6: ML Monitoring auto-refresh (60s)
        dcc.Interval(id="ml-monitoring-refresh-interval", interval=60000, n_intervals=0, disabled=True),
        # Issue #458 M6: ML Monitoring stores (skeleton - callbacks validate at startup)
        dcc.Store(id="ml-monitoring-metrics-store", storage_type="memory", data=None),
        dcc.Store(id="ml-monitoring-alerts-store", storage_type="memory", data=None),
        dcc.Store(id="ml-monitoring-tab-activated-trigger", storage_type="memory", data=None),
        dbc.Button(id="ml-monitoring-refresh-btn", style={"display": "none"}),  # Placeholder for callback Input
        # Issue #458 M6: ML Monitoring output placeholders (skeleton - callbacks validate at startup)
        html.Span(id="ml-monitoring-entropy", style={"display": "none"}),
        html.Span(id="ml-monitoring-outlier-rate", style={"display": "none"}),
        html.Span(id="ml-monitoring-confidence", style={"display": "none"}),
        html.Span(id="ml-monitoring-pending", style={"display": "none"}),
        html.Span(id="ml-monitoring-coverage", style={"display": "none"}),
        html.Span(id="ml-monitoring-coverage-classified", style={"display": "none"}),
        html.Span(id="ml-monitoring-coverage-total", style={"display": "none"}),
        html.Span(id="ml-monitoring-last-update", style={"display": "none"}),
        html.Div(id="ml-monitoring-alerts-container", style={"display": "none"}),
        dcc.Graph(id="ml-monitoring-distribution-chart", style={"display": "none"}),
        html.Div(id="ml-monitoring-high-risk-table", style={"display": "none"}),
        dcc.Graph(id="ml-monitoring-trend-chart", style={"display": "none"}),
        # Issue #465/#466: Precision and Drift KPIs
        html.Span(id="ml-monitoring-precision", style={"display": "none"}),
        dbc.Badge(id="ml-monitoring-precision-status", style={"display": "none"}),  # Badge supports color prop
        html.Span(id="ml-monitoring-drift", style={"display": "none"}),
        dbc.Badge(id="ml-monitoring-drift-status", style={"display": "none"}),  # Badge supports color prop
        # Issue #477: Duplicates VentaLibre stores (skeleton - callbacks validate at startup)
        dcc.Store(id="duplicates-page-store", storage_type="memory", data=1),
        dcc.Store(id="duplicates-merge-data-store", storage_type="memory", data=None),
        dcc.Store(id="duplicates-tab-activated-trigger", storage_type="memory", data=None),
        dbc.Button(id="duplicates-refresh-btn", style={"display": "none"}),
        dbc.Button(id="duplicates-prev-page", style={"display": "none"}),
        dbc.Button(id="duplicates-next-page", style={"display": "none"}),
        dbc.Button(id="duplicates-merge-cancel", style={"display": "none"}),
        dbc.Button(id="duplicates-merge-confirm", style={"display": "none"}),
        # Issue #477: Duplicates VentaLibre output placeholders
        html.Span(id="duplicates-pending-count", style={"display": "none"}),
        html.Span(id="duplicates-merged-count", style={"display": "none"}),
        html.Span(id="duplicates-different-count", style={"display": "none"}),
        html.Span(id="duplicates-coverage-pct", style={"display": "none"}),
        html.Div(id="duplicates-groups-container", style={"display": "none"}),
        html.Span(id="duplicates-pagination-info", style={"display": "none"}),
        html.Span(id="duplicates-current-page", style={"display": "none"}),
        html.Span(id="duplicates-last-update", style={"display": "none"}),
        dbc.Modal(id="duplicates-merge-modal", is_open=False, style={"display": "none"}),
        html.Div(id="duplicates-merge-preview", style={"display": "none"}),
        # Issue #480: Duplicates VentaLibre filter components (skeleton)
        dbc.Button(id="duplicates-apply-filters-btn", style={"display": "none"}),
        dbc.Button(id="duplicates-clear-filters-btn", style={"display": "none"}),
        dcc.Dropdown(id="duplicates-brand-filter", style={"display": "none"}),
        dbc.Select(id="duplicates-sort-by", style={"display": "none"}),
        dbc.Select(id="duplicates-sort-order", style={"display": "none"}),
        dbc.Input(id="duplicates-date-from", style={"display": "none"}),
        # Issue #475: Manual duplicate search components (skeleton)
        dbc.Input(id="duplicates-search-input", style={"display": "none"}),
        dbc.Select(id="duplicates-search-threshold", style={"display": "none"}),
        dbc.Button(id="duplicates-search-btn", style={"display": "none"}),
        html.Div(id="duplicates-search-results", style={"display": "none"}),
        # Issue #449: Classification NECESIDAD stores (skeleton - callbacks validate at startup)
        dcc.Store(id="classification-tab-activated-trigger", storage_type="memory", data=None),
        dcc.Store(id="classification-page-store", storage_type="memory", data=1),
        dcc.Store(id="classification-categories-store", storage_type="memory", data=None),
        dcc.Store(id="classification-queue-store", storage_type="memory", data=None),
        # Issue #449: Classification NECESIDAD buttons (skeleton)
        dbc.Button(id="classification-refresh-btn", style={"display": "none"}),
        dbc.Button(id="classification-prev-page", style={"display": "none"}),
        dbc.Button(id="classification-next-page", style={"display": "none"}),
        dbc.Select(id="classification-priority-filter", style={"display": "none"}),
        # Issue #449: Classification NECESIDAD output placeholders
        html.Span(id="classification-coverage-pct", style={"display": "none"}),
        html.Span(id="classification-p1-count", style={"display": "none"}),
        html.Span(id="classification-p2-count", style={"display": "none"}),
        html.Span(id="classification-p3-count", style={"display": "none"}),
        html.Span(id="classification-accuracy-pct", style={"display": "none"}),
        html.Span(id="classification-pagination-info", style={"display": "none"}),
        html.Span(id="classification-current-page", style={"display": "none"}),
        html.Span(id="classification-last-update", style={"display": "none"}),
        html.Div(id="classification-table-container", style={"display": "none"}),
        # Issue #449: Keywords management stores (skeleton - callbacks validate at startup)
        dcc.Store(id="keywords-tab-activated-trigger", storage_type="memory", data=None),
        dcc.Store(id="keywords-page-store", storage_type="memory", data=1),
        dcc.Store(id="keywords-data-store", storage_type="memory", data=None),
        dcc.Store(id="keywords-edit-id-store", storage_type="memory", data=None),
        dcc.Store(id="keywords-categories-store", storage_type="memory", data=None),
        dcc.Store(id="keywords-apply-results-store", storage_type="memory", data=None),  # Phase 3: Apply modal
        # Issue #449: Keywords management buttons (skeleton)
        dbc.Button(id="keywords-add-btn", style={"display": "none"}),
        dbc.Button(id="keywords-refresh-btn", style={"display": "none"}),
        dbc.Button(id="keywords-prev-page", style={"display": "none"}),
        dbc.Button(id="keywords-next-page", style={"display": "none"}),
        dbc.Button(id="keywords-preview-btn", style={"display": "none"}),
        dbc.Button(id="keywords-modal-cancel-btn", style={"display": "none"}),
        dbc.Button(id="keywords-modal-save-btn", style={"display": "none"}),
        dbc.Button(id="keywords-preview-modal-close", style={"display": "none"}),
        dbc.Select(id="keywords-type-filter", style={"display": "none"}),
        dbc.Select(id="keywords-category-filter", style={"display": "none"}),
        dbc.Input(id="keywords-search-input", style={"display": "none"}),
        dbc.Checklist(id="keywords-show-inactive", style={"display": "none"}),
        # Issue #449: Keywords form fields (skeleton)
        dbc.Input(id="keywords-form-keyword", style={"display": "none"}),
        dbc.Select(id="keywords-form-type", style={"display": "none"}),
        dbc.Select(id="keywords-form-category", style={"display": "none"}),
        dbc.Input(id="keywords-form-priority", style={"display": "none"}),
        dbc.Textarea(id="keywords-form-notes", style={"display": "none"}),
        # Issue #449: Keywords output placeholders (skeleton)
        html.Div(id="keywords-stats-container", style={"display": "none"}),
        html.Div(id="keywords-table-container", style={"display": "none"}),
        html.Span(id="keywords-pagination-info", style={"display": "none"}),
        html.Div(id="keywords-preview-container", style={"display": "none"}),
        html.Div(id="keywords-preview-modal-body", style={"display": "none"}),
        html.Span(id="keywords-modal-title", style={"display": "none"}),
        # Note: Modals are defined in layouts/admin.py, not skeleton (they need full structure)
        # Issue #459: Category Aliases management stores (skeleton - callbacks validate at startup)
        dcc.Store(id="aliases-tab-activated-trigger", storage_type="memory", data=None),
        dcc.Store(id="aliases-page-store", storage_type="memory", data=1),
        dcc.Store(id="aliases-data-store", storage_type="memory", data=None),
        dcc.Store(id="aliases-edit-id-store", storage_type="memory", data=None),
        # Issue #459: Aliases management buttons (skeleton)
        dbc.Button(id="aliases-add-btn", style={"display": "none"}),
        dbc.Button(id="aliases-refresh-btn", style={"display": "none"}),
        dbc.Button(id="aliases-prev-page", style={"display": "none"}),
        dbc.Button(id="aliases-next-page", style={"display": "none"}),
        dbc.Button(id="aliases-modal-cancel-btn", style={"display": "none"}),
        dbc.Button(id="aliases-modal-save-btn", style={"display": "none"}),
        dbc.Input(id="aliases-search-input", style={"display": "none"}),
        dbc.Checklist(id="aliases-show-inactive", style={"display": "none"}),
        # Issue #459: Aliases form fields (skeleton)
        dbc.Input(id="aliases-form-source", style={"display": "none"}),
        dbc.Input(id="aliases-form-target", style={"display": "none"}),
        dbc.Textarea(id="aliases-form-reason", style={"display": "none"}),
        # Issue #459: Aliases output placeholders (skeleton)
        html.Div(id="aliases-stats-container", style={"display": "none"}),
        html.Div(id="aliases-table-container", style={"display": "none"}),
        html.Span(id="aliases-pagination-info", style={"display": "none"}),
        html.Span(id="aliases-modal-title", style={"display": "none"}),
        dbc.Modal(id="aliases-modal", is_open=False),  # Issue #459: Modal placeholder
        # Issue #521: Curated Groups management stores (skeleton - callbacks validate at startup)
        dcc.Store(id="curated-groups-tab-activated-trigger", storage_type="memory", data=None),
        dcc.Store(id="curated-groups-pagination-store", storage_type="memory", data={"page": 1, "page_size": 20}),
        dcc.Store(id="curated-groups-selected-group-store", storage_type="memory", data=None),
        dcc.Store(id="curated-groups-edit-id", storage_type="memory", data=None),
        # Issue #521: Curated Groups buttons (skeleton)
        dbc.Button(id="curated-groups-add-btn", style={"display": "none"}),
        dbc.Button(id="curated-groups-refresh-btn", style={"display": "none"}),
        dbc.Button(id="curated-groups-prev-page", style={"display": "none"}),
        dbc.Button(id="curated-groups-next-page", style={"display": "none"}),
        dbc.Button(id="curated-groups-modal-cancel-btn", style={"display": "none"}),
        dbc.Button(id="curated-groups-modal-save-btn", style={"display": "none"}),
        dbc.Button(id="curated-members-modal-close-btn", style={"display": "none"}),
        dbc.Button(id="curated-members-add-btn", style={"display": "none"}),
        # Issue #521: Curated Groups form fields (skeleton)
        dbc.Input(id="curated-groups-search-input", style={"display": "none"}),
        dbc.Select(id="curated-groups-l1-filter", style={"display": "none"}),
        dbc.Select(id="curated-groups-source-filter", style={"display": "none"}),
        dbc.Checklist(id="curated-groups-show-inactive", style={"display": "none"}),
        dbc.Input(id="curated-groups-form-name", style={"display": "none"}),
        dbc.Textarea(id="curated-groups-form-description", style={"display": "none"}),
        dbc.Select(id="curated-groups-form-l1", style={"display": "none"}),
        dbc.Input(id="curated-groups-form-l2", style={"display": "none"}),
        dbc.Input(id="curated-members-form-ean", style={"display": "none"}),
        dbc.Input(id="curated-members-form-cn", style={"display": "none"}),
        dbc.Input(id="curated-members-form-name", style={"display": "none"}),
        # Issue #521: Curated Groups output placeholders (skeleton)
        html.Div(id="curated-groups-stats-container", style={"display": "none"}),
        html.Div(id="curated-groups-table-container", style={"display": "none"}),
        html.Span(id="curated-groups-pagination-info", style={"display": "none"}),
        html.Span(id="curated-groups-modal-title", style={"display": "none"}),
        html.Span(id="curated-members-modal-title", style={"display": "none"}),
        html.Div(id="curated-members-group-info", style={"display": "none"}),
        html.Div(id="curated-members-list-container", style={"display": "none"}),
        dbc.Modal(id="curated-groups-modal", is_open=False),  # Issue #521: Modal placeholder
        dbc.Modal(id="curated-members-modal", is_open=False),  # Issue #521: Members modal
        dcc.ConfirmDialog(id="curated-groups-delete-confirm"),  # Issue #521: Delete confirmation
        dcc.Store(id="curated-groups-delete-group-id", storage_type="memory", data=None),
        # Componentes de Catalog Management (catalog_management.py)
        html.Div(id="admin-cima-count", style={"display": "none"}),
        html.Div(id="admin-cima-last-sync", style={"display": "none"}),
        html.Div(id="admin-cima-status", style={"display": "none"}),
        html.Div(id="admin-nomenclator-count", style={"display": "none"}),
        html.Div(id="admin-nomenclator-last-sync", style={"display": "none"}),
        html.Div(id="admin-nomenclator-status", style={"display": "none"}),
        html.Div(id="admin-sales-enrichment-section", style={"display": "none"}),
        html.Div(id="admin-activity-cima-sync", style={"display": "none"}),
        html.Div(id="admin-activity-nomenclator-sync", style={"display": "none"}),
        html.Div(id="admin-activity-catalog-update", style={"display": "none"}),
        html.Div(id="admin-last-update-timestamp", style={"display": "none"}),
        html.Div(id="admin-sync-progress-bar-container", style={"display": "none"}),
        html.Div(id="admin-next-update-countdown", style={"display": "none"}),
        html.Div(id="admin-analysis-results", style={"display": "none"}),
        # Componentes de Admin Tabs (admin_tabs.py)
        dcc.Tabs(id="admin-tabs", value="health-dashboard", style={'display': 'none'}),
        html.Div(id="admin-tab-content", style={"display": "none"}),
        # Issue #485: Sub-tabs dentro de Tab Sistema (dbc.Tabs usa active_tab)
        dbc.Tabs(id="system-sub-tabs", active_tab="tools", style={'display': 'none'}),
        # Componentes de Sync Operations (sync_operations.py)
        html.I(id="sync-cima-icon", style={"display": "none"}),
        html.Span(id="sync-cima-text", style={"display": "none"}),
        dbc.Button(id="admin-sync-cima-button", style={"display": "none"}),
        html.I(id="sync-nomenclator-icon", style={"display": "none"}),
        html.Span(id="sync-nomenclator-text", style={"display": "none"}),
        dbc.Button(id="admin-sync-nomenclator-button", style={"display": "none"}),
        html.I(id="sync-all-icon", style={"display": "none"}),
        html.Span(id="sync-all-text", style={"display": "none"}),
        dbc.Button(id="admin-sync-all-button", style={"display": "none"}),
        html.Div(id="sync-history-display", style={"display": "none"}),
        html.Div(id="scheduled-syncs-panel", style={"display": "none"}),
        dbc.Switch(id="enable-scheduled-sync-switch", value=False, style={"display": "none"}),
        dcc.Dropdown(id="sync-schedule-dropdown", style={"display": "none"}),
        html.Div(id="sync-conflicts-resolution", style={"display": "none"}),
        dbc.Button(id="resolve-conflicts-button", style={"display": "none"}),
        # Componentes de Dangerous Tools (dangerous_tools.py)
        html.Div(id="dangerous-operation-description", style={"display": "none"}),
        html.Div(id="dangerous-operation-confirmation-instruction", style={"display": "none"}),
        dbc.Button(id="admin-delete-sales-button", style={"display": "none"}),
        dbc.Button(id="admin-clean-catalog-button", style={"display": "none"}),
        dbc.Button(id="admin-vacuum-db-button", style={"display": "none"}),
        dbc.Button(id="admin-reindex-button", style={"display": "none"}),
        dbc.Button(id="dangerous-operation-cancel", style={"display": "none"}),
        dbc.Button(id="dangerous-operation-execute", disabled=True, style={"display": "none"}),
        dbc.Input(id="dangerous-operation-confirmation", value="", style={"display": "none"}),
        html.Div(id="admin-dangerous-operations-result", style={"display": "none"}),
        dbc.Modal(id="dangerous-operations-modal", is_open=False, style={"display": "none"}),
        # Componentes de Health Monitoring (health_monitoring.py)
        html.Div(id="system-health-metrics", style={"display": "none"}),
        html.Div(id="performance-charts", style={"display": "none"}),
        html.Div(id="alert-management-panel", style={"display": "none"}),
        html.Div(id="render-specific-metrics", style={"display": "none"}),
        html.Div(id="database-health-status", style={"display": "none"}),
        dbc.Button(id="health-manual-refresh-btn", style={"display": "none"}),  # Botón refresh manual
        # Health dashboard components (detailed metrics)
        html.H3(id="health-cpu-usage", style={"display": "none"}),
        html.H3(id="health-memory-usage", style={"display": "none"}),
        html.H3(id="health-connections", style={"display": "none"}),
        html.I(id="health-backend-icon", style={"display": "none"}),
        dbc.Badge(id="health-backend-status", style={"display": "none"}),
        html.I(id="health-postgres-icon", style={"display": "none"}),
        dbc.Badge(id="health-postgres-status", style={"display": "none"}),
        html.Span(id="health-uptime", style={"display": "none"}),
        html.Span(id="health-render-service", style={"display": "none"}),
        html.Span(id="health-render-region", style={"display": "none"}),
        html.Span(id="health-last-deploy", style={"display": "none"}),
        html.Span(id="health-alerts-count", style={"display": "none"}),
        html.Div(id="health-alerts-list", style={"display": "none"}),
        # Componentes de análisis de catálogo (catalog_management.py - botones de análisis)
        dbc.Button(id="admin-coverage-report-button", style={"display": "none"}),
        dbc.Button(id="admin-duplicates-button", style={"display": "none"}),
        dbc.Button(id="admin-orphans-button", style={"display": "none"}),
        dbc.Button(id="admin-clean-orphans-button", disabled=True, style={"display": "none"}),
        # ===================================================================
        # ADMIN PANEL FASE 3 COMPONENTS (Global placeholders - REGLA #0.5)
        # Issue #348 FASE 3: User Management, Database, Tools
        # ===================================================================
        # Stores para gestión de usuarios
        dcc.Store(id='admin-users-data-store', storage_type='memory', data=None),
        dcc.Store(id='admin-selected-user-store', storage_type='memory', data=None),
        dcc.Store(id='admin-user-form-mode-store', storage_type='memory', data='create'),
        dcc.Store(id='admin-user-form-data-store', storage_type='memory', data=None),  # Issue #365: Preservar datos formulario multi-tab
        dcc.Store(id='admin-user-operation-result', storage_type='memory', data=None),
        # Tab USERS - placeholders
        html.Div(id='admin-users-table-container', style={'display': 'none'}),
        html.Div(id='admin-storage-usage-chart', style={'display': 'none'}),
        html.Div(id='admin-user-form-content', style={'display': 'none'}),
        html.Div(id='admin-user-form-errors', style={'display': 'none'}),
        html.Span(id='admin-delete-user-name', style={'display': 'none'}),
        html.Span(id='admin-restore-user-name', style={'display': 'none'}),
        html.Span(id='admin-user-modal-title', style={'display': 'none'}),
        html.I(id='admin-user-modal-icon', style={'display': 'none'}),
        html.Span(id='admin-user-modal-save-text', style={'display': 'none'}),
        html.I(id='admin-user-modal-save-icon', style={'display': 'none'}),  # Issue #417: Loading state icon
        html.I(id='admin-password-toggle-icon', style={'display': 'none'}),  # Issue #417: Password visibility toggle
        # User management buttons
        # NOTE: "nonexistent object" warnings in console are EXPECTED (REGLA #0.5 - Timing)
        # Components exist in skeleton but hidden until user navigates to /admin
        # Callbacks have proper guards (PreventUpdate + pathname check) - see Issue #330
        dbc.Button(id='admin-password-toggle-btn', style={'display': 'none'}),  # Issue #417: Password visibility toggle
        dbc.Button(id='admin-create-user-btn', style={'display': 'none'}),
        dbc.Button(id='admin-refresh-users-btn', style={'display': 'none'}),
        dbc.Select(id='admin-user-role-filter', options=[], style={'display': 'none'}),
        dbc.Select(id='admin-user-subscription-filter', options=[], style={'display': 'none'}),
        dbc.Checklist(id='admin-user-include-deleted', options=[], value=[], style={'display': 'none'}),
        # Modal controls (User modal)
        dbc.Button(id='admin-user-modal-cancel', style={'display': 'none'}),
        dbc.Button(id='admin-user-modal-save', style={'display': 'none'}),
        dbc.Tabs(id='admin-user-form-tabs', active_tab="user-tab", style={'display': 'none'}),
        # Modal controls (Delete modal)
        dbc.Button(id='admin-delete-user-modal-cancel', style={'display': 'none'}),
        dbc.Button(id='admin-delete-user-modal-confirm', style={'display': 'none'}),
        # Modal controls (Restore modal)
        dbc.Button(id='admin-restore-user-modal-cancel', style={'display': 'none'}),
        dbc.Button(id='admin-restore-user-modal-confirm', style={'display': 'none'}),
        # Pattern-matching action buttons (placeholders for skeleton - Issue #348 FASE 3.1)
        html.Button(id={"type": "edit-user-btn", "index": "skeleton"}, style={'display': 'none'}),
        html.Button(id={"type": "delete-user-btn", "index": "skeleton"}, style={'display': 'none'}),
        html.Button(id={"type": "restore-user-btn", "index": "skeleton"}, style={'display': 'none'}),
        # Tab DATABASE - placeholders (nuevos componentes)
        html.Div(id='admin-backup-list-container', style={'display': 'none'}),
        html.Div(id='admin-database-stats-container', style={'display': 'none'}),
        html.Span(id='admin-db-total-size', style={'display': 'none'}),
        html.Span(id='admin-db-total-records', style={'display': 'none'}),
        html.Span(id='admin-db-active-connections', style={'display': 'none'}),
        dbc.Button(id='admin-create-backup-btn', style={'display': 'none'}),
        # Tab TOOLS - Audit logs placeholders
        html.Div(id='admin-audit-logs-container', style={'display': 'none'}),
        dbc.Select(id='admin-audit-action-filter', style={'display': 'none'}),
        dbc.Select(id='admin-audit-limit-filter', style={'display': 'none'}),
        dbc.Button(id='admin-refresh-audit-logs-btn', style={'display': 'none'}),
        # User form inputs (Issue #348 FASE 3.1 - Form placeholders)
        # CRITICAL: Usar visibility hidden en lugar de display none para que IDs existan en DOM
        # cuando callbacks dinámicos del modal los referencian como State
        html.Div([
            dbc.Input(id='admin-user-email-input', type='email'),
            dbc.Input(id='admin-user-full-name-input', type='text'),
            dbc.Input(id='admin-user-password-input', type='password'),
            dbc.Select(id='admin-user-role-select', options=[]),
            dbc.Select(id='admin-user-subscription-select', options=[]),
            dbc.Checklist(id='admin-user-is-active-check', options=[], value=[]),
            # Pharmacy form inputs (Issue #348 FASE 3.1 - Pharmacy tab placeholders)
            dbc.Input(id='admin-pharmacy-name-input', type='text'),
            dbc.Input(id='admin-pharmacy-nif-input', type='text'),
            dbc.Input(id='admin-pharmacy-address-input', type='text'),
            dbc.Input(id='admin-pharmacy-city-input', type='text'),
            dbc.Input(id='admin-pharmacy-postal-code-input', type='text'),
            dbc.Input(id='admin-pharmacy-phone-input', type='tel'),
            dcc.Dropdown(id='admin-pharmacy-erp-input'),
        ], style={'visibility': 'hidden', 'position': 'absolute', 'top': '-9999px'}),
        # User modals (Issue #348 FASE 3.1 - Modal placeholders)
        dbc.Modal(id='admin-user-modal', is_open=False, style={'display': 'none'}),
        dbc.Modal(id='admin-delete-user-modal', is_open=False, style={'display': 'none'}),
        dbc.Modal(id='admin-restore-user-modal', is_open=False, style={'display': 'none'}),
        # Admin session/settings stores (Issue #348 FASE 3.1)
        dcc.Store(id='admin-session-store', storage_type='session', data=None),
        dcc.Store(id='admin-settings-store', storage_type='local', data=None),
        dcc.Store(id='admin-catalog-cache', storage_type='memory', data=None),
        # ===================================================================
        # PRESCRIPTION PAGE SKELETON (Issue #400 - Sprint 2)
        # Global placeholders para callbacks de análisis de prescripción
        # Estos componentes están ocultos y son reemplazados por /prescription
        # ===================================================================
        # Tabs placeholder (REGLA #0.5: Skeleton Pattern para Inputs de callbacks)
        dbc.Tabs(id='prescription-tabs', active_tab='tab-overview', style={'display': 'none'}),
        # Stores globales (datos compartidos entre callbacks)
        dcc.Store(id='prescription-overview-store', data=None, storage_type='session'),
        # REGLA #11 Fix: Stores de trigger separados para evitar Input duplicado
        dcc.Store(id='prescription-products-trigger-store', data=None, storage_type='session'),
        dcc.Store(id='prescription-evolution-trigger-store', data=None, storage_type='session'),
        # Issue #XXX: Nuevos stores para layout simplificado (tabla + evolución)
        dcc.Store(id='prescription-date-range-store', data=None, storage_type='session'),
        dcc.Store(id='prescription-products-store', data=None, storage_type='memory'),
        dcc.Store(id='prescription-temporal-drill-store', data={"level": "quarter", "path": []}, storage_type='session'),
        dcc.Store(id='prescription-atc-distribution-store', data=None, storage_type='session'),
        dcc.Store(id='prescription-waterfall-store', data=None, storage_type='session'),
        dcc.Store(id='prescription-top-contributors-store', data=None, storage_type='session'),  # Issue #436 Fase 2: Top Contributors
        dcc.Store(id='prescription-category-products-store', data={}, storage_type='memory'),  # Issue #441: Cache de productos por categoría
        dcc.Store(id='prescription-category-render-trigger', data=None, storage_type='memory'),  # REGLA #11 Fix: Trigger separado para render
        dcc.Store(id='prescription-ingredient-products-store', data={}, storage_type='memory'),  # Issue #441: Cache de productos por principio activo
        dcc.Store(id='prescription-ingredient-render-trigger', data=None, storage_type='memory'),  # REGLA #11 Fix: Trigger separado para render
        dcc.Store(id='prescription-stats-store', data=None, storage_type='memory'),  # Admin prescription management
        dcc.Store(id='prescription-tab-activated-trigger', data=None, storage_type='memory'),  # Fix REGLA #11: Trigger para prescription tab
        dcc.Store(id='prescription-reference-stats-trigger', data=None, storage_type='memory'),  # Fix REGLA #11: Trigger para reference list stats
        dcc.Store(id='manual-review-tab-activated-trigger', data=None, storage_type='memory'),  # Issue #447: Trigger para manual-review tab (manual_review stats)
        # Issue #16 Fase 2: Stores adicionales para gestión de clasificación (Skeleton Pattern)
        dcc.Store(id='prescription-unclassified-data-store', data=None, storage_type='memory'),
        dcc.Store(id='prescription-upload-result-store', data=None, storage_type='memory'),
        dcc.Store(id='prescription-batch-result-store', data=None, storage_type='memory'),
        dcc.Store(id='prescription-selected-category-store', data=None, storage_type='memory'),
        dcc.Store(id='prescription-all-products-store', data=None, storage_type='memory'),
        dcc.Store(id='prescription-scroll-trigger', data=None, storage_type='memory'),
        dcc.Download(id="prescription-download-csv"),  # Export CSV de clasificaciones
        # Issue #445: Reference List Upload (listados oficiales)
        html.Div(id='reference-list-stats-mini', style={'display': 'none'}),
        html.Div(id='reference-list-upload-status', style={'display': 'none'}),
        dcc.Upload(id='reference-list-upload-excel', style={'display': 'none'}),
        dbc.Checkbox(id='reference-list-truncate-checkbox', style={'display': 'none'}),
        dcc.Store(id='prescription-filters-store', data={
            "date_from": None,
            "date_to": None,
            "categories": [],
            "atc_level": 1
        }, storage_type='session'),  # session para mantener filtros en F5
        # HTML placeholders ocultos (evitar "nonexistent object")
        html.Div(id='prescription-evolution-container', style={'display': 'none'}),
        html.Div(id='atc-distribution-container', style={'display': 'none'}),
        html.Div(id='waterfall-analysis-container', style={'display': 'none'}),
        dcc.Graph(id='waterfall-analysis-chart', figure={}, style={'display': 'none'}),  # Skeleton for clickData Input
        # Issue #458: Product waterfall placeholders (skeleton pattern)
        html.Div(id='product-waterfall-placeholder', style={'display': 'none'}),
        html.Div(id='product-waterfall-container', style={'display': 'none'}),
        dcc.Graph(id='product-waterfall-chart', figure={}, style={'display': 'none'}),
        html.Div(id='product-waterfall-category-badge', style={'display': 'none'}),
        # Issue #458: Re-added as simple placeholder for combined callback (REGLA #0.5)
        # Empty placeholder - content replaced by callback, avoids timing issues on page navigation
        html.Div(id='top-contributors-container', style={'display': 'none'}),
        html.Div(id='atc-coverage-dynamic-text', style={'display': 'none'}),
        # Click-to-filter desde waterfall chart (skeleton placeholders)
        # Componentes definidos en layouts/prescription.py, placeholders aquí para evitar timing issues
        dcc.Store(id='contributors-category-filter-store', data=None, storage_type='memory'),
        html.Div(id='contributors-filter-row', style={'display': 'none'}),  # Contenedor del filtro
        html.Div(id='contributors-category-filter-indicator', style={'display': 'none'}),  # Alert interno
        html.Strong(id='contributors-filter-category-name', style={'display': 'none'}),  # Nombre categoría
        html.Button(id='contributors-clear-filter-btn', style={'display': 'none'}),  # Botón limpiar
        # Nuevo layout: Tabla productos + Evolución temporal
        html.Div(id='prescription-products-table-container', style={'display': 'none'}),
        dcc.Graph(id='prescription-evolution-chart', figure={}, style={'display': 'none'}),
        dbc.Badge(id='prescription-drill-level-badge', style={'display': 'none'}),
        html.Div(id='prescription-drill-breadcrumb', style={'display': 'none'}),
        dbc.Button(id='prescription-drill-up-btn', style={'display': 'none'}),
        # Ribbon Chart: Selector Top N categorías
        dbc.Select(id='ribbon-top-n', value='8', style={'display': 'none'}),
        dbc.RadioItems(id='evolution-chart-type', value='ribbon', options=[], style={'display': 'none'}),  # Legacy
        # Issue #436: Context Panel placeholders (treemap + métricas)
        dcc.Graph(id='prescription-context-treemap', figure={}, style={'display': 'none'}),
        html.Div(id='prescription-context-total-sales', style={'display': 'none'}),
        html.Div(id='prescription-context-total-percentage', style={'display': 'none'}),
        html.Div(id='prescription-context-total-units', style={'display': 'none'}),
        html.Div(id='prescription-context-units-percentage', style={'display': 'none'}),
        html.Div(id='prescription-context-percentage-value', style={'display': 'none'}),
        html.Div(id='prescription-context-atc-coverage', style={'display': 'none'}),
        # Inputs placeholders (componentes de filtros)
        # Issue #420: FREE tier banner - ELIMINADO del skeleton (ver generics-date-restriction-banner)
        dcc.Dropdown(id='prescription-categories-filter', style={'display': 'none'}),
        dcc.Dropdown(id='prescription-evolution-categories-filter', style={'display': 'none'}),  # Filtro local del gráfico de evolución
        dcc.Store(id='prescription-categories-filter-store', data=None, storage_type='memory'),  # Issue #449: Store para sincronizar filtro con accordion
        dcc.Dropdown(id='prescription-atc-codes-filter', style={'display': 'none'}),  # Issue #441
        dcc.Dropdown(id='atc-level-filter', style={'display': 'none'}),
        # Issue #441: Dos dropdowns para selección de períodos de comparación
        dcc.Dropdown(id='waterfall-period-base', options=[], style={'display': 'none'}),
        dcc.Dropdown(id='waterfall-period-comparison', options=[], style={'display': 'none'}),  # Legacy - mantener para compatibilidad
        # Nueva UX waterfall: selector tipo + auto-cálculo período
        dbc.RadioItems(id='waterfall-comparison-type', value='qoq', options=[], style={'display': 'none'}),
        html.Div(id='waterfall-current-period-display', style={'display': 'none'}),  # Issue #444
        html.Div(id='waterfall-period-comparison-display', style={'display': 'none'}),
        dcc.Store(id='waterfall-period-comparison-store', data=None, storage_type='memory'),
        # Issue #489: Stores para Tab 3 Seasonality (Tendencias y Estacionalidad)
        dcc.Store(id='seasonality-heatmap-store', data=None, storage_type='session'),
        dcc.Store(id='seasonality-monthly-index-store', data=None, storage_type='session'),
        dcc.Store(id='seasonality-kpis-store', data=None, storage_type='session'),
        # Issue #532: ATC filter store for seasonality tab
        dcc.Store(id='seasonality-atc-filter-store', data=None, storage_type='session'),
        # Issue #498: Stores para STL Decomposition y Forecast
        dcc.Store(id='seasonality-decomposition-store', data=None, storage_type='session'),
        dcc.Store(id='seasonality-forecast-store', data=None, storage_type='session'),
        # Issue #532: category-patterns-store removed - replaced by ATC filter
        # Issue #500: Store para Stock-out Risk Matrix
        dcc.Store(id='seasonality-stockout-risk-store', data=None, storage_type='session'),
        # Issue #501: Store para Anomaly Detection
        dcc.Store(id='seasonality-anomalies-store', data=None, storage_type='session'),
        # Issue #502: Stores para Export y Badges
        dcc.Store(id='seasonality-export-store', data=None, storage_type='session'),
        dcc.Store(id='seasonality-badges-store', data=None, storage_type='session'),
        # Issue #502: Export button placeholder
        html.Div(id='seasonality-export-button-container', style={'display': 'none'}),
        dcc.Download(id='seasonality-csv-download'),
        # Issue #502: Tab badge placeholders (REGLA #0.5 - skeleton pattern)
        html.Span(id='tab-seasonality-anomaly-badge', style={'display': 'none'}),
        html.Span(id='tab-seasonality-peak-badge', style={'display': 'none'}),
        # Issue #500: Dropdown días para reposición (Input de callback)
        dcc.Dropdown(id='stockout-days-restock', value=7, style={'display': 'none'}),
        # Issue #532: ATC filter dropdown placeholder (skeleton pattern)
        dcc.Dropdown(id='seasonality-atc-filter', value=None, style={'display': 'none'}),
        dbc.Button(id='prescription-apply-filters-btn', style={'display': 'none'}),
        # KPIs placeholders
        html.Div(id='prescription-total-sales-value', style={'display': 'none'}),
        html.Div(id='prescription-total-units-value', style={'display': 'none'}),
        html.Div(id='prescription-percentage-value', style={'display': 'none'}),
        html.Div(id='prescription-atc-coverage-value', style={'display': 'none'}),
        # Evolution chart type selector - MOVED to layouts/prescription.py (Issue #441)
        # Comparison period selector (YoY/MoM/QoQ)
        dbc.RadioItems(id='prescription-comparison-period', value='yoy', style={'display': 'none'}),
        # Issue #436 Fase 2: Top Contributors controls (REGLA #0.5 - skeleton pattern)
        dcc.Dropdown(id='top-contributors-limit', value=10, style={'display': 'none'}),
        dbc.RadioItems(id='top-contributors-direction', value='all', style={'display': 'none'}),
        # Issue #436: Employee filter placeholder (PRO feature)
        dcc.Dropdown(id='prescription-employee-filter', style={'display': 'none'}),
        html.Span(id='prescription-employee-filter-badge', style={'display': 'none'}),  # Badge PRO tier
        html.Div(id='prescription-employee-tooltip', style={'display': 'none'}),
        # Issue #439: Laboratory filter placeholder
        dcc.Dropdown(id='prescription-laboratories-filter', style={'display': 'none'}),
        # Issue #441: Acordeón de categorías (skeleton para pattern-matching callback)
        dbc.Accordion(id='prescription-categories-accordion', style={'display': 'none'}),
        dbc.Accordion(id='prescription-contributors-accordion', style={'display': 'none'}),  # Issue #441: Acordeón principios activos
        # DatePickerRange para prescription (patrón de generics-date-range)
        dcc.DatePickerRange(id='prescription-date-range', style={'display': 'none'}),
        # Issue #444: RangeSlider para navegación visual de fechas
        # Issue #540: updatemode='mouseup' reduce callbacks durante drag
        html.Div(dcc.RangeSlider(id='prescription-date-slider', min=0, max=1, value=[0, 1], updatemode='mouseup'), style={'display': 'none'}),
        # ===================================================================
        # CLUSTERING PAGE SKELETON (Issue #458 - UMAP 2D Visualization)
        # Global placeholders for clustering callbacks (REGLA #0.5)
        # These components are hidden and replaced by /clustering layout
        # ===================================================================
        # Stores for UMAP data and ghost points
        dcc.Store(id='umap-data-store', storage_type='session', data=None),
        dcc.Store(id='ghost-points-store', storage_type='session', data=[]),
        dcc.Store(id='selected-product-store', storage_type='memory', data=None),
        # Placeholders for UMAP UI components (inputs/outputs)
        dcc.Dropdown(id='umap-necesidad-filter', options=[], style={'display': 'none'}),
        dcc.Dropdown(id='umap-color-by', value='necesidad', style={'display': 'none'}),
        dcc.Checklist(id='umap-options', options=[], value=[], style={'display': 'none'}),
        html.Div(id='umap-scatter-container', style={'display': 'none'}),
        dcc.Graph(id='umap-scatter', figure={}, style={'display': 'none'}),
        html.Div(id='product-detail-panel', style={'display': 'none'}),
        # KPI placeholders
        html.Span(id='kpi-total-products', style={'display': 'none'}),
        html.Span(id='kpi-with-coords', style={'display': 'none'}),
        html.Span(id='kpi-coverage', style={'display': 'none'}),
        html.Span(id='kpi-verified', style={'display': 'none'}),
        html.Span(id='umap-version-badge', style={'display': 'none'}),
        # Ghost point move controls (placeholder for W15)
        dbc.Button(id='btn-move-product', style={'display': 'none'}),
        dcc.Dropdown(id='target-cluster-dropdown', options=[], style={'display': 'none'}),
        # ===================================================================
        # VENTA LIBRE PAGE SKELETON (Issue #461 - REGLA #0.5)
        # Dashboard de ventas OTC por categoría NECESIDAD
        # ===================================================================
        # Tabs placeholder (REGLA #0.5: Skeleton Pattern para Outputs de callbacks)
        # Requerido por context_navigation.py callbacks que usan Output("ventalibre-tabs", "active_tab")
        dbc.Tabs(id='ventalibre-tabs', active_tab='tab-analisis', style={'display': 'none'}),
        # Stores
        dcc.Store(id='ventalibre-data-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-products-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-selected-category-store', storage_type='memory', data=None),
        dcc.Store(id='ventalibre-kpis-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-chart-type-store', storage_type='session', data='treemap'),  # Issue #492
        # Issue #493: Stores para Tab "Categorías y Marcas"
        dcc.Store(id='ventalibre-brands-data-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-selected-necesidad-store', storage_type='memory', data=None),
        dcc.Store(id='ventalibre-brand-duel-store', storage_type='memory', data=None),
        dcc.Store(id='ventalibre-brands-chart-type-store', storage_type='session', data='treemap'),
        # Tab 3: Producto y Surtido - Stores (Issue #494)
        dcc.Store(id='ventalibre-selected-product-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-product-ficha-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-alternatives-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-complementary-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-competitive-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-recommendation-store', storage_type='session', data=None),
        # Interval para carga inicial
        dcc.Interval(id='ventalibre-load-interval', interval=500, n_intervals=0, max_intervals=1, disabled=True),
        # Toggle buttons para tipo de gráfico (Issue #492)
        dbc.Button(id='ventalibre-chart-treemap-btn', style={'display': 'none'}),
        dbc.Button(id='ventalibre-chart-bars-btn', style={'display': 'none'}),
        # Treemap container and graph
        html.Div(id='ventalibre-treemap-container', style={'display': 'none'}),
        dcc.Graph(id='ventalibre-treemap', figure={}, style={'display': 'none'}),
        # Executive summary container
        html.Div(id='ventalibre-executive-summary-container', style={'display': 'none'}),
        # Products container and components
        html.Div(id='ventalibre-products-container', style={'display': 'none'}),
        dbc.Badge(id='ventalibre-category-badge', children='Todas', style={'display': 'none'}),
        dbc.Button(id='ventalibre-clear-category', style={'display': 'none'}),  # Clear category filter
        dbc.Input(id='ventalibre-search', type='text', style={'display': 'none'}),
        # Filters (usando componentes reutilizables con prefijo ventalibre-)
        dcc.Dropdown(id='ventalibre-category-filter', options=[], style={'display': 'none'}),
        # Employee filter skeleton - callbacks reference value
        dcc.Dropdown(id='ventalibre-employee-filter', options=[], style={'display': 'none'}),
        # DatePickerRange skeleton - callbacks reference start_date/end_date
        dcc.DatePickerRange(
            id='ventalibre-date-range',
            style={'display': 'none'},
            persistence=True,
            persistence_type='session',
        ),
        # RangeSlider no soporta style directamente - envolver en Div oculto
        html.Div(
            dcc.RangeSlider(id='ventalibre-date-slider', min=0, max=100, step=1, marks={}, value=[0, 100]),
            style={'display': 'none'}
        ),
        # Paginación
        dbc.Button(id='ventalibre-prev-page', style={'display': 'none'}),
        dbc.Button(id='ventalibre-next-page', style={'display': 'none'}),
        html.Span(id='ventalibre-page-info', children='', style={'display': 'none'}),
        html.Div(id='ventalibre-pagination-info', style={'display': 'none'}),
        # Modal feedback
        dbc.Modal(id='ventalibre-feedback-modal', is_open=False),
        html.Div(id='ventalibre-feedback-content', style={'display': 'none'}),
        dcc.Store(id='ventalibre-feedback-product-store', storage_type='memory', data=None),
        dbc.Button(id='ventalibre-feedback-submit', style={'display': 'none'}),
        dbc.Button(id='ventalibre-feedback-cancel', style={'display': 'none'}),
        dcc.Dropdown(id='ventalibre-feedback-category', options=[], style={'display': 'none'}),
        dbc.Textarea(id='ventalibre-feedback-comment', style={'display': 'none'}),
        # Alerts
        html.Div(id='ventalibre-alerts-container', style={'display': 'none'}),
        # ===================================================================
        # Issue #505: L2 DRILL-DOWN SKELETON - Subcategorías L2
        # ===================================================================
        dcc.Store(id='ventalibre-l2-data-store', storage_type='memory', data=None),
        dcc.Store(id='ventalibre-selected-l1-for-l2', storage_type='memory', data=None),
        # L2 Modal skeleton
        dbc.Modal(id='ventalibre-l2-modal', is_open=False),
        html.Span(id='ventalibre-l2-modal-title', children='', style={'display': 'none'}),
        dbc.Button(id='ventalibre-l2-modal-close', style={'display': 'none'}),
        html.Div(id='ventalibre-l2-treemap-container', style={'display': 'none'}),
        dcc.Graph(id='ventalibre-l2-treemap', figure={}, style={'display': 'none'}),
        html.Div(id='ventalibre-l2-coverage-badge', style={'display': 'none'}),
        html.Div(id='ventalibre-l2-archetype-legend', style={'display': 'none'}),
        html.Div(id='ventalibre-l2-quadrant-container', style={'display': 'none'}),
        # W14a: Categorías minoritarias (treemap principal)
        dbc.Badge(id='ventalibre-other-count', children='0', style={'display': 'none'}),
        dbc.Button(id='ventalibre-toggle-other', style={'display': 'none'}),
        html.I(id='ventalibre-toggle-icon', className='fas fa-chevron-down', style={'display': 'none'}),
        dbc.Collapse(id='ventalibre-other-collapse', is_open=False),
        html.Div(id='ventalibre-other-categories-table', style={'display': 'none'}),
        html.Div(id='ventalibre-other-section', style={'display': 'none'}),
        # ===================================================================
        # Issue #506: INSIGHTS TAB SKELETON - Motor de Insights Automáticos
        # ===================================================================
        # Stores para datos de insights
        dcc.Store(id='ventalibre-insights-data-store', storage_type='session', data=None),
        dcc.Store(id='ventalibre-insights-filter-store', storage_type='memory', data=None),
        # Interval para auto-refresh (5 min)
        dcc.Interval(
            id='ventalibre-insights-refresh-interval',
            interval=5 * 60 * 1000,  # 5 minutos
            n_intervals=0,
            disabled=True,  # Se activa cuando se carga la tab
        ),
        # Refresh button skeleton
        dbc.Button(id='ventalibre-insights-refresh-btn', n_clicks=0, style={'display': 'none'}),
        # Container para insights
        html.Div(id='ventalibre-insights-container', style={'display': 'none'}),
        # Stats badges
        html.Span(id='ventalibre-insights-total-value', children='--', style={'display': 'none'}),
        html.Span(id='ventalibre-insights-count-high', children='0', style={'display': 'none'}),
        html.Span(id='ventalibre-insights-count-medium', children='0', style={'display': 'none'}),
        html.Span(id='ventalibre-insights-count-low', children='0', style={'display': 'none'}),
        html.Span(id='ventalibre-insights-suppressed-count', children='', style={'display': 'none'}),
        # Retry button (error state)
        dbc.Button(id='ventalibre-insights-retry-btn', style={'display': 'none'}),
        # ===================================================================
        # Issue #512: CONTEXT NAVIGATION SKELETON - Menú Contextual Treemap
        # ===================================================================
        # Store para datos de right-click (set by JavaScript)
        dcc.Store(id='ventalibre-context-click-store', storage_type='memory', data=None),
        # Store para contexto de navegación (preserva filtros entre tabs)
        dcc.Store(id='ventalibre-navigation-context-store', storage_type='session', data=None),
        # Interval para polling cambios de JavaScript
        dcc.Interval(
            id='ventalibre-context-interval',
            interval=250,  # 250ms polling (reduced CPU overhead)
            n_intervals=0,
            disabled=False,
        ),
        # Download component para CSV export
        dcc.Download(id='ventalibre-context-csv-download'),
        # Modal skeleton (el contenido se crea en layouts/ventalibre.py)
        # Los botones de navegación se crean en el modal
        # ===================================================================
        # Issue #493: BRANDS TAB SKELETON - Tab "Categorías y Marcas"
        # ===================================================================
        # Selector de categoría NECESIDAD
        dcc.Dropdown(id='ventalibre-brands-necesidad-dropdown', options=[], style={'display': 'none'}),
        # Toggle buttons treemap/barras para marcas
        dbc.Button(id='ventalibre-brands-treemap-btn', style={'display': 'none'}),
        dbc.Button(id='ventalibre-brands-bars-btn', style={'display': 'none'}),
        # Gráficos
        dcc.Graph(id='ventalibre-brands-chart', figure={}, style={'display': 'none'}),
        dcc.Graph(id='ventalibre-value-quadrant-chart', figure={}, style={'display': 'none'}),
        dcc.Graph(id='ventalibre-market-evolution-chart', figure={}, style={'display': 'none'}),
        dcc.Graph(id='ventalibre-price-boxplot-chart', figure={}, style={'display': 'none'}),
        # Tabla de marcas
        html.Div(id='ventalibre-brands-table-container', style={'display': 'none'}),
        # HHI indicator
        html.Div(id='ventalibre-hhi-indicator', style={'display': 'none'}),
        # HHI Matrix - Scatter plot Issue #539
        dcc.Graph(id='ventalibre-hhi-matrix-chart', figure={}, style={'display': 'none'}),
        html.Div(id='ventalibre-hhi-matrix-table', style={'display': 'none'}),
        # Brand Duel
        dbc.Button(id='ventalibre-brand-duel-toggle-btn', style={'display': 'none'}),
        dbc.Collapse(id='ventalibre-brand-duel-collapse', is_open=False),
        dcc.Dropdown(id='ventalibre-brand-duel-dropdown-a', options=[], style={'display': 'none'}),
        dcc.Dropdown(id='ventalibre-brand-duel-dropdown-b', options=[], style={'display': 'none'}),
        html.Div(id='ventalibre-brand-duel-content', style={'display': 'none'}),
        # ===================================================================
        # Issue #491: TAB 1 "ANÁLISIS" - Evolution Section placeholders
        # RadioItems as hidden skeleton (replaced by visible ones in layout)
        # ===================================================================
        dbc.RadioItems(id='ventalibre-evolution-period', value=12, options=[], className='d-none'),
        dbc.RadioItems(id='ventalibre-contributors-direction', value='all', options=[], className='d-none'),
        html.Div(id='ventalibre-evolution-container', style={'display': 'none'}),
        html.Div(id='ventalibre-yoy-container', style={'display': 'none'}),
        html.Div(id='ventalibre-contributors-container', style={'display': 'none'}),
        # ===================================================================
        # Issue #494: TAB 3 "PRODUCTO Y SURTIDO" - Visual containers
        # These are Output targets of callbacks, so skeleton is REQUIRED
        # ===================================================================
        # Product search dropdown (CRITICAL: used as Input/Output in product_analysis.py)
        dcc.Dropdown(id='ventalibre-product-search', options=[], value=None, style={'display': 'none'}),
        html.Div(id='ventalibre-product-search-status', style={'display': 'none'}),
        # Ficha and KPIs containers
        html.Div(id='ventalibre-product-ficha-container', style={'display': 'none'}),
        html.Div(id='ventalibre-product-kpis-container', style={'display': 'none'}),
        # Recommendation toggle and collapse
        dbc.Button(id='ventalibre-recommendation-toggle', style={'display': 'none'}),
        dbc.Collapse(id='ventalibre-recommendation-collapse', is_open=False),
        # Other containers
        html.Div(id='ventalibre-alternatives-container', style={'display': 'none'}),
        html.Div(id='ventalibre-complementary-container', style={'display': 'none'}),
        html.Div(id='ventalibre-competitive-container', style={'display': 'none'}),
        html.Div(id='ventalibre-decision-box-container', style={'display': 'none'}),
        dcc.Graph(id='ventalibre-context-treemap', figure={}, style={'display': 'none'}),
        dbc.Button(id='ventalibre-feedback-btn', style={'display': 'none'}),
        # Issue #511: Sistema de Doble Reporte (Dirección + Suelo)
        dcc.Download(id='ventalibre-direccion-pdf-download'),
        dcc.Download(id='ventalibre-suelo-excel-download'),
        dcc.Download(id='ventalibre-action-plan-download'),  # Issue #513: Plan de Acción Word
        dbc.Button(id='ventalibre-download-pdf-btn', style={'display': 'none'}),
        dbc.Button(id='ventalibre-download-excel-btn', style={'display': 'none'}),
        dbc.Button(id='ventalibre-download-action-plan-btn', style={'display': 'none'}),  # Issue #513
        html.Div(id='ventalibre-reports-toast'),  # Toast container for report errors
        # Issue #512: Context Menu Modal skeleton (REGLA #0.5)
        dbc.Modal(id='ventalibre-context-menu-modal', is_open=False),
        html.Span(id='ventalibre-context-menu-title', style={'display': 'none'}),
        html.Div(id='ventalibre-context-category-badge', style={'display': 'none'}),
        # ===================================================================
        # NOTA: Tab Gestión eliminado (Issue #488) - funcionalidades movidas a /admin
        # ===================================================================
        # INVENTORY TAB SKELETON (Issue #471 - REGLA #0.5)
        # Pestañas de inventario reutilizables en /prescription y /ventalibre
        # Stores para product_type (medicamento vs venta_libre)
        # ===================================================================
        dcc.Store(id='prescription-inv-product-type', data='medicamento'),
        dcc.Store(id='ventalibre-inv-product-type', data='venta_libre'),
        # ===================================================================
        # Interval para detectar cambios de viewport cada segundo
        dcc.Interval(
            id="viewport-interval",
            interval=get_optimized_interval("viewport"),
            n_intervals=0,
            disabled=not should_enable_component("viewport"),  # Deshabilitar en producción
        ),
        # Interval para actualizar progreso (optimizado para rate limit)
        dcc.Interval(
            id="progress-update-interval",
            interval=get_optimized_interval("progress_update"),
            n_intervals=0,
            disabled=True,  # Deshabilitado por defecto
        ),
        # Sidebar lateral fijo
        html.Div(id="sidebar-container"),
        # Contenedor principal con margen para el sidebar
        html.Div(
            [
                # Navbar funcional con nombre de farmacia (Issue #154)
                # Reemplaza la franja decorativa de 4px por navbar de 64px (4px + 60px)
                html.Div(id="navbar-container"),
                # Banner de estado del sistema - DESHABILITADO (causa sincronizaciones no deseadas)
                # create_system_status_banner(),
                # Contenedor principal con padding uniforme y margen superior para navbar
                html.Div(
                    [
                        # Contenido de la página
                        html.Div(
                            id="page-content",
                            children=[create_loading_spinner(size="lg", text="Cargando aplicación...")],
                            className="fade-in",
                        )
                    ],
                    style={
                        "padding": "2rem",
                        "paddingTop": "5rem",  # Espacio para navbar 64px + margen (Issue #154)
                        "minHeight": "100vh",
                    },
                ),
                # Status de conexión global
                html.Div(
                    id="backend-status",
                    className="position-fixed",
                    style={"bottom": "20px", "right": "20px", "zIndex": "1000", "maxWidth": "300px"},
                ),
            ],
            id="main-content-container",
            style={
                "marginLeft": "280px",
                "width": "calc(100% - 280px)",  # Ocupar todo el ancho menos el sidebar
                "minHeight": "100vh",
                "backgroundColor": "#f8f9fa",
                "overflowX": "hidden",  # Prevenir scroll horizontal
            },
        ),
        # Intervalos para actualizaciones automáticas (optimizado para rate limit)
        dcc.Interval(id="health-check-interval", interval=get_optimized_interval("health_check"), n_intervals=0),
        # Interval para auth guard (Issue #187)
        # Verifica errores 401 para redirección rápida a login
        # Intervalo: 5s dev, 8s prod - Balance entre UX y recursos
        dcc.Interval(id="auth-guard-interval", interval=get_optimized_interval("auth_guard"), n_intervals=0),
        # Botón para toggle del sidebar en móviles y tablets
        html.Button(
            html.I(className="fas fa-bars"),
            id="sidebar-toggle",
            className="btn btn-primary",
            style={
                "position": "fixed",
                "top": "15px",
                "left": "15px",
                "zIndex": "1050",
                "width": "50px",
                "height": "50px",
                "borderRadius": "8px",
                "boxShadow": "0 4px 15px rgba(13, 110, 253, 0.3)",
                "transition": "all 0.3s ease",
                "fontSize": "1.1rem",
                "display": "none",  # Controlado por callback
            },
        ),
        # Overlay para cerrar sidebar en móviles/tablets
        html.Div(
            id="sidebar-overlay",
            className="sidebar-overlay",
            style={
                "position": "fixed",
                "top": "0",
                "left": "0",
                "right": "0",
                "bottom": "0",
                "background": "rgba(0, 0, 0, 0.5)",
                "zIndex": "1044",
                "opacity": "0",
                "visibility": "hidden",
                "transition": "opacity 0.3s ease, visibility 0.3s ease",
                "display": "none",  # Controlado por callback
            },
        ),
    ],
    className="d-flex",
)

# Añadir los componentes UX del catálogo al layout
# CRÍTICO: Asignar layout completo ANTES de registrar callbacks
# para evitar race conditions en Render (Issue: partner-discount-slider not found)
final_layout = add_catalog_ux_components(base_layout)

# ===================================================================
# VALIDACIÓN DE LAYOUT COMPLETO (Issue: /generics ReferenceErrors)
# ===================================================================
# Validar que TODOS los IDs críticos estén presentes ANTES de asignar layout
# Esto previene race conditions donde callbacks se registran antes de que
# componentes existan en el DOM, causando ReferenceErrors en producción
#
# CRÍTICO: SIEMPRE fallar si validación falla (no graceful degradation)
# Razón: Layout incompleto causa errores user-facing silenciosos peores que crash
from utils.layout_validator import validate_layout_completeness

validate_layout_completeness(final_layout)
logger.info(f"[APP] Layout validation PASSED - all critical IDs present")

# ===================================================================
# ASIGNACIÓN DE LAYOUT (DESPUÉS de validación)
# ===================================================================
app.layout = final_layout

# ===================================================================
# REGISTRO DE CALLBACKS (DESPUÉS de asignar layout completo)
# ===================================================================
# Esto previene que Dash congele la lista de IDs antes de que todos los componentes existan
register_all_callbacks(app)
register_catalog_callbacks(app)

# Pivot 2026: Register local security callbacks (only in KAIFARMA_LOCAL mode)
if IS_LOCAL_MODE:
    from callbacks.security_local import register_security_local_callbacks
    register_security_local_callbacks(app)
    logger.info("[APP] Local security callbacks registered (PIN lock screen)")

# ===========================================
# SISTEMA RESPONSIVE CON DASH CALLBACKS
# ===========================================

# Clientside callback para detectar viewport dimensions
app.clientside_callback(
    """
    function(n_intervals) {
        return {
            'width': window.innerWidth,
            'height': window.innerHeight,
            'timestamp': Date.now()
        };
    }
    """,
    dash.Output("viewport-store", "data"),
    dash.Input("viewport-interval", "n_intervals"),  # Trigger each second
    prevent_initial_call=False,  # Required: Must capture initial viewport dimensions
)


# Callback principal responsive - maneja todos los dispositivos
@app.callback(
    [
        dash.Output("sidebar-container", "style"),
        dash.Output("sidebar-container", "className"),
        dash.Output("sidebar-toggle", "style"),
        dash.Output("sidebar-overlay", "style"),
        dash.Output("sidebar-state-store", "data"),
    ],
    [
        dash.Input("viewport-store", "data"),
        dash.Input("sidebar-toggle", "n_clicks"),
        dash.Input("sidebar-overlay", "n_clicks"),
        dash.Input("url", "pathname"),
        dash.Input("auth-state", "data"),  # FIX #3: Añadir auth-state como input
    ],
    [dash.State("sidebar-state-store", "data")],
    prevent_initial_call=False,  # Required: Must render sidebar immediately on page load
)
def handle_responsive_layout(viewport_data, toggle_clicks, overlay_clicks, pathname, auth_state, sidebar_state):
    """
    Maneja layout responsive para sidebar en todos los dispositivos.
    Oculta sidebar en páginas públicas (landing, auth) SOLO para usuarios no autenticados.

    Breakpoints:
    - Móvil: ≤767px - Sidebar overlay
    - Tablet: 768-1199px - Sidebar colapsible
    - Desktop: ≥1200px - Sidebar fijo

    Issue #150 Fix: PreventUpdate guard para evitar re-renders innecesarios en tablets.
    - Viewport interval cambiado de 1s a 5s (80% reducción de re-renders)
    - PreventUpdate cuando el device type no cambia
    - Defensive checks para sidebar_state

    Issue #222 Fix: Verificar autenticación antes de ocultar sidebar
    - Si usuario autenticado en página pública (ej: "/" después de login), NO ocultar sidebar
    - El redirect a /home manejado en routing.py mostrará el sidebar correctamente

    Issue #230 Fix: SIEMPRE ocultar sidebar en landing page ("/")
    - La landing page NUNCA debe mostrar sidebar (autenticado o no)
    - El redirect a /home se maneja en routing.py (dcc.Location refresh=True)
    - Evita flicker de sidebar durante transición post-login
    """
    # FIX #4: Hardcode especial para landing page (máxima prioridad)
    # La landing "/" es semánticamente diferente de otras rutas públicas
    # (es landing de marketing, no pantalla de auth)
    if pathname == "/":
        return (
            {"display": "none"},
            "",
            {"display": "none"},
            {"display": "none"},
            {"open": False, "device": "desktop", "public": True},
        )

    # Para OTRAS páginas públicas (/auth/login, /auth/register, etc.)
    # verificar autenticación para decidir si mostrar sidebar
    if pathname in PUBLIC_ROUTES:
        # Verificar autenticación usando helper centralizado
        authenticated = is_user_authenticated(auth_state)

        # Solo ocultar si NO está autenticado
        if not authenticated:
            return (
                {"display": "none"},
                "",
                {"display": "none"},
                {"display": "none"},
                {"open": False, "device": "desktop", "public": True},
            )
        # Si está autenticado en otras rutas públicas, continuar normalmente

    # Defensive checks: Asegurar que siempre tenemos datos válidos
    if not viewport_data:
        viewport_data = {"width": 1440, "height": 900}

    if not sidebar_state or not isinstance(sidebar_state, dict):
        sidebar_state = {"open": False, "device": "desktop"}

    width = viewport_data.get("width", 1440)

    # Determinar dispositivo
    if width <= 767:
        device = "mobile"
    elif width <= 1199:
        device = "tablet"
    else:
        device = "desktop"

    # Detectar qué input fue triggered
    if ctx.triggered:
        prop_id = ctx.triggered[0]["prop_id"]

        # Issue #150 Fix: PreventUpdate guard para optimización de performance en tablets
        # Solo re-renderizar si el device type realmente cambió (cruce de breakpoint)
        if "viewport-store" in prop_id:
            old_device = sidebar_state.get("device", "desktop")
            if old_device == device:
                # Device no cambió, evitar re-render innecesario (crítico para tablets 768px)
                raise dash.exceptions.PreventUpdate

        # Toggle del botón hamburguesa
        if "sidebar-toggle" in prop_id and toggle_clicks:
            sidebar_state["open"] = not sidebar_state.get("open", False)

        # Click en overlay para cerrar
        elif "sidebar-overlay" in prop_id and overlay_clicks:
            sidebar_state["open"] = False

    # Actualizar device en el state
    sidebar_state["device"] = device

    # ===========================================
    # DESKTOP: Sidebar fijo (≥1200px)
    # ===========================================
    if device == "desktop":
        sidebar_style = {
            "position": "fixed",
            "top": "0",
            "left": "0",
            "height": "100vh",
            "width": "280px",
            "zIndex": "1040",
            "transform": "translateX(0)",
            "transition": "transform 0.3s ease",
            "backgroundColor": "#ffffff",
            "borderRight": "1px solid #dee2e6",
            "overflowY": "auto",
        }
        sidebar_className = "sidebar d-flex flex-column"

        # Botón toggle oculto en desktop
        toggle_style = {"display": "none"}

        # Overlay oculto en desktop
        overlay_style = {"display": "none"}

        # En desktop, sidebar siempre abierto
        sidebar_state["open"] = True
        sidebar_state["public"] = False  # No es página pública

    # ===========================================
    # TABLET: Sidebar colapsible (768-1199px)
    # ===========================================
    elif device == "tablet":
        is_open = sidebar_state.get("open", False)

        sidebar_style = {
            "position": "fixed",
            "top": "0",
            "left": "0",
            "height": "100vh",
            "width": "280px",
            "zIndex": "1045",
            "transform": "translateX(0)" if is_open else "translateX(-100%)",
            "transition": "transform 0.3s ease",
            "backgroundColor": "white",
            "borderRight": "1px solid #dee2e6",
            "boxShadow": "5px 0 15px rgba(0, 0, 0, 0.2)" if is_open else "none",
            "overflowY": "auto",
        }
        # CRITICAL: Add sidebar-open class when open for CSS to show inner sidebar
        sidebar_className = "sidebar d-flex flex-column sidebar-open" if is_open else "sidebar d-flex flex-column"

        # Botón toggle visible en tablet
        toggle_style = {
            "position": "fixed",
            "top": "15px",
            "left": "15px",
            "zIndex": "1050",
            "width": "50px",
            "height": "50px",
            "borderRadius": "8px",
            "boxShadow": "0 4px 15px rgba(13, 110, 253, 0.3)",
            "transition": "all 0.3s ease",
            "fontSize": "1.1rem",
            "background": "#0d6efd",
            "color": "white",
            "border": "none",
            "display": "block",
        }

        # Overlay para cerrar (visible solo cuando sidebar está abierto)
        overlay_style = {
            "position": "fixed",
            "top": "0",
            "left": "0",
            "right": "0",
            "bottom": "0",
            "background": "rgba(0, 0, 0, 0.5)",
            "zIndex": "1044",
            "opacity": "1" if is_open else "0",
            "visibility": "visible" if is_open else "hidden",
            "transition": "opacity 0.3s ease, visibility 0.3s ease",
            "display": "block" if is_open else "none",
        }

        sidebar_state["public"] = False  # No es página pública

    # ===========================================
    # MÓVIL: Sidebar overlay (≤767px)
    # ===========================================
    else:  # mobile
        is_open = sidebar_state.get("open", False)

        sidebar_style = {
            "position": "fixed",
            "top": "0",
            "left": "0",
            "height": "100vh",
            "width": "280px",
            "zIndex": "1045",
            "transform": "translateX(0)" if is_open else "translateX(-100%)",
            "transition": "transform 0.3s ease",
            "backgroundColor": "white",
            "borderRight": "1px solid #dee2e6",
            "boxShadow": "5px 0 15px rgba(0, 0, 0, 0.2)" if is_open else "none",
            "overflowY": "auto",
        }
        # CRITICAL: Add sidebar-open class when open for CSS to show inner sidebar
        sidebar_className = "sidebar d-flex flex-column sidebar-open" if is_open else "sidebar d-flex flex-column"

        # Botón toggle visible en móvil (más pequeño)
        toggle_style = {
            "position": "fixed",
            "top": "10px",
            "left": "10px",
            "zIndex": "1050",
            "width": "45px",
            "height": "45px",
            "borderRadius": "8px",
            "boxShadow": "0 4px 15px rgba(13, 110, 253, 0.3)",
            "transition": "all 0.3s ease",
            "fontSize": "1rem",
            "background": "#0d6efd",
            "color": "white",
            "border": "none",
            "display": "block",
        }

        # Overlay para cerrar (visible solo cuando sidebar está abierto)
        overlay_style = {
            "position": "fixed",
            "top": "0",
            "left": "0",
            "right": "0",
            "bottom": "0",
            "background": "rgba(0, 0, 0, 0.5)",
            "zIndex": "1044",
            "opacity": "1" if is_open else "0",
            "visibility": "visible" if is_open else "hidden",
            "transition": "opacity 0.3s ease, visibility 0.3s ease",
            "display": "block" if is_open else "none",
        }

        sidebar_state["public"] = False  # No es página pública

    return sidebar_style, sidebar_className, toggle_style, overlay_style, sidebar_state


# Callback para renderizar navbar con nombre de farmacia (Issue #154)
@app.callback(
    dash.Output("navbar-container", "children"),
    [
        dash.Input("pharmacy-info-store", "data"),
        dash.Input("sidebar-state-store", "data"),
        dash.Input("url", "pathname"),
        dash.Input("auth-state", "data"),  # Issue #154: Pasar auth_state para user menu
    ],
    prevent_initial_call=False,  # Required: Must render navbar on initial page load
)
def render_navbar_with_pharmacy_name(pharmacy_info, sidebar_state, pathname, auth_state):
    """
    Renderiza el navbar con nombre de farmacia y user menu.

    Issue #154: Mostrar nombre de farmacia permanentemente y user menu dropdown.

    Args:
        pharmacy_info: Información de la farmacia (con pharmacy_name)
        sidebar_state: Estado del sidebar (device, public)
        pathname: Ruta actual
        auth_state: Estado de autenticación del usuario (para user menu)

    Returns:
        Navbar component con nombre de farmacia, user menu, o vacío para páginas públicas
    """
    from components.navigation import create_navbar, create_fallback_navbar
    from utils.auth_helpers import AuthStateInconsistentError, is_user_authenticated

    if not sidebar_state:
        sidebar_state = {"device": "desktop", "public": False}

    # VERIFICACIÓN 0: Landing page ("/") - NO mostrar navbar
    # La landing page tiene su propio diseño completo con login integrado
    # No necesita navbar superior (ni siquiera fallback)
    if pathname == "/":
        return html.Div()  # Navbar vacío para landing

    # VERIFICACIÓN 1: Páginas públicas explícitas (login, register, etc.)
    # CRITICAL: Devolver navbar fallback con navbar-logout-button skeleton
    # para prevenir "nonexistent object" error en callback de logout
    if sidebar_state.get("public", False):
        return create_fallback_navbar()

    # VERIFICACIÓN 2: Usuario NO autenticado (proactiva)
    # Si el usuario no está autenticado, mostrar navbar fallback
    # incluso si sidebar_state no tiene public=True
    if not is_user_authenticated(auth_state):
        logger.debug("[NAVBAR] User not authenticated - showing fallback navbar")
        return create_fallback_navbar()

    # Usuario autenticado: Renderizar navbar completo
    # Extraer nombre de farmacia del store
    pharmacy_name = None
    if pharmacy_info and isinstance(pharmacy_info, dict):
        pharmacy_name = pharmacy_info.get("pharmacy_name")

    # Renderizar navbar con nombre de farmacia, auth_state (user menu), y página actual
    try:
        return create_navbar(pharmacy_name=pharmacy_name, auth_state=auth_state, current_page=pathname)
    except AuthStateInconsistentError as e:
        # Estado inconsistente detectado - auth guard redirigirá a login
        logger.warning(f"[NAVBAR] Auth state inconsistente, mostrando fallback navbar: {e}")
        # ✅ FIX: Navbar fallback en lugar de vacío para evitar "pantalla sin navegación"
        return create_fallback_navbar()  # Navbar con botón "Iniciar Sesión"


# Callback para navegación dinámica con progreso
@app.callback(
    [dash.Output("sidebar-container", "children"), dash.Output("progress-update-interval", "disabled")],
    [
        dash.Input("url", "pathname"),
        dash.Input("sidebar-progress-store", "data"),
        dash.Input("catalog-status-store", "data"),
        dash.Input("catalog-progress-store", "data"),
        dash.Input("auth-state", "data"),
        dash.Input("storage-usage-store", "data"),  # Issue #420: FREE tier storage indicator
        dash.Input("auth-ready", "data"),  # Issue #438: Wait for auth-ready before rendering sidebar
    ],
    prevent_initial_call=False,  # Required: Must render sidebar on initial page load
)
def update_navigation(pathname, progress_data, catalog_status, catalog_progress, auth_state, storage_usage, auth_ready):
    """
    Actualizar sidebar según la página actual y estado de progreso.
    No renderiza sidebar en páginas públicas SOLO para usuarios no autenticados.

    Args:
        pathname: Ruta actual
        progress_data: Datos del progreso de actualización
        catalog_status: Estado del catálogo para el badge
        catalog_progress: Progreso específico del catálogo (si está delegado)
        auth_state: Estado de autenticación del usuario
        storage_usage: Datos de uso de almacenamiento (Issue #420)
        auth_ready: Signal indicating auth state is ready (Issue #438)

    Returns:
        Sidebar actualizado con página activa resaltada y progreso si hay
    """
    # FIX #4: Hardcode especial para landing page (máxima prioridad)
    # SIEMPRE retornar sidebar vacío en landing page ("/")
    if pathname == "/":
        return html.Div(), True  # Sidebar vacío, interval deshabilitado

    # Issue #438: Wait for auth-ready signal on protected routes
    # FIX: Also check auth_state to avoid blocking when user is already authenticated
    # This fixes sidebar not rendering after Ctrl+F5 when auth_state loads before auth_ready
    if pathname not in PUBLIC_ROUTES and not auth_ready and not is_user_authenticated(auth_state):
        logger.debug(f"[update_navigation] Waiting for auth-ready on {pathname}")
        return html.Div(), True  # Return empty sidebar while waiting for auth

    # Para OTRAS páginas públicas, verificar autenticación
    if pathname in PUBLIC_ROUTES:
        # Verificar autenticación usando helper centralizado
        authenticated = is_user_authenticated(auth_state)

        # Solo retornar sidebar vacío si NO está autenticado
        if not authenticated:
            return html.Div(), True  # Retornar div vacío y deshabilitar interval
        # Si está autenticado en otras rutas públicas, continuar normalmente

    # Mapear rutas a páginas
    page_mapping = {
        "/home": "home",
        "/dashboard": "dashboard",
        "/upload": "upload",
        "/config": "config",
        "/prescription": "prescription",
        "/generics": "generics",
        "/ventalibre": "ventalibre",
    }

    current_page = page_mapping.get(pathname, "home")

    # Si hay delegación al sistema de catálogo, usar esos datos
    if progress_data and progress_data.get("delegate_to") == "catalog-progress-store":
        # Combinar información del progreso general con el específico del catálogo
        if catalog_progress and catalog_progress.get("active"):
            progress_data = {
                "active": True,
                "task": "catalog_sync",
                "message": catalog_progress.get("message", "Actualizando catálogo..."),
                "progress": catalog_progress.get("progress", 0),
            }

    # Habilitar/deshabilitar interval según si hay tarea activa
    interval_disabled = not (progress_data and progress_data.get("active", False))

    return create_sidebar(current_page, progress_data, catalog_status, auth_state, storage_usage), interval_disabled


# Callback para ajustar el contenido principal según el dispositivo
@app.callback(
    dash.Output("main-content-container", "style"),
    [dash.Input("viewport-store", "data"), dash.Input("sidebar-state-store", "data")],
    prevent_initial_call=False,  # Required: Must adjust main content margins on initial render
)
def adjust_main_content_style(viewport_data, sidebar_state):
    """
    Ajusta el estilo del contenido principal según el dispositivo.
    En páginas públicas, ocupa todo el ancho.
    """
    if not viewport_data:
        viewport_data = {"width": 1440}

    if not sidebar_state:
        sidebar_state = {"device": "desktop"}

    # Si es página pública, ocupar todo el ancho
    if sidebar_state.get("public", False):
        return {
            "marginLeft": "0",
            "width": "100%",
            "minHeight": "100vh",
            "backgroundColor": "#ffffff",
            "overflowX": "hidden",
            "transition": "none",
        }

    width = viewport_data.get("width", 1440)
    device = sidebar_state.get("device", "desktop")

    # Desktop: margen izquierdo para sidebar fijo
    if device == "desktop":
        return {
            "marginLeft": "280px",
            "width": "calc(100% - 280px)",
            "minHeight": "100vh",
            "backgroundColor": "#f8f9fa",
            "overflowX": "hidden",
            "transition": "margin-left 0.3s ease",
        }

    # Tablet y móvil: sin margen (sidebar overlay)
    else:
        return {
            "marginLeft": "0",
            "width": "100%",
            "minHeight": "100vh",
            "backgroundColor": "#f8f9fa",
            "overflowX": "hidden",
            "transition": "margin-left 0.3s ease",
        }


# Callback para inicializar el progreso si hay actualización en curso
@app.callback(
    [
        dash.Output("sidebar-progress-store", "data", allow_duplicate=True),
        dash.Output("progress-update-interval", "disabled", allow_duplicate=True),
    ],
    dash.Input("url", "pathname"),
    prevent_initial_call="initial_duplicate",
)
def initialize_progress_on_load(pathname):
    """
    Detecta si hay una actualización en curso al cargar la página
    y configura el sidebar-progress-store correctamente
    """
    from utils.request_coordinator import get_system_status

    # NO ejecutar en páginas públicas (landing, login)
    if pathname in ["/", "/login", "/auth/login"]:
        return dash.no_update, dash.no_update

    try:
        # Usar RequestCoordinator para evitar rate limiting
        data = get_system_status()
        if data:

            # Extraer información de inicialización
            init_data = data.get("initialization", {})
            components = init_data.get("components", {})

            # Verificar nomenclátor
            nomen_status = components.get("nomenclator", {})
            if nomen_status.get("status") == "initializing" and nomen_status.get("progress", 0) < 100:
                return {
                    "active": True,
                    "task": "nomenclator_update",
                    "message": "Actualizando nomenclátor oficial...",
                }, False  # Habilitar interval

            # Verificar CIMA
            cima_status = components.get("cima", {})
            if cima_status.get("status") == "initializing" and cima_status.get("progress", 0) < 100:
                return {"active": True, "task": "cima_update", "message": "Actualizando base de datos CIMA..."}, False

            # Verificar catálogo
            catalog_status = components.get("catalog", {})
            if catalog_status.get("status") == "initializing" and catalog_status.get("progress", 0) < 100:
                return {"active": True, "task": "catalog_sync", "message": "Sincronizando catálogo completo..."}, False
    except:
        pass

    # No hay actualización en curso - devolver store vacío
    return {"active": False}, True  # Store con active=False, interval deshabilitado


# Función principal para ejecutar la aplicación
def run_app():
    """
    Ejecutar la aplicación xfarma.
    Configura el entorno y inicia el servidor Dash.
    """
    # Fix Windows console encoding for UTF-8
    import sys

    if sys.platform == "win32":
        sys.stdout.reconfigure(encoding="utf-8")

    # Configurar modo debug según entorno
    debug_mode = os.getenv("ENVIRONMENT", "development") == "development"

    # Configurar host y puerto
    host = os.getenv("FRONTEND_HOST", "0.0.0.0")
    port = int(os.getenv("FRONTEND_PORT", 8050))

    print("Iniciando xfarma frontend...")
    print(f"   Dashboard: http://{host if host != '0.0.0.0' else 'localhost'}:{port}")
    print(f"   Backend: {BACKEND_URL}")
    print(f"   Modo: {'Desarrollo' if debug_mode else 'Producción'}")
    print("   Responsive: Habilitado")

    # Ejecutar servidor
    app.run(debug=debug_mode, host=host, port=port, dev_tools_hot_reload=debug_mode, dev_tools_ui=debug_mode)


# Punto de entrada
if __name__ == "__main__":
    run_app()
