# Frontend - CLAUDE.md

## Contexto del Frontend

Aplicación Dash que proporciona la interfaz para kaiFarma.

**Stack**: Dash 3.3.0 + Bootstrap + Plotly
**Arquitectura**: Single-worker local (Windows) + opcional Hub web
**Auth**: License Key + PIN local (no OAuth)

### Pivot 2026
- **Local**: UI corre en `localhost:8050`, acceso vía PIN
- **Datos**: SQLite local (offline-first), snapshots desde Hub
- **Sin multi-worker**: Instalación local usa 1 worker → patrones simplificados

## ⚡ Quick Commands

| Acción | Comando |
|--------|---------|
| **Validar callbacks** | `python frontend/utils/validate_callbacks.py` |
| **Validar componentes (railguards)** | `python frontend/utils/dash_component_validator.py --all` |
| **Instalar git hooks** | `.\scripts\install-git-hooks.ps1` |
| **Tests E2E** | `.\scripts\test-e2e.ps1` |
| **Tests E2E debug** | `.\scripts\test-e2e.ps1 -Debug` |
| **Tests E2E rápidos** | `.\scripts\test-e2e.ps1 -Fast` |
| **Buscar ID componente** | `grep -r "mi-component-id" frontend/` |
| **Ver logs frontend** | `.\scripts\logs.ps1 frontend` |
| **Iniciar frontend** | `cd frontend && python app.py` |

## 🏗️ Estructura de Directorios

```
frontend/
├── app.py                 # Entry point (skeleton global)
├── components/            # Componentes UI reutilizables (23+)
├── layouts/              # Páginas principales (dashboard, upload, auth, etc.)
├── callbacks/            # Lógica interactiva Dash (12 archivos)
├── utils/                # Helpers (auth, formateo, validadores)
├── assets/              # CSS, JS, imágenes estáticas
├── docs/                # Documentación específica frontend
│   ├── DASH_DETAILED_PATTERNS.md    # Patrones avanzados completos
│   ├── DASH_ANTIPATTERNS.md         # Antipatrones + troubleshooting
│   ├── CALLBACK_PATTERNS.md         # Debugging de callbacks
│   └── API_ENDPOINTS.md             # 101 endpoints backend
└── tests/
    ├── e2e/             # Tests E2E con Playwright
    └── unit/            # Tests unitarios frontend
```

## 🚨 REGLAS CRÍTICAS DASH

### OBLIGATORIO: Leer Primero
**ANTES** de trabajar en frontend, leer `frontend/docs/DASH_FRAMEWORK_GUIDE.md` para entender diferencias con desarrollo web tradicional.

---

### REGLA #0: Importaciones Absolutas (CRÍTICO)
```python
# ✅ SIEMPRE usar importaciones absolutas desde root
from utils.auth import auth_manager
from components.navigation import create_navbar
from callbacks.dashboard import register_dashboard_callbacks

# ❌ NUNCA usar importaciones relativas con ..
# from ..utils.auth import auth_manager  # ERROR en Docker

# Razón: app.py ejecuta desde /app en Docker
# Python añade /app al sys.path
```

**Excepción**: Importaciones relativas SOLO entre archivos del mismo paquete directo:
```python
# En utils/auth.py:
from .config import BACKEND_URL  # OK (mismo paquete utils/)
```

---

### REGLA #0.5: Skeleton Pattern (Multi-Página) ⭐ CRÍTICO

**Problema**: Apps multi-página generan errores "nonexistent object" porque callbacks se validan antes de renderizar páginas.

**Solución**: Global Skeleton Pattern en `app.py`

#### Arquitectura Obligatoria
```
app.py (Layout Global)
├── Componentes comunes (navbar, sidebar)
├── dcc.Location(id='url')
├── html.Div(id='page-content')
└── SKELETON PLACEHOLDERS (TODOS los Input/Output de callbacks)
    ├── dcc.Store globales (auth-state, upload-state, etc.)
    ├── dcc.Interval deshabilitados (disabled=True)
    └── html.Div ocultos (style={'display': 'none'})
```

#### Reglas Críticas
| Componente | Global (app.py) | Página (layouts/*.py) |
|------------|-----------------|----------------------|
| **dcc.Store** | ✅ ÚNICO | ❌ NO duplicar |
| **dcc.Interval** | ✅ ÚNICO (disabled=True) | ❌ NO duplicar |
| **html.Div** | ✅ Placeholder oculto | ✅ Funcional visible (mismo ID) |
| **dcc.Upload** | ✅ Placeholder oculto | ✅ Funcional visible (mismo ID) |

**Ver patrón completo**: `frontend/docs/DASH_DETAILED_PATTERNS.md#skeleton-pattern`

---

### Timing Issues (Issue #330)

**Error común**: "A nonexistent object was used in an Output"

**Causa**: Routing manual + validación de callbacks antes de renderizar páginas

**Mitigación implementada**:
- ✅ `suppress_callback_exceptions=True` en app.py
- ✅ Guards con `ctx.triggered` y validación de pathname
- ✅ Skeleton pattern (REGLA #0.5)
- ✅ `PreventUpdate` en callbacks

**Estrategias Anti-Timing**:

| Estrategia | Estado | Efectividad |
|------------|--------|-------------|
| **PreventUpdate** | ✅ Implementado | Excelente |
| **dcc.Store Global** | ✅ Implementado | Excelente |
| **Skeleton Pattern** | ✅ Implementado | Buena |
| **dcc.Loading** | ⚠️ Parcial | Buena |

**Patrón Guard**:
```python
@app.callback(...)
def my_callback(...):
    if not ctx.triggered:
        raise PreventUpdate

    trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

    if trigger_id == "url" and pathname != "/my-page":
        return cleanup_outputs()

    from utils.auth_helpers import is_user_authenticated
    if not is_user_authenticated(auth_state):
        raise PreventUpdate

    return process_data()
```

**Ver timing completo**: `frontend/docs/DASH_DETAILED_PATTERNS.md#timing-issues`

---

### Railguards Dash (DASH001-006) 🛡️

Sistema de 3 capas de protección contra antipatrones:

#### 1️⃣ Validador Estático
```bash
# Validar archivo específico
python frontend/utils/dash_component_validator.py frontend/components/mi_componente.py

# Validar todo el frontend
python frontend/utils/dash_component_validator.py --all

# Modo strict (salir con código 1 si hay errores - CI/CD)
python frontend/utils/dash_component_validator.py --all --strict
```

#### Antipatrones Detectados

| Código | Problema | Detección |
|--------|----------|-----------|
| **DASH001** | Listas sin wrapper | ✅ AST |
| **DASH002** | Outputs duplicados (REGLA #11) | ✅ AST |
| **DASH003** | `dash.callback_context` deprecado | ✅ AST |
| **DASH004** | Imports circulares | ✅ AST |
| **DASH005** | Atributos HTML en dbc.* | ✅ AST |
| **DASH006** | `prevent_initial_call=False` + API sin auth | ✅ AST |
| **DASH007** | `auth_manager.restore` sin `auth_headers=` explícito | 🔍 Manual |

**Ejemplo DASH006 (Timeouts 180s)**:
```python
# ❌ INCORRECTO: Timeout en sesiones no autenticadas
@app.callback(
    Output("result", "children"),
    Input("data-store", "data"),
    prevent_initial_call=False  # Se ejecuta en page load
)
def callback(data):
    # Sin verificación auth → API call → 401 → retries → timeout
    response = request_coordinator.make_request(...)
    return ...

# ✅ CORRECTO: Verificación proactiva
@app.callback(
    Output("result", "children"),
    Input("data-store", "data"),
    State("auth-state", "data"),  # Agregar State
    prevent_initial_call=False
)
def callback(data, auth_state):
    from utils.auth_helpers import is_user_authenticated

    if not is_user_authenticated(auth_state):
        raise PreventUpdate  # NO hacer API calls

    response = request_coordinator.make_request(...)
    return ...
```

#### 2️⃣ Pre-commit Hook Automático
```bash
# Instalar hooks (una vez)
.\scripts\install-git-hooks.ps1

# A partir de ahí, git commit valida automáticamente
git commit -m "feat: nuevo componente"
# → Validador se ejecuta automáticamente
```

#### 3️⃣ Validación en CI/CD (Futuro)
GitHub Actions ejecutará validador en cada PR.

**Ver antipatrones completos**: `frontend/docs/DASH_ANTIPATTERNS.md`

---

### Auth Management (CRÍTICO)

#### Token Management Pattern

**REGLA**: NUNCA leer tokens directamente de `auth-tokens-store`

```python
# ❌ ANTI-PATRÓN (causa auth failures)
@app.callback(
    ...,
    State("auth-tokens-store", "data")
)
def my_callback(tokens_data):
    # FALLA: tokens_data contiene tokens ENCRIPTADOS
    if tokens_data and "access_token" in tokens_data:  # Nunca True
        token = tokens_data["access_token"]  # Clave no existe

# ✅ CORRECTO: Usar auth_manager
from utils.auth import auth_manager

@app.callback(...)
def my_callback():
    access_token = auth_manager.get_access_token()  # Ya desencriptado
    if access_token:
        # Validación exitosa
        pass
```

**Orden obligatorio de registro**:
```python
# frontend/callbacks/__init__.py
def register_all_callbacks(app):
    register_auth_context_callbacks(app)  # 1️⃣ PRIMERO: Desencripta
    register_auth_guard_callbacks(app)     # 2️⃣ SEGUNDO: Guarda
    register_auth_callbacks(app)           # 3️⃣ TERCERO: General
    # ... resto
```

#### Auth Guard con Token Refresh (Issue #187)

**Características**:
- ✅ Renovación proactiva ANTES de expirar (< 3 min restantes)
- ✅ Sesiones largas (hasta 7 días sin relogin)
- ✅ Emergency refresh si token expira entre intervalos
- ✅ Logout solo si refresh_token expiró (7 días)

**Timing**:
- Access token: 15 min
- Refresh token: 7 días
- Renovación proactiva: A los 12 min
- Auth guard interval: Cada 5-8s
- Oportunidades de renovación: ~36 intentos antes de expiración

#### Verificación Auth Proactiva (Issue #302)

**Patrón obligatorio para callbacks con intervals + API calls**:
```python
@callback(
    Output(...),
    Input("some-interval", "n_intervals"),
    State("auth-state", "data"),  # ✅ OBLIGATORIO
    prevent_initial_call=True
)
def my_callback(n_intervals, auth_state):  # ✅ OBLIGATORIO
    from utils.auth_helpers import is_user_authenticated

    if not is_user_authenticated(auth_state):
        logger.debug("[CALLBACK] User not authenticated - skipping API calls")
        raise PreventUpdate  # CRÍTICO

    response = request_coordinator.make_request(...)
    return ...
```

**Cuándo aplicar**:
- ✅ Callbacks con `dcc.Interval` (polling)
- ✅ Callbacks con `Input("url", "pathname")` (navegación)
- ✅ Cualquier callback que haga API calls protegidas

#### Autenticación Local (REGLA #7.6)

**Arquitectura Híbrida (Pivot 2026)**: En instalación local single-worker, los problemas de multi-worker ya no aplican. Sin embargo, mantenemos buenas prácticas para compatibilidad con Hub.

**Modelo de Auth Local**:
- License Key + PIN (no OAuth)
- Validación local instantánea
- Heartbeat al Hub cada 24h (si online)

```python
# ✅ PATRÓN LOCAL: Autenticación simplificada
from utils.auth_helpers import is_pin_validated, get_license_status

@app.callback(
    Output(...),
    Input(...),
    State("local-auth-state", "data"),
)
def my_callback(..., auth_state):
    # 1. Verificar PIN validado (local, instantáneo)
    if not is_pin_validated(auth_state):
        raise PreventUpdate

    # 2. Verificar licencia (cache local, no requiere internet)
    license_status = get_license_status()
    if license_status.get("expired"):
        return show_license_expired_message()

    # 3. Procesar normalmente
    return process_data()
```

**Nota para Hub**: Si desarrollas para el Hub (multi-worker), sigue usando `auth_headers` explícitos:
```python
# Para callbacks que llaman al Hub API (si aplica)
auth_headers = get_auth_headers_from_tokens(auth_tokens)
response = request_coordinator.make_request(
    "/api/v1/endpoint",
    auth_headers=auth_headers,
)
```

**Reglas**:
1. En local: PIN validation es suficiente
2. Para Hub API: Usar auth_headers explícitos
3. License status se cachea localmente (no requiere internet constante)

**Ver auth completo**: `frontend/docs/DASH_DETAILED_PATTERNS.md#auth-patterns`

---

### Callback Patterns

**Ver documentación completa**: `frontend/docs/CALLBACK_PATTERNS.md`

**Validación obligatoria**:
```bash
python frontend/utils/validate_callbacks.py
```

**Reglas básicas**:
- Un Input → Un Callback (REGLA #11 del CLAUDE.md principal)
- States ilimitados (no disparan callbacks)
- Guards con `ctx.triggered` y `PreventUpdate`
- Verificación auth proactiva para API calls

**Dos patrones de decorador coexisten**:

| Patrón | Decorador | Cuándo usar |
|--------|-----------|-------------|
| **Standard (preferido)** | `@app.callback` dentro de `register_*_callbacks(app)` | Nuevos callbacks |
| **Module-level (legacy)** | `@callback` standalone + `register_*_callbacks(app)` con `pass` | Archivos existentes: `catalog_*.py`, `generics/optimal_partners.py`, `generics/homogeneous_detail.py` |

```python
# ✅ STANDARD (preferido para nuevos archivos)
def register_foo_callbacks(app):
    @app.callback(Output(...), Input(...))
    def my_callback(...):
        ...

# ⚠️ MODULE-LEVEL (legacy, no modificar patrón en archivos existentes)
def register_foo_callbacks(app):
    pass  # Callbacks se registran abajo al importar

@callback(Output(...), Input(...))  # Sin app.
def my_callback(...):
    ...
```

---

### Feedback Visual (OBLIGATORIO)

**Patrones centralizados**:

```python
# ✅ TOASTS: Usar toast_manager
from components.toast_manager import (
    success_toast,
    error_toast,
    warning_toast,
    info_toast
)

# ✅ SKELETON LOADERS: Importar de skeleton_loader.py
from components.skeleton_loader import (
    create_skeleton_loader,
    create_card_skeleton,
    create_table_skeleton,
    create_chart_skeleton
)

# ✅ ERRORES: Usar admin_feedback_components.py
from components.admin_feedback_components import (
    create_friendly_error_alert,
    create_loading_button
)

# ❌ NO HACER: Toast directo o duplicar skeleton loaders
```

**Checklist**:
- ✅ Toasts: `toast_manager`
- ✅ Skeleton Loaders: `skeleton_loader.py` (NO duplicar)
- ✅ Errores: `create_friendly_error_alert()`
- ✅ Loading States: `create_loading_button()`

---

### REGLA #19: Consumo de Measures en Frontend

**Obligatorio** para dashboards analytics. Consumir medidas del backend en lugar de calcular inline.

```python
# ❌ INCORRECTO: Calcular KPIs en callback
@callback(Output("kpi-total", "children"), Input("filters-store", "data"))
def update_kpi(filters):
    df = load_sales_data()
    total = df['pvp'].sum() * df['quantity'].sum()  # Lógica duplicada!
    return f"€{total:,.2f}"

# ✅ CORRECTO: Consumir endpoint de measures
@callback(Output("kpi-total", "children"), Input("filters-store", "data"))
def update_kpi(filters):
    response = request_coordinator.make_request(
        "GET",
        f"{BACKEND_URL}/api/v1/measures/total-sales",
        params=filters
    )
    return f"€{response['value']:,.2f}"
```

**Endpoints disponibles**:
- `GET /api/v1/measures/` - Lista de medidas disponibles
- `GET /api/v1/measures/{measure_id}` - Calcular una medida con filtros
- `GET /api/v1/generic-analysis/` - Análisis agregado

**FilterContext** (parámetros comunes):
```python
params = {
    "pharmacy_id": str(pharmacy_id),
    "date_from": "2025-01-01",
    "date_to": "2025-12-31",
    "laboratory_ids": ["lab1", "lab2"],  # Opcional
    "product_type": "medicamento"         # Opcional
}
```

**Ver**: `CLAUDE.md` REGLA #19, ADR-003

---

## 🎨 Patrones de Diseño

### Layout Responsive
```python
# ✅ SIEMPRE usar columnas responsive
dbc.Row([
    dbc.Col([contenido_principal], width=12, lg=8),  # Mobile full, desktop 2/3
    dbc.Col([sidebar_content], width=12, lg=4)       # Mobile full, desktop 1/3
])

# ❌ NO usar anchos fijos
# dbc.Col([...], width=6)  # MAL - no responsive
```

### Formateo de Números (OBLIGATORIO)
```python
# ✅ SIEMPRE usar helpers centralizados
from frontend.utils.helpers import (
    format_currency,      # €1.234,56
    format_number,        # 1.234
    format_percentage,    # 12,34%
    format_compact_number # 1,2K
)

# ❌ NUNCA formatear manualmente
# f"{value:,.2f}"  # MAL - no sigue convenciones españolas
```

### Llamadas a APIs
```python
# ✅ Patrón estándar CON AUTENTICACIÓN JWT
import requests
from frontend.utils.config import BACKEND_URL

def fetch_sales_data(pharmacy_id, date_from, date_to):
    try:
        token = auth_manager.get_access_token()  # JWT obligatorio

        response = requests.get(
            f"{BACKEND_URL}/api/v1/sales/enriched",
            params={...},
            headers={"Authorization": f"Bearer {token}"},
            timeout=30
        )
        response.raise_for_status()
        return response.json()
    except requests.RequestException as e:
        logger.error("api_call_failed", endpoint="sales/enriched", error=str(e))
        return {"data": [], "error": str(e)}
```

---

## 🎯 Verificación Visual Obligatoria

**Protocolo POST-CAMBIO**:
1. ✅ Verificar datos reales (no "--")
2. ✅ Probar callbacks principales
3. ✅ Verificar responsive en tablet (768px) - Móvil no soportado
4. ✅ Capturar screenshot para evidencia
5. ✅ Solo entonces marcar como completado

**Browser Testing**:
```python
mcp__playwright__browser_navigate("http://localhost:8050/dashboard")
mcp__playwright__browser_snapshot()  # DOM structure
mcp__playwright__browser_resize(768, 1024)  # Tablet test (mínimo soportado)
mcp__playwright__browser_take_screenshot()  # Evidence
```

---

## 📊 Visualizaciones con Plotly

**Configuración estándar**:
```python
import plotly.express as px

fig = px.bar(data, x='categoria', y='ventas', title='Ventas por Categoría')

# ✅ Personalización español
fig.update_layout(
    title={'text': 'Ventas por Categoría', 'x': 0.5, 'font': {'size': 18}},
    xaxis_title='Categoría',
    yaxis_title='Ventas (€)',
    height=400,
    margin=dict(t=60, b=60, l=60, r=60)
)

# ✅ Formato español
fig.update_traces(texttemplate='€%{y:,.0f}', textposition='outside')

# ✅ Responsive
dcc.Graph(
    figure=fig,
    config={'responsive': True, 'displayModeBar': False},
    style={'height': '400px'}
)
```

---

## 🚫 Antipatrones Frontend

### ❌ NO HACER
```python
# No usar CSS externo para elementos críticos
# No asumir que los datos están disponibles
# No usar IDs hardcodeados sin verificar
# No formatear números manualmente
value_text = f"{value:,.2f}€"  # MAL - no español

# No leer auth-tokens-store directamente
State("auth-tokens-store", "data")  # MAL - tokens encriptados
```

### ✅ HACER EN SU LUGAR
```python
# Usar estilos inline para elementos críticos
style={'display': 'block' if condition else 'none'}

# Verificar datos antes de mostrar
if data and len(data) > 0:
    return create_chart(data)
return html.P("No hay datos disponibles")

# Validar IDs antes de usar
# python frontend/utils/validate_callbacks.py

# Usar helpers centralizados
value_text = format_currency(value)

# Usar auth_manager para tokens
access_token = auth_manager.get_access_token()
```

---

## 🧪 Testing Frontend

### Tests E2E con Playwright
**Ver guía completa**: `frontend/tests/e2e/README.md`

```powershell
# Ejecutar tests E2E (navegador visible)
.\scripts\test-e2e.ps1

# Modo debug (navegador + acciones lentas + tracing)
.\scripts\test-e2e.ps1 -Debug

# Tests rápidos (solo smoke)
.\scripts\test-e2e.ps1 -Fast

# Browser específico
.\scripts\test-e2e.ps1 -Browser firefox
```

**Tests disponibles**:
- `test_dashboard.py`: Carga, navegación, datos
- `test_upload.py`: Flujo de subida de archivos
- `test_workflows.py`: Workflows farmacéuticos completos

**Resultados**:
- Screenshots: `frontend/test-results/`
- Videos (fallos): `frontend/test-results/videos/`
- Traces (debug): `frontend/test-results/traces/`

---

## 📁 Key Documentation

### Documentación Local (Organizada por Tipo)

**Dash Framework**:
- `frontend/docs/DASH_FRAMEWORK_GUIDE.md` - Guía completa framework Dash
- `frontend/docs/DASH_DETAILED_PATTERNS.md` - Patrones avanzados + Auth ⭐ NUEVO
- `frontend/docs/DASH_ANTIPATTERNS.md` - DASH001-006 + Troubleshooting ⭐ NUEVO
- `frontend/docs/CALLBACK_PATTERNS.md` - Debugging de callbacks

**APIs y Backend**:
- `frontend/docs/API_ENDPOINTS.md` - 156 endpoints backend
- `docs/DATA_CATALOG.md` - Database schema (20 models)
- `docs/SCALING_PATTERNS.md` - Cache local, sync Hub

**Testing**:
- `frontend/tests/e2e/README.md` - Guía E2E Playwright
- `backend/tests/README.md` - Tests unitarios

**Utilidades**:
- `frontend/utils/helpers.py` - Funciones formateo
- `frontend/utils/validate_callbacks.py` - Validación callbacks
- `frontend/utils/dash_component_validator.py` - Railguards

### Recursos Externos
- [Dash Bootstrap Components](https://dash-bootstrap-components.opensource.faculty.ai/)
- [Plotly Python Documentation](https://plotly.com/python/)
- [Bootstrap Classes Reference](https://getbootstrap.com/docs/5.1/utilities/)

---

## 📊 Project Stats Frontend

**Arquitectura**:
- **Componentes**: 47 (admin, catalog, system, UI, filters, prescription, generics)
- **Layouts**: 12 páginas (dashboard, upload, auth, generics, admin, settings, prescription, landing, homepage, admin_prescription_tab, admin_prescription_panel, config)
- **Callbacks**: 37 archivos (incluye módulos prescription/, generics/, admin/)
- **Tests E2E**: 3 suites principales (Playwright)

**Coverage**:
- **Tests E2E**: Dashboard, upload, workflows farmacéuticos
- **Responsive**: Desktop (1440px), Tablet (768px). Móvil (<768px) no soportado.
- **Performance**: < 10s carga inicial
- **Accesibilidad**: Tests automáticos con Playwright

**Callback Modules (2026-01)**:
- `callbacks/prescription/` - 6 módulos: contributors, data_loading, evolution, filters, overview, products
- `callbacks/generics/` - 8 módulos: analysis, context_display, data_loading, homogeneous, homogeneous_detail, optimal_partners, partners
- `callbacks/admin/` - 14 módulos: admin_tabs, catalog_management, curated_groups, keywords, aliases, etc.
- `callbacks/ventalibre/` - 8 módulos (ADR-004): data_loading, treemap, l2_drilldown, brands, gestion, etc.

**Componentes Nuevos (2026-01 - ADR-004)**:
- `ventalibre/necesidad_treemap.py` - Treemap L1 NECESIDAD con drill-down
- `ventalibre/l2_drilldown.py` - Modal de subcategorías L2
- `ventalibre/categories.py` - Constantes L1/L2 (21 L1, 28 L2)
- `ventalibre/productos_table.py` - Tabla productos con columna L2
- `ventalibre/brand_analysis.py` - Análisis por marca detectada
- `clustering/` - *(deprecated ADR-004)* - UMAP scatter, controles

**Componentes (2025-12)**:
- `employee_filter.py` - Filtro empleados con gating PRO (Issue #402)
- `waterfall_analysis.py` - Análisis YoY/MoM/QoQ (Issue #458)
- `atc_distribution_treemap.py` - Treemap distribución ATC
- `feature_flags.py` - Gestión feature flags
- `system_status_unified.py` - Estado sistema unificado

**Componentes (2025-10)**:
- `admin_feedback_components.py` - Feedback acciones admin
- `system_health_dashboard.py` - Dashboard salud sistema
- `catalog_control.py` - Controles manuales catálogo
- `skeleton_loader.py` - Skeleton loaders centralizados
- `toast_manager.py` - Sistema toasts centralizado

---

> **Frontend CLAUDE.md** - Dash application guide for kaiFarma
> Last updated: 2026-01-08 (Pivot 2026: Arquitectura Híbrida, auth local simplificado)
> Documentación detallada: `frontend/docs/DASH_DETAILED_PATTERNS.md` + `DASH_ANTIPATTERNS.md`
